/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * UDP protocol specific code
 */

#include <general.h>
#include <protospec.h>


/*  */

udp_conn_request(addr, a_ch, errind)
    struct address_t *addr;
    channel_t *a_ch;
    error_t   *errind;
{
    int s;
    struct sockaddr_in myaddr;
    
    pprintf("udp_conn_request(0x%x, 0x%x, 0x%x)\n", addr, a_ch, errind);
    
    if ((s = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) == NOTOK) {
	eprintf(EF_SYSCALL,  COMMUNICATION, "socket call", "udp_conn_request",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    myaddr.sin_addr.s_addr = INADDR_ANY;
    myaddr.sin_port = 0;
    myaddr.sin_family = AF_INET;

    /* BUG FIX */
    bzero(myaddr.sin_zero, sizeof (myaddr.sin_zero));

    if (bind(s, &myaddr, sizeof(myaddr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "bind call", "udp_conn_request",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    pprintf("udp_conn_request: to %d:0x%x:%d\n", addr->sin_family,
	    addr->sin_addr.s_addr, addr->sin_port);

    a_ch->sock = s;
    a_ch->addr = *addr; /* struct copy */
    a_ch->send_seq = a_ch->recv_seq = 0;

    *errind = OK;
} /* udp_conn_request */

/*  */


udp_create_channel(addr, a_ch, errind)
    struct address_t *addr;
    channel_t *a_ch;
    error_t   *errind;
{
    int s;
    struct sockaddr_in myaddr;
    
    pprintf("udp_create_channel(0x%x, 0x%x, 0x%x)\n",	addr, a_ch, errind);
    
    if ((s = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) == NOTOK) {
	eprintf(EF_SYSCALL,  COMMUNICATION, "socket call", "udp_create_channel",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    myaddr.sin_addr.s_addr = INADDR_ANY;
    myaddr.sin_port = 0;
    myaddr.sin_family = AF_INET;

    /* BUG FIX */
    bzero(myaddr.sin_zero, sizeof (myaddr.sin_zero));

    if (bind(s, &myaddr, sizeof(myaddr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "bind call", "udp_create_channel",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    a_ch->sock = s;
    a_ch->addr = myaddr; /* INADDR_ANY */
    a_ch->send_seq = a_ch->recv_seq = 0;

    *errind = OK;
} /* udp_create_channel */

udp_conn_channel(ch, errind)
    channel_t ch;
    error_t   *errind;
{
    pprintf("udp_conn_channel(%d, 0x%x)\n", ch.sock, errind);

    *errind = OK;
} /* udp_conn_channel */

/*  */

udp_await_conn_ind(server, a_ch, errind)
    struct server_t *server;	
    channel_t *a_ch;
    error_t *errind;
{    
    pprintf("udp_await_conn_ind(server %d)\n", server->sock);

    a_ch->sock = server->sock;
    a_ch->send_seq = a_ch->recv_seq = 0;

    *errind = OK;
} /* udp_await_conn_ind */

/*  */

udp_disc_request(ch, errind)
    channel_t ch;
    error_t   *errind;
{
    pprintf("udp_disc_request(%d, 0x%x)\n", ch.sock, errind);
    
    if (close(ch)== NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "close call", "udp_disc_request",
	       getsyserr());
  	*errind = NOTOK;
	return;
    }
    *errind = OK;
} /* udp_disc_request */

/*  */

udp_await_disc_ind(ch, errind)
    channel_t ch;
    error_t *errind;
{
    pprintf("udp_await_disc_ind(ch %d, 0x%x)\n", ch.sock, errind);

    *errind = OK;
} /* udp_await_disc_ind */

/*  */

udp_create_server(aa_server, aa_addr, errind)
    struct server_t **aa_server;
    struct address_t **aa_addr;
    error_t   *errind;
{
    int s;
    struct sockaddr_in myaddr;
    int len = sizeof(myaddr);
    
    
    pprintf("udp_create_server(0x%x, 0x%x, 0x%x)\n",
		aa_server, aa_addr, errind);

    if ((s = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "socket call", "udp_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    myaddr.sin_addr.s_addr = INADDR_ANY;
    myaddr.sin_port = 0;
    myaddr.sin_family = AF_INET;

    /* BUG FIX */
    bzero(myaddr.sin_zero, sizeof (myaddr.sin_zero));

    if (bind(s, &myaddr, sizeof(myaddr)) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "bind call", "udp_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }

    if (getsockname(s, &myaddr, &len) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "getsockname", "udp_create_server",
	       getsyserr());
	*errind = NOTOK;
	return;
    }
    if (len != sizeof(myaddr)) {
	eprintf(EF_SYSCALL, COMMUNICATION, "getsockname", "udp_create_server",
	       "\nBad length on returned address");
	*errind = NOTOK;
	return;
    }
    if (get_myipaddr(&myaddr.sin_addr) == NOTOK) {
	eprintf(EF_IN3, INTERNAL, "init_myipaddr", "CreateServer");
	*errind = NOTOK;
	return;
    }

    pprintf("Created socket %d with address: 0x%x:%d\n",
		s, myaddr.sin_addr.s_addr, myaddr.sin_port);

    *aa_server = (struct server_t *)malloc(sizeof(struct server_t));
    if (*aa_server == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "CreateServer");
	*errind = NOTOK;
	return;
    }
    *aa_addr = (struct address_t *)malloc(sizeof(struct address_t));
    if (*aa_addr == NULL) {
	eprintf(EF_IN3, INTERNAL, RESOURCE, "CreateServer");
	*errind = NOTOK;
	free((char *)*aa_server);
	return;
    }
    (*aa_server)->sock = s;
    **aa_addr = myaddr;	/* struct copy */
    *errind = OK;
} /* udp_create_server */

/*  */

udp_destroy_server(a_server, errind)
    struct server_t *a_server;
    error_t   *errind;
{
    
    pprintf("udp_destroy_server(0x%x, 0x%x)\n", a_server, errind);

    if (close(a_server->sock) == NOTOK) {
	eprintf(EF_SYSCALL, COMMUNICATION, "close", "udp_destroy_server",
	       getsyserr());
	free((char *)a_server);
	*errind = NOTOK;
	return;
    }

    free((char *)a_server);
	
    *errind = OK;
} /* udp_destroy_server */

/*  */

udp_report_error(err, str)
    error_t *err;
    char *str;
{
    
    pprintf("udp_report_error(0x%x)\n", err);
    
    if (err == NULL) {
	eprintf(EF_SYSCALL, COMMUNICATION, "Error", str,
	       getsyserr());
    } else {
	eprintf(EF_SYSCALL, COMMUNICATION, "UDP protocol", str,
	       getsyserr());
    }
} /* udp_report_error */

