/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * The demon server: only serves one benchmark client at a time
 */
#include <general.h>
#include <ipc.h>
#include <ipc_pres.h>

/*  */

#define HOSTNAMLEN 40
char LocalHost[HOSTNAMLEN];

static bcpaddr_t myaddr;

static bcpaddr_t ALL_PROCS = { BA_CLIENTS | BA_SERVERS, 0 };

demon_server()
{
    msg_t request, response;
    msg_t *rq = &request, *rp = &response;
    state_t state = IDLE;
    int secs;
    
    tprintf("demon_server()\n");

    if (gethostname(LocalHost, HOSTNAMLEN) == NOTOK) {
	logmsg("Internal error: gethostname\n");
	return;
    }

    rq->msg_datalen = 0;
    
    if (hostname2addr(LocalHost, &myaddr) == NOTOK) {
	logmsg("Internal error: hostname2addr failed!\n");
	return;
    }
    dprintf("myaddr 0x%x:0x%x\n", myaddr.ba_addr, myaddr.ba_index);
    
    if (bcp_init(myaddr) == NOTOK) {
	logmsg("Can't initialize the Benchmark Control Protocol\n");
	return;
    }

    while (1) {
	secs = parentstate2timeout(state);
    	if (await_recv_msg(rq, secs) == NOTOK) {
	    dprintf(EF_IN3, INTERNAL, "await_recv_msg", "demon_server");
	    continue;
	}

	rp->msg_datalen = 0;
	rp->msg_to = rq->msg_from;

	iprintf("%s %s in state %s\n",
		"Received a",
		code2str(rq->msg_code),
		state2str(state));
	
	switch (rq->msg_code) {

	case MSG_TIMEOUT:
	    switch (handle_timeout(&state, rq, rp)) { 	
	    case OK:
		break;
	    case DONE:
		(void)send_msg(rp);
		break;
	    case NOTOK:
		eprintf(EF_IN3, INTERNAL, "handle_timeout", "demon_server");
		state = IDLE;
		break;
	    }
	    break;
	
	case MSG_RESET:
	    /* can't do this right - refuse */
	    rp->msg_code = NOTOK;
	    (void)send_msg(rp);
	    break;

	case MSG_CREATE:
	case MSG_DESTROY:
	case MSG_ALARM:
	    /* should't get this - report the message */
	    eprintf(EF_IN3, PROTOCOL, "Received a bad message",
		    "demon_server");
	    dprintf("\tCode %s, from 0x%x:%d\n", code2str(rq->msg_code),
		    rq->msg_from.ba_addr, rq->msg_from.ba_index);
	    dprintf("\tin state %s\n", state2str(state));
	    rp->msg_code = MSG_PROERROR;
	    state = ABORTING;
	    (void)send_msg(rp);
	    break;
	
	case MSG_PROERROR:
	    eprintf("Received a MSG_PROERROR message\n");
	    dprintf("\tCode %s, from 0x%x:%d\n", code2str(rq->msg_code),
		    rq->msg_from.ba_addr, rq->msg_from.ba_index);
	    dprintf("\tin state %s\n", state2str(state));
	    eprintf("Protocol error: resetting benchmark\n");
	    state = ABORTING;
	    (void)send_abort(rp);
	    break;

	case MSG_FATAL:
	    eprintf("Benchmark killed by an external signal\n");
	    state = ABORTING;
	    (void)send_abort(rp);
	    rp->msg_from.ba_addr = BA_SERVERS | BA_CLIENTS;
	    rp->msg_to = myaddr;
	    rp->msg_code = MSG_DESTROY;
	    (void)bcp_send_msg(rp);
	    state = IDLE;
	    break;

	case MSG_ABORT:
	    eprintf("%s: %s %s in state %s - ignored\n",
		    INTERNAL, "Received a",
		code2str(rq->msg_code),
		state2str(state));
	    break;

	case MSG_CLOSED:
	    dprintf("Code %s from %s to %s\n",
	    code2str(rq->msg_code),
	    bcpaddr2str(rq->msg_from),
		    bcpaddr2str(rq->msg_to));
	    
	    if (rq->msg_from.ba_addr & (BA_SERVER|BA_CLIENT)) {
		/* forward to bench */
		rq->msg_to.ba_addr = BA_BENCH;
		(void)bcp_send_msg(rq);
		
		if (set_state(rq->msg_from, UNKNOWN) == NOTOK) {
		    dprintf(EF_IN3, INTERNAL, "set_state", "demon_server");
		    state = ABORTING;
		    (void)send_abort(rp);
		    break;
		}

		if (check_state(ALL_PROCS, UNKNOWN) == OK) {
		    dprintf("Closing the connection to BENCH\n");
		    rp->msg_to.ba_addr = BA_BENCH;
		    rp->msg_code = MSG_CLOSE;
		    (void)send_msg(rp);
		    (void)shutdown(rp);
		    state = IDLE;
		}
		break;		    
	    }
	    /* from BENCH */
	    (void)shutdown(rp);
	    state = IDLE;
	    break;

	case MSG_SPEC:
	    stprintf("Sets the state for all to INIT\n");
	    rq->msg_to.ba_addr = BA_CLIENTS | BA_SERVERS;
	    (void)set_state(rq->msg_to, INIT);
	    break;
	    
	default:
	    eprintf("%s: Received %s from %s in state %s\n",
		    INTERNAL, code2str(rq->msg_code),
		    bcpaddr2str(rq->msg_from), state2str(state));
	    break;
	}
	if (rq->msg_datalen) {
	    dprintf("bench-server: freeing rq data\n");
	    free(rq->msg_data);
	    rq->msg_datalen = 0;
	}
	if (rp->msg_datalen) {
	    dprintf("bench-server: freeing rp data\n");
	    free(rp->msg_data);
	    rp->msg_datalen = 0;
	}
	stprintf("State %s\n", state2str(state));
    } /* while (1) */
    /* NOTREACHED */
} /* demon_server */

/*  */

static int send_abort(msg)
    msg_t *msg;
{
    msg->msg_code = MSG_ABORT;
    msg->msg_to.ba_addr = BA_BENCH;
    msg->msg_datalen = 0;
    return send_msg(msg);
} /* send_abort */


int send_started()
{
    msg_t msgs, *msg = &msgs;
    
    tprintf("send_started()\n");
    
    msg->msg_to.ba_addr = BA_BENCH;
    msg->msg_code = MSG_STARTED;
    msg->msg_datalen = 0;
    (void)send_msg(msg);
} /* send_started */

/*  */

int shutdown(msg)
    msg_t *msg;
{
    tprintf("shutdown(0x%x)\n", msg);

    msg->msg_from.ba_addr = BA_SERVERS | BA_CLIENTS;
    msg->msg_to = myaddr;
    msg->msg_code = MSG_DESTROY;
    (void)bcp_send_msg(msg);
    /* force all channels to close */
    msg->msg_to.ba_addr = BA_CLIENTS | BA_SERVERS;
    msg->msg_code = MSG_CLOSE;
    (void)send_msg(msg);
    (void)close_readends();

    return OK;
} /* shutdown */
