/* The SPIMS software is covered by a license. The use of the software */
/* represents acceptance of the terms and conditions in the license. */
/* ****************************************************************** */
/* Copyright (c) 1989, Swedish Institute of Computer Science */
/*
 * Support for the data structures being built by the parser
 */


#include <general.h>

#include <y.tab.h>

/*  */

extern char DefClientHost[], DefServerHost[];

/*
 * Forward declarations:
 */

char *spaces();
char *int2mode();

void string_free();

/*  LOCAL DATA */

static int col = 0,
    	   colinc = 3;

/*  */

struct spec *spec_create(rpt,be,tl)
    int rpt;
    struct bench *be;
    double tl;
{
    struct spec *sp;

    sp = (struct spec *)malloc(sizeof(*sp));
    if (sp == NULLSP) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "spec_create",
	       "Malloc failed");
	return NULL;
    }

    sp->sp_number = rpt;
    sp->sp_bench = be;
    sp->sp_timelimit = tl;
    sp->sp_next = NULLSP;
    return sp;
} /* spec_create */

struct spec *spec_free(sp)
    struct spec *sp;
{
    struct spec *res;

    if (sp == NULLSP) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "spec_free",
	       "Null pointer");
	return;
    }
    res = spec_next(sp);
    bench_free(spec_bench(sp));
    free((char *)sp);
    return res;
} /* spec_free */

void specs_free(sp)
    struct spec *sp;
{
    while (sp != NULLSP) 
	sp = spec_free(sp);
} /* specs_free */

struct spec *spec_append(list,single)
    struct spec *list, *single;
{
    struct spec *p = list;

    if (list == NULLSP) {
	single->sp_next = NULLSP;
	return single;
    }
	
    while (p->sp_next != NULLSP)
	p = p->sp_next;
    p->sp_next = single;
    single->sp_next = NULLSP;
    return list;
} /* spec_append */


void spec_print(sp)
    struct spec *sp;
{
    if (sp == NULLSP) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "spec_print",
	       "Null pointer");
	return;
    }
    printf("%sSpecification: runs %d", spaces(col), spec_number(sp));
    if (!timelimit_unlim(spec_timelimit(sp)))
	printf(" timelimit: %.2f seconds\n", spec_timelimit(sp));
    else
	printf("\n");
    col += colinc;
    bench_print(spec_bench(sp));
    col -= colinc;
} /* spec_print */


void specs_print(sp)
    struct spec *sp;
{
    if (sp == NULLSP) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "specs_print",
	       "Null pointer");
	return;
    }

    spec_print(sp);
    if (sp->sp_next != NULLSP)
	specs_print(sp->sp_next);
} /* specs_print */

/*  */

struct bench *basic_create(se, dt, ti, hs)
    struct symbol *se;
    struct datasp *dt;
    struct timing *ti;
    struct hostspec *hs;
{
    struct bench *be;

    be = (struct bench *)malloc(sizeof(*be));
    if (be == NULLBE) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "basic_create",
	       "Malloc failed");
	return NULL;
    }

    be->be_tag = BET_BASIC;
    be->be_symbol = se;
    be->be_data = dt;
    be->be_timing = ti;
    be->be_hosts = hs;
    be->be_background = be->be_priority = 0;
    
    return be;
} /* basic_create */


struct bench *comp_create(se, pt, def_hs)
    struct symbol *se;
    struct part *pt;
    struct hostspec *def_hs;
{
    struct bench *be;
    int i;

    be = (struct bench *)malloc(sizeof(*be));
    if (be == NULLBE) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "comp_create",
	       "Malloc failed");
	return NULL;
    }

    be->be_tag = BET_COMP;
    be->be_symbol = se;
    be->be_parts = pt;
    be->be_number = 0;

    for (;pt != NULLPT; pt = part_next(pt)) {
	hostspec_setdefault(basic_hosts(part_bench(pt)), def_hs);
	be->be_number++;
    }
    hostspec_free(def_hs);
    return be;
} /* comp_create */


void bench_free(be)
    struct bench *be;
{
    if (be == NULLBE) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "bench_free",
	       "NULL pointer");
	return;
    }
    if (bench_isbasic(be))
	basic_free(be);
    else
	comp_free(be);
} /* bench_free */

static basic_free(be)
    struct bench *be;
{
    hostspec_free(basic_hosts(be));
    datasp_free(basic_data(be));
    if (basic_timing(be) != NULLTI)
	timing_free(basic_timing(be));
    free((char *)be);
} /* basic_free */

static comp_free(be)
    struct bench *be;
{
    parts_free(comp_parts(be));
    free((char *)be);
} /* comp_free */


extern char protostr[];
	
void bench_print(be)
    struct bench *be;
{
    if (be == NULLBE) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "bench_print",
	       "Null pointer");
	return;
    }
    if (bench_symbol(be) == NULLSE) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "bench_print",
	       "Null symbol pointer");
	return;
    }
    if (bench_isbasic(be)) {
	printf("%sBasic %s from %s to %s, %s protocol\n",
	       spaces(col), bench_name(be),
	       (basic_client(be) == NULL) ? "(unspec)": basic_client(be),
	       (basic_server(be) == NULL) ? "(unspec)": basic_server(be),
	       protostr);
	col += colinc;
	if (basic_background(be) || basic_priority(be) != 0) {
	    printf("%s%s with priority %d\n", spaces(col),
		   basic_background(be) ? "background" : "foreground",
		   basic_priority(be));
	}
	datasp_print(basic_data(be));
	if (basic_timing(be) == NULLTI)
  	    printf("%sNo timing specification\n", spaces(col));
	else
	    timings_print(basic_timing(be));
	col -= colinc;
    } else {
	printf("%sComposer %s: number of parts %d\n",
		spaces(col), bench_name(be), comp_number(be));
	col += colinc;
	parts_print(comp_parts(be));
	col -= colinc;
    }
} /* bench_print */

int part_numprocesses(pt)
    struct part *pt;
{
    int i;
    
    tprintf("part_numprocesses(0x%x)\n", pt);

    i = 0;
    
    while (pt != NULLPT) {
	i += part_number(pt);
	pt = part_next(pt);
    }
    return i;
} /* part_numprocesses */

/*  */

struct hostspec *hostspec_create(client, server)
    char *client, *server;
{
    struct hostspec *hs;

    hs = (struct hostspec *)malloc(sizeof(*hs));
    if (hs == NULLHS) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "hostspec_create",
	       "Malloc failed");
	return NULL;
    }

    if (client == NULL) {
	client = DefClientHost;
	hs->client_isdefault = 1;
    } else
	hs->client_isdefault = 0;
    
    hs->client = (char *)malloc(strlen(client)+1);
    if (hs->client == NULL) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "hostspec_create",
		"Malloc failed");
	return NULL;
    }
    strcpy(hs->client, client);
    
    if (server == NULL) {
	server = DefServerHost;
	hs->server_isdefault = 1;
    } else
	hs->server_isdefault = 0;
    
    hs->server = (char *)malloc(strlen(server)+1);
    if (hs->server == NULL) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "hostspec_create",
		"Malloc failed");
	return NULL;
    }
    strcpy(hs->server, server);
    
    return hs;
} /* hostspec_create */

void hostspec_free(hs)
    struct hostspec *hs;
{
    if (hs->server != NULL)
	free((char *)hs->server);
    if (hs->client != NULL)
	free((char *)hs->client);
    free((char *)hs);
} /* hostspec_free */

void hostspec_print(hs)
    struct hostspec *hs;
{
    printf("%sFrom %s to %s\n",
		spaces(col), hs->client, hs->server);
} /* hostspec_print */

void hostspec_setdefault(hs, def_hs)
    struct hostspec *hs, *def_hs;
{
    if (hs->client_isdefault) {
	hs->client_isdefault = 0;

	hs->client = (char *)malloc(strlen(def_hs->client)+1);
	if (hs->client == NULL) {
	    eprintf(EF_IN4X, INTERNAL, "Resource shortage",
		    "hostspec_setdefault",
		    "Malloc failed");
	    return;
	}
	strcpy(hs->client, def_hs->client);
    }
    
    if (hs->server_isdefault) {
	hs->server_isdefault = 0;
	
	hs->server = (char *)malloc(strlen(def_hs->server)+1);
	if (hs->server == NULL) {
	    eprintf(EF_IN4X, INTERNAL, "Resource shortage",
		    "hostspec_setdefault",
		    "Malloc failed");
	    return;
	}
	strcpy(hs->server, def_hs->server);
    }
} /* hostspec_setdefault */
	
/*  */

struct part *part_create(num, be)
    int num;
    struct bench *be;
{
    struct part *pt;

    pt = (struct part*)malloc(sizeof(*pt));
    if (pt == NULLPT) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "part_create",
	       "Malloc failed");
	return NULL;
    }

    pt->pt_number = num;
    pt->pt_bench = be;
    pt->pt_next = NULLPT;
    return pt;
} /* part_create */

struct part *part_free(pt)
    struct part *pt;
{
    struct part *res;

    if (pt == NULLPT) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "part_free",
	       "Null pointer");
	return;
    }
    res = part_next(pt);
    bench_free(part_bench(pt));
    free((char *)pt);
    return res;
} /* part_free */

void parts_free(pt)
    struct part *pt;
{
    while (pt != NULLPT) 
	pt = part_free(pt);
} /* parts_free */

struct part *part_append(list,single)
    struct part *list, *single;
{
    struct part *p = list;

    if (list == NULLPT) {
	single->pt_next = NULLPT;
	return single;
    }
	
    while (p->pt_next != NULLPT)
	p = p->pt_next;
    p->pt_next = single;
    single->pt_next = NULLPT;
    return list;
} /* part_append */


void part_print(pt)
    struct part *pt;
{
    if (pt == NULLPT) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "part_print",
	       "Null pointer");
	return;
    }
    printf("%sPart: number of replicas %d\n", spaces(col), part_number(pt));
    col += colinc;
    bench_print(part_bench(pt));
    col -= colinc;
} /* part_print */


void parts_print(pt)
    struct part *pt;
{
    if (pt == NULLPT) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "parts_print",
	       "Null pointer");
	return;
    }

    part_print(pt);
    if (pt->pt_next != NULLPT)
	parts_print(pt->pt_next);
} /* parts_print */

/*  */

struct datasp *datasp_create(type, ss, src, dst)
    int type;
    struct sizespec *ss;
    struct access *src, *dst;
{
    struct datasp *dt;

    dt = (struct datasp *)malloc(sizeof(*dt));
    if (dt == NULLDT) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "datasp_create",
	       "Malloc failed");
	return NULL;
    }

    dt->dt_type = type;
    dt->dt_size = ss;
    dt->dt_src = src;
    dt->dt_dst = dst;
    return dt;
} /* datasp_create */


void datasp_free(dt)
    struct datasp *dt;
{
    if (dt == NULLDT) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "datasp_free",
	       "Null pointer");
	return;
    }
    sizespec_free(datasp_size(dt));
    access_free(datasp_src(dt));
    access_free(datasp_dst(dt));
    free((char *)dt);
} /* datasp_free */
    
    
void datasp_print(dt)
    struct datasp *dt;
{
    char *typestr;
    
    if (dt == NULLDT) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "datasp_print",
	       "Null pointer");
	return;
    }
    typestr = symbol_lookup_token(datasp_type(dt));
    if (typestr == NULL) {
	printf("%sData: type %d\n", spaces(col), datasp_type(dt));
    } else {
	printf("%sData: type %s\n", spaces(col), typestr);
    }
    col += colinc;
    sizespec_print(datasp_size(dt));
    if (datasp_src(dt) == NULLAC)
	printf("%sNo source for data.\n", spaces(col));
    else {
	printf("%sSource:", spaces(col));
	access_print(datasp_src(dt));
    }
    
    if (datasp_dst(dt) == NULLAC)
	printf("%sNo destination for data.\n", spaces(col));
    else {
	printf("%sDestination:", spaces(col));
	access_print(datasp_dst(dt));
    }
    col -= colinc;
    
} /* datasp_print */

/*  */

struct access *access_create(type, fname)
    int type;
    char *fname;
{
    struct access *ac;
    char *filename;

    ac = (struct access *)malloc(sizeof(*ac));
    if (ac == NULLAC) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "access_create",
	       "Malloc failed");
	return NULL;
    }


    ac->ac_type = type;
    if (type == ACT_FILE) {
        filename = (char *)malloc(strlen(fname) + 1);
	if (filename == NULL) {
	    eprintf(EF_IN4X, INTERNAL, "Resource shortage", "access_create",
		    "Malloc failed");
	    free((char *)ac);
	    return NULL;
	}
	
	strcpy(filename, fname);
	ac->ac_filename = filename;
    }
    else
	ac->ac_filename = NULL;
    return ac;
} /* access_create */


void access_free(ac)
    struct access *ac;
{
    if (ac == NULLAC) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "access_free",
	       "Null pointer");
	return;
    }
    if (access_type(ac) == ACT_FILE && access_filename(ac) != NULL) {
	dprintf("Freeing the file name: %s\n", access_filename(ac));
	string_free(access_filename(ac));
    }
    free((char *)ac);
} /* access_free */

void access_print(ac)
    struct access *ac;
{
    if (ac == NULLAC) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "access_print",
	       "Null pointer");
	return;
    }
    switch (access_type(ac)) {
    case ACT_FILE:
	printf("FILE  %s\n", access_filename(ac));
	break;
    case ACT_MEM:
        printf("MEMORY\n");
	break;
    case ACT_VM:
        printf("VIRTUAL MEMORY\n");
	break;
    }
} /* access_print */

/*  */

struct sizespec *sizespec_create(sendmsg, recvmsg, iterations)
    struct msize *sendmsg, *recvmsg;
    int iterations;
{
    struct sizespec *ss;

    ss = (struct sizespec *)malloc(sizeof(*ss));
    if (ss == NULLSS) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "sizespec_create",
	       "Malloc failed");
	return NULL;
    }

    ss->ss_sendmsg = sendmsg;
    ss->ss_recvmsg = recvmsg;
    ss->ss_iterations = iterations;
    return ss;
} /* sizespec_create */


void sizespec_free(ss)
    struct sizespec *ss;
{
    if (ss == NULLSS) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "sizespec_free",
	       "Null pointer");
	return;
    }
    free((char *)ss);
} /* sizespec_free */
	 
    
void sizespec_print(ss)
    struct sizespec *ss;
{
    if (ss == NULLSS) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "sizespec_print",
	       "Null pointer");
	return;
    }
    printf("%sSize spec: iterations %d\n",
	    spaces(col), sizespec_iterations(ss));
    col += colinc;
    msize_print(sizespec_sendmsg(ss));
    msize_print(sizespec_recvmsg(ss));
    col -= colinc;
} /* sizespec_print */

/*  */

#define TIMING_NVAL	1009

struct timing *timing_create(dist, nvalue, values)
    struct dist *dist;
    int nvalue;
    double *values;
{
    struct timing *ti;

    ti = (struct timing *)malloc(sizeof(*ti));
    if (ti == NULLTI) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "timing_create",
	       "Malloc failed");
	return NULL;
    }

    ti->ti_dist = dist;
    ti->ti_nvalue = nvalue;
    ti->ti_values = values;

    return ti;
} /* timing_create */

void timing_generate(ti)
    struct timing *ti;
{
    int i;
    
    if (ti == NULLTI) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "timing_generate",
	       "Null pointer");
	return;
    }
    if (ti->ti_nvalue != 0 && ti->ti_values != NULL)
	free((char *)ti->ti_values);
    ti->ti_nvalue = TIMING_NVAL;
    ti->ti_values = (double *)malloc(TIMING_NVAL * sizeof(double));
    if (ti->ti_values == NULL) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "timing_generate",
	       "Malloc failed");
	free((char *)ti);
	return;
    }

    for (i = 0; i < ti->ti_nvalue; i++) {
	ti->ti_values[i] = (dist_proc(timing_dist(ti)))(dist_param(timing_dist(ti)));
	dprintf("timing_generate: values[%d] = %f\n", i, ti->ti_values[i]);
    }
} /* timing_generate */

struct timing *timing_free(ti)
    struct timing *ti;
{
    struct timing *res;

    if (ti == NULLTI) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "timing_free",
	       "Null pointer");
	return;
    }
    res = timing_next(ti);
    dist_free(timing_dist(ti));
    if (timing_nvalue(ti))
	free((char *)timing_values(ti));
    free((char *)ti);
    return res;
} /* timing_free */

void timings_free(ti)
    struct timing *ti;
{
    while (ti != NULLTI) 
	ti = timing_free(ti);
} /* timings_free */

struct timing *timing_append(list,single)
    struct timing *list, *single;
{
    struct timing *p = list;

    if (list == NULLTI) {
	single->ti_next = NULLTI;
	return single;
    }
	
    while (p->ti_next != NULLTI)
	p = p->ti_next;
    p->ti_next = single;
    single->ti_next = NULLTI;
    return list;
} /* timing_append */


void timing_print(ti)
    struct timing *ti;
{
    int i;
    
    if (ti == NULLTI) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "timing_print",
	       "Null pointer");
	return;
    }
    printf("%sTiming: number of values %d\n", spaces(col), timing_nvalue(ti));
    col += colinc;
    dist_print(timing_dist(ti));
    for (i = 0; i < timing_nvalue(ti); i++) {
	dprintf("%svalues[%d] = %f\n", spaces(col), i, timing_values(ti)[i]);
    }
    col -= colinc;
} /* timing_print */


void timings_print(ti)
    struct timing *ti;
{
    if (ti == NULLTI) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "timings_print",
	       "Null pointer");
	return;
    }

    while (ti != NULLTI) {
	timing_print(ti);
	ti = timing_next(ti);
    }
} /* timings_print */

/*  */

#define MSIZE_NVAL	1009

struct msize *msize_create(type, constant, dist, nvalue, values)
    int type;
    int constant;
    struct dist *dist;
    int nvalue;
    int *values;
{
    struct msize *ms;
    int i, max;

    ms = (struct msize *)malloc(sizeof(*ms));
    if (ms == NULLMS) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "msize_create",
	       "Malloc failed");
	return NULL;
    }

    switch (type) {
    case MS_CONSTANT:
	ms->ms_dist = NULLDI;
	ms->ms_values = (int *)malloc(1 * sizeof(int));
	if (ms->ms_values == NULL) {
	    eprintf(EF_IN4X, INTERNAL, "Resource shortage", "msize_create",
		    "Malloc failed");
	    free((char *)ms);
	    return NULL;
	}
	ms->ms_nvalue = 1;
	ms->ms_values[0] = constant;
	ms->ms_maxval = constant;
	break;
	
    case MS_DIST:
	ms->ms_dist = dist;
	ms->ms_nvalue = nvalue;
	ms->ms_values = values;
	
	max = 0;
	for (i = 0; i < ms->ms_nvalue; i++) {
	    if (ms->ms_values[i] > max)
		max = ms->ms_values[i];
	}
	ms->ms_maxval = max;
	break;

    default:
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "msize_create",
		"Illegal tag");
	return NULL;
    }
    
    return ms;
} /* msize_create */

void msize_generate(ms)
    struct msize *ms;
{
    int i, max;

    if (ms == NULLMS) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "msize_generate",
	       "Null pointer");
	return;
    }
    if (ms->ms_dist != NULLDI) {
	if (ms->ms_nvalue != 0 && ms->ms_values != NULL)
	    free((char *)ms->ms_values);
	ms->ms_nvalue = MSIZE_NVAL;

	ms->ms_values = (int *)malloc(MSIZE_NVAL * sizeof(int));
	if (ms->ms_values == NULL) {
	    eprintf(EF_IN4X, INTERNAL, "Resource shortage", "msize_generate",
		    "Malloc failed");
	    free((char *)ms);
	    return;
	}

	max = 0;
	for (i = 0; i < ms->ms_nvalue; i++) {
	    ms->ms_values[i] = (int)(dist_proc(msize_dist(ms)))(dist_param(msize_dist(ms)));
	    dprintf("msize_generate: values[%d] = %d\n", i, ms->ms_values[i]);
	    if (ms->ms_values[i] > max)
		max = ms->ms_values[i];
	}
	ms->ms_maxval = max;
    }
} /* msize_generate */


void msize_free(ms)
    struct msize *ms;
{
    if (ms == NULLMS) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "msize_free",
	       "Null pointer");
	return;
    }
    dist_free(msize_dist(ms));
    if (msize_nvalue(ms))
	free((char *)msize_values(ms));
    free((char *)ms);
} /* msize_free */

void msize_print(ms)
    struct msize *ms;
{
    int i;
    
    if (ms == NULLMS) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "msize_print",
	       "Null pointer");
	return;
    }
    printf("%sMessage size: number of values %d\n",
	   spaces(col), msize_nvalue(ms));
    col += colinc;
    if (msize_dist(ms) == NULLDI) {
	printf("%sConstant value %d\n", spaces(col), msize_values(ms)[0]);
    } else {
	dist_print(msize_dist(ms));
	for (i = 0; i < msize_nvalue(ms); i++) {
	    dprintf("%svalues[%d] = %d\n",
		    spaces(col), i, msize_values(ms)[i]);
	}
	printf("%smaximum value %d\n", spaces(col), msize_maxval(ms));
    }
    col -= colinc;
} /* msize_print */

/*  */

struct dist *dist_create(se, pm)
    struct symbol *se;
    struct param *pm;
{
    struct dist *di;

    di = (struct dist *)malloc(sizeof(*di));
    if (di == NULLDI) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "dist_create",
	       "Malloc failed");
	return NULL;
    }

    di->di_symbol = se;
    di->di_param = pm;
    return di;
} /* dist_create */
    

void dist_free(di)
    struct dist *di;
{
    if (di == NULLDI) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "dist_free",
	       "Null pointer");
	return;
    }
    params_free(dist_param(di));
    free((char *)di);
} /* dist_free */


void dist_print(di)
    struct dist *di;
{
    if (di == NULLDI) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "dist_print",
	       "Null pointer");
	return;
    }
    if (dist_symbol(di) == NULLSE) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "dist_print",
	       "Null symbol pointer");
	return;
    }

    printf("%sDistribution: %s\n",
	    spaces(col), dist_name(di));
    col += colinc;
    params_print(dist_param(di));
    col -= colinc;
} /* dist_print */

/*  */

struct param *param_create(val)
    double val;
{
    struct param *pm;

    pm = (struct param *)malloc(sizeof(*pm));
    if (pm == NULLPM) {
	eprintf(EF_IN4X, INTERNAL, "Resource shortage", "param_create",
	       "Malloc failed");
	return NULL;
    }

    pm->pm_value = val;
    pm->pm_next = NULLPM;
    return pm;
} /* param_create */


struct param *param_free(pm)
    struct param *pm;
{
    struct param *res;

    if (pm == NULLPM) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "param_free",
	       "Null pointer");
	return;
    }
    res = param_next(pm);
    free((char *)pm);
    return res;
} /* param_free */

void params_free(pm)
    struct param *pm;
{
    while (pm != NULLPM) 
	pm = param_free(pm);
} /* params_free */


struct param *param_append(list,single)
    struct param *list, *single;
{
    struct param *p = list;

    if (p == NULLPM) {
	single->pm_next = NULLPM;
	return single;
    }
    while (p->pm_next != NULLPM)
	p = p->pm_next;
    p->pm_next = single;
    single->pm_next = NULLPM;
    return list;
} /* param_append */


int param_nvalues(pm)
    struct param *pm;
{
    int res;

    for (res = 0; pm != NULLPM; res++, pm = param_next(pm))
	;
    return res;
} /* param_nvalues */


double param_getvalue(pm, index)
    struct param *pm;
    int index;
{
    int i;

    for (i = 0; i < index && pm != NULLPM; i++)
	pm = param_next(pm);
    if (pm == NULLPM) {
	eprintf(EF_IN3, INTERNAL, PARAMETER, "param_getvalue");
	return 0.0;
    }
    dprintf("param_getvalue(%d) -> %f\n", index, param_value(pm));
    return param_value(pm);
} /* param_getvalue */

void param_print(pm)
    struct param * pm;
{
    if (pm == NULLPM) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "param_print",
	       "Null pointer");
	return;
    }
    printf(" %f", param_value(pm));
} /* param_print */


void params_print(pm)
    struct param * pm;
{
    if (pm == NULLPM) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "params_print",
	       "Null pointer");
	return;
    }

    printf("%s(", spaces(col));
    while (1) {
    	param_print(pm);
	pm = pm->pm_next;
	if (pm == NULLPM) {
	    printf(")\n");
	    break;
	} else
	    printf(",");
    }
} /* params_print */


#define BUFFSIZE 128

static char buff[BUFFSIZE];
static char bigspace[] =
"                                                                            ";
    
char *spaces(n)
    int n;
{
    int max = 2*strlen(bigspace);
    
    strcpy(buff, bigspace);
    strcat(buff, bigspace);
    if (n >= max)
	n = max - 1;
    buff[n] = '\0';
    return buff;
}

/*  */
/*
 * This should be in lex.c, but since demon will not load without it ...
 * This relies on getstring() in lex.c malloc()ing the returned string.
 */
   
void string_free(str)
    char *str;
{
    if (str == NULL) {
	eprintf(EF_IN4X, INTERNAL, PARAMETER, "string_free",
	       "Null pointer");
	return;
    }
    free((char *)str);
} /* string_free */
