/*
 * Copyright (C) 1992, Board of Trustees of the University of Illinois.
 *
 * Permission is granted to copy and distribute source with out fee.
 * Commercialization of this product requires prior licensing
 * from the National Center for Supercomputing Applications of the
 * University of Illinois.  Commercialization includes the integration of this 
 * code in part or whole into a product for resale.  Free distribution of 
 * unmodified source and use of NCSA software is not considered 
 * commercialization.
 *
 */


#include <stdio.h>

#include <X11/Xlib.h>
#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Xatom.h>
#include <X11/cursorfont.h>
#include <Xm/ScrollBar.h>

#include "list.h"
#include "view.h"
#include "viewer.h"
#include "doodle.h"
#include "mode.h"
#include "net.h"


extern Pixmap FindDHash();


extern NetPort *outP;



extern Pixmap drawPix;
extern Pixmap doodlePix;
extern Pixmap scratchPix;
extern Pixmap row_arrow, col_arrow;
extern int SdrawWidth, SdrawHeight, SdrawAscent;
extern Widget	rootWidget;
Widget	rowwin, colwin;
Widget	drawArea;
extern GC	drawAreaGC;
extern GC	invGC;
extern GC	textGC;
extern GC	invtextGC;
extern Display	*myDpy;

void ScrollYChanged();
void ScrollXChanged();
void RowsExposed();
void ColsExposed();
void SheetExposed();
void InitDrawAreaSel();
void SheetSelOn();
void SheetSelOff();
void SelRow();
void SelCol();
void UnSelect();


struct sheet_rec {
	int row, col;
	int modified;
	struct sheet_rec *next;
};

struct sheet_rec *SheetList = NULL;


void
ScrollYChanged(w, client_data, call_data)
	Widget w;
	caddr_t client_data;
	caddr_t call_data;
{
	XmScrollBarCallbackStruct *sc = (XmScrollBarCallbackStruct *)call_data;
	View *V = (View *)client_data;
	XWindowAttributes attr;
	int oldy;

	oldy = V->Scroll_y;
	V->Scroll_y = sc->value - 1;
	V->text_y = V->text_y + oldy - V->Scroll_y;


	if (V->Scroll_y > oldy)
	{
		XGetWindowAttributes(myDpy, XtWindow(V->drawArea), &attr);
		XCopyArea(myDpy, XtWindow(V->drawArea), XtWindow(V->drawArea),
			drawAreaGC,
			0, (V->Scroll_y - oldy),
			attr.width, (attr.height - V->Scroll_y + oldy), 0, 0);
		SheetExposed(V, 0, (attr.height - V->Scroll_y + oldy),
			attr.width, (V->Scroll_y - oldy));

		XGetWindowAttributes(myDpy, XtWindow(V->rowwin), &attr);
		XCopyArea(myDpy, XtWindow(V->rowwin), XtWindow(V->rowwin), drawAreaGC,
			0, (V->Scroll_y - oldy),
			attr.width, (attr.height - V->Scroll_y + oldy), 0, 0);
		RowsExposed(V, 0, (attr.height - V->Scroll_y + oldy),
			attr.width, (V->Scroll_y - oldy));
	}
	else
	{
		XGetWindowAttributes(myDpy, XtWindow(V->drawArea), &attr);
		XCopyArea(myDpy, XtWindow(V->drawArea), XtWindow(V->drawArea),
			drawAreaGC,
			0, 0, attr.width, (attr.height - oldy + V->Scroll_y),
			0, (oldy - V->Scroll_y));
		SheetExposed(V, 0, 0, attr.width, (oldy - V->Scroll_y));

		XGetWindowAttributes(myDpy, XtWindow(V->rowwin), &attr);
		XCopyArea(myDpy, XtWindow(V->rowwin), XtWindow(V->rowwin), drawAreaGC,
			0, 0, attr.width, (attr.height - oldy + V->Scroll_y),
			0, (oldy - V->Scroll_y));
		RowsExposed(V, 0, 0, attr.width, (oldy - V->Scroll_y));
	}
}


void
ScrollXChanged(w, client_data, call_data)
	Widget w;
	caddr_t client_data;
	caddr_t call_data;
{
	XmScrollBarCallbackStruct *sc = (XmScrollBarCallbackStruct *)call_data;
	View *V = (View *)client_data;
	XWindowAttributes attr;
	int oldx;

	oldx = V->Scroll_x;
	V->Scroll_x = sc->value - 1;
	V->text_x = V->text_x + oldx - V->Scroll_x;
	V->ret_x = V->ret_x + oldx - V->Scroll_x;


	if (V->Scroll_x > oldx)
	{
		XGetWindowAttributes(myDpy, XtWindow(V->drawArea), &attr);
		XCopyArea(myDpy, XtWindow(V->drawArea), XtWindow(V->drawArea),
			drawAreaGC,
			(V->Scroll_x - oldx), 0,
			(attr.width - V->Scroll_x + oldx), attr.height, 0, 0);
		SheetExposed(V, (attr.width - V->Scroll_x + oldx), 0,
			(V->Scroll_x - oldx), attr.height);

		XGetWindowAttributes(myDpy, XtWindow(V->colwin), &attr);
		XCopyArea(myDpy, XtWindow(V->colwin), XtWindow(V->colwin), drawAreaGC,
			(V->Scroll_x - oldx), 0,
			(attr.width - V->Scroll_x + oldx), attr.height, 0, 0);
		ColsExposed(V, (attr.width - V->Scroll_x + oldx), 0,
			(V->Scroll_x - oldx), attr.height);
	}
	else
	{
		XGetWindowAttributes(myDpy, XtWindow(V->drawArea), &attr);
		XCopyArea(myDpy, XtWindow(V->drawArea), XtWindow(V->drawArea),
			drawAreaGC,
			0, 0, (attr.width - oldx + V->Scroll_x), attr.height,
			(oldx - V->Scroll_x), 0);
		SheetExposed(V, 0, 0, (oldx - V->Scroll_x), attr.height);

		XGetWindowAttributes(myDpy, XtWindow(V->colwin), &attr);
		XCopyArea(myDpy, XtWindow(V->colwin), XtWindow(V->colwin), drawAreaGC,
			0, 0, (attr.width - oldx + V->Scroll_x), attr.height,
			(oldx - V->Scroll_x), 0);
		ColsExposed(V, 0, 0, (oldx - V->Scroll_x), attr.height);
	}
}


void
RowsExposed(V, x, y, width, height)
	View *V;
	int x, y;
	unsigned int width, height;
{
	int i;
	int row, ypos, r1, r2;
	char buf[S_DIGITS + 1];

	r1 = (V->Scroll_y + y) / SdrawHeight;
	r2 = (V->Scroll_y + y + height) / SdrawHeight;
	ypos = (r1 * SdrawHeight) + SdrawAscent - V->Scroll_y;

	row = r1;
	while (row <= r2)
	{
		if (row < V->cData->ydim)
		{
			sprintf(buf, "%*d ", S_INDEX - 1, row);
		}
		else
		{
			for (i=0; i<S_INDEX; i++)
			{
				buf[i] = ' ';
			}
			buf[S_INDEX] = '\0';
		}
		XDrawImageString(myDpy, XtWindow(V->rowwin), textGC,
                        0, ypos, buf, S_INDEX - 1);
		XClearArea(myDpy, XtWindow(V->rowwin),
			(SdrawWidth * (S_INDEX - 1)), (ypos - SdrawAscent),
			SdrawWidth, SdrawHeight, False);
		if ((row >= V->area_y1)&&(row <= V->area_y2))
		{
			XCopyArea(myDpy, row_arrow, XtWindow(V->rowwin),
				drawAreaGC, 0, 0, SdrawWidth, (SdrawHeight / 2),
				(SdrawWidth * (S_INDEX - 1)),
				(ypos - SdrawAscent + (SdrawHeight / 4)));
		}
		if (row == V->pt_y)
		{
			XCopyArea(myDpy, row_arrow, XtWindow(V->rowwin),
				drawAreaGC, 0, 0, SdrawWidth, (SdrawHeight / 2),
				(SdrawWidth * (S_INDEX - 1)),
				(ypos - SdrawAscent + (SdrawHeight / 4)));
		}
		ypos += SdrawHeight;
		row++;
	}
}


void
ColsExposed(V, x, y, width, height)
	View *V;
	int x, y;
	unsigned int width, height;
{
	int i;
	int col, xpos, c1, c2;
	char buf[S_DIGITS + 1];

	c1 = (V->Scroll_x + x) / (SdrawWidth * S_DIGITS);
	c2 = (V->Scroll_x + x + width) / (SdrawWidth * S_DIGITS);
	xpos = (c1 * SdrawWidth * S_DIGITS) - V->Scroll_x;

	col = c1;
	while (col <= c2)
	{
		if (col < V->cData->xdim)
		{
			sprintf(buf, "%*d", S_DIGITS, col);
		}
		else
		{
			for (i=0; i<S_DIGITS; i++)
			{
				buf[i] = ' ';
			}
			buf[S_DIGITS] = '\0';
		}
		XDrawImageString(myDpy, XtWindow(V->colwin), textGC,
                        xpos, SdrawAscent, buf, S_DIGITS);
		XClearArea(myDpy, XtWindow(V->colwin),
			xpos, SdrawHeight,
			(SdrawWidth * S_DIGITS), (SdrawHeight / 2), False);
		if ((col >= V->area_x1)&&(col <= V->area_x2))
		{
			XCopyArea(myDpy, col_arrow, XtWindow(V->colwin),
				drawAreaGC, 0, 0, SdrawWidth, (SdrawHeight / 2),
				(xpos + SdrawWidth * (S_DIGITS - 1)), SdrawHeight);
		}
		if (col == V->pt_x)
		{
			XCopyArea(myDpy, col_arrow, XtWindow(V->colwin),
				drawAreaGC, 0, 0, SdrawWidth, (SdrawHeight / 2),
				(xpos + SdrawWidth * (S_DIGITS - 1)), SdrawHeight);
		}
		xpos = xpos + (SdrawWidth * S_DIGITS);
		col++;
	}
}


void
RedrawRowCol(V, row, col)
	View *V;
	int row, col;
{
	int xpos, ypos;
	int i, val;
	float fval;
	Pixmap dmap;
	Pixmap clipmask;
	Pixmap nopix;
	char buf[S_DIGITS + 1];

	nopix = (Pixmap)0;
	ypos = (row * SdrawHeight) + SdrawAscent - V->Scroll_y;
	xpos = (col * SdrawWidth * S_DIGITS) - V->Scroll_x;
	if ((col < V->cData->xdim)&&(row < V->cData->ydim))
	{
		if (V->cData->type == D_CHAR)
		{
			val = (int)*((unsigned char *) (V->cData->buff +
				(row * V->cData->xdim) + col));
			sprintf(buf, "%*d", S_DIGITS, val);
		}
		else if (V->cData->type == D_INT)
		{
			val = (int)*((int *) (((int *)V->cData->fbuff) +
				(row * V->cData->xdim) + col));
			sprintf(buf, "%*d", S_DIGITS, val);
		}
		else if (V->cData->type == D_FLOAT)
		{
			fval = (float)*((float *) (V->cData->fbuff +
				(row * V->cData->xdim) + col));
			sprintf(buf, " % 1.*e", S_DIGITS - 8, fval);
		}
		dmap = FindDHash(V->Doodles, col, row, &clipmask);
	}
	else
	{
		for (i=0; i<S_DIGITS; i++)
		{
			buf[i] = ' ';
		}
		buf[S_DIGITS] = '\0';
		dmap = nopix;
	}
	if ((row >= V->area_y1)&&(row <= V->area_y2)&&
	    (col >= V->area_x1)&&(col <= V->area_x2))
	{
		XDrawImageString(myDpy, XtWindow(V->drawArea),
			invtextGC, xpos, ypos, buf, S_DIGITS);
	}
	else if ((row == V->pt_y)&&(col == V->pt_x))
	{
		XDrawImageString(myDpy, XtWindow(V->drawArea),
			invtextGC, xpos, ypos, buf, S_DIGITS);
	}
	else
	{
		XDrawImageString(myDpy, XtWindow(V->drawArea),
			textGC, xpos, ypos, buf, S_DIGITS);
	}
	if ((dmap != (Pixmap)0)&&(V->viewOverlay))
	{
		XSetClipMask(myDpy, drawAreaGC, clipmask);
		XSetClipOrigin(myDpy, drawAreaGC, xpos, (ypos - SdrawAscent));
		XCopyArea(myDpy, dmap, XtWindow(V->drawArea), drawAreaGC, 0, 0,
			(SdrawWidth * S_DIGITS), SdrawHeight, xpos, (ypos - SdrawAscent));
		XSetClipMask(myDpy, drawAreaGC, None);
	}
}


void
AddSheetBuffer(V, row, col)
	View *V;
	int row, col;
{
	struct sheet_rec *sptr;
	struct sheet_rec *tptr;

	if (SheetList == NULL)
	{
		SheetList = (struct sheet_rec *)
			MALLOC(sizeof(struct sheet_rec));
		SheetList->row = row;
		SheetList->col = col;
		SheetList->modified = 1;
		SheetList->next = NULL;
	}
	else if ((SheetList->row == row)&&(SheetList->col == col))
	{
		SheetList->modified = 1;
	}
	else
	{
		sptr = SheetList;
		while (sptr->next != NULL)
		{
			if ((sptr->next->row == row)&&(sptr->next->col == col))
			{
				sptr->next->modified = 1;
				break;
			}
			sptr = sptr->next;
		}
		if (sptr->next == NULL)
		{
			sptr->next = (struct sheet_rec *)
				MALLOC(sizeof(struct sheet_rec));
			sptr = sptr->next;
			sptr->row = row;
			sptr->col = col;
			sptr->modified = 1;
			sptr->next = NULL;
		}
	}
}


void
DrainSheetBuffer(V)
	View *V;
{
	struct sheet_rec *sptr;
	struct sheet_rec *tptr;

	sptr = SheetList;
	while ((sptr != NULL)&&(sptr->modified != 1))
	{
		tptr = sptr;
		sptr = sptr->next;
		RedrawRowCol(V, tptr->row, tptr->col);
		FREE((char *)tptr);
	}

	if (sptr == NULL)
	{
		SheetList = NULL;
	}
	else
	{
		SheetList = sptr;
		SheetList->modified = 0;
	}

	sptr = SheetList;
	while (sptr->next != NULL)
	{
		if (sptr->next->modified)
		{
			sptr = sptr->next;
			sptr->modified = 0;
		}
		else
		{
			tptr = sptr->next;
			sptr->next = tptr->next;
			RedrawRowCol(V, tptr->row, tptr->col);
			FREE((char *)tptr);
		}
	}
}


void
FlushSheetBuffer(V)
	View *V;
{
	struct sheet_rec *sptr;
	struct sheet_rec *tptr;

	sptr = SheetList;
	while (sptr != NULL)
	{
		tptr = sptr;
		sptr = sptr->next;
		RedrawRowCol(V, tptr->row, tptr->col);
		FREE((char *)tptr);
	}
	SheetList = NULL;
}


void
RealBufferSheetExposed(V, x, y, width, height)
	View *V;
	int x, y;
	unsigned int width, height;
{
	int col, c1, c2;
	int row, r1, r2;

	r1 = (V->Scroll_y + y) / SdrawHeight;
	r2 = (V->Scroll_y + y + height) / SdrawHeight;

	c1 = (V->Scroll_x + x) / (SdrawWidth * S_DIGITS);
	c2 = (V->Scroll_x + x + width) / (SdrawWidth * S_DIGITS);

	for (row = r1; row <= r2; row++)
	{
		for (col = c1; col <= c2; col++)
		{
			AddSheetBuffer(V, row, col);
		}
	}
	DrainSheetBuffer(V);
}


void
RealSheetExposed(V, x, y, width, height)
	View *V;
	int x, y;
	unsigned int width, height;
{
	int col, c1, c2;
	int row, r1, r2;

	r1 = (V->Scroll_y + y) / SdrawHeight;
	r2 = (V->Scroll_y + y + height) / SdrawHeight;

	c1 = (V->Scroll_x + x) / (SdrawWidth * S_DIGITS);
	c2 = (V->Scroll_x + x + width) / (SdrawWidth * S_DIGITS);

	for (row = r1; row <= r2; row++)
	{
		for (col = c1; col <= c2; col++)
		{
			RedrawRowCol(V, row, col);
		}
	}
}


void
BufferSheetExposed(V, x, y, width, height)
	View *V;
	int x, y;
	unsigned int width, height;
{
	RealBufferSheetExposed(V, x, y, width, height);

	if (V->Mode == TEXTINSERT)
	{
		RealSheetExposed(V, V->text_x, (V->text_y - SdrawAscent),
			SdrawWidth, SdrawHeight);

		XFillRectangle(myDpy, XtWindow(V->drawArea), invGC,
			V->text_x, (V->text_y - SdrawAscent), SdrawWidth, SdrawHeight);
	}
}


void
SheetExposed(V, x, y, width, height)
	View *V;
	int x, y;
	unsigned int width, height;
{
	RealSheetExposed(V, x, y, width, height);

	if (V->Mode == TEXTINSERT)
	{
		RealSheetExposed(V, V->text_x, (V->text_y - SdrawAscent),
			SdrawWidth, SdrawHeight);

		XFillRectangle(myDpy, XtWindow(V->drawArea), invGC,
			V->text_x, (V->text_y - SdrawAscent), SdrawWidth, SdrawHeight);
	}
}


void
CBDrawAreaRowsExposed(w, client_data, event)
	Widget w;
	caddr_t client_data;
	XEvent *event;
{
	XExposeEvent *ExEvent = (XExposeEvent *)event;
	View *V = (View *)client_data;

	RowsExposed(V, ExEvent->x, ExEvent->y, ExEvent->width, ExEvent->height);
}


void
CBDrawAreaColsExposed(w, client_data, event)
	Widget w;
	caddr_t client_data;
	XEvent *event;
{
	XExposeEvent *ExEvent = (XExposeEvent *)event;
	View *V = (View *)client_data;

	ColsExposed(V, ExEvent->x, ExEvent->y, ExEvent->width, ExEvent->height);
}


void
CBDrawAreaExposed(w, client_data, event)
	Widget w;
	caddr_t client_data;
	XEvent *event;
{
	XExposeEvent *ExEvent = (XExposeEvent *)event;
	View *V = (View *)client_data;

	if (V->isUp == False)
	{
		V->isUp = True;
	}

	SheetExposed(V, ExEvent->x, ExEvent->y, ExEvent->width, ExEvent->height);
}


void
CBDrawAreaResize(w, client_data, event)
	Widget w;
	caddr_t client_data;
	XEvent *event;
{
	View *V = (View *)client_data;
	Arg argList[10];
	Cardinal i;
	int wi, h, redraw;
	int Ww, Wh;
	XWindowAttributes attr;

	if (event->type != ConfigureNotify)
	{
		return;
	}

	redraw = 0;
	wi = V->cData->xdim * SdrawWidth * S_DIGITS;
	h = V->cData->ydim * SdrawHeight;

	XGetWindowAttributes(myDpy, XtWindow(V->drawArea), &attr);
	Ww = attr.width;
	Wh = attr.height;
	if (Ww > wi)
	{
		Ww = wi;
	}
	if (Wh > h)
	{
		Wh = h;
	}

	i = 0;
	if (V->Scroll_y > (h - (Wh - 1)))
	{
		redraw = 1;
		V->text_y = V->text_y + V->Scroll_y - (h - (Wh - 1));
		V->Scroll_y = h - (Wh - 1);
		XtSetArg(argList[i], XmNvalue, V->Scroll_y); i++;
	}
	XtSetArg(argList[i], XmNsliderSize, (Wh - 1)); i++;
	XtSetValues(V->vscroll, argList, i);

	i = 0;
	if (V->Scroll_x > (wi - (Ww - 1)))
	{
		redraw = 1;
		V->text_x = V->text_x + V->Scroll_x - (wi - (Ww - 1));
		V->ret_x = V->ret_x + V->Scroll_x - (wi - (Ww - 1));
		V->Scroll_x = wi - (Ww - 1);
		XtSetArg(argList[i], XmNvalue, V->Scroll_x); i++;
	}
	XtSetArg(argList[i], XmNsliderSize, (Ww - 1)); i++;
	XtSetValues(V->hscroll, argList, i);

	if (redraw)
	{
		XClearArea(myDpy, XtWindow(V->rowwin), 0, 0, 0, 0, True);
		XClearArea(myDpy, XtWindow(V->colwin), 0, 0, 0, 0, True);
		XClearArea(myDpy, XtWindow(V->drawArea), 0, 0, 0, 0, True);
	}
}


void
UnSelect(V, type)
	View *V;
	int type;
{
	int i, j;
	int val;
	float fval;
	int xpos, ypos;
	Pixmap dmap;
	Pixmap clipmask;
	char buf[S_DIGITS + 1];
	int wx1, wx2, wy1, wy2;
	XWindowAttributes attr;

	XGetWindowAttributes(myDpy, XtWindow(V->drawArea), &attr);
	wx1 = V->Scroll_x / (SdrawWidth * S_DIGITS);
	wx2 = (V->Scroll_x + attr.width) / (SdrawWidth * S_DIGITS);
	wy1 = V->Scroll_y / SdrawHeight;
	wy2 = (V->Scroll_y + attr.height) / SdrawHeight;

	/*
	 * unselect the selected number..
	 * This is done by redrawing the number unhighlighted, restoring
	 * and doodles that were eresed by the redraw, and erasing the
	 * pointer arrows from the row and column windows.
	 */
	if ((type == AREASEL)&&(V->area_x1 >= 0)&&(V->area_y1 >= 0))
	{
	    for (j = V->area_y1; j <= V->area_y2; j++)
	    {
		if ((j < wy1)||(j > wy2))
		{
			continue;
		}
		for (i = V->area_x1; i <= V->area_x2; i++)
		{
			if ((i < wx1)||(i > wx2))
			{
				continue;
			}
			if (V->cData->type == D_CHAR)
			{
				val = (int)*((unsigned char *) (V->cData->buff +
					(j * V->cData->xdim) + i));
				sprintf(buf, "%*d", S_DIGITS, val);
			}
			else if (V->cData->type == D_INT)
			{
				val = (int)*((int *) (((int *)V->cData->fbuff) +
					(j * V->cData->xdim) + i));
				sprintf(buf, "%*d", S_DIGITS, val);
			}
			else if (V->cData->type == D_FLOAT)
			{
				fval = (float)*((float *) (V->cData->fbuff +
					(j * V->cData->xdim) + i));
				sprintf(buf, " % 1.*e", S_DIGITS - 8, fval);
			}
			xpos = (i * SdrawWidth * S_DIGITS) - V->Scroll_x;
			ypos = (j * SdrawHeight) + SdrawAscent - V->Scroll_y;
			XDrawImageString(myDpy, XtWindow(V->drawArea), textGC,
				xpos, ypos, buf, S_DIGITS);
			dmap = FindDHash(V->Doodles, i, j, &clipmask);
			if ((dmap != (Pixmap)0)&&(V->viewOverlay))
			{
				XSetClipMask(myDpy, drawAreaGC, clipmask);
				XSetClipOrigin(myDpy, drawAreaGC,
					xpos, (ypos - SdrawAscent));
				XCopyArea(myDpy, dmap, XtWindow(V->drawArea),
					drawAreaGC, 0, 0,
					(SdrawWidth * S_DIGITS), SdrawHeight,
					xpos, (ypos - SdrawAscent));
				XSetClipMask(myDpy, drawAreaGC, None);
			}
			XClearArea(myDpy, XtWindow(V->colwin),
				(xpos + SdrawWidth * (S_DIGITS - 1)), SdrawHeight,
				SdrawWidth, (SdrawHeight / 2), False);
			XClearArea(myDpy, XtWindow(V->rowwin),
				(SdrawWidth * (S_INDEX - 1)), (ypos - SdrawAscent + (SdrawHeight / 4)),
				SdrawWidth, (SdrawHeight / 2), False);
		}
	    }
	    V->area_x1 = -1;
	    V->area_y1 = -1;
	    V->area_x2 = -1;
	    V->area_y2 = -1;

	    if (V->Mode == TEXTINSERT)
	    {
		RealSheetExposed(V, V->text_x, (V->text_y - SdrawAscent),
			SdrawWidth, SdrawHeight);

		XFillRectangle(myDpy, XtWindow(V->drawArea), invGC,
			V->text_x, (V->text_y - SdrawAscent), SdrawWidth, SdrawHeight);
	    }
	}
	else if ((type == POINTSEL)&&(V->pt_x >= 0)&&(V->pt_y >= 0))
	{
		if ((V->pt_x >= wx1)&&(V->pt_x <= wx2)&&
		    (V->pt_y >= wy1)&&(V->pt_y <= wy2))
		{
			if (V->cData->type == D_CHAR)
			{
				val = (int)*((unsigned char *) (V->cData->buff +
					(V->pt_y * V->cData->xdim) + V->pt_x));
				sprintf(buf, "%*d", S_DIGITS, val);
			}
			else if (V->cData->type == D_INT)
			{
				val = (int)*((int *) (((int *)V->cData->fbuff) +
					(V->pt_y * V->cData->xdim) + V->pt_x));
				sprintf(buf, "%*d", S_DIGITS, val);
			}
			else if (V->cData->type == D_FLOAT)
			{
				fval = (float)*((float *) (V->cData->fbuff +
					(V->pt_y * V->cData->xdim) + V->pt_x));
				sprintf(buf, " % 1.*e", S_DIGITS - 8, fval);
			}
			xpos = (V->pt_x * SdrawWidth * S_DIGITS) - V->Scroll_x;
			ypos = (V->pt_y * SdrawHeight) + SdrawAscent - V->Scroll_y;
			XDrawImageString(myDpy, XtWindow(V->drawArea), textGC,
				xpos, ypos, buf, S_DIGITS);
			dmap = FindDHash(V->Doodles, V->pt_x, V->pt_y,
				&clipmask);
			if ((dmap != (Pixmap)0)&&(V->viewOverlay))
			{
				XSetClipMask(myDpy, drawAreaGC, clipmask);
				XSetClipOrigin(myDpy, drawAreaGC,
					xpos, (ypos - SdrawAscent));
				XCopyArea(myDpy, dmap, XtWindow(V->drawArea),
					drawAreaGC, 0, 0,
					(SdrawWidth * S_DIGITS), SdrawHeight,
					xpos, (ypos - SdrawAscent));
				XSetClipMask(myDpy, drawAreaGC, None);
			}
			XClearArea(myDpy, XtWindow(V->colwin),
				(xpos + SdrawWidth * (S_DIGITS - 1)), SdrawHeight,
				SdrawWidth, (SdrawHeight / 2), False);
			XClearArea(myDpy, XtWindow(V->rowwin),
				(SdrawWidth * (S_INDEX - 1)),
				(ypos - SdrawAscent + (SdrawHeight / 4)),
				SdrawWidth, (SdrawHeight / 2), False);
		}
		V->pt_x = -1;
		V->pt_y = -1;

		if (V->Mode == TEXTINSERT)
		{
			RealSheetExposed(V, V->text_x, (V->text_y - SdrawAscent),
				SdrawWidth, SdrawHeight);

			XFillRectangle(myDpy, XtWindow(V->drawArea), invGC,
				V->text_x, (V->text_y - SdrawAscent),
				SdrawWidth, SdrawHeight);
		}
	}
}


/*
 * Move the viewable portion of the spreadsheet, so that the
 * passed window coordinates are visible.
 */
void
MoveSheet(V, w, x1, y1)
	View *V;
	Widget w;
	int x1, y1;
{
	XWindowAttributes attr;

	XGetWindowAttributes(myDpy, XtWindow(V->drawArea), &attr);

	if ((x1 < 0)||(x1 >= attr.width)||(y1 < 0)||(y1 >= attr.height))
	{
		Arg argList[10];
		Cardinal argcnt;
		int oldsx, oldsy;

		oldsx = V->Scroll_x;
		oldsy = V->Scroll_y;
		V->Scroll_x = (x1 + oldsx) - (attr.width / 2) +
			(SdrawWidth * S_DIGITS);
		V->Scroll_y = (y1 + oldsy) - (attr.height / 2) +
			SdrawHeight;
		if (V->Scroll_x > ((V->cData->xdim * SdrawWidth * S_DIGITS) -
				attr.width))
		{
			V->Scroll_x = (V->cData->xdim * SdrawWidth * S_DIGITS) -
				attr.width;
		}
		if (V->Scroll_y > ((V->cData->ydim * SdrawHeight) - attr.height))
		{
			V->Scroll_y = (V->cData->ydim * SdrawHeight) - attr.height;
		}
		if (V->Scroll_x < 0)
		{
			V->Scroll_x = 0;
		}
		if (V->Scroll_y < 0)
		{
			V->Scroll_y = 0;
		}

		V->text_y = V->text_y + oldsy - V->Scroll_y;
		V->text_x = V->text_x + oldsx - V->Scroll_x;
		V->ret_x = V->ret_x + oldsx - V->Scroll_x;

		argcnt = 0;
		XtSetArg(argList[argcnt], XmNvalue, V->Scroll_y + 1); argcnt++;
		XtSetValues(V->vscroll, argList, argcnt);

		argcnt = 0;
		XtSetArg(argList[argcnt], XmNvalue, V->Scroll_x + 1); argcnt++;
		XtSetValues(V->hscroll, argList, argcnt);

		XClearArea(myDpy, XtWindow(V->rowwin), 0, 0, 0, 0, True);
		XClearArea(myDpy, XtWindow(V->colwin), 0, 0, 0, 0, True);
		XClearArea(myDpy, XtWindow(V->drawArea), 0, 0, 0, 0, True);
	}
}


void
InitDrawAreaSel(V, w, x, y)
	View *V;
	Widget w;
	int x, y;
{
	int col;
	int row;

	/*
	 * If their is already a selected area unselect it.
	 */
	UnSelect(V, AREASEL);
	V->areaSelect = True;

	/*
	 * Find the row in the dataset of the mouse click.
	 */
	row = (V->Scroll_y + y) / SdrawHeight;

	/*
	 * Find the column in the dataset of the mouse click.
	 */
	col = (V->Scroll_x + x) / (SdrawWidth * S_DIGITS);

	SheetSelOn(V, w, col, row);

	V->area_x1 = col;
	V->area_x2 = col;
	V->area_y1 = row;
	V->area_y2 = row;

	if (V->Mode == TEXTINSERT)
	{
		RealSheetExposed(V, V->text_x, (V->text_y - SdrawAscent),
			SdrawWidth, SdrawHeight);

		XFillRectangle(myDpy, XtWindow(V->drawArea), invGC,
			V->text_x, (V->text_y - SdrawAscent),
			SdrawWidth, SdrawHeight);
	}
}


void
DrawAreaSel(V, w, x, y)
	View *V;
	Widget w;
	int x, y;
{
	int col, row;
	int i, j;
	int oldx1, oldx2, oldy1, oldy2;
	int x1, x2, y1, y2;
	int xs, xe, ys, ye;
	int wx1, wx2, wy1, wy2;
	XWindowAttributes attr;

	XGetWindowAttributes(myDpy, XtWindow(V->drawArea), &attr);
	wx1 = V->Scroll_x / (SdrawWidth * S_DIGITS);
	wx2 = (V->Scroll_x + attr.width) / (SdrawWidth * S_DIGITS);
	wy1 = V->Scroll_y / SdrawHeight;
	wy2 = (V->Scroll_y + attr.height) / SdrawHeight;

	if (V->area_x1 <= V->area_x2)
	{
		oldx1 = V->area_x1;
		oldx2 = V->area_x2;
	}
	else
	{
		oldx1 = V->area_x2;
		oldx2 = V->area_x1;
	}

	if (V->area_y1 <= V->area_y2)
	{
		oldy1 = V->area_y1;
		oldy2 = V->area_y2;
	}
	else
	{
		oldy1 = V->area_y2;
		oldy2 = V->area_y1;
	}

	/*
	 * Find the row in the dataset of the mouse click.
	 */
	row = (V->Scroll_y + y) / SdrawHeight;
	if (row < 0)
		row = 0;
	if (row >= V->cData->ydim)
		row = V->cData->ydim - 1;

	/*
	 * Find the column in the dataset of the mouse click.
	 */
	col = (V->Scroll_x + x) / (SdrawWidth * S_DIGITS);
	if (col < 0)
		col = 0;
	if (col >= V->cData->xdim)
		col = V->cData->xdim - 1;

	if (col >= V->area_x1)
	{
		x1 = V->area_x1;
		x2 = col;
	}
	else
	{
		x1 = col;
		x2 = V->area_x1;
	}

	if (row >= V->area_y1)
	{
		y1 = V->area_y1;
		y2 = row;
	}
	else
	{
		y1 = row;
		y2 = V->area_y1;
	}

	if (oldx1 < x1)
	{
		xs = oldx1;
	}
	else
	{
		xs = x1;
	}
	if (x2 > oldx2)
	{
		xe = x2;
	}
	else
	{
		xe = oldx2;
	}

	if (oldy1 < y1)
	{
		ys = oldy1;
	}
	else
	{
		ys = y1;
	}
	if (y2 > oldy2)
	{
		ye = y2;
	}
	else
	{
		ye = oldy2;
	}

	for (j = ys; j <= ye; j++)
	{
	    if ((j < wy1)||(j > wy2))
	    {
		continue;
	    }
	    for (i = xs; i <= xe; i++)
	    {
		int inold, innew;

		if ((i < wx1)||(i > wx2))
		{
			continue;
		}
		inold = 0;
		innew = 0;
		if ((i >= oldx1)&&(i <= oldx2)&&(j >= oldy1)&&(j <= oldy2))
		{
			inold = 1;
		}
		if ((i >= x1)&&(i <= x2)&&(j >= y1)&&(j <= y2))
		{
			innew = 1;
		}
		if ((inold)&&(!innew))
		{
			SheetSelOff(V, w, i, j, AREASEL);
		}
		else if ((innew)&&(!inold))
		{
			SheetSelOn(V, w, i, j);
		}
	    }
	}

	for (j = y1; j <= y2; j++)
	{
		if ((j >= wy1)&&(j <= wy2))
		{
			SelRow(V, w, j);
		}
	}
	for (i = x1; i <= x2; i++)
	{
		if ((i >= wx1)&&(i <= wx2))
		{
			SelCol(V, w, i);
		}
	}

	V->area_x2 = col;
	V->area_y2 = row;

	if (V->Mode == TEXTINSERT)
	{
		RealSheetExposed(V, V->text_x, (V->text_y - SdrawAscent),
			SdrawWidth, SdrawHeight);

		XFillRectangle(myDpy, XtWindow(V->drawArea), invGC,
			V->text_x, (V->text_y - SdrawAscent),
			SdrawWidth, SdrawHeight);
	}
}



void
DoneDrawAreaSel(V, w, x, y)
	View *V;
	Widget w;
	int x, y;
{
	int temp;

	if ((V->area_x1 != -1)||(V->area_x2 != -1)||(V->area_y1 != -1)||(V->area_y2 != -1))
	{
		if (V->area_x2 < V->area_x1)
		{
			temp = V->area_x1;
			V->area_x1 = V->area_x2;
			V->area_x2 = temp;
		}
		if (V->area_y2 < V->area_y1)
		{
			temp = V->area_y1;
			V->area_y1 = V->area_y2;
			V->area_y2 = temp;
		}

		if (V->area_x1 < 0)
			V->area_x1 = 0;
		if (V->area_y1 < 0)
			V->area_y1 = 0;
		if (V->area_x2 >= V->cData->xdim)
			V->area_x2 = V->cData->xdim - 1;
		if (V->area_y2 >= V->cData->ydim)
			V->area_y2 = V->cData->ydim - 1;

		if (V->ispub)
		{
			NetSendAreaSelect(outP, V->cData->name, "HISTOGRAM",
					  V->area_x1, V->area_y1,
					  V->area_x2, V->area_y2);
		}
	}
}


void
SelRow(V, w, row)
	View *V;
	Widget w;
	int row;
{
	int ypos;

	/*
	 * Find the y position of the string in the current view window.
	 */
	ypos = (row * SdrawHeight) + SdrawAscent - V->Scroll_y;

	if (row < V->cData->ydim)
	{
		XCopyArea(myDpy, row_arrow, XtWindow(V->rowwin),
			drawAreaGC, 0, 0, SdrawWidth, (SdrawHeight / 2),
			(SdrawWidth * (S_INDEX - 1)), (ypos - SdrawAscent + (SdrawHeight / 4)));
	}
}

void
SelCol(V, w, col)
	View *V;
	Widget w;
	int col;
{
	int xpos;

	/*
	 * Find the x position of the string in the current view window.
	 */
	xpos = (col * SdrawWidth * S_DIGITS) - V->Scroll_x;

	if (col < V->cData->xdim)
	{
		XCopyArea(myDpy, col_arrow, XtWindow(V->colwin),
			drawAreaGC, 0, 0, SdrawWidth, (SdrawHeight / 2),
			(xpos + SdrawWidth * (S_DIGITS - 1)), SdrawHeight);
	}
}





/*
 * Highlights the number at the row and column passed.
 */
void
SheetSelOn(V, w, col, row)
	View *V;
	Widget w;
	int col, row;
{
	int xpos;
	int ypos;
	int val;
	float fval;
	Pixmap dmap;
	Pixmap clipmask;
	char buf[S_DIGITS + 1];

	/*
	 * Find the y position of the string in the current view window.
	 */
	ypos = (row * SdrawHeight) + SdrawAscent - V->Scroll_y;

	/*
	 * Find the x position of the string in the current view window.
	 */
	xpos = (col * SdrawWidth * S_DIGITS) - V->Scroll_x;

	/*
	 * If the click is withing the dataset, select the number.
	 * This is done by redrawing the number highlighted, restoring
	 * and doodles that were eresed by the redraw, and drawing
	 * pointer arrows in the row and column windows.
	 */
	if ((row < V->cData->ydim)&&(col < V->cData->xdim))
	{
		if (V->cData->type == D_CHAR)
		{
			val = (int)*((unsigned char *) (V->cData->buff +
				(row * V->cData->xdim) + col));
			sprintf(buf, "%*d", S_DIGITS, val);
		}
		else if (V->cData->type == D_INT)
		{
			val = (int)*((int *) (((int *)V->cData->fbuff) +
				(row * V->cData->xdim) + col));
			sprintf(buf, "%*d", S_DIGITS, val);
		}
		else if (V->cData->type == D_FLOAT)
		{
			fval = (float)*((float *) (V->cData->fbuff +
				(row * V->cData->xdim) + col));
			sprintf(buf, " % 1.*e", S_DIGITS - 8, fval);
		}
		xpos = (col * SdrawWidth * S_DIGITS) - V->Scroll_x;
		ypos = (row * SdrawHeight) + SdrawAscent - V->Scroll_y;
		XDrawImageString(myDpy, XtWindow(V->drawArea), invtextGC,
			xpos, ypos, buf, S_DIGITS);
		dmap = FindDHash(V->Doodles, col, row, &clipmask);
		if ((dmap != (Pixmap)0)&&(V->viewOverlay))
		{
			XSetClipMask(myDpy, drawAreaGC, clipmask);
			XSetClipOrigin(myDpy, drawAreaGC,
				xpos, (ypos - SdrawAscent));
			XCopyArea(myDpy, dmap, XtWindow(V->drawArea),
				drawAreaGC, 0, 0, (SdrawWidth * S_DIGITS), SdrawHeight,
				xpos, (ypos - SdrawAscent));
			XSetClipMask(myDpy, drawAreaGC, None);
		}
		XCopyArea(myDpy, col_arrow, XtWindow(V->colwin),
			drawAreaGC, 0, 0, SdrawWidth, (SdrawHeight / 2),
			(xpos + SdrawWidth * (S_DIGITS - 1)), SdrawHeight);
		XCopyArea(myDpy, row_arrow, XtWindow(V->rowwin),
			drawAreaGC, 0, 0, SdrawWidth, (SdrawHeight / 2),
			(SdrawWidth * (S_INDEX - 1)), (ypos - SdrawAscent + (SdrawHeight / 4)));
	}
}


/*
 * Unhighlights the number at the row and column passed.
 */
void
SheetSelOff(V, w, col, row, type)
	View *V;
	Widget w;
	int col, row;
	int type;
{
	int val;
	float fval;
	int xpos, ypos;
	Pixmap dmap;
	Pixmap clipmask;
	char buf[S_DIGITS + 1];

	if (type == AREASEL)
	{
		if ((col == V->pt_x)&&(row == V->pt_y))
		{
			return;
		}
	}
	else if (type == POINTSEL)
	{
		if ((col >= V->area_x1)&&(col <= V->area_x2)&&
		    (row >= V->area_y1)&&(row <= V->area_y2))
		{
			return;
		}
	}
	/*
	 * unselect the selected number..
	 * This is done by redrawing the number unhighlighted, restoring
	 * and doodles that were eresed by the redraw, and erasing the
	 * pointer arrows from the row and column windows.
	 */
	if ((col >= 0)&&(row >= 0))
	{
		if (V->cData->type == D_CHAR)
		{
			val = (int)*((unsigned char *) (V->cData->buff +
				(row * V->cData->xdim) + col));
			sprintf(buf, "%*d", S_DIGITS, val);
		}
		else if (V->cData->type == D_INT)
		{
			val = (int)*((int *) (((int *)V->cData->fbuff) +
				(row * V->cData->xdim) + col));
			sprintf(buf, "%*d", S_DIGITS, val);
		}
		else if (V->cData->type == D_FLOAT)
		{
			fval = (float)*((float *) (V->cData->fbuff +
				(row * V->cData->xdim) + col));
			sprintf(buf, " % 1.*e", S_DIGITS - 8, fval);
		}
		xpos = (col * SdrawWidth * S_DIGITS) - V->Scroll_x;
		ypos = (row * SdrawHeight) + SdrawAscent - V->Scroll_y;
		XDrawImageString(myDpy, XtWindow(V->drawArea), textGC,
			xpos, ypos, buf, S_DIGITS);
		dmap = FindDHash(V->Doodles, col, row, &clipmask);
		if ((dmap != (Pixmap)0)&&(V->viewOverlay))
		{
			XSetClipMask(myDpy, drawAreaGC, clipmask);
			XSetClipOrigin(myDpy, drawAreaGC,
				xpos, (ypos - SdrawAscent));
			XCopyArea(myDpy, dmap, XtWindow(V->drawArea),
				drawAreaGC, 0, 0, (SdrawWidth * S_DIGITS), SdrawHeight,
				xpos, (ypos - SdrawAscent));
			XSetClipMask(myDpy, drawAreaGC, None);
		}
		XClearArea(myDpy, XtWindow(V->colwin),
			(xpos + SdrawWidth * (S_DIGITS - 1)), SdrawHeight,
			SdrawWidth, (SdrawHeight / 2), False);
		XClearArea(myDpy, XtWindow(V->rowwin),
			(SdrawWidth * (S_INDEX - 1)), (ypos - SdrawAscent + (SdrawHeight / 4)),
			SdrawWidth, (SdrawHeight / 2), False);
	}
}


void
MakeCurrentSheet(V)
	View *V;
{
	Cdata *d;
	Arg argList[10];
	Cardinal i;
	int width, height;
	int Ww, Wh;
	XWindowAttributes attr;

	d = V->cData;

	if (!CdataSearchByName(d->name))
	{
		CdataAddEntry(d);
		RegisterData(d);
	}

	XGetWindowAttributes(myDpy, XtWindow(V->drawArea), &attr);

/*
	DrawAreaNewSize(d->xdim, d->ydim, attr.width, attr.height);
*/
	Ww = attr.width;
	Wh = attr.height;

	width = d->xdim * SdrawWidth * S_DIGITS;
	height = d->ydim * SdrawHeight;

	if (Ww > width)
	{
		Ww = width;
	}
	if (Wh > height)
	{
		Wh = height;
	}

	i = 0;
	XtSetArg(argList[i], XmNminimum, 1); i++;
	XtSetArg(argList[i], XmNmaximum, height); i++;
	XtSetArg(argList[i], XmNvalue, 1); i++;
	XtSetArg(argList[i], XmNsliderSize, (Wh - 1)); i++;
	XtSetValues(V->vscroll, argList, i);

	i = 0;
	XtSetArg(argList[i], XmNminimum, 1); i++;
	XtSetArg(argList[i], XmNmaximum, width); i++;
	XtSetArg(argList[i], XmNvalue, 1); i++;
	XtSetArg(argList[i], XmNsliderSize, (Ww - 1)); i++;
	XtSetValues(V->hscroll, argList, i);

	V->Scroll_x = 0;
	V->Scroll_y = 0;
	V->area_x1 = -1;
	V->area_y1 = -1;
	V->area_x2 = -1;
	V->area_y2 = -1;
	V->pt_x = -1;
	V->pt_y = -1;

	XClearArea(myDpy, XtWindow(V->rowwin), 0, 0, 0, 0, True);
	XClearArea(myDpy, XtWindow(V->colwin), 0, 0, 0, 0, True);
	XClearArea(myDpy, XtWindow(V->drawArea), 0, 0, 0, 0, True);

	if (d->hasPalette)
	{
		SetPalette(d->V, d->pal);
	}
}

