/*
 Copyright (c) 2003 RIPE

 All Rights Reserved

 Permission to use, copy, modify, and distribute this software and its
 documentation for any purpose and without fee is hereby granted,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of the author not be
 used in advertising or publicity pertaining to distribution of the
 software without specific, written prior permission.

 THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING
 ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS; IN NO EVENT SHALL
 AUTHOR BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY
 DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN
 AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

 $Id: conntrack.c,v 1.9 2003/01/28 10:54:21 can Exp $
*/

#include "conntrack.h"

static gint maxConnUnique = 1;
static gint maxConnUniqueWarn = 1;
static GStaticMutex connTrackMutex = G_STATIC_MUTEX_INIT;

/*
 * Utility function for automatically destroying hash table
 */
GDestroyNotify connTrackKeyValFree(gpointer data)
{
    g_free(data);
    return (NULL);
}

/*
 * Checks if a client can connect, records and returns NULL if yes.
 * Otherwise returns the error message to print.
 */
gchar *connCanConnect(GHashTable * conn, gchar * newConn)
{
    gint newVal;
    gpointer val;
    gchar *newConnZeroPadded, *result = NULL;

    newConnZeroPadded = sockInfo_zeropad(newConn);
    g_static_mutex_lock(&connTrackMutex);
    val = g_hash_table_lookup(conn, newConnZeroPadded);
    if (val == NULL) {
        if (maxConnUnique > 0) {
            g_hash_table_insert(conn, newConnZeroPadded, g_strdup("1"));
        } else {
            result = (gchar *) getConf("TOO_MANY_CONCURRENT_CONNECTIONS");
        }
    } else {
        newVal = atoi(val) + 1;
        g_hash_table_remove(conn, newConnZeroPadded);
        g_hash_table_insert(conn, newConnZeroPadded,
                            g_strdup_printf("%d", newVal));
        if (newVal > maxConnUnique) {
            result = "";
        } else if (newVal > maxConnUniqueWarn) {
            result = (gchar *) getConf("TOO_MANY_CONCURRENT_CONNECTIONS");
        }
    }
    g_static_mutex_unlock(&connTrackMutex);
    return (result);
}

/*
 * Must be called when client disconnects, so that the counter is decreased
 */
void connDisconnect(GHashTable * conn, gchar * newConn)
{
    gpointer val;
    gint newVal;
    gchar *newConnZeroPadded;

    newConnZeroPadded = sockInfo_zeropad(newConn);
    g_static_mutex_lock(&connTrackMutex);
    val = g_hash_table_lookup(conn, newConnZeroPadded);
    if (val != NULL) {
        newVal = atoi(val) - 1;
        g_hash_table_remove(conn, newConnZeroPadded);
        if (newVal > 0) {
            g_hash_table_insert(conn, g_strdup(newConnZeroPadded),
                                g_strdup_printf("%d", newVal));
        }
    } else {
        g_warning("in connDisconnect: hash table returns NULL for %s",
                  newConnZeroPadded);
    }
    g_static_mutex_unlock(&connTrackMutex);
    g_free(newConnZeroPadded);
    return;
}

/*
 * Initialize the connection tracking
 */
GHashTable *connTrackInit(void)
{
    GHashTable *result;

    result =
        g_hash_table_new_full(g_str_hash, g_str_equal,
                              (GDestroyNotify) & connTrackKeyValFree,
                              (GDestroyNotify) & connTrackKeyValFree);
    maxConnUnique = atoi((gchar *) getConf("MAXCONNUNIQUE"));
    maxConnUniqueWarn = atoi((gchar *) getConf("MAXCONNUNIQUEWARN"));
    return (result);
}

/*
 * Dispose of connection tracking
 */
void connTrackDone(GHashTable * conn)
{
    g_hash_table_destroy(conn);
}

/*
 * Log connections one by one
 */
GHFunc connLogConnection(gpointer key, gpointer value, gpointer user_data)
{
    g_message("current_conn from: %s concurrent: %s", (gchar *) key,
              (gchar *) value);
    return (NULL);
}

/*
 * Dump the contents of the hash
 */
void connTrackDump(GHashTable * conn)
{
    g_message("Contents of the current connection table follows");
    g_hash_table_foreach(conn, (GHFunc) & connLogConnection, NULL);
    g_message("Contents of the current connection table finished");
}
