%{
/*
     fc - Filter compiler software for the Drawbridge package
     Copyright (C) 1993 David K. Hess, Douglas Lee Schales, David R. Safford

     Please see the file `COPYING' for the complete copyright notice.
*/
#include <sys/types.h>
#include <stdio.h>
#include <memory.h>
#include <netdb.h>
#include <netinet/in.h>
#include "chario.h"
#include "hosts.h"
#include "groups.h"
#include "services.h"

extern int parse_error;

%}
     
%union {
     unsigned int int32;
     char *str;
     struct hostrange hostrange;
     struct service service;
}

%token DEFINE HOST NETWORK INCLUDE REJECT ALLOW
%token STRING IN OUT INOUT BYTEVAL INTEGER
%token SRC DST
%type <int32> netmask ip_address hostspec
%type <str> STRING IN OUT INOUT
%type <int32> inout dirspec
%type <int32> BYTEVAL
%type <int32> INTEGER integer
%type <hostrange> machine_spec
%type <service> service_spec service service_range
%type <int32> protocol
%%

file
        : stmt ';' file
        | stmt ';'
        | stmt error {
	     fprintf(stderr, "\"%s\", line %d: syntax error.\n",
		     getfilename(), getlinenum());
	}
;

stmt
	: /* */
        | host_def
	| group_def
	| network_def
	| include
        | reject
        | allow
;

reject
        : REJECT ip_address netmask {
	     add_reject($2, $3);
	}
;

allow
        : ALLOW ip_address netmask service_list {
	     add_allow($2, $3);
	}

group_def
	: DEFINE STRING service_list {
	     make_group($2); /* service stack has list of services */
	}
;

network_def
	: NETWORK machine_spec netmask service_list {
	     do_network($2, $3); /* service stack has list of services */
	}
;

host_def
        : HOST hostspec service_list {
	     if($2 == 0){
		  printf("Host == 0, %d\n", getlinenum());
	     }
	     else
		  do_host($2); /* service stack has list of services */
	}
;

include
	: INCLUDE STRING {
	     includefile($2);
	}
;

service_list
	: service_entry
	| service_list ',' service_entry
;

service_entry
	: '<' dirspec service_spec inout '>' {
	     push2_service($3, $4, $2, FP_ACC);
	}
        | '<' '!' dirspec service_spec inout '>' {
	     push2_service($4, $5, $3, FP_REJ);
	}
        | STRING {
	     push_group($1);
	     free($1);
	}
;

service_spec
     : service
     | service_range
;

service
     : integer {
	  $$.start = $$.end = $1;
	  $$.protocol = IPPROTO_TCP;
     }
     | integer '/' protocol {
	  $$.start = $$.end = $1;
	  $$.protocol = $3;
     }
     | STRING {
	  struct servent *se;
	  if((se = getservbyname($1, "tcp"))){
	       $$.start = $$.end = se->s_port;
	       $$.protocol = IPPROTO_TCP;
	  }
	  else {
	       fprintf(stderr, "\"%s\", line %d: unknown service: %s/tcp\n", 
		       getfilename(), getlinenum(),$1);
	       parse_error = 1;
	  }
     }
     | STRING '/' protocol {
	  struct servent *se;
	  struct protoent *pe;

	  if((pe = getprotobynumber($3)))
	       if((se = getservbyname($1, pe->p_name))){
		    $$.start = $$.end = se->s_port;
		    $$.protocol = $3;
	       }
	       else {
		    fprintf(stderr, "\"%s\", line %d: unknown service: %s/%s\n",
			    getfilename(), getlinenum(), $1, pe->p_name);
		    parse_error = 1;
	       }
	  else {
	       fprintf(stderr, "\"%s\", line %d: unknown service: %s/%d\n",
		       getfilename(), getlinenum(), $1, $3);
	  }
     }
;

service_range     
     : integer '-' integer {
	  $$.start = $1;
	  $$.end = $3;
	  $$.protocol = IPPROTO_TCP;
     }
     | integer '-' integer '/' protocol {
	  $$.start = $1;
	  $$.end = $3;
	  $$.protocol = $5;
     }
;

dirspec
        : SRC '=' { $$=FP_SRC; }
        | DST '=' { $$=FP_DST; }
        | /* */   { $$=FP_DST; }
;

protocol
     : integer {
	  $$ = $1;
     }
     | STRING {
	  struct protoent *pe;
	  if((pe = getprotobyname($1))){
	       $$ = pe->p_proto;
	  }
	  else {
	       fprintf(stderr, "\"%s\", line %d: unknown protocol: %s\n",
		       getfilename(), getlinenum(), $1);
	       parse_error = 1;
	  }
     }
     | error {
	  fprintf(stderr,
		  "\"%s\", line %d: expecting protocol\n",
		  getfilename(), getlinenum());
	  parse_error = 1;
     }
;

integer
     : INTEGER {
	  $$ = $1;
     }
     | BYTEVAL {
	  $$ = $1;
     }
;

inout
	: IN {
	     $$ = FP_IN+1;
	}
	| OUT {
	     $$ = FP_OUT+1;
	}
	| INOUT {
	     $$ = (FP_IN+1)|(FP_OUT+1);
	}
        | error {
	     fprintf(stderr,
		     "\"%s\", line %d: expecting one of 'in', 'out', or 'in-out'\n",
		     getfilename(), getlinenum());
	     parse_error = 1;
	}
;

hostspec
	: ip_address {
	     $$ = $1;
	}
	| STRING {
	     struct hostent *he;
	     struct {
		  int ipaddr:32;
	     } host;
	     if((he = gethostbyname($1))){
		  memcpy(&host, he->h_addr_list[0], 4);
		  $$ = host.ipaddr;
	     }
	     else {
		  fprintf(stderr, "\"%s\", line %d: unknown host: %s\n", 
			  getfilename(), getlinenum(), $1);
		  $$ = 0xFFFFFFFF;
		  parse_error = 1;
	     }
	}
;

machine_spec
	: ip_address {
	     $$.start = $$.end = $1;
	}
	| ip_address '-' ip_address {
	     $$.start = $1;
	     $$.end = $3;
	}
;

netmask
	: ip_address
;

ip_address
	: BYTEVAL '.' BYTEVAL '.' BYTEVAL '.' BYTEVAL {
	     $$ = htonl($1 << 24 | $3 << 16 | $5 << 8 | $7);
	}
;

