/*
 * Copyright (c) 1993,1995
 *	Texas A&M University.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Texas A&M University
 *	and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * Developers:
 *             David K. Hess, Douglas Lee Schales, David R. Safford
 */
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/fcntl.h>
#include <malloc.h>
#include <memory.h>
#include "chario.h"
#include "macros.h"
#include "hosts.h"
#include "services.h"

struct hosttable *hostlist = (struct hosttable *)0;

static struct reject *rejectlistbuf;
static int reject_count;
static int rejectlistsize;

static struct allow *allowlistbuf;
static int allow_count;
static int allowlistsize;

static int compare_addr(struct reject *, struct reject *);

extern int parse_error;

void
init_hosts(void)
{
     hostlist = (struct hosttable *)0;
     rejectlistsize = 8;
     rejectlistbuf = (struct reject *)
	  malloc(sizeof(struct reject)*rejectlistsize);
     if(!rejectlistbuf){
	  fprintf(stderr, "%s, line %d: malloc failed\n",
		  __FILE__, __LINE__);
	  exit(1);
     }
     reject_count = 0;
     
     allowlistsize = 8;
     allowlistbuf = (struct allow *)
	  malloc(sizeof(struct allow)*allowlistsize);
     if(!allowlistbuf){
	  fprintf(stderr, "%s, line %d: malloc failed\n",
		  __FILE__, __LINE__);
	  exit(1);
     }
     allow_count = 0;
}

unsigned int
getnetmask(unsigned int hostaddr)
{
     unsigned int result;
     if(IN_CLASSA(hostaddr))
	  result = IN_CLASSA_NET;
     else if(IN_CLASSB(hostaddr))
	  result = IN_CLASSB_NET;
     else if(IN_CLASSC(hostaddr))
	  result = IN_CLASSC_NET;
     return result;
}

static int
compare_addr(struct reject *x, struct reject *y)
{
	if ((x->ipaddr & x->netmask & y->netmask) == (y->ipaddr & y->netmask & x->netmask)) { 
		/* y is a child of x or vice versa. Sort by mask. Largest first. */
		return y->netmask - x->netmask;
	}
	else 
		return (x->ipaddr & x->netmask) - (y->ipaddr & y->netmask);
}

static int
allow_comp(struct allow *x, struct allow *y)
{
	/* printf("%08X %08X %08X %08X\n",x->ipaddr,x->netmask,y->ipaddr,y->netmask); */
	if ((x->ipaddr & x->netmask & y->netmask) == (y->ipaddr & y->netmask & x->netmask)) { 
		/* printf("yep\n"); */
		/* y is a child of x or vice versa. Sort by mask. Largest first. */
		return y->netmask - x->netmask;
	}
	else  { 
		/* printf("whoops\n"); */
		return (x->ipaddr & x->netmask) - (y->ipaddr & y->netmask);
	}
}

void
writehosttables(int byteswap)
{
     int fd;
     char *filename;
     struct hosttable *rove;
     struct in_addr ha;
     char buffer[20];

     for(rove=hostlist;rove;rove=rove->next){
	  unsigned int network = htonl(rove->network);
	  memcpy(&ha.s_addr, &network, 4);
	  if (byteswap) {
		  sprintf(buffer,"%08X.NET",rove->network);
		  filename = buffer;
	  }
	  else {
		  filename = inet_ntoa(ha); 
	  }
	  /* filename = inet_ntoa(htonl(rove->network)); */
	  if((fd = open(filename, O_WRONLY|O_CREAT|O_TRUNC, 0600)) != -1){
	       /*
	       network = ntohl(network);
	       if(byteswap){
		    unsigned long p = ltob(network);
		    write(fd, &p, 4);
	       }
	       else {
		    write(fd, &network, 4);
	       }
	       */
               if(byteswap)
                      network = ltob(network);
               write(fd, &network, sizeof(network));

	       /* No need to byteswap these */
	       write(fd, rove->table, rove->size);
	       close(fd);
	  }
	  else
	       perror(filename);
     }

     qsort(rejectlistbuf, reject_count, sizeof(struct reject), (void *) compare_addr);
     if (byteswap) {
	filename = "REJECT.TBL";
     }
     else {
	filename = "reject_table";
     }
     if((fd = open(filename, O_WRONLY|O_CREAT|O_TRUNC, 0600)) != -1){
            int i;
            unsigned long t, u;
            for(i=0;i<reject_count;i++){
                   t = htonl(rejectlistbuf[i].ipaddr);
                   u = htonl(rejectlistbuf[i].netmask);
                   if (byteswap)
                       t = ltob(t),
                       u = ltob(u);
                   write(fd, &t, sizeof(t));
                   write(fd, &u, sizeof(u));
                }
            close(fd);

	/*
	  if(byteswap){
	       int i;
	       unsigned long t, u;
	       for(i=0;i<reject_count;i++){
		    t = ltob(rejectlistbuf[i].ipaddr);
		    u = ltob(rejectlistbuf[i].netmask);
		    write(fd, &t, 4);
		    write(fd, &u, 4);
	       }
	  }
	  else {
	       write(fd, rejectlistbuf, reject_count * sizeof(struct reject));
	  }
	  close(fd);
	*/
     }
     else
	  perror("reject_table");

     qsort(allowlistbuf, allow_count, sizeof(struct allow), (void *) allow_comp);
     if (byteswap) {
	filename = "ALLOW.TBL";
     }
     else {
	filename = "allow_table";
     }
     if((fd = open(filename, O_WRONLY|O_CREAT|O_TRUNC, 0600)) != 01){
            int i, j;
            unsigned long t, u;
            unsigned short p;
            for(i=0;i<allow_count;i++){
                    t = htonl(allowlistbuf[i].ipaddr);
                    u = htonl(allowlistbuf[i].netmask);
                    if (byteswap)
                        t = ltob(t),
                        u = ltob(u);
                    write(fd, &t, sizeof(t));
                    write(fd, &u, sizeof(u));
                    for(j=0;j<CLASS_SIZE;j++){
                            p = htons(allowlistbuf[i].ports[j].start);
                            if (byteswap)
                                p = swab(p);
                            write(fd, &p, sizeof(p));
                            p = htons(allowlistbuf[i].ports[j].end);
                            if (byteswap)
                                p = swab(p);
                            write(fd, &p, sizeof(p));
                    }
            }
            close(fd);

	/*
	  if(byteswap){
	       int i, j;
	       unsigned long t, u;
	       unsigned short p;
	       for(i=0;i<allow_count;i++){
		    t = ltob(allowlistbuf[i].ipaddr);
		    u = ltob(allowlistbuf[i].netmask);
		    write(fd, &t, 4);
		    write(fd, &u, 4);
		    for(j=0;j<CLASS_SIZE;j++){
			 p = swab(allowlistbuf[i].ports[j].start);
			 write(fd, &p, 2);
			 p = swab(allowlistbuf[i].ports[j].end);
			 write(fd, &p, 2);
		    }
	       }
	  }
	  else {
	       write(fd, allowlistbuf, allow_count * sizeof(struct allow));
	  }
	  close(fd);
	*/
     }
     else
	  perror("allow_table");
}
 
void
do_host(unsigned int host)
{
     int class_entry;

     if(host == 0){
	  fprintf(stderr, "%s, line %d: internal error: do_host(host == 0)\n",
		  getfilename(), getlinenum());
	  return;
     }

     if(host != 0xFFFFFFFF){
	  struct hosttable *hrove;
	  unsigned int network;
	  unsigned int netmask;
	  unsigned int hostaddr;

	  netmask = getnetmask(host);
	  network = host & netmask;
	  for(hrove = hostlist;hrove;hrove=hrove->next)
	       if(network == hrove->network)
		    break;
	  if(!hrove){
	       int size;
	       hrove = (struct hosttable *)malloc(sizeof(struct hosttable));
	       if(!hrove){
		    fprintf(stderr, "%s, line %d: malloc failed\n",
			    __FILE__, __LINE__);
		    exit(1);
	       }
	       hrove->network = network;
	       size = (~netmask)+1;
	       hrove->size = size;
	       hrove->table = (unsigned char *)malloc(size);
	       if(!hrove->table){
		    fprintf(stderr, "%s, line %d: malloc failed\n",
			    __FILE__, __LINE__);
		    exit(1);
	       }
	       memset(hrove->table, 0, size);
	       hrove->next = hostlist;
	       hostlist = hrove;
	  }
	  class_entry = make_class();

	  hostaddr = host & (~netmask);
	  hrove->table[hostaddr] = class_entry;
     }

     destroy_all_services();
}

void
do_network(struct hostrange hr, unsigned int mask)
{
     struct hosttable *hrove;
     unsigned int network;
     unsigned int netmask;
     unsigned int hostaddr;
     unsigned int start, end;
     int class_entry;

     netmask = getnetmask(hr.start);
     network = hr.start & netmask;

     if (mask < netmask) {
	  fprintf(stderr, "%s, line %d: mask must be equal to or larger than network's class mask\n",
		  getfilename(), getlinenum());
	  parse_error = 1;
	  return;
     }

     for(hrove=hostlist;hrove;hrove=hrove->next)
	  if(network == hrove->network)
	       break;
     if(!hrove){
	  int size;
	  hrove = (struct hosttable *)malloc(sizeof(struct hosttable));
	  if(!hrove){
	       fprintf(stderr, "%s, line %d: malloc failed\n",
		       __FILE__, __LINE__);
	       exit(1);
	  }
	  hrove->network = network;
	  size = (~netmask)+1;
	  hrove->size = size;
	  hrove->table = (unsigned char *)malloc(size);
	  if(!hrove->table){
	       fprintf(stderr, "%s, line %d: malloc failed\n",
		       __FILE__, __LINE__);
	       exit(1);
	  }
	  memset(hrove->table, 0, size);
	  hrove->next = hostlist;
	  hostlist = hrove;
     }
     class_entry = make_class();

     if(hr.end == hr.start){
	  start = hr.start & ~netmask;
	  end = start + ~mask;
     }
     else {
	  start = hr.start & ~netmask;
	  end = hr.end & ~netmask;
     }

     for(hostaddr=start;hostaddr<=end;hostaddr++){
	  hrove->table[hostaddr] = class_entry;
     }
     destroy_all_services();
}


void
add_reject(unsigned int ipaddr, unsigned int mask)
{
     if(reject_count == rejectlistsize){
	  rejectlistsize *= 2;
	  rejectlistbuf = (struct reject *)
	       realloc(rejectlistbuf,
		       rejectlistsize*sizeof(struct reject));
	  if(!rejectlistbuf){
	       fprintf(stderr, "%s, line %d: realloc failed\n",
		       __FILE__, __LINE__);
	       exit(1);
	  }
     }
     rejectlistbuf[reject_count].ipaddr = ipaddr;
     rejectlistbuf[reject_count].netmask = mask;
     reject_count++;
}

void
add_allow(unsigned int ipaddr, unsigned int mask)
{
     struct classentry in_src[CLASS_SIZE];
     struct classentry in_dst[CLASS_SIZE];
     struct classentry out_dst[CLASS_SIZE];
     struct classentry udp_in[CLASS_SIZE];

     if(allow_count == allowlistsize){
	  allowlistsize *= 2;
	  allowlistbuf = (struct allow *)
	       realloc(allowlistbuf,
		       allowlistsize*sizeof(struct allow));
	  if(!allowlistbuf){
	       fprintf(stderr, "%s, line %d: realloc failed\n",
		       __FILE__, __LINE__);
	       exit(1);
	  }
     }
     allowlistbuf[allow_count].ipaddr = ipaddr;
     allowlistbuf[allow_count].netmask = mask;
     gen_class(in_src, in_dst, out_dst, udp_in);
     destroy_all_services();
     if(in_src[0].start || in_dst[0].start || udp_in[0].start)
	  fprintf(stderr, "INFO: \"%s\", line %d: 'allow' only supports 'out' TCP ports\n",
		  getfilename(), getlinenum());
     memcpy(allowlistbuf[allow_count].ports,
	    out_dst, CLASS_SIZE * sizeof(struct classentry));
     allow_count++;
}
