
/* Copyright (c) 1992 Vincent Cate
 * All Rights Reserved.
 *
 * Permission to use and modify this software and its documentation
 * is hereby granted, provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative works
 * or modified versions, and any portions thereof, and that both notices
 * appear in supporting documentation.  This software or any derivate works
 * may not be sold or distributed without prior written approval from
 * Vincent Cate.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND VINCENT CATE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL
 * VINCENT CATE BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Users of this software agree to return to Vincent Cate any improvements
 * or extensions that they make and grant Vincent Cate the rights to
 * redistribute these changes.
 *
 */


/* Bunch of small routines */

#include "alexincs.h"
#include "alex.h"

/* See alex.h before touching these */
char *ATYPES[] = {"AOK", "AFAIL",   "AFILE",      "ADIR",   "AERRORMESSAGE", "SHADOWFILE",
                         "ALINK",   "NOSUCHFILE", "AOLD",   "NOPWD",
                         "SOFTERROR", "AEOF",     "AHOST",  "AUPDATE", "AWORKING", "ANFSSTALE",
                         "NOSUCHHOST", "CONNECTING", "MAKELINKSTOPARENT", "MAKELINKSTOALEX",
                         "REMOVEALWAYS", "REMOVEMAYBE", "REMOVENEVER",
                         "ADOMAIN", "NOANONFTP", "NEVERDOFTP",
                         "UNIX",    "ABSUNIX",    "CDMODE", "RELMODE",
                         "DECVMS",  "IBM",        "TOPS20", "DECTEN", "HOSTALIAS",
                         "HPUX", "DECVMS2", "HENSA", "NOVEL", "UNKNOWN", NULL };

  

char LogBaseName[100]="/usr/alexsrvr/logs/alex.log";  /* changes */
char LogName[MAXPATH];

SetLogBaseName(s)
char *s;
{
    (void) strcpy(LogBaseName, s);
}

UnlinkLog()
{
    (void) unlink(LogName);
}


extern LogReal(s)
char *s;
{
    int Status;
    static int LogNum=0;
    static int NumLinesInLog=0;
    static char LogName[MAXPATH];
    int pid;

    if (NumLinesInLog > 60000) {            /* 20,000 is about 1 MB most of the time */
        if ((debugLog != NULL) && (debugLog !=stderr)) {
            fputs("\n\nClosing debugLog\n", debugLog);
            (void) fclose(debugLog);
            pid = vfork();
            if (pid == 0) {
                execl("/bin/nice", "nice", COMPRESSPATH, "-f", LogName, 0);
                fprintf(stderr, "Log  execl of compress failed ");
                _exit(-1);     /* exits ok for failed children only */
            }

            NumLinesInLog=0;
            LogNum++;
            debugLog=NULL;

#ifdef PERIODICDEATH 
            if (LogNum == 20) {
                (void) kill(getpid(), SIGHUP);     /* Alex believes in reincarnation, so  */
                                                   /* he thinks of death as a fine thing. */
            }    
#endif
        }
    }

    if (debugLog==NULL) {
        (void) sprintf(LogName, "%s.%d", LogBaseName, LogNum);
        debugLog=fopen(LogName, "w");
        if (debugLog!=NULL) {
           if (getuid() == (uid_t) 0) {
               (void) chown(LogName, ALEXUIDVAR, ALEXGIDVAR);    
           }
        } else {
            (void) fprintf(stderr,"ERROR Log can not open alex.log file %s\n\n", LogName);
            (void) fflush(stderr);
            if (!unlink(LogName)) {
                (void) fprintf(stderr,"ERROR Log can not unlink %s", LogName);
            }
            debugLog=fopen(LogName, "w");
            if (debugLog==NULL) {
                (void) fprintf(stderr,
                   "Can not open Log file %s - using stderr instead\n\n",
                    LogName);
                debugLog=stderr;
            }
        }
    }

    clearerr(debugLog);
    fputs(s, debugLog);
#ifndef FEWFFLUSH
    (void) fflush(debugLog);
#endif
    Status=ferror(debugLog);
    NumLinesInLog++;
    if (Status) {
        fprintf(stderr, "Log can not write to Log %s\n", LogName);
        abort();
    }
}

/* Thanks to hannken@eis.cs.tu-bs.de (Juergen Hannken-Illjes) for an example
 *      of using varargs which the following is based on.
 *
 * Main Logging function:
 *      First arg is a loging level
 *      Second is a format or string
 *      Existance and type of following depend on second arg
 * 
 * Processes variable argument list and calls LogReal() if debuglevel is enough
 */
/*VARARGS2*/               /* is there a portable way to declare type of first 2 args? */
extern ToLog(va_alist)
va_dcl
{
  va_list ap;
  int level;
  char buf[10000];
  char *fmt;

  va_start(ap);
  level = va_arg(ap,int);
  if (level>DEBUGLEVEL) {              /* is this above our cutoff?   */
      va_end(ap);
      return;                          /* if so just return           */
  }
  fmt = va_arg(ap,char *);
  vsprintf(buf,fmt,ap);
  LogReal(buf);                        /* else really send to the log */
  va_end(ap);
}



extern error1(s)
char *s;
{
     extern int sys_nerr;
     extern char *sys_errlist[];

     if (errno > 1 /* && errno < sys_nerr */) {
         ToLog(DBERROR, "ERROR %s : %s\n", s, sys_errlist[errno]);
     } else {
         ToLog(DBERROR, "ERROR BUG not a valid error number %s\n", s);
     }
     errno=0;

#ifdef FEWFFLUSH
     flushLog();
#endif 
 
}

extern error2(s1, s2)
char *s1, *s2;
{
     char tmp[1000];

     (void) sprintf(tmp,"%s %s", s1, s2);
     error1(tmp);
}

extern flushLog()
{
      (void) fflush(debugLog);
}


/*
 *  Input string with space separated tokens
 *  copy Nth token into token (first token is token 0)
 *  Return pointer to space or 0 that terminated the token
 */
extern char *GetNthToken(s, n, token)
char *s;
int n;
char *token;
{
    int i;

    if (s == NULL) {
        ToLog(DBERROR, "GetNthToken ERROR BUG should never get a null pointer");
        if (token != NULL) {
            *token=0;                               /* terminate token */
        }
        return(NULL);
    }

    while((*s != 0) && isspace(*s)) {       /* skip leading spaces if any */
        s++;
    }

    for (i=0; i<n; i++) {                      /* skip n tokens */

        if (*s == '"') {                       /* skip a quoted token */
            while ((*s != '"') && (*s != 0)) {
                s++;
            }
        } else {
            while((*s != 0) && !isspace(*s))   /* skip a plain token */
                s++;
        }

        while((*s != 0) && isspace(*s))        /* skip spaces */
            s++;
    }

    if (*s == '"') {
        s++;                                          /* skip the quote                    */
        while ((*s != 0)   && (*s != '"')) {          /* copy till quot or end             */
            *token++ = *s++;                                    
        }
        s++;                                          /* point to char after closing quote */
    } else {
        while ((*s != 0)   && !(isspace(*s))) {       /* copy till space or end */
            *token++ = *s++;
        }
    }
    *token=0;                               /* terminate token     */

    while((*s != 0) && isspace(*s)) {       /* skip any following spaces so caller knows when done*/
        s++;
    }
    return(s);                              /* so we coult start up again where we left off */
}


/*
 *   copy the next non-space stuff in s to token
 *   The return value is a pointer to the space after this token
 */

extern char *GetNextToken(s, token)
char *s, *token;
{
    return(GetNthToken(s, 0, token));
}

/*  Copies the last token from a string
 *
extern GetLastToken(ins, outs)
char *ins, *outs;
{
    int i, j;

    if ((ins == NULL) || (outs == NULL)) {
        Log("GetLastToken ERROR we hate NULL pointers");
        return;
    }

    i=strlen(ins);

    for(i--; (i>0) && isspace(ins[i]) ; i--) {       /* ignore spaces at the end * /
        ;
    }

    while( (i>0) && (!(isspace(ins[i]))) ) {         /* back up to start or space * /
        i--;
    }

    if (isspace(ins[i]))  i++;                       /* want stuff after this space * /

    for(j=0; (ins[i] != 0); j++) {
        outs[j] = ins[i++];                          /* copy out the token * /
    }

    outs[j]=0;                                       /* and terminate it * /
}
*/


extern int CountOccurances(s, c)
char *s, c;
{
    int Result;

    if (s==NULL) return(0);

    for( Result=0; (*s != 0); s++) {
        if (*s == c) Result++;
    }
    return(Result);
}



/*  Close all files except stdin, stdout, stderr
 */
CloseFDsAfter20()
{
    int i, nfds;

    fh_fdclose();

    nfds = getdtablesize();

    for (i=20; i<nfds; i++) {
        (void) close(i);            /* should mostly fail */
    }
}


#define VFOPENMAX 200
int   OpenCount=0;                     /* global only used by afopen and afclose */
FILE *FOpenFiles[VFOPENMAX];           /* these 2 should be a struct */
char FOpenPaths[VFOPENMAX][MAXPATH];   /* no malloc                  */

FILE *afopen(path, type, srcfile, srcline)
char *path, *type;
char *srcfile;
int  srcline;
{
    int i, j;

    if (OpenCount > STRANGEOPENCOUNT) {
        for (j=0; j<VFOPENMAX; j++) {
            if (FOpenFiles[j] != NULL) {
                Log2("afopen list ", FOpenPaths[j]);   
            }
        }
    }

    for (i=0; (i<VFOPENMAX) && (FOpenFiles[i] != NULL); i++) {        /* find a spot for next */
        ;
    }
    sprintf(FOpenPaths[i], "%13s %4d %s", srcfile, srcline, path);
    ToLog(DBOPEN, "fopen - OpenCount = %10d    %s\n", OpenCount, path);

    FOpenFiles[i]=fopen(path, type);

    if (FOpenFiles[i] != NULL) {
        OpenCount++;
    } else {
        if ((errno == EMFILE) || (errno == EMFILE)) {
            Log2("afopen too many open files - aborting ", path);
            abort();
        }
    }

    return(FOpenFiles[i]);
}


int afclose(file, srcfile, srcline)
FILE *file;
char *srcfile;
int  srcline;
{
     int Result;
     int i;

     if (file==NULL) {
         ToLog(DBERROR, "afclose error NULL file");
         return(AFAIL);
     }

     for (i=0; (i < VFOPENMAX) && (FOpenFiles[i] != file); i++) {
        ;
     }

     Result=EOF;
     if (file != FOpenFiles[i]) {
         ToLog(DBERROR, "afclose ERROR BUG file not in open list %s %d\n", srcfile, srcline);
         Result=fclose(file);
     } else {
         Result=fclose(file);                        /* we really did an open of this */
         if (Result == EOF) {
             ToLog(DBERROR, "afclose ERROR could not close %s -- %s %d \n", 
                                 FOpenPaths[i], srcfile, srcline);
         }
         ToLog(DBOPEN, "afclose - %s\n", FOpenPaths[i]); /* even if we did not close, is closed */
         FOpenFiles[i]=NULL;
         FOpenPaths[i][0]=0;     
         OpenCount--;
     }

     if (Result != AOK) {
         Result = AFAIL;
     }

     LogT("afclose returning", Result);
     return(Result);
}



/*
 *    For outputting types for debugging purposes
 */
extern char *ATypeToString(type)
int type;
{
    if ((AOK <= type) && (type <= UNKNOWN)) {
        return(ATYPES[type]);
    } else {
        ToLog(DBERROR, "ATypeToString ERROR BUG %d\n", type);
        return("ERROR ATypeToString notes buggy code");
    }
}

PrintTypes()
{
    int i;

    for (i=AOK; i<= UNKNOWN; i++) {
        LogN(ATypeToString(i), i);
    }
}

extern int StringToAType(s)
char *s;
{
    int type;

    for (type=0; type<= UNKNOWN; type++) {
        if (streql(ATYPES[type], s)) {
            return(type);
        }
    }

    if (streql("DOMAIN", s)) {       /* reverse compatability patch */
        return(ADOMAIN);
    }

    Log2("StringToAType ERROR no such type ", s);

    return(AFAIL);
}


/*            Input               Output
 *        
 *        /                     
 *        /foo/bar/             /foo/bar
 *        /foo/bar/baz          /foo/bar
 */
extern int PathToDir(Path, dir)
char *Path, *dir;
{
    int i;

    AlexAssert(Path[0]=='/');

    for(i=0; *dir++ = *Path++; i++) {      /* copy on way to end              */
        ;
    }
    dir--;                                 /* was pointing past the 0         */
                                           /* so now *dir==0 and i==strlen()  */

                                           /* change to i>1 for / to stay /   */
    while((i > 0) && (*dir != '/')) {      /* back up a through last part     */
        dir--;
        i--;
    }

    *dir = 0;

    return(AOK);
}

/*
 *   Input   /foo/bar/baz
 *   Output  baz
 */
extern int PathToFileName(Path, name)
char *Path, *name;
{
     Path=rindex(Path, '/');
     if (Path == 0) {
         return(AFAIL);
     }

     Path++;
     (void) strcpy(name,Path);
     return(AOK);
}

/*  Returns AOK if everything is.
 */
extern int TouchFile(Path)
char *Path;
{
    int Result;

    FILE *tmpfile;

    Log2("TouchFile ", Path);

    Result=AFAIL;
    tmpfile=AFOPEN(Path, "a");
    if (tmpfile != NULL) {
        Result=AOK;
        (void) AFCLOSE(tmpfile);
    } else {
        Log2("TouchFile could not open file", Path);
    }

    return(Result);
}



/*  Do an lstat and return  AFAIL, ADIR, AFILE, ALINK
 *
 *  Also a full status is updated
 */
extern int AlexStat(Path, StatBufPtr)
char *Path;
struct stat *StatBufPtr;
{
    int Status, Result, FileType;

    Status = lstat(Path, StatBufPtr);
    if (Status != 0) {
        Result=AFAIL;
    } else {
        FileType = StatBufPtr->st_mode & S_IFMT;
        if (FileType == S_IFDIR) {
            Result=ADIR;
        } else if (FileType == S_IFLNK) {
            Result=ALINK;
        } else {
            Result=AFILE;
        }
    }

    Log2(ATypeToString(Result), Path);
    return(Result);
}

/* 
 *    If the file exists 
 *        then we return the size in bytes, 
 *        else we return -1.
 */
extern int SizeOfCachedFile(Path)
char *Path;
{
    struct stat buf;                       /* not for use by caller */
    int Status;
    int Result;

    Status=AlexStat(Path, &buf);
    if (Status == AFILE) {
        Result=buf.st_size;
    } else {
        Result= -1;
    }

    LogN("SizeOfCachedFile ", Result);
    return(Result);
}

/* Only returns AFAIL, ADIR, AFILE, ALINK 
 */
extern int LiteStat(Path)
char *Path;
{
    struct stat buf;                       /* not for use by caller */

    return(AlexStat(Path, &buf));          /* just return result    */
}

/*  This returns Greenich time in seconds since Jan 1 1970 
 *  This is Unix/NFS standard
 */
extern double TimeInSeconds()
{
    struct timeval tp;
    struct timezone tzp;
    double Result;

    (void) gettimeofday(&tp, &tzp);

    Result= tp.tv_sec + (tp.tv_usec / 1000000.0);

    return(Result);
}


    

extern char *DateStr(seconds, buf)
time_t seconds;
char *buf;
{
    struct tm ug;

    ug = *(localtime(&seconds));

    fdate(buf,"%3Wday %3Month %2day %0hour:%min:%sec %year", &ug);

    return(buf);
}

/*  returns a pointer to a string with the current date in it
 */
extern char *CurrentDateStr()
{
    static char buf[100];
    struct timeval tp;
    struct timezone tzp;

    (void) gettimeofday(&tp,&tzp);
    DateStr(tp.tv_sec, buf);

    return(buf);
}



/*  Adds to the Log a string like:
 *      Wed Aug 21 02:14:44 1991
 */
extern double TimeStampLog(level)
int level;
{
    char buf[255];
    struct timeval tp;
    struct timezone tzp;
    double Result;

    (void) gettimeofday(&tp,&tzp);
    ToLog(level, "%s\n", DateStr(tp.tv_sec, buf));

    Result= tp.tv_sec + (tp.tv_usec / 1000000.0);
    return(Result);
}



/*  Returns 1 if s has Pattern in it (case insensitive) else 0.
 */
extern int HasStringNoCase(s, Pattern)
char *s, *Pattern;
{
    int len;

    len=strlen(Pattern);
    for (; (s!=NULL) && (*s != 0); s++) {
        if (strncasecmp(s, Pattern, len) == 0) {
            Log2("FOUND a PATTERN", s);
            return(1);
        }
    }
    return(0);
}

extern int HasString(s, Pattern)
char *s, *Pattern;
{
    int len;

    len=strlen(Pattern);
    for (; (s!=NULL) && (*s != 0); s++) {
        if (strncmp(s, Pattern, len) == 0) {
            Log2("FOUND a PATTERN", s);
            return(1);
        }
    }
    return(0);
}




extern int HasChar(s, c)
char *s;
char  c;
{
    if (c == 0) return(0);

    while ((*s != 0) && (*s != c)) {
       s++;
    }

    if (*s == c) {
        return(1);
    } else {
        return(0);
    }
}



extern int WhereString(s, Pattern)
char *s, *Pattern;
{
    int len, i;

    if (s==NULL) return(-1);

    len=strlen(Pattern);
    for (i=0; s[i] != 0; i++) {
        if (strncasecmp(&s[i], Pattern, len) == 0) {
            /* Log2("WhereString FOUND a PATTERN", &s[i]); */
            return(i);
        }
    }
    return(-1);
}

/*  Removes an extra trailing slash
 */
extern int RemoveTrailingSlash(Path)
char *Path;
{
    int len;

    len=strlen(Path);
    if (len>=2) {                             /* / ok but // would be bad */
        if ((Path[len-1] == '/') || (Path[len-1] == '\\' )) {
            Path[len-1] = 0;
            return(1);
        }
    }
    return(0);
}


/*  Expected to clip string at a CR
 *  returns AOK if it does.
 */
extern int ChopAtCR(s)
char *s;
{
    if (s==NULL) return(AFAIL);

    while((*s != 0) && (*s != '\n')) {
        s++;
    }

    if (*s == '\n') {
        *s = 0;
        return(AOK);
    }

    return(AFAIL);
}


extern void ToLower(s)
char *s;
{
    for (; *s != 0; s++) {
        if (isupper(*s)) {
            *s=tolower(*s);
        }
    }
}




extern int AlarmIgnore()
{
    return(0);
}


/*  Input:    Path=  "/edu/cmu/cs/sam"
 *  Output:   name=  "sam.cs.cmu.edu"
 *
 *  Does not return if input is bad.
 */
extern void PathToHostName(Path, name)
char *Path, *name;
{
    char LIFO[30][200];
    int seg, i;

    if ((Path == NULL) || (name==NULL)) {
        ToLog(DBERROR, "PathToHostName does not play will NULL pointers.\n");
        AlexAssert(FALSE);
    }

    seg=0;
    i=0;
    if (*Path != '/') {
        ToLog(DBERROR, "PathToHostName ERROR %s\n", Path);
        AlexAssert(FALSE);
    }

    for (; *Path != 0; Path++) {
        if (*Path == '/') {
            LIFO[seg][i]=0;
            seg++;
            i=0;
        } else {
            LIFO[seg][i] = *Path;
            i++;
        }
    }
    LIFO[seg][i]=0;

    for (; seg>=0; seg--) {
        (void) strcpy(name, LIFO[seg]);
        name += strlen(LIFO[seg]);
        *name++ = '.';
    }
    name--;
    name--;
    *name = 0;
}


/*  Input:     "sam.cs.cmu.edu"
 *  Returns:   "/edu/cmu/cs/sam"
 */
extern int HostNameToPath(Name, Path)
char *Name, *Path;
{
    char LIFO[20][100];
    int length, seg, outi, cur, Result;

    seg=0;
    outi=0;
    Result=AOK;

    length=strlen(Name);

    for (cur=0; cur<length; cur++) {
        if (Name[cur] == '.') {
            LIFO[seg][outi]=0;
            seg++;
            outi=0;
        } else {
            LIFO[seg][outi]=Name[cur];
            outi++;
        }
    }
    LIFO[seg][outi]=0;

    if (seg==0) Result=AFAIL;

    cur=0;
    for (; seg>=0; seg--) {
        Path[cur++]='/';
        (void) strcpy(&Path[cur], LIFO[seg]);
        cur += strlen(LIFO[seg]);
    }
    Path[cur]=0;

    Log2("HostNameToPath returning ", Path);
    return(Result);
}




extern int LineToTokens(lineptr, Tokens)
char *lineptr;
char Tokens[MAXTOKENS][MAXTOKENLEN];
{
    int i;

    for (i=0; i<MAXTOKENS; i++) {
        Tokens[i][0]=0;
    }

    for (i=0; (i<MAXTOKENS) && (lineptr != NULL) && (*lineptr != 0); i++) {
        lineptr=GetNextToken(lineptr, Tokens[i]);
    }

    return(i);
}




extern ClearParsedDirEntry(Entry)
struct ParsedDir *Entry;
{
    Entry->Name[0]=0;
    Entry->Type= -1;
#ifdef EXECUTABLE
    Entry->Executable = 0;
#endif
    Entry->Inode=0;
    Entry->Size=0;
    Entry->Date=0;
    Entry->SymLink[0]=0;
    Entry->Stale=0;
    Entry->CacheStatus=0;
    Entry->SizeTrusted=0;
    Entry->PartsInHostName=0;
}

unsigned int InodeNext;

/*  Output information about a file in standard form
 */
extern int OneLineOut(AlexInfoFile, Current)
FILE *AlexInfoFile;
struct ParsedDir *Current;
{
    int Status=0;

    if (Current->Name[0]==0) {
        ToLog(DBERROR, "OneLineOut ERROR zero length file name %d\n", Current->Inode);
        return(AFAIL);
    }

/*
 *   if ((Current->Inode == 0) || (Current->Inode == BOGUSINODE)) {
 *       ToLog(DBERROR, "OneLineOut ERROR BUG bogusinode %s %d\n", Current->Name, Current->Inode);
 *       return(AFAIL);
 *   }
 */

    if (Current->SymLink[0] != 0) {
        Status=fprintf(AlexInfoFile, "\"%s\" %d %9u %9u %9u \"%s\"\n",
                       Current->Name, Current->Type, Current->Size,
                       Current->Date, Current->Inode, Current->SymLink);
    } else {
#ifndef EXECUTABLE
        Status=fprintf(AlexInfoFile, "\"%s\" %d %9u %9u %9u \n",
                       Current->Name, Current->Type, Current->Size,
                       Current->Date, Current->Inode);
#else
        /* use a fake symlink "-1" to note the file executable */
        if (Current->Executable) {
             Status=fprintf(AlexInfoFile, "\"%s\" %d %9u %9u %9u \"-1\"\n",
                     Current->Name, Current->Type, Current->Size,
                     Current->Date, Current->Inode);
        } else {
             Status=fprintf(AlexInfoFile, "\"%s\" %d %9u %9u %9u \n",
                     Current->Name, Current->Type, Current->Size,
                     Current->Date, Current->Inode);
        }
#endif
    }

    if (Status==EOF) {
         ToLog(DBERROR, "ERROR OneLineOut could not write %s\n", Current->Name);
         return(AFAIL);
    } else {
         return(AOK);
    }
}



/*  If there is a trailing   "/.."  remove it.
 *  Return 1 if we removed something, otherwise 0
 */
extern int RemoveSlashDotDot(path)
char *path;
{
    if (strlen(path) < 4) {
        return(0);
    }
    while(*path!=0) {
       path++;
    }
    path-=3;
    if ((path[0]== '/') && (path[1]=='.') && (path[2]=='.')) {
        path--;
        while(*path != '/') {
           *path-- = 0;
        }
        *path = 0;         /* and the "/"    */
        return(1);
    }
    return(0);
}

/*  "./foo"        -->  "foo"  
 *  "foo/bar"      -->  "bar"
 *  "foo/bar/baz"  -->  "baz"
 */
extern int RemoveThroughSlash(s)
char *s;
{
    char *Name;

    Name=rindex(s, '/');

    if (Name != NULL) {                  /* if found a slash            */
        Name++;                          /* point to char after slash   */
        (void) strcpy(s, Name);          /* copy from there overwriting */
    }
}


/*  If there is a trailing   "/."  remove it.
 *  Return 1 if we removed something, otherwise 0
 */
extern int RemoveSlashDot(path)
char *path;
{
    while(*path!=0) {
       path++;
    }
    path-=2;
    if ((path[0]== '/') && (path[1]=='.')) {
        path[0]=0;
        return(1);
    }
    return(0);
}




/*  Destructive.
 *  Removes     /.   foo/..   and changes /foo/ to /foo
 *  Returns:
 *         ADIR     if we did any simplification
 *                  (note that any simplificaiton indicats path is a directory)
 *
 *         AFAIL    if did not start with CACHEDIRVAR
 */
extern int SimplifyPath(path)
char *path;
{
    int Result;

    if (strncmp(path, CACHEDIRVAR, strlen(CACHEDIRVAR)) != 0) {
        Log2("SimplifyPath working on out of bounds file", path);
/*        return(AFAIL);  */
    }


    Result=AOK;
    Result+=RemoveSlashDot(path);
    Result+=RemoveSlashDotDot(path);
    Result+=RemoveTrailingSlash(path);
    if (Result>0) {
        Result=ADIR;
        Log2("Simplified path a bit", path);
    }

    return(Result);
}



/*
 *  Return the time the file was last modified (unix seconds)
 */
extern double MTimeOfFile(Path)
char *Path;
{
    struct stat buf;
    double Result;

    if (lstat(Path, &buf)) {
        Result= -1.0;
    } else {
        Result= buf.st_mtime;
    }

    return(Result);
}


/*  How long has file been around?
 *  returns -1 if does not exist
 */
extern double SecsSinceWrite(Path)
char *Path;
{
    double Result, mt;

    mt=MTimeOfFile(Path);
    
    if (mt > 0) {
        Result=TimeInSeconds() - mt;
    } else {
        Result= -1;
    }

    LogN(Path, (int) Result);

    return(Result);
}

CheckLogDir()
{
    int Status, Result, FileType;
    struct stat StatBuf;                       

    Status = stat(LOGDIR, &StatBuf);

    if (Status != 0) {
        Result=AFAIL;
    } else {
        FileType = StatBuf.st_mode & S_IFMT;
        if (FileType == S_IFDIR) {
            Result=ADIR;
        } else {
            Result=AFILE;
        }
    }                               /* above does  Result ~= LiteStat(LOGDIR); without Log */

    switch (Result) {
        case ADIR:   Log("CheckLogDir LOGDIR was there ");
                     break;

        case AFILE:  fprintf(stderr, "Fatal ERROR LOGDIR %s is a file not a dir\n",LOGDIR);
                     exit(-1);
                   
        default:
        case AFAIL:  Status=mkdir(LOGDIR, 0755);
                     if (Status != 0) {
                          fprintf(stderr, "Fatal ERROR Can not make LOGDIR \n");
                          exit(-1);
                     }
                     Status= chown(LOGDIR, ALEXUIDVAR, ALEXGIDVAR);             /* should be roo
t now */
                     if (Status != 0) {
                          fprintf(stderr, "Fatal ERROR Can not chown LOGDIR %d\n", Status);
                          exit(-1);
                     }
                     Status= chmod(LOGDIR, 0755);                         /* should be root now*/
                     if (Status != 0) {
                          fprintf(stderr, "Fatal ERROR Can not chown LOGDIR %d", Status);
                          exit(-1);
                     }
                     Log("CheckLogDir  made LOGDIR");
                     break;
    }

    Log("CheckLogDir done");
}


int ALEXUIDVAR, ALEXGIDVAR;
double InitTime;

InitAlexUidVar()
{
    struct passwd *pw;

    if ((ALEXUID <0) || (ALEXGID<0)) {
        pw=getpwnam(ALEXSRVR);
        if (pw == NULL) {
            fprintf(stderr, "InitAlexUidVar Fatal ERROR could not run getpwnam");
            exit(-1);
        }
    }
    if (ALEXUID <0) {
        ALEXUIDVAR=pw->pw_uid;
    } else {
        ALEXUIDVAR=ALEXUID;
    }
    if (ALEXGID <0) {
        ALEXGIDVAR=pw->pw_gid;
    } else {
        ALEXGIDVAR=ALEXGID;
    }

    InitTime = TimeInSeconds();

}


SetUidGid()
{
    if ((ALEXUIDVAR < 0) || (ALEXGIDVAR < 0)) {
        fprintf(stderr, "ERROR Can not setgid");
        exit(-1);
    }

    if (setgid((gid_t) ALEXGIDVAR)) {
        fprintf(stderr, "ERROR Can not setgid");
        exit(-1);
    }

    if (setuid((uid_t) ALEXUIDVAR)) {
        fprintf(stderr, "ERROR Can not setuid");
        exit(-1);
    }

}


/* These are nearly constants.  Only loaded up right at Init time */
int     CACHEDIRLEN;          /*   10                             */
char    ROOTALEXINFOSTR[100]="";
char    CACHEDIRVAR[MAXPATH]="";  
char    SERVERHOSTNAME[100]="";          /* name of host running Alex */

InitReadOnlyVariables()
{
    int Status;
    static int AlreadyInited=0;
    char myname[100];
    struct hostent *hp;


    if (AlreadyInited) {
        Log("InitReadOnlyVariables already been called");
        return;
    }
    AlreadyInited=1;
   

    InitAlexUidVar();
    CheckLogDir();                  /* no "Logs" till after this */ 
    LogN("InitReadOnlyVariables ALEXUIDVAR ", ALEXUIDVAR);
    LogN("InitReadOnlyVariables ALEXGIDVAR ", ALEXGIDVAR);


    Log2("InitReadOnlyVariables    CACHEDIR", CACHEDIR);


    Status=LiteStat(CACHEDIR);
    switch (Status) {
        case ADIR:    (void) strcpy(CACHEDIRVAR, CACHEDIR);
                      break;

        case ALINK:   Status = readlink(CACHEDIR, CACHEDIRVAR, MAXPATH);
                      AlexAssert(Status > strlen("/alex-cache"));
                                                /* what we followed link to must end in /alex-cache */
                      CACHEDIRVAR[Status]=0;             /* is not terminated by readlink */
                      break;

        case AFAIL:  (void) strcpy(CACHEDIRVAR, CACHEDIR);
                      Status=mkdir(CACHEDIRVAR, 0755);
                      if (Status != 0) {
                          ToLog(DBERROR, "Fatal ERROR Can not make CACHEDIRVAR %d", Status);
                          exit(-1);
                      }
                      Status= chown(CACHEDIRVAR, ALEXUIDVAR, ALEXGIDVAR);             /* should be root now */
                      if (Status != 0) {
                          LogN("Fatal ERROR Can not chown CACHEDIRVAR", Status);
                          exit(-1);
                      }
                      Status= chmod(CACHEDIRVAR, 0755);                         /* should be root now */
                      if (Status != 0) {
                          LogN("Fatal ERROR Can not chown CACHEDIRVAR", Status);
                          exit(-1);
                      }
      

                      break;
        default:
                      Log("Fatal ERROR CACHEDIR not usable");
                      exit(-1);
    }

    Log2("InitReadOnlyVariables CACHEDIRVAR", CACHEDIRVAR);
    CACHEDIRLEN=strlen(CACHEDIRVAR);
    LogN("InitReadOnlyVariables  CACHEDIRLEN = ", CACHEDIRLEN);

    PathToDir(CACHEDIRVAR, ROOTALEXINFOSTR);
    (void) strcat(ROOTALEXINFOSTR, SLASHALEXINFO);
    Log2("ROOTALEXINFOSTR ", ROOTALEXINFOSTR);

    if(gethostname(myname, sizeof(myname)) < 0 || (hp = gethostbyname(myname)) == NULL) {
        sprintf(SERVERHOSTNAME, "UnkownHost");
    } else {
        sprintf(SERVERHOSTNAME, hp->h_name);
    }

    if(!HasString(SERVERHOSTNAME, ".")) {
        (void) strcat(SERVERHOSTNAME, ".");
        (void) strcat(SERVERHOSTNAME, LOCALUSERDOMAIN);   /* should not need this */
    }
    ToLower(SERVERHOSTNAME);

    Log("InitReadOnlyVariables is done");
}


extern int CopyAFile(From, To)
char *From, *To;
{
    FILE *FromFile, *ToFile;
    int Status;

    FromFile=fopen(From, "r");

    if (FromFile == NULL) {
        return(AFAIL);
    }

    ToFile=fopen(To, "w");

    if (ToFile == NULL) {
        return(AFAIL);
    }

    Status=ffilecopy(FromFile, ToFile);
    if (Status != AOK) {
        Log2("CopyAFile ERROR ", From);
        Status = AFAIL;
    }

    return(Status);
}


MallocDeath(s)
char *s;
{
    flushLog();
    error1(s);
    flushLog();
    ToLog(DBERROR, "MallocDeath ERROR BUG %s\n", s);
    flushLog();
    TimeStampLog(1);
    flushLog();

    abort();
}


/* Make a file named Path with just the string s in it 
 */
int StringIntoFile(Path, s)
char *Path, *s;
{
    FILE *File;
    int Result, Status;

    File=AFOPEN(Path, "w+");
    if (File == NULL) {
        error2("StringIntoFile could not open", Path);
        Result=AFAIL;
    } else {
        clearerr(File);

        while (*s != 0) {
            fputc(*s, File);
            s++;
        }
        fputc('\n', File);

        Status=ferror(File);

        Result=AFCLOSE(File);

        if ((Status != 0) || (Result != 0)) {
            error2("StringIntoFile could write to or close ", Path);
            Result=AFAIL;
        }
    }

    return(Result);
}


/*  Read a string from a file
 */
int StringFromFile(Path, s)
char *Path, *s;
{
    FILE *File;
    int Result, c;

    File=AFOPEN(Path, "r");
    if (File == NULL) {
        error2("StringFromFile could not open", Path);
        return(AFAIL);
    }

    c=fgetc(File);
    while (c != '\n' && c != 0 && c != EOF) {
        *s++=c;
        c=fgetc(File);
    }
    *s=0;

    Result=AFCLOSE(File);
    if (Result != 0) {
        error2("StringFromFile could not close", Path);
        Result=AFAIL;
    }

    return(Result);
}

/* 
 *  Returns AOK if the 2 files exist and have the same contents,
 *     else returns AFAIL.
 */
extern int FilesAreEqual(Path1, Path2)
char *Path1, *Path2;
{
    FILE *File1, *File2;
    int  Result, Status, c1, c2, Done;

    ToLog(10, "FilesAreEqual with %s and %s \n", Path1, Path2);

    File1=AFOPEN(Path1, "r");
    if (File1 == NULL) {
        Log2("FilesAreEqual could not open", Path1);
        return(AFAIL);
    }

    File2=AFOPEN(Path2, "r");
    if (File2 == NULL) {
        Log2("FilesAreEqual could not open", Path2);
        return(AFAIL);
    }

    Done=0;
    Result=AOK;                                           /* no difference so far */
    while (!Done) {
        c1=fgetc(File1);
        c2=fgetc(File2);
        if (c1 != c2) {
            Result=AFAIL;                                 /* a difference         */
        }
        if ((c1 == EOF) || (c2 == EOF)) {
            Done=1;
        }
    }

    Status=AFCLOSE(File1);
    if (Status != 0) {
        error2("FilesAreEqual could not close", Path1);
        Result=AFAIL;
    }

    Status=AFCLOSE(File2);
    if (Status != 0) {
        error2("FilesAreEqual could not close", Path2);
        Result=AFAIL;
    }

    LogT("FilesAreEqual returning ", Result);
    return(Result);
}

/* returns 0 if the last n characters of each string are the same.
 *   Used for things like comparing   cc.fsu.edu and naming.cc.fsu.edu
 *   to realize the symlink would be going lower in tree.
 */
extern int StrLastNCmp(s1, s2, n)
char *s1, *s2;
int n;
{
    int s1len, s2len, Result;

    s1len=strlen(s1);
    s2len=strlen(s2);
 
    if (s1len < n) {
        return(AFAIL);
    }

    if (s2len < n) {
        return(AFAIL);
    }

    Result=strncmp(&s1[s1len - n], &s2[s2len - n], n);

    if (Result != AOK)  {
        Result=AFAIL;
    }

    LogT("StrLastNCmp returning ", Result);
    
    return(Result);
}


/* Returns AOK if we can find a number in the string s (which we put into u), 
 *     else returns AFAIL
 */
extern int StringToUnsigned(s, u)
register char *s;
unsigned int  *u;
{
    register unsigned int n;
    int GotSomething;

    n = 0;
    GotSomething=0;

    while (isspace(*s)) {
        s++;
    }
    
    while(*s >= '0' && *s <= '9') {
        n = n*10 + *s++ - '0';
        GotSomething++;
    }

    *u=n;

    if (GotSomething > 0) {
        return(AOK);
    } else {
        return(AFAIL);
    }
}


/* remove as man of the c chars from the end of s as there are */
NoTrailingC(s, c)
char *s, c;
{
    while (*s != 0)  s++;

    s--;

    while (*s == c) {
        *s= 0;
        s--;
    }
}



/* Only returns if the malloc works and we have copied the string */
char *strsave(str)
char *str;
{
    int i;
    char *ptr;

    i=strlen(str)+1;
    ptr = malloc( (unsigned) i);
    if (ptr) {
        (void) strcpy(ptr,str);
    } else {
        MallocDeath("strsave");          /* does not return - exits */
    }
    return(ptr);
}


/*  must be a library that you can call so we don't do silly execl  XXXXX
 */
extern int RecursiveRm(DirPath, SrcFile, SrcLine, WaitTillDone)
char *DirPath;
char *SrcFile;
int   SrcLine;
int WaitTillDone;
{
    int pid, pid2;
    char RootPath[100];

    ToLog(DBMAJOR, "RecursiveRm Called by %s %d with %s\n", SrcFile, SrcLine, DirPath);
/*    if (!streql(SrcFile, "evictor.c")) {
 *       ToLog(DBMAJOR, "RecursiveRm only takes orders from evictor these days\n");
 *       return(AOK);       
 *   }
 */
    (void) strcpy(RootPath, CACHEDIRVAR);
    (void) strcat(RootPath, "/");

    if (strncmp(DirPath, RootPath, strlen(RootPath)) != 0) {
         ToLog(DBERROR, "RecursiveRm ERROR tried to rm -r something not below CACHEDIRVAR %s\n", DirPath);
         return(AFAIL);
    }

    if (HasString(DirPath, "..")) {
         ToLog(DBERROR, "RecursiveRm ERROR tried to rm -r something with .. in it %s\n", DirPath);
         return(AFAIL);
    }

    pid = fork();
    if (pid == 0) {
        execl("/bin/rm", "rm", "-rf", DirPath, 0);
        error1("execl of /bin/rm failed");
        _exit(-1);     /* exits ok for failed children only */
    }
    LogN("RecursiveRm pid of rm is ", pid);

    if (WaitTillDone) {
       for (pid2=wait((union wait *) 0); 
            (pid2 != pid) && (pid2 != -1); pid2=wait((union wait *) 0) ) {
            ToLog(DBERROR, "RecursiveRm ERROR did not get same pid after waiting %d\n", pid2);
       }
    }

    Log2("RecursiveRm done with", DirPath);
    return(AOK);
}



#define MAXSORTLINES  100000     /* 100,000 items in a directory max */
#define MAXSORTSTRING  500
static int
 pstrcmp(p1, p2)     /* compare strings through pointers */
char *p1, *p2;          /* void * for ANSI C */
{
        return strcmp(*(char **)p1, *(char **)p2);
}



/* reads in a file, sorts it, and writes it back out.
 */
extern int SortFile(Path)
char *Path;
{
    char *Line[MAXSORTLINES], s[MAXSORTSTRING];
    int  NumLines, i, Result;
    FILE *TmpFile;
    
    Log("SortTmpFile entering");

    TmpFile=AFOPEN(Path, "rw");

    if (TmpFile == NULL) {
        Log2("SortTmpFile could not open ", Path);
        return(AFAIL);
    } else {
        Log2("SortTmpFile has opened ", Path);
    }

    for (NumLines=0; fgets(s, MAXSORTSTRING, TmpFile) != NULL ; NumLines++) {
        if (NumLines>=MAXSORTLINES) {
            Log("SortTmpFile at MAXSORTLINES dropping rest of file");    /* says for each dropped */
        } else {
            Line[NumLines]=strsave(s);
        }
    }
   
    if (NumLines==0) {
        Log2("SortTmpFile has empty file", Path);
        Result=AFAIL;
    } else {
        LogN("SortTmpFile read in this many lines", NumLines);

        qsort((char *) Line, NumLines, sizeof(char *), pstrcmp);

        LogN("SortTmpFile done with sort", NumLines);
        rewind(TmpFile);
        for (i=0; (i<NumLines) && (fputs(Line[i], TmpFile) != NULL) ; i++) {
            free(Line[i]);
        }
        Log("SortTmpFile done with write");
        Result=AOK;
    }
    
    (void) AFCLOSE(TmpFile);
    LogT("SortTmpFile done ", Result);
    return(Result);
}



