
/* Copyright (c) 1992 Vincent Cate
 * All Rights Reserved.
 *
 * Permission to use and modify this software and its documentation
 * is hereby granted, provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative works
 * or modified versions, and any portions thereof, and that both notices
 * appear in supporting documentation.  This software or any derivate works
 * may not be sold or distributed without prior written approval from
 * Vincent Cate.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND VINCENT CATE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL
 * VINCENT CATE BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Users of this software agree to return to Vincent Cate any improvements
 * or extensions that they make and grant Vincent Cate the rights to
 * redistribute these changes.
 *
 */

/* Todo:
 *
 * Does not handle places with filenames with spaces.  It would not
 * be hard to do but then places that put extra stuff on the line would
 * mess up (but that seems more fair really).

 *
 *


 */


/* Exports:
 *    DirToInfo()    -   reads in a .alex.dir file and writes out a .alex.info
 */


#ifdef EXECUTABLE
/*
For security reasons I did not want to have executables show up as executable.
However, Durand wanted it and I said if he hacked it up I would include
the patchs.  Here are the comments from email from  durand alain denis 
<Alain.Durand@inria.fr> where he sent in EXECUTABLE hack.

The right way to do this is to add another field to .alex.info.  Too many 
things to do.

>Basically, the idea is to catch the "x" bit when converting from
>.alex.dir to .alex.info (in dirtoinfo.c, in XXXTokensToParsedDir),
>faking a link with a special "-1" linkname to keep track of the information
>(in alexlib.c OneLineOut) and then, when reading the .alex.info file,
>(in readinfo.c ParseOneLine) tracking down this fake link,
> marking the file executable (Current->Executable) and then giving the
>file the correct rights (ATypeToStatMode) according to its executable flag.
>
>At first, I wanted to add an field in the .alex.info files to store the
>executable state of the files, but then I couldn't figure out a way to
>restart alex... I tried to modify the HostList, RootAlexInfo,... accordingly
>but it didn't work. So I use this trick to fake a link "-1". This have also
>the advantage to stay compatible with the original version...
>

This is ugly, for sure.  XXXX
*/
#endif

#include "alexincs.h"
#include "alex.h"

#ifdef ONLYSHOWREADABLE
#define ONLYSHOWREADABLEINT     1
#else
#define ONLYSHOWREADABLEINT	0
#endif

/* returns 1 if not a normal file or directory 
 */
int SpecialFile(Current)
struct ParsedDir *Current;
{
    int Result;

    Result=(streql(Current->Name, ".")        ||
            streql(Current->Name, "..")       ||
            streql(Current->Name, ALEXERROR)  ||
            streql(Current->Name, ALEXUPDATE) ||
            streql(Current->Name, ALEXDIR)    ||
            streql(Current->Name, ALEXINFO));

    return(Result);
}

int SpecialFileOrError(Current)
struct ParsedDir *Current;
{
    int Result;

    Result=(SpecialFile(Current) || (Current->Type == AERRORMESSAGE));

    return(Result);
}


int StringToTimeInt(s, Now)
char *s;
struct timeb *Now;
{
    struct timeb  Then;
    int status, Result;

    Log2("StringToTimeInt input is  ", s);

    status=StringToTimeb(s, Now, &Then);

    if (status != AOK) {
        LogN("StringToTimeInt got back bad status and time of ", (int) Then.time);
        Result=Now->time;
    } else {
        Result=Then.time;
        if (Result > Now->time+60) {           
            LogN("StringToTimeInt  error future modification time after parsing ", Result);
            Log2("StringToTimeInt  error future ", s);
            LogN("Zone was ", (int) Then.timezone);
            if (Result < (Now->time + (Then.timezone*60))) {   /* Could it be Greenwich? */
                Log("StringToTimeInt XXX fudging timezone to Greenwich ");
                Result -= Then.timezone*60;
            } else {
                Log("StringToTimeInt setting time to Now");
                Result=Now->time - (Now->time % 60);            /* round back an hour          */
                                                    /* otherwise caching fails badly as file  */
                                                    /* looks like it is changing all the time */
            }
        }
    }

    LogN("StringToTimeInt returning ", Result);
    return(Result);
}

 


char Month[12][5] ={"Jan", "Feb", "Mar", "Apr", "May", "Jun",
                    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

int IsAMonth(token)
char *token;
{
    int i;

    for (i=0; i<12; i++) {
        if (strcasecmp(token, Month[i]) == 0) {
            return(1);
        }
    }

    /* Log2("Was expecting a month but got ", token); */
    return(0);
}





int IsThreeSepNums(token, sep)
char *token;
char sep;
{
    int n1, n2, n3, Result;

    n1=n2=n3=0;
    Result=1;
    while((*token != 0) && (isdigit(*token))) {
        n1=1;
        token++;
    }

    if (*token++ != sep) Result=0;

    while((*token != 0) && (isdigit(*token))) {
        n2=1;
        token++;
    }

    if (*token++ != sep) Result=0;
    
    while((*token != 0) && (isdigit(*token))) {
        n3=1;
        token++;
    }

    if (n1 && n2 && n3 && Result) {
        return(1);
    } else {
        return(0);
    }
}

/*  returns true if token is of the form     7/11/90 */
int IsSlashDate(token)
char *token;
{
    return(IsThreeSepNums(token, '/'));
}

/*  returns true if token is of the form     11:01:15 */
int IsTimeWithSecs(token)
char *token;
{
    return(IsThreeSepNums(token, ':'));
}

/* 
ABCODE   FIL      V         56       2669         16  7/11/90 11:01:15 PDH515
CGRID    GIF      V       8192          3          5  7/30/91  7:50:27 PDH515

FUSION   89-00012 V         80        299          3  5/03/89 14:27:15 FUSION
FUSION   89-00013 V         80       2346         28  5/03/89 14:30:35 FUSION

/alex/edu/columbia/cc/cuvmb
$$READ   ME       V         73         70          1 12/04/91 15:58:46 TCM301
$CUANON  DOC      V         72         33          1  2/15/89 17:56:46 TCM301

 */

int IsIBM(Tokens, NumTokens) 
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
{
    return((NumTokens >= 7) && IsSlashDate(Tokens[6]) && IsTimeWithSecs(Tokens[7]));
}


/*
/alex/uk/ac/hensa/micros
Files matching pattern '*' in directory '.' :

00contents                    719 bytes   4 Feb 1993 16:15
job.txt                      2005 bytes   4 Feb 1993 16:11
newsfile                     1240 bytes   4 Feb 1993 16:15


Directories matching pattern '*':

chest                    <DIRECTORY>     12 Feb 1993 14:25
cti                      <DIRECTORY>      3 Nov 1992 12:08
docs                     <DIRECTORY>     25 Nov 1992 14:45
tools                    <DIRECTORY>      3 Nov 1992 23:50
x                        <DIRECTORY>      3 Nov 1992 23:51
*/

int IsHENSA(Tokens, NumTokens)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
{
    if (NumTokens<1) return 0;
    if (streql(Tokens[0],"Files")) return 1;
    if (streql(Tokens[0],"Directories")) return 1;
    if (NumTokens>2 && streql(Tokens[1],"<DIRECTORY>")) return 1;
    if (NumTokens>3 && streql(Tokens[2],"bytes")) return 1;
    return 0;
}

/*
 *    Novel.  From /alex/edu/usu/netlab2:
- [R----F--] jrd                  1646       May 07 21:43    index
d [R----F--] jrd                   512       Aug 24 20:06    netwire
d [R----F--] jrd                   512       Aug 21 18:44    pktdrvr


      or /alex/com/wordperfect/ftp

d [R----F--] scotteh               512       Aug 24 20:05    msdos
- [R----F--] scotteh                74       Jun 29 17:12    read.me
0     1       2                     3         4   5  6        7
*/

int IsNOVEL(Tokens, NumTokens)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
{
    if (NumTokens != 8) return(0);
    if ((Tokens[0][0] != 'd') && (Tokens[0][0] != '-') || Tokens[0][1] != 0) return(0);
    if (Tokens[1][0] != '[') return(0);
    Log("IsNovel");
    return(1);
}

int NOVELTokensToParsedDir(Tokens, NumTokens, Current, Now)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
struct ParsedDir *Current;
struct timeb *Now;
{
    int Result;
    char TimeForParsing[200];

    LogN("NOVELTokensToParsedDir", NumTokens);

    (void) strcpy(Current->Name, Tokens[7]);          /* filename */

    if (Tokens[0][0] == 'd') {
        Current->Type=ADIR; 
    } else {
        Current->Type=AFILE; 
#ifdef EXECUTABLE
        Current->Executable = 0;                      /* Not Yet Implemented */
#endif
    }

    Current->Size=atoi(Tokens[3]);  

    (void) strcpy(TimeForParsing, Tokens[4]);
    (void) strcat(TimeForParsing, " ");
    (void) strcat(TimeForParsing, Tokens[5]);
    (void) strcat(TimeForParsing, " ");
    (void) strcat(TimeForParsing, Tokens[6]);
    Current->Date = (unsigned int) StringToTimeInt(TimeForParsing, Now);
    LogN("NOVELTokensToParsedDir Date ", Current->Date);

    if (strlen(Current->Name) > 1) {
        Result=AOK;
    } else {
        Result=AFAIL;
    }

    return(Result);
}

/*
PS:<ANONYMOUS>
    1752(07)  3-Nov-90 14:49:55 00-README.TXT.2
    1298(07) 30-Jun-90 20:18:15 ACCOUNTS.INFO.5
   10823(07) 24-Oct-87 17:05:57 BINTNXVMS.C.1
 */

int IsTOPS20(Tokens, NumTokens)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
{
    return((NumTokens >=3) && HasChar(Tokens[0], '(') && IsTimeWithSecs(Tokens[2]));  /* had '\(' XXX */
}


/*

/alex/il/ac/tau/vm
FUSION   89-00008 V         80       2281         26  5/03/89 14:25:54 FUSION
FUSION   89-00009 V         80       2061         24  5/03/89 14:26:18 FUSION
FUSION   89-00010 V         80       2025         21  5/03/89 14:26:41 FUSION
FUSION   89-00011 V         80       2264         25  5/03/89 14:27:08 FUSION
FUSION   89-00012 V         80        299          3  5/03/89 14:27:15 FUSION
FUSION   89-00013 V         80       2346         28  5/03/89 14:30:35 FUSION

/alex/edu/columbia/cc/cuvmb
$$READ   ME       V         73         70          1 12/04/91 15:58:46 TCM301
$CUANON  DOC      V         72         33          1  2/15/89 17:56:46 TCM301
$CULOCAL DOC      V         72         36          1  2/06/89 16:58:02 TCM301
$CUMXHOS DOC      V         73         47          1  9/05/89 13:17:52 TCM301

*/

int IBMTokensToParsedDir(Tokens, NumTokens, Current, Now)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
struct ParsedDir *Current;
struct timeb *Now;
{
    int Result;
    char TimeForParsing[200];

    LogN("IBMTokensToParsedDir", NumTokens);

    (void) strcpy(Current->Name, Tokens[0]);          /* basename */

    if (streql(Tokens[1], "DISK")) {                  /* extension */
        Current->Type=ADIR; 
    } else {
        Current->Type=AFILE; 
#ifdef EXECUTABLE
                Current->Executable = 0;        /* Not Yet Implemented */
#endif
        (void) strcat(Current->Name, ".");
        (void) strcat(Current->Name, Tokens[1]);
    }

    ToLower(Current->Name);

    Current->Size=atoi(Tokens[3]) * atoi(Tokens[4]);           /* just a guess */


    (void) strcpy(TimeForParsing, Tokens[6]);
    (void) strcat(TimeForParsing, " ");
    (void) strcat(TimeForParsing, Tokens[7]);
    Current->Date = (unsigned int) StringToTimeInt(TimeForParsing, Now);


    if (strlen(Current->Name) > 1) {
        Result=AOK;
    } else {
        Result=AFAIL;
    }

    return(Result);
}



int UNKNOWNTokensToParsedDir(Tokens, NumTokens, Current)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
struct ParsedDir *Current;
{
    int wasadir;

    LogN("UNKNOWNTokensToParsedDir", NumTokens);

    if (NumTokens < 1) return(AFAIL);

    if ((strcasecmp(Tokens[0], "Total") == 0) && (NumTokens <= 2)) {
        Log("UNKNOWNTokensToParsedDir skipping Total ");
        if (NumTokens == 2) {
            Log2("UNKNOWNTokensToParsedDir full line was     Total ", Tokens[1]);
        }
        return(AFAIL);
    }

    if ((NumTokens == 2) && (strcasecmp(Tokens[1], "unreadable") == 0)) {
        Log("UNKNOWNTokensToParsedDir combining unreadable ");
        (void) strcpy(Current->Name, "ALEX Message: ");
        (void) strcat(Current->Name, Tokens[0]);
        (void) strcat(Current->Name, " ");
        (void) strcat(Current->Name, Tokens[1]);
        Current->Type=AFILE;
        Current->Size=0;   
        return(AOK);                         /* maybe we play with protections someday  XXX */
    }

    (void) strcpy(Current->Name, Tokens[0]);
    wasadir=RemoveTrailingSlash(Current->Name);

    if (wasadir) {
        Current->Type=ADIR;
    } else {
        Current->Type=AFILE;
#ifdef EXECUTABLE
                Current->Executable = 0;        /* Not Yet Implemented */
#endif
    }

    Current->Size=1;            /* just one byte to get things started */

    if (Current->Name[0] != '/') {
        return(AOK);
    } else {   
        return(AFAIL);
    }
}

/*
DECVMS
ANNOUNCEMENTS.TXT;1        11   2-MAY-1991 15:31 [ANONYMOUS] (RWED,RWED,RE,)
00-DIRECTORY.FULL;2      4842  25-NOV-1992 19:21 AUI$TEX (RE,RWED,RE,RE)
                          ^-- 512 byte blocks
DECVMS2
.NEWSRC;1            10-OCT-1991 11:20:33    23606/47     (RWED,RWED,RE,)
                                                ^-- bytes


 */
int DECVMSTokensToParsedDir(Tokens, NumTokens, Current, Now)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
struct ParsedDir *Current;
struct timeb *Now;
{
    char TimeForParsing[MAXTOKENLEN];
    char *tmp;
    int loc, j, NumBlocks;

    LogN("DECVMSTokensToParsedDir", NumTokens);

    (void) strcpy(Current->Name, Tokens[0]);

    for (tmp=Current->Name; *tmp != 0; tmp++) {
        if (isupper(*tmp)) {
            *tmp=tolower(*tmp);
        }
        if (*tmp==';') {
            *tmp=0;                        /* we don't need no stinking semicolons */
        }
    }

    loc=WhereString(Current->Name, "]");
    if (loc>=0) {                          /* if has directory path get rid of it */
        for (j=0; j<(loc+1); j++) {
            Current->Name[j]=Current->Name[j+loc+1];
        }
    }

    loc=WhereString(Current->Name, ".dir");         /* use as type info but get rid of it */
    if (loc>=0) {
        Current->Name[loc]=0;                           /* chop him */
        Current->Type=ADIR;
    } else {
        Current->Type=AFILE;
#ifdef EXECUTABLE
                Current->Executable = 0;        /* Not Yet Implemented */
#endif

    }

    if (HasChar(Tokens[3], '/')) {                      /* is this DECVMS2 */
        (void) strcpy(TimeForParsing, Tokens[1]);
        (void) strcat(TimeForParsing, " ");
        (void) strcat(TimeForParsing, Tokens[2]);
        Current->Date = (unsigned int) StringToTimeInt(TimeForParsing, Now);
        Current->Size= atoi(Tokens[3]);             /* sizes almost right - modulo cr/lf -> lf */
    } else {
        (void) strcpy(TimeForParsing, Tokens[2]);
        (void) strcat(TimeForParsing, " ");
        (void) strcat(TimeForParsing, Tokens[3]);
        Current->Date = (unsigned int) StringToTimeInt(TimeForParsing, Now);
        NumBlocks=atoi(Tokens[1]);                  /* use blocks to estimate size  */
        if (NumBlocks > 1) {
            Current->Size=512*(NumBlocks-1);        /* much better to be low than high with NFS */
        } else {
            if (Current->Type==ADIR) {
                Current->Size=512;         
            } else {
                Current->Size=1;                    /* much better to be low than high with NFS */
            }
        }
    }

    return(AOK);
}


/*  Input:   "foo.bar"
 *  Output:  "foo"
 */
CutAtLastDot(s)
char *s;
{
    if (HasString(s, ".")) {
        while(*s != 0) s++;            /* go to end      */
        while(*s != '.') s--;          /* back up to '.' */
        *s=0;                          /* cut it there   */
    }
}


/*  Input  00README.TXT.1,7    
 *    or   MAC.DIRECTORY.1
 *
 *  Output: 00readme.txt       return(0)
 *    or    mac                return(1)
 */
int DECTENTokensToParsedDir(Tokens, NumTokens, Current)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
struct ParsedDir *Current;
{
    int i;

    LogN("DECTENTokensToParsedDir", NumTokens); 
    (void) strcpy(Current->Name, Tokens[0]);
 
    ToLower(Current->Name);

    i=WhereString(Current->Name, ".directory");
    if (i>=0) {
        Current->Name[i]=0;                               /* chop him */
        Current->Type=ADIR;
    } else {
        CutAtLastDot(Current->Name);
        Current->Type=AFILE;
#ifdef EXECUTABLE
        Current->Executable = 0;        /* Not Yet Implemented */
#endif
    }

    return(AOK);
}

int HENSATokensToParsedDir(Tokens, NumTokens, Current, Now)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
struct ParsedDir *Current;
struct timeb *Now;
{
    char TimeForParsing[MAXTOKENLEN];
    int i=0;

    LogN("HENSATokensToParsedDir", NumTokens);

    if (NumTokens<6) return (AFAIL);
    (void) strcpy(Current->Name, Tokens[0]);
    if (streql(Tokens[1],"<DIRECTORY>")) {
	Current->Type=ADIR;
	Current->Size=512;
        i=2;
    } else {
        if (streql(Tokens[2],"bytes")) {
             Current->Type=AFILE;
#ifdef EXECUTABLE
             Current->Executable = 0;   /* Not Yet Implemented */
#endif
             Current->Size=atoi(Tokens[1]);
             i=3;
        }
    }
    if (i) {
	sprintf(TimeForParsing,"%s-%s-%s %s",Tokens[i],Tokens[i+1],Tokens[i+2],Tokens[i+3]);
	Current->Date = (unsigned int) StringToTimeInt(TimeForParsing, Now);
	return(AOK);
    }
    return(AFAIL);
}


/*  Input:    00-README.TXT.2
 *
 *  Output:   00-readme.txt
 */

int TOPS20TokensToParsedDir(Tokens, NumTokens, Current)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
struct ParsedDir *Current;
{
    LogN("TOPS20TokensToParsedDir", NumTokens);

    (void) strcpy(Current->Name, Tokens[NumTokens+1]);
    ToLower(Current->Name);
    CutAtLastDot(Current->Name);
    Current->Type=AFILE;
#ifdef EXECUTABLE
        Current->Executable = 0;        /* Not Yet Implemented */
#endif

    return(AOK);
}

/*  Returns true if s is of the form "drwxr-xr-x"
 *
 *  Some day I should profile alex and see how slow and silly this is XXXX
 *     d  if the entry is a directory;
 *     b  if the entry is a block-type special file;
 *     c  if the entry is a character-type special file;
 *     l  if the entry is a symbolic link;
 *     s  if the entry is a socket, or
 *     -  if the entry is a plain file.
 *
 *     r  if the file is readable;
 *     w  if the file is writable;
 *     x  if the file is executable;
 *     -  if the indicated permission is not granted.
 */

/* This should become a fast table lookup XXXX */
int IsUnixProtections(s, OnlyIfReadable)
char *s;
int  OnlyIfReadable;
{
    int IsDir;

    /* if (strlen(s) != 10) return(0);       No because it fucks up below 
     *     drwxrwsr-x130 root     274          3072 Sep 12 08:37 pub         */

    if (strlen(s) < 10) return(0); 

    IsDir = (*s == 'd');

    if (!HasChar("dbcls-", *s++)) return(0);

    if (!HasChar("r-",  *s++)) return(0);
    if (!HasChar("w-",  *s++)) return(0);
    if (!HasChar("xsS-", *s++)) return(0);

    if (OnlyIfReadable) {       /* was  ifdef ONLYSHOWREADABLE */
        if (!HasChar("r",  *s++)) return(0);
        if (!HasChar("w-",  *s++)) return(0);
        if (IsDir) {
           if (!HasChar("xsS", *s++)) return(0);
        } else {
           if (!HasChar("xsS-", *s++)) return(0);
        }

        if (!HasChar("r",  *s++)) return(0);
        if (!HasChar("w-",  *s++)) return(0);
        if (IsDir) {
           if (!HasChar("xstS", *s++)) return(0);
        } else {
           if (!HasChar("xstS-", *s++)) return(0);
        }
    } else { 
        if (!HasChar("r-",  *s++)) return(0);
        if (!HasChar("w-",  *s++)) return(0);
        if (!HasChar("xsS-", *s++)) return(0);

        if (!HasChar("r-",  *s++)) return(0);
        if (!HasChar("w-",  *s++)) return(0);
        if (!HasChar("xstS-", *s++)) return(0);   
    }
    return(1);                          /* really looks like Unix */
}

IsUNIX(Tokens, NumTokens, OnlyIfReadable)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
int OnlyIfReadable;
{
    return((NumTokens > 2) && (IsUnixProtections(Tokens[0], OnlyIfReadable)));
}

IsHPUX(Tokens, NumTokens, OnlyIfReadable)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
int OnlyIfReadable;
{
    return((NumTokens > 2) && (IsUnixProtections(Tokens[1], OnlyIfReadable)));
}



int IsTime(s)
char *s;
{
    if (isdigit(s[0]) && isdigit(s[1]) && (s[2] == ':') && isdigit(s[3]) && isdigit(s[4])) {
        return(1);
    } else {
        return(0);
    }
}


/*
CLDATA:[ANONYMOUS_FTP.FILES]

00README.TXT;7              6   9-APR-1991 18:14 [ANONYMOUS] (RWED,RE,RE,RE)
ALEX.DIR;1                  8  18-OCT-1990 07:20 [ANONYMOUS] (RWED,RE,RE,RE)
ANNOUNCEMENTS.TXT;1        11   2-MAY-1991 15:31 [ANONYMOUS] (RWED,RWED,RE,)

*/

IsDECVMS(Tokens, NumTokens)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
{
    return((NumTokens >=4) && HasChar(Tokens[0], ';') && IsTime(Tokens[3])); 
}

/*
.NEWSRC;1            10-OCT-1991 11:20:33    23606/47     (RWED,RWED,RE,)
FTPD.LOG;2367        22-OCT-1991 21:08:36        0/0      (RWED,RWED,RE,)
M51X.GIF;2           29-NOV-1990 12:15:29   118327/232    (RWED,RWED,RE,RE)
M57.GIF;1            31-MAR-1991 11:38:21    47425/93     (RWED,RWED,RE,RE)
*/

IsDECVMS2(Tokens, NumTokens)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
{
    return ((NumTokens >= 4) &&  HasChar(Tokens[0], ';') && IsTime(Tokens[2])  && 
                  HasChar(Tokens[3], '/'));
}

  
       
/* 
 *  Returns 1 if is right type 
 */ 
int IsRightType(Tokens, NumTokens, TypeOfDir) 
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
int TypeOfDir;
{
    switch(TypeOfDir) {
        case UNIX:     return(IsUNIX(Tokens, NumTokens, ONLYSHOWREADABLEINT)); 
        case HPUX:     return(IsHPUX(Tokens, NumTokens, ONLYSHOWREADABLEINT)); 
        case DECVMS:   return(IsDECVMS(Tokens, NumTokens) || IsDECVMS2(Tokens, NumTokens)); 
        case HENSA:    return(IsHENSA(Tokens,NumTokens));
        case NOVEL:    return(IsNOVEL(Tokens,NumTokens));
#ifndef ONLYSAFEPARSE
        case IBM:      return(IsIBM(Tokens, NumTokens)); 
        case TOPS20:   return(IsTOPS20(Tokens, NumTokens));   
/*        case DECTEN:   return(IsDECTEN(Tokens, NumTokens)); */
        case UNKNOWN:   return(NumTokens > 0);
#endif
        default:        return(0); 
    }
}

int TokensToParsedDir(Tokens, NumTokens, TypeOfDir, Current, Now, DirPath, HostPath)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
int TypeOfDir;
struct ParsedDir *Current;
struct timeb *Now;
char *DirPath;
char *HostPath;
{
    int Result;
    char buf[100];

    switch(TypeOfDir) {
        case UNIX:     Result=UNIXTokensToParsedDir(Tokens, NumTokens, Current, Now, 
                                              DirPath, HostPath, 0);
                       break;
        case HPUX:     Result=UNIXTokensToParsedDir(Tokens, NumTokens, Current, Now, 
                                              DirPath, HostPath, 1);
                       break;
        case DECVMS:   Result=DECVMSTokensToParsedDir(Tokens, NumTokens, Current, Now);
                       break;
        case HENSA:    Result=HENSATokensToParsedDir(Tokens, NumTokens, Current, Now);
                       break;
        case NOVEL:    Result=NOVELTokensToParsedDir(Tokens, NumTokens, Current, Now);
                       break;

#ifndef ONLYSAFEPARSE
        case IBM:      Result=IBMTokensToParsedDir(Tokens, NumTokens, Current, Now);
                       break;
        case TOPS20:   Result=TOPS20TokensToParsedDir(Tokens, NumTokens, Current);
                       break;
        case DECTEN:   Result=DECTENTokensToParsedDir(Tokens, NumTokens, Current);
                       break;
#endif
        case UNKNOWN:  Result=UNKNOWNTokensToParsedDir(Tokens, NumTokens, Current);
                       break;
        default:
             AlexAssert(FALSE);
    }

    Log2("TokensToParsedDir Date", DateStr((time_t) Current->Date, buf));
    return(Result);
}


 


/*  Do this democratically.  We vote on it and whoever gets the most votes wins.
 */
int DetermineTypeOfDir(DirFile)
FILE *DirFile;
{
    char line[MAXPATH];
    int TypeVotes[MaxHostType+3];
    char Tokens[MAXTOKENS][MAXTOKENLEN];
    int i, Winner, InputOk, NumTokens, NumLines;

    for (i=0; i<=UNKNOWN; i++) {
        TypeVotes[i]=0;
    }

    InputOk=1; 
    NumLines=0;
    for (i=0; i<10 && InputOk; i++) {
        if (fgets(line, MAXPATH, DirFile) == NULL) {
            InputOk=0;
        } else {
            NumLines++;

            NumTokens=LineToTokens(line, Tokens);
            if (NumTokens>0) {                               
                if (IsUNIX(Tokens, NumTokens, 0)) {
                    TypeVotes[UNIX]++;

                } else if  (IsDECVMS(Tokens, NumTokens)) {
                    TypeVotes[DECVMS]++;

                } else if  (IsDECVMS2(Tokens, NumTokens)) {
                    TypeVotes[DECVMS]++;

                } else if  (IsHPUX(Tokens, NumTokens, 0)) {
                    TypeVotes[HPUX]++;

                } else if (IsHENSA(Tokens, NumTokens)) {
                    TypeVotes[HENSA]++;

                } else if (IsNOVEL(Tokens, NumTokens)) {
                    TypeVotes[NOVEL]++;

#ifndef ONLYSAFEPARSE
                } else if (IsTOPS20(Tokens, NumTokens)) {
                    TypeVotes[TOPS20]++;

                } else if  (IsIBM(Tokens, NumTokens)) {
                    TypeVotes[IBM]++;
#endif
                }

              /*  TypeVotes[DECTEN]+=IsDECTEN(Tokens, NumTokens);  */
            }
        }
    }        

    Winner=UNKNOWN;
    for (i=0; i<=UNKNOWN; i++) {
        if (TypeVotes[i] > TypeVotes[Winner]) { 
            Winner=i;
        }
    }

    LogT("DetermineTypeOfDir returning ", Winner);
    LogN("DetermineTypeOfDir winner had ", TypeVotes[Winner]);
    if (Winner==UNKNOWN) {
        LogN("DetermineTypeOfDir lines checked ", NumLines);
    }

    return(Winner);
}

/*
 *  Input "alexdir" is a path for a file of the form:

drwxr-xr-x  2 1092     tty           512 Mar 27 01:57 china-related
-r--r--r--  1 root     tty         20182 Nov  5  1988 virus.patch
-r--r--r--341 root     tty        220680 Nov  5  1988 foobar
lrwxr-xr-x  1 50       21             23 Aug 15 16:04 comp.sources.3b1 -> usenet/comp.sources.3b1
-rw-r--r--  2 vac      warp       61574 Aug 26 02:59 space-companies
drwxrwxr-x1693 7        21          39936 Aug 11 23:46 faces
drwxr-xr-x  2 vac      warp        1024 Aug 18 23:51 space-investors.mail
-rw-r--r--  1 1092     tty         28575 May  7 15:54 comm1.c
drwxrwsr-x  2 27847    101          9216 Jul 24 04:44 gif
drwxr-sr-x  2 27847    101           512 Nov 30  1990 gray
dr-xr-xr-x 23 ftp           daemon        512 Jun 20 15:41 pub
drwxrwxrwx  2 ftp           none          512 May 21 01:37 tmp
lrwxrwxrwx  1 root     wheel           1 May 29 11:33 ftp -> .
drwxrwxrwx   2 3             80 Jul 19 11:33 incomming
dr-xr-xr-x   4 3            240 Oct 10  1990 pub
lrwxrwxrwx  1 root     wheel          11 May 29 11:30 news.pub -> .r/news.pub
            0  1        2              3  4  5   6     7
                                                  .   
                                                  50 

HPUX:
dir        drwxrwxrwx+  2 librarianftp          1024 Feb  2 16:35 drop_box

 *  "name" is a string like "virus.patch"
 *
 */

UNIXTokensToParsedDir(Tokens, NumTokens, Current, Now, DirPath, HostPath, ProtIndex)
char Tokens[MAXTOKENS][MAXTOKENLEN];
int NumTokens;
struct ParsedDir *Current;
struct timeb *Now;
char *DirPath, *HostPath;
{
    int LookingForMonth, MonthIndex, i, Result;
    char TimeForParsing[MAXTOKENLEN];
    char FilePath[MAXPATH];

    LookingForMonth=1;
    for (i=2; (i<NumTokens) && LookingForMonth; i++) {
        if (IsAMonth(Tokens[i])) {
            LookingForMonth=0;
            MonthIndex=i;
        }
    }

    if (LookingForMonth) {
        Log2("UNIXTokensToParsedDir ERROR no month", Tokens[0]);
        return(AFAIL);
    }

    if (MonthIndex+3 >= NumTokens) {
        LogN("UNIXTokensToParsedDir ERROR not enough tokens", NumTokens);
        return(AFAIL);
    }

    Current->Size = atoi(Tokens[MonthIndex-1]);                 /* on Unix we get a real size */

    (void) strcpy(TimeForParsing, Tokens[MonthIndex]);          /* month */
    (void) strcat(TimeForParsing, " ");
    (void) strcat(TimeForParsing, Tokens[MonthIndex+1]);        /* day */
    (void) strcat(TimeForParsing, " ");
    (void) strcat(TimeForParsing, Tokens[MonthIndex+2]);        /* time or year */

    Current->Date = (unsigned int) StringToTimeInt(TimeForParsing, Now);

    (void) strcpy(Current->Name, Tokens[MonthIndex+3]);         /* name */

    RemoveThroughSlash(Current->Name);                          /* "./foo" --> "foo"  */

    Result=AOK;
    switch(Tokens[ProtIndex][0]) {
        case 'd': Current->Type=ADIR;   
#ifdef ONLYSAFEPARSE
                  if (NumTokens > MonthIndex+4) {
                      LogN("UNIXTokensToParsedDir ERROR too many tokens", NumTokens);
                      Result=AFAIL;
                  }
#endif
                  break;

        case '-': Current->Type=AFILE;  
#ifdef EXECUTABLE
                if (Tokens[ProtIndex][3] == 'x') { /* -rwx */
                    Current->Executable = 1;
                } else {
                    Current->Executable = 0;
                }
#endif

#ifdef ONLYSAFEPARSE
                  if (NumTokens > MonthIndex+4) {
                      LogN("UNIXTokensToParsedDir ERROR too many tokens", NumTokens);
                      Result=AFAIL;
                  }
#endif
                  break;

        case 'l': Current->Type=ALINK; 
                               if (MonthIndex+5 >= NumTokens) {
                                   LogN("UNIXTokensToParsedDir ERROR not enough tokens", NumTokens);
                                   Result=AFAIL;
                               } else {
                                   (void) strcpy(Current->SymLink, Tokens[MonthIndex+5]);
                                   (void) strcpy(FilePath, DirPath);
                                   (void) strcat(FilePath, "/");
                                   (void) strcat(FilePath, Current->Name);
                                   Result=OkSymLink(FilePath, Current->SymLink, HostPath);
                               }
                                        break;
        default:
            Log("UNIXTokensToParsedDir ERROR did not understand file type");
            Result=AFAIL;
            break;
    }

                                            /* should never happen any more  XXXX  */
    if (Current->Name[0] == '/') {
         Log2("UNIXTokensToParsedDir ERROR can not deal with leading / ", Current->Name);
         Result=AFAIL;
    }

    LogT("UNIXTokensToParsedDir returning ", Result);
    return(Result);
}


#if (defined(__386BSD__) || defined(ALPHAOSF))
/*  Emulate this.
 */
void ftime(struct timeb *Now)
{
  time_t _now=time(NULL);
  struct tm *_Now=localtime(&_now);
  Now->time=_now;
  Now->millitm=0;
  Now->timezone=_Now->tm_gmtoff/60;
  Now->dstflag=_Now->tm_isdst;
}
#endif


/*  Caller will call at most once per .alex.info even with failure.
 */
int GetRemoteNow(HostPtr, Now)
struct HostEntry *HostPtr;
struct timeb *Now;
{
    char Name[100];
    int TimeZone;
    int OldTimeZone;
    double SecsSinceChecked;

    AlexAssert(HostPtr != NULL);

    ftime(Now);

    OldTimeZone=HostPtr->TimeZone;
    TimeZone=HostPtr->TimeZone;

    SecsSinceChecked = TimeInSeconds() - HostPtr->TimeLastCheck;

    if (((TimeZone >= RemTimeZoneFAIL) && (SecsSinceChecked > 300)) || 
                       ((TimeInSeconds() - HostPtr->TimeLastCheck) > TRUSTTIMEZONE))  {
        PathToHostName(HostPtr->Path, Name);
        HostPtr->Type = UNKNOWN;         /* next time we DoAnFtp we will check type */
        TimeZone=RemTimeZone(Name);
        TotalRTZCalls++;                 /* perf stuff */
        HostPtr->TimeZone = TimeZone;    /* Save in cache - could be RemTimeZoneFAIL   */
        HostPtr->TimeLastCheck = (int) TimeInSeconds();
    }


    if (TimeZone >= RemTimeZoneFAIL) {      /* failed so ignore value */
        Now->timezone=OldTimeZone;          /* return old zone        */
    } else {
        Now->timezone=TimeZone;             /* return real one */
    }

    LogN("GetRemoteNow returning ", (int) Now->timezone);
    return(AOK);
}



char *CopyTillSlash(in, out)
char *in;
char *out;
{
    char *tmp;
   
    tmp=out; 
    while ((*in != 0) && (*in != '/')) {
        *tmp++ = *in++;
    }
    *tmp=0;

    return(out);
}



/*  In the case of a HostAlias we need to get the symlink 
 */
int HostToParsedDir(Host, HostName, Current)
struct HostEntry *Host;
char *HostName;
struct ParsedDir *Current;
{
    int Result;

    ClearParsedDirEntry(Current);
    (void) strcpy(Current->Name, HostName);

    Result=AOK;
    Current->SymLink[0]=0;                                   /* set up defaults */
    if (Host->Type != HOSTALIAS) {
        Current->Type=AHOST;                                 /* Any host -> AHOST (mostly ADIR) */
    } else {
        Current->Type=ALINK;                                 /* HOSTALIASes are ALINKs     */
        AlexAssert(Host->SymLink != NULL);                   /* We check when read in file */
        (void) strcpy(Current->SymLink, Host->SymLink);
    }

    Current->Inode=0;
    Current->Size=512;
    Current->Date=TimeInSeconds();

    return(Result);
}



int OutputHostsBefore(AIfile, NextHost, DirPWS, DPWSlen, Name)
FILE *AIfile;
int NextHost;
char *DirPWS;
int DPWSlen;
char *Name;
{
    static char LastHostName[100];
    static int  LastNextHost=-1;

    char HostName[100], FullName[MAXPATH];
    struct ParsedDir Current;
    int Status, Done;

    (void) strcpy(FullName, DirPWS);             /* dir path with slash */
    (void) strcat(FullName, Name);

    Log2("OutputHostsBefore with ", FullName);

    if (NextHost != LastNextHost) {              /* need to know if continuing or startin over */
        LastHostName[0]=0;
        LastNextHost=NextHost;
    }

    Done=0;
    for( ; !Done && NextHost<HostsInTable; ) {
        Status = strcmp(HostsTable[NextHost]->Path, FullName);
        if (Status >= 0) {                       
            Done=1;
        } else {
            Status=strncmp(HostsTable[NextHost]->Path, DirPWS, DPWSlen);
            if (Status == 0) {                                             /* is in dir */
                CopyTillSlash(&(HostsTable[NextHost]->Path[DPWSlen]), HostName);

                if (!streql(HostName, LastHostName)) {
                    Log2("OutputHostsBefore found ", HostName);
                    Status=HostToParsedDir(HostsTable[NextHost], HostName, &Current);
                    if (Status == AOK) {
                        if ((HostsTable[NextHost]->Type == HOSTALIAS) &&
                            (HostsTable[NextHost]->PathLen > DPWSlen+strlen(HostName))) {
                            Current.Type=ADOMAIN;
                            Current.SymLink[0]=0;
                        }
                        OneLineOut(AIfile, &Current);
                    }
                    (void) strcpy(LastHostName,HostName);
                }

            }
            NextHost++;
        }
    }
            
    LogN("OutputHostsBefore returning ", NextHost);
    LastNextHost=NextHost;

    return(NextHost);
}


/*  If there is a .alex.error in DirPath output it to the Sream AIfile
 *     in the correct format for a .alex.info.
 *
 *  Having an AERRORMESSAGE should shorten the UpdateDate or it should be ignored
 * 
 *  Return:
 *            AOK - no .alex.error file found 
 *  AERRORMESSAGE - added an error message to stream
 *          AFAIL - error occurred
 *
 */
int AddAnyErrorToStream(AIfile, DirPath)
FILE *AIfile;
char *DirPath;
{
    char AlexErrorPath[MAXPATH];
    struct ParsedDir Current;
    int Status;

    (void) strcpy(AlexErrorPath, DirPath);
    (void) strcat(AlexErrorPath, SLASHALEXERROR);

    Status=AOK;
    if (LiteStat(AlexErrorPath) == AFILE) {
        ClearParsedDirEntry(&Current);

        Status=StringFromFile(AlexErrorPath, Current.Name);
        if (Status != AOK) {
            ToLog(DBERROR, "AddAnyErrorToStream ERROR BUG .alex.error empty %s\n", AlexErrorPath);
        } else {
            Log2("AddAnyErrorToStream got error string ", Current.Name);
            Current.Inode=InodeNext++;                           /* allocate a new inode */
            Current.Size=0;
            Current.Date=TimeInSeconds();
            Current.Type=AERRORMESSAGE;
            Current.CacheStatus=0;
            Current.SymLink[0]=0;
            Status=OneLineOut(AIfile, &Current);
            if (Status == AOK) {
                Status = AERRORMESSAGE;
            } 
        }
    }

    LogT("AddAnyErrorToStream  returning ", Status);
    return(Status);
}

/*  Returns 0 if none of parents are same,
 *    else number of ../ to have in symlink.
 */
int NumUpForSameDir(DirPath, ParentDirPath, DirName)
char *DirPath, *ParentDirPath, *DirName;
{
    char WorkingDirPath[MAXPATH], Tmp[MAXPATH];
    char ParentDirName[MAXPATH];
    char ParentAlexDir[MAXPATH], AlexDir[MAXPATH];
    int  Result, NumUp;

    Result=0;
    NumUp=1;

    (void) strcpy(WorkingDirPath, ParentDirPath);

    while (!Result && strlen(WorkingDirPath) > CACHEDIRLEN && HasString(WorkingDirPath, DirName)) {
        ToLog(6, "NumUpForSameDir checking for looping %s\n", DirPath);
        PathToFileName(WorkingDirPath, ParentDirName); 
        if (streql(DirName, ParentDirName)) {
            ToLog(DBRPC, "NumUpForSameDir checking for looping %s \n", DirPath);
            (void) strcpy(AlexDir, DirPath);
            (void) strcat(AlexDir, SLASHALEXDIR);
            (void) strcpy(ParentAlexDir, WorkingDirPath);
            (void) strcat(ParentAlexDir, SLASHALEXDIR);
            if (FilesAreEqual(AlexDir, ParentAlexDir) == AOK) {
                ToLog(DBMAJOR, "NumUpForSameDir for sure we are looping %s \n", DirPath);
                Result=NumUp;
            }
        }
        PathToDir(WorkingDirPath, Tmp);
        (void) strcpy(WorkingDirPath, Tmp);
        NumUp++;
    }

    return(Result);
}
    

/*  DirPath = path for directory we are updating .alex.info in
 *
 *  We go to parent of this to get "." and ".."
 *  We also output entries for ".alex.dir" and ".alex.info" (info is fixed up later)
 */
int OutputExtras(AIfile, DirPath, UidStr, NumUp)
FILE *AIfile;
char *DirPath;
char *UidStr;
int  *NumUp;
{
    struct ParsedDir Current, DotCurrent, DotDotCurrent;
    char TmpPath[MAXPATH], Name[MAXPATH], AlexInfoDir[MAXPATH], AlexInfoPath[MAXPATH];
    struct ActiveAlexInfo AAI;
    struct stat StatBuf;
    static char CACHEDIRPARENT[MAXPATH]="";
    int    Result, Status, NeedDot, NeedDotDot, OutputCurrent;

    Log2("OutputExtras ", DirPath);

    if (CACHEDIRPARENT[0] == 0) {
        PathToDir(CACHEDIRVAR, CACHEDIRPARENT);
        Log2("OutputExtras CACHEDIRPARENT ", CACHEDIRPARENT);
    }

    ClearParsedDirEntry(&Current);
    PathToFileName(DirPath, Name);                    /* name to look for in the .alex.info */
    PathToDir(DirPath, AlexInfoDir);                  /* directory .alex.info would be in   */

    if (streql(AlexInfoDir, CACHEDIRPARENT)) {
        strcpy(AlexInfoPath, ROOTALEXINFO);           
    } else {
        (void) strcpy(AlexInfoPath, AlexInfoDir);
        (void) strcat(AlexInfoPath, SLASHALEXINFO);
    }

    if (LiteStat(AlexInfoPath) != AFILE) {          
        Log2("OutputExtras ERROR parent had bad .alex.info file ", AlexInfoPath);
        return(AFAIL);
    }

    if (ALEXINFOINOPEN(AlexInfoPath, &AAI, UidStr, NORECURSION) != AOK) {
        Log2("OutputExtras ERROR could not open .alex.info file ", AlexInfoPath);
        return(AFAIL);
    }

    NeedDot=1;                 /* we read through our parents .alex.info to get our . and .. */
    NeedDotDot=1;              
    OutputCurrent=0;
    while((AlexInfoInNext(&AAI, &Current) == AOK) && (NeedDot || NeedDotDot)) {
        if (NeedDot && streql(Current.Name, Name)) {
            Log2("OutputExtras found listing for current in parent ", Name);
            DotCurrent=Current;
            (void) strcpy(Current.Name, ".");
            NeedDot=0;
            OutputCurrent=1;
        }

        if (!OutputCurrent && NeedDotDot && streql(Current.Name, ".")) {
            Log2("OutputExtras found parents info ", Current.Name);
            DotDotCurrent=Current;
            (void) strcpy(Current.Name, "..");
            NeedDotDot=0;
            OutputCurrent=1;
        }

        if (OutputCurrent) {
            Status=OneLineOut(AIfile, &Current);
            if (Status != AOK) {
                ToLog(DBERROR, "OutputExtras ERROR BUG can not OneLineOut AIfile");
                (void) AlexInfoInClose(&AAI);
                return(AFAIL);
            }
            OutputCurrent=0;
        }
    }

    (void) AlexInfoInClose(&AAI);                                    /* done with parent */

    if (NeedDot || NeedDotDot) {
        ToLog(DBERROR, "OutputExtras ERROR BUG did not get both . and .. ");
        return(AOLD);                                                /* update parent */
    }

    Result=Status;
    if ((Result == AOK)   
                                          /* check for things like sgi.com:sgi sgi.sgi.com sgi.sgi.com:sgi */
           && (DotCurrent.Type != AHOST) 
        /* && (DotDotCurrent.Date == DotCurrent.Date) 
         * && (DotDotCurrent.Size == DotCurrent.Size) */  ) { /* for now don't filter on date/size */
        *NumUp=NumUpForSameDir(DirPath, AlexInfoDir, Name);
        if (*NumUp > 0) {
            Result = MAKELINKSTOPARENT;
        }
    }

    if ((Result == AOK) && streql(Name, "alex") && (strlen(DirPath) > 10)) {
        Result = MAKELINKSTOALEX;
    }    

    if (DEBUGLEVEL >= DBDOTALEX) {
        (void) strcpy(TmpPath, DirPath);
        (void) strcat(TmpPath, SLASHALEXDIR);
        Status=lstat(TmpPath, &StatBuf);
        if (Status == 0) {
            ClearParsedDirEntry(&Current);
            Current.Inode=InodeNext++;
            (void) strcpy(Current.Name, ALEXDIR);
            Current.Size=StatBuf.st_size;
            Current.Date=StatBuf.st_mtime;
            Current.Type=AFILE;
            Current.CacheStatus=1;
            Current.SymLink[0]=0;
            OneLineOut(AIfile, &Current);
            if (Status != AOK) {
                ToLog(DBERROR, "OutputExtras ERROR BUG can not OneLineOut AIfile");
                return(AFAIL);
            }
        } else {
            Log2("OutputExtras - error no .alex.dir for ", TmpPath);
        }

        ClearParsedDirEntry(&Current);
        Current.Inode=BOGUSINODE;                    /* temporary values - we fix later */
        (void) strcpy(Current.Name, ALEXINFO);
        Current.Size=BOGUSINODE;
        Current.Date=BOGUSINODE;
        Current.Type=AFILE;
        Current.CacheStatus=1;
        Current.SymLink[0]=0;
        Status=OneLineOut(AIfile, &Current);

        if (Status != AOK) {
            ToLog(DBERROR, "OutputExtras ERROR BUG can not OneLineOut AIfile");
            return(AFAIL);
        }
    }


    LogT("OutputExtras returning ", Result);
    return(Result);
}
   


int OutputMetaData(AIfile, Version, UpdateDate, NewestDate, TypeOfDir,
             PartsInHostName, AddAnyErrorStatus)
FILE *AIfile;
int  Version;
unsigned int UpdateDate, NewestDate;
int TypeOfDir, PartsInHostName, AddAnyErrorStatus;
{
    int Status, SizeOk;

    Status=AOK;
    if (TypeOfDir == UNIX) {
       SizeOk=1;
    } else {
       SizeOk=0;
    }


    if (fprintf(AIfile, "%c%c %d\n", METACHAR, VERSCHAR, Version) == EOF) {
        error1("OutputMetaData could not write Version ");
        Status=AFAIL;

    } else if (fprintf(AIfile, "%c%c %u\n", METACHAR, DATECHAR, UpdateDate) == EOF) {
        error1("OutputMetaData could not write UpdateDate ");
        Status=AFAIL;

    } else if (fprintf(AIfile, "%c%c %u\n", METACHAR, NEWCHAR, NewestDate) == EOF) {
        error1("OutputMetaData could not write NewestDate ");
        Status=AFAIL;

    } else if (fprintf(AIfile, "%c%c %d\n", METACHAR, SIZECHAR, SizeOk) == EOF) {
        error1("OutputMetaData could not write SizeOk ");
        Status=AFAIL;

    } else if (fprintf(AIfile, "%c%c %d\n", METACHAR, HOSTPARTSCHAR, PartsInHostName) == EOF) {
        error1("OutputMetaData could not write PartsInHostName ");
        Status=AFAIL;

    } else if (AddAnyErrorStatus == AERRORMESSAGE) {
        if (fprintf(AIfile, "%c%c\n", METACHAR, ERRORCHAR) == EOF) {
            error1("OutputMetaData could not write ERRORCHAR");
            Status=AFAIL;
        }
    }



    LogT("OutputMetaData returning ", Status);
    return(Status);
}

int TotalStaleFiles=0;
int TotalStaleSecs=0;

/*  Performance reporting 
 *  XXXX YYYY
 *
 *  Called when old .alex.info entry for a file was out of date.
 *  Input includes directory and the ParsedDir for new entry in .alex.info.
 *
 *  We only get a reasonable approximation of staleness stats.
 *  File could change, read stale, then change again so looking like read was ok.
 *  Could have read while stale, then been flushed from cache.
 *
 *  Return 
 *      AFAIL if not in cache - (caller removes if is in cache)
 */
int ReportStaleStats(Dir, CurrentPtr)
char *Dir;
struct ParsedDir *CurrentPtr;
{
    char Path[MAXPATH];
    struct stat StatBuf;
    int ModTimeMinusATime, ATimeMinusMTime, StaleNess;
    int Result;

    (void) strcpy(Path, Dir);
    (void) strcat(Path, "/");
    (void) strcat(Path, CurrentPtr->Name);

    Result=AlexStat(Path, &StatBuf);

    if (Result == AFAIL) {
        ToLog(DBPERF, "ReportStaleStats  stale but not in cache \n", Path);
    } else {
        ToLog(DBPERF, "ReportStaleStats RawData mtime=%d  atime=%d  modtime=%d  %s \n", 
                  (int) StatBuf.st_mtime, (int) StatBuf.st_atime, (int) CurrentPtr->Date, Path);

        ModTimeMinusATime = (int) (CurrentPtr->Date - StatBuf.st_atime);
        ATimeMinusMTime = (int) StatBuf.st_atime - StatBuf.st_mtime;

        StaleNess=0;
        if (ATimeMinusMTime < 0) {
            ToLog(DBPERF, "ReportStaleStats  updated after last read %d\n", ATimeMinusMTime);
        } else {
            if (ModTimeMinusATime > 0) {            
                ToLog(DBPERF, "ReportStaleStats changed after we read file\n");  /* StaleNess==0 */
            } else if (ModTimeMinusATime < 0) {
                ToLog(DBPERF, "ReportStaleStats should have changed before we read stale data\n");
                StaleNess= abs(ModTimeMinusATime);
                if (ATimeMinusMTime < StaleNess) {
                    StaleNess=ATimeMinusMTime;
                    ToLog(DBPERF, "ReportStaleStats StaleData used closer after last update %d\n",
                                                              StaleNess);
                } else {
                    ToLog(DBPERF, "ReportStaleStats StaleData used before current update%d \n",
                                                              StaleNess);
                }
            }
        }
  
        if (CurrentPtr->Date < StatBuf.st_mtime) {
            ToLog(DBPERF, "ReportStaleStats  the mod time is before we FTPed it\n");
            StaleNess=0;
        }

        if (StaleNess > 60) { 
            ToLog(DBPERF, "ReportStaleStats StaleDataInfo  %d seconds == %f days  for %s \n",
                      StaleNess, StaleNess/86400.0, Path);
            TotalStaleFiles++;
            TotalStaleSecs += StaleNess;
        } 
    }
    return(Result);
}


int RmDirNameIfThere(dir, name)
char *dir, *name;
{
    char tmp[MAXPATH];
    int Result, Type;

    (void) strcpy(tmp, dir);
    (void) strcat(tmp, "/");
    (void) strcat(tmp, name);


    Type=LiteStat(tmp);                            /* not from .alex.info */

    switch (Type) {
        case ADIR:     Log2("RmDirNameIfThere is RecursiveRming ", tmp);
                       Result=RecursiveRm(tmp, __FILE__, __LINE__, DONOTWAIT);
                       break;

        case AFILE:    Log2("RmDirNameIfThere is unlinking ", tmp);
                       Result=unlink(tmp);
                       break;

        case AFAIL:    Log2("RmDirNameIfThere was not there ", tmp);
                       break;

        default:       Result=AFAIL;
                       LogT("RmDirNameIfThere ERROR bad type ", Type);
    }

    if (Result != AOK) {
        Log2("RmDirNameIfThere did not remove ", tmp);
        Result=AFAIL;
    } else {
        Log2("RmDirNameIfThere removed ", tmp);
    }

    return(Result);
}



/*  Assumes files are sorted so we must be sure they are !!!!
 *
 *  Input:   filenames for .alex.info.old, .alex.info.tmp and .alex.info
 *                 the .alex.info.old can be /dev/null
 *
 *  Function:
 *           Remove files and directories that were around but are no longer
 *           Patchs up Inode numbers in cases where files already had inodes 
 *
 *  Algorithm:
 *           work down the 2 sorted lists sort of like a merge
 *  New:
 *        If there is a .alex.error, then there is not a new .alex.info, and we
 *        really want to keep the old info as it is better than the new.  So,
 *        we should add an error message, but keep old stuff around.
 */
int FixInodesAndRemoveOld(AlexInfoOld, AlexInfoTmp, AlexInfo, UidStr, 
                      UpdateDate, NewestDate, TypeOfDir, PartsInHostName, AddAnyErrorStatus)
char *AlexInfoOld, *AlexInfoTmp, *AlexInfo, *UidStr;
unsigned int UpdateDate, NewestDate;
int PartsInHostName;
int TypeOfDir, AddAnyErrorStatus;
{
    struct ActiveAlexInfo OldAIF, TmpAIF;
    FILE *AIfile;
    char dir[MAXPATH];
    int Status, TmpFull;
    struct ParsedDir CurrentOld, CurrentTmp, Current;
    long AlexInfoLoc, FileLength;
    struct stat StatBuf;

    Log2("FixInodesAndRemoveOld ", AlexInfoOld);

    PathToDir(AlexInfo, dir);

    if (ALEXINFOINOPEN(AlexInfoOld, &OldAIF, UidStr, NORECURSION) != AOK) {
        error2("FixInodesAndRemoveOld could not open", AlexInfoOld);
        return(AFAIL);
    } else {
        Log2("FixInodesAndRemoveOld has file open for read ", AlexInfoOld);
    }

    if (ALEXINFOINOPEN(AlexInfoTmp, &TmpAIF, UidStr, NORECURSION) != AOK) {
        Log2("FixInodesAndRemoveOld could not open", AlexInfoTmp);
        (void) AlexInfoInClose(&OldAIF);
        (void) AlexInfoInClose(&TmpAIF);
        return(AFAIL);
    } else {
        Log2("FixInodesAndRemoveOld has file open for read ", AlexInfoTmp);
    }

    AIfile=AFOPEN(AlexInfo,"w+");
    if (AIfile==NULL) {
        error2("FixInodesAndRemoveOld could not open", AlexInfo);
        (void) AlexInfoInClose(&OldAIF);
        (void) AlexInfoInClose(&TmpAIF);
        return(AFAIL);
    } else {
        OutputMetaData(AIfile, ALEXINFOVERSION, UpdateDate, NewestDate, TypeOfDir, 
                PartsInHostName, AddAnyErrorStatus);
    }
    

    CurrentOld.Name[0]=0;
    CurrentTmp.Name[0]=0;
    AlexInfoLoc= -1;
    TmpFull=0;
    while (1) {
        if((AlexInfoLoc == -1) && streql(CurrentTmp.Name, ALEXINFO)) {   
            AlexInfoLoc=ftell(AIfile);                /* remember this spot we fix later */
            LogN("FixInodesAndRemoveOld just did ftell ", (int) AlexInfoLoc);
        }

        Status=strcmp(CurrentTmp.Name, CurrentOld.Name);
        ToLog(10, "FixInodesAndRemoveOld : %s : %d : %s   t=%d\n", CurrentTmp.Name,
                                Status, CurrentOld.Name, (int) TmpFull);

        if (Status == 0) {                                                   /* matched !!!!!! */
            if (TmpFull) {
                if (!SpecialFile(&CurrentTmp)) {
                    CurrentTmp.Inode=CurrentOld.Inode;        /* was already around use inode #  */

                    if (CurrentTmp.Date != CurrentOld.Date) {          /* if has been updated    */
                        if ((CurrentTmp.Type == AFILE) ||              /* and is a file or type has changed */
                            (SimpleType(CurrentTmp.Type) != SimpleType(CurrentOld.Type))) { 
                            Log2("FixInodesAndRemoveOld thinks this is old ", CurrentTmp.Name);
                            if (ReportStaleStats(dir, &CurrentTmp) != AFAIL) {
                                (void) RmDirNameIfThere(dir, CurrentOld.Name); /* then remove from cache */
                            }
                        } else {
                            switch (CurrentTmp.Type) {
                                case ADOMAIN:
                                case ADIR:
                                case AHOST:
                                    Log2("FixInodesAndRemoveOld subdirectory has changed", CurrentTmp.Name);
            /*  should keep track of this - flag as stale XXXXXX YYYYYY */
                                    break;
                                case ALINK:
                                    Log2("FixInodesAndRemoveOld have a new link value", CurrentTmp.Name);
                                    break;
                                case AERRORMESSAGE:
                                    Log2("FixInodesAndRemoveOld have an errormessage ", CurrentTmp.Name);
                                    break;
                                default:
                                    ToLog(DBERROR, "FixInodesAndRemoveOld ERROR BUG %s\n", CurrentTmp.Name);
                                    break;
                            }
                        }
                    }
                }          
                OneLineOut(AIfile, &CurrentTmp);
                TmpFull=0;
            }

            if(AlexInfoInNext(&TmpAIF, &CurrentTmp) != AOK) break;   /* check CurrentTmp first */
            TmpFull=1;
            if(AlexInfoInNext(&OldAIF, &CurrentOld) != AOK) break;
            Log2(CurrentTmp.Name, CurrentOld.Name);

        } else if (Status > 0) {
            if (!SpecialFileOrError(&CurrentOld)) {
                if (AddAnyErrorStatus == AERRORMESSAGE) {
                     if (CurrentOld.Type <= ADIR) {
                         OneLineOut(AIfile, &CurrentOld);         /* if no new .alex.info copy old */
                     }
                } else {
                     Log2("FixInodesAndRemoveOld Greater so rming ", CurrentOld.Name);
                    (void) RmDirNameIfThere(dir, CurrentOld.Name);       /* gone so remove it */
                }
            }

            if(AlexInfoInNext(&OldAIF, &CurrentOld) != AOK) break;
            Log2("Just a new old ", CurrentOld.Name);
        } else {
            Log2("FixInodesAndRemoveOld less so advancing past ", CurrentTmp.Name);
            if (TmpFull) {
                CurrentTmp.Inode=InodeNext++;                           /* allocate a new inode */
                OneLineOut(AIfile, &CurrentTmp);
                TmpFull=0;
            }
            if(AlexInfoInNext(&TmpAIF, &CurrentTmp) != AOK) break;
            Log2("Just a new Tmp ", CurrentTmp.Name);
            TmpFull=1;
        }      
    }                                                                                 /* while */


    while(AlexInfoInNext(&OldAIF, &CurrentOld) == AOK) {             /* deal with rest of Old */
        Log2("FixInodesAndRemoveOld rest-of-old ", CurrentOld.Name);
        if (!SpecialFileOrError(&CurrentOld)) {
            if (AddAnyErrorStatus == AERRORMESSAGE) {
                 if (CurrentOld.Type <= ADIR) {
                     OneLineOut(AIfile, &CurrentOld);                    /* if no new .alex.info copy old */
                 }
            } else {
                (void) RmDirNameIfThere(dir, CurrentOld.Name);       /* gone so remove it */
            }
        }
    }
    (void) AlexInfoInClose(&OldAIF);


    do {                                                             /* deal with rest Tmp */
        if (TmpFull) {
            Log2("FixInodesAndRemoveOld rest-of-tmp ", CurrentTmp.Name);
            if((AlexInfoLoc == -1) && streql(CurrentTmp.Name, ALEXINFO)) {   
                Log("FixInodesAndRemoveOld about to ftell (2) ");
                AlexInfoLoc=ftell(AIfile);                /* remember this spot we fix later */
                LogN("FixInodesAndRemoveOld just did ftell (3) ", (int) AlexInfoLoc);
            } else {
                if (!SpecialFile(&CurrentTmp)) {
                    CurrentTmp.Inode=InodeNext++;                    /* allocate a new inode */
                }
            }
            OneLineOut(AIfile, &CurrentTmp);
        }
        TmpFull=1;                                               /* if we loop we will have data */
    } while(AlexInfoInNext(&TmpAIF, &CurrentTmp) == AOK);         

    (void) AlexInfoInClose(&TmpAIF);
    Status = AOK;

    fflush(AIfile);
    if (DEBUGLEVEL >=  DBDOTALEX) {
        Log("FixInodesAndRemoveOld is about to go back and fix AlexInfo");
        Status=lstat(AlexInfo, &StatBuf);
        if (Status == 0) {
            ClearParsedDirEntry(&Current);
            (void) strcpy(Current.Name, ALEXINFO);
            Current.Inode=InodeNext++;
            FileLength = ftell(AIfile);              /* we are at the end */
            Current.Size=FileLength;                 /* we know the real size now */
            Current.Date=StatBuf.st_mtime;
            Current.Type=AFILE;
            Current.CacheStatus=1;
            Current.SymLink[0]=0;
            if ((AlexInfoLoc > 0) && (AlexInfoLoc <FileLength)) {
                if (fseek(AIfile, AlexInfoLoc, 0)) {      /* we saw one go there */
                    LogN("FixInodesAndRemoveOld fseek failed ", (int) AlexInfoLoc);
                }
            } else {
                ToLog(DBERROR, "FixInodesAndRemoveOld ERROR BUG Bogus AlexInfoLoc %d\n ",
                                                        (int) AlexInfoLoc);
                Status=AFAIL;
            }
            if (Status == AOK) {
                Status=OneLineOut(AIfile, &Current);   /* since size if fixed field this works */
            }
        } else {
            Log2("FixInodesAndRemoveOld ERROR could not update size for ", AlexInfo);
            Status=AFAIL;
        }
    } else {
        if (AlexInfoLoc > 0) {
            ToLog(DBERROR, "FixInodesAndRemoveOld ERROR %s has a .alex.info at %d\n", AlexInfo, AlexInfoLoc);
        }
    }

    (void) AFCLOSE(AIfile);

    LogT("FixInodesAndRemoveOld returning ", Status);
    return(Status);
}





/*  returns 1 if we have no trouble with this name 
 */
int NotForbiddenName(Name)
char *Name;
{
    return(!streql(Name, ".")      &&  !streql(Name, "..") && 
           !streql(Name, ALEXINFO) &&  !streql(Name, ALEXDIR));
}


/*  WhereTo is either MAKELINKSTOALEX or MAKELINKSTOPARENT
 *    if TOPARENT then NumUp contains number of ../ to add.
 */
ModifyCurrentToLink(PtrCurrent, WhereTo, NumUp)
struct ParsedDir *PtrCurrent;
int WhereTo, NumUp;
{
    int i;

    PtrCurrent->Type=ALINK;

    switch (WhereTo) {
        case MAKELINKSTOALEX:   (void) strcpy(PtrCurrent->SymLink, ALEXPATH);
                                (void) strcat(PtrCurrent->SymLink, "/");
                                (void) strcat(PtrCurrent->SymLink, PtrCurrent->Name);
                                break;

        case MAKELINKSTOPARENT: PtrCurrent->SymLink[0]=0;
                                for (i=0; i<NumUp; i++) {
                                    (void) strcat(PtrCurrent->SymLink, "../");
                                }
                                (void) strcat(PtrCurrent->SymLink, PtrCurrent->Name);
                                break;

        default:                ToLog(DBERROR, "ModifyCurrentToLink got bogus argument %d", WhereTo);
                                exit(-1);
    }
}

int ParentHasNewAlexDir(DirPath)
char *DirPath;
{
    char ParentAlexDir[MAXPATH];

    PathToDir(DirPath, ParentAlexDir);
    (void) strcat(ParentAlexDir, SLASHALEXDIR);

    if (SecsSinceWrite(ParentAlexDir) < SECSTILLRETRY) {
        return(1);
    } else {
        return(0);
    }
}

char *AlexTops[] =  {"ar", "arpa", "at", "au", "be", "br", "ca", "ch", "com", "de", "dk", "ec",
                     "edu", "es", "fi", "fr", "gov", "gr", "hk", "ie", "il", "it", "jp",  "kr", 
                     "mil", "net", "nl", "no", "nz", "org", "se", "sg", "tw", "uk", "za", NULL };

/*  return 1 if Name is in above list */
int IsAnAlexTop(Name) 
char *Name;
{
    int i;

    for (i=0; AlexTops[i] != NULL; i++) {
        if (streql(AlexTops[i], Name)) {
            return(1);
        } 
    }
    return(0);
}

/*  Input is *directory* to play with and HostPtr to host that is parent of this tree.
 *  If we don't have a HostPtr and there is no .alex.dir we just make a 
 *  .alex.info from the known hosts/domains.
 *
 *  We parse the .alex.dir returned from the ftp dir command once and create
 *  a .alex.info file which is in a standard easy to parse form.
 *
 *  .alex.dir | conversion | sort > .alex.dir.tmp        (saving UpdateDate, NewestDate)
 *  FixInodesAndRemoveOld(Old, Tmp, UpdateDate, NewestDate) ==>  .alex.info.new
 *  mv .alex.info.new .alex.info 
 *
 */
extern int DirToInfo(DirPath, HostPtr, UidStr)
char *DirPath;
struct HostEntry *HostPtr;
char *UidStr;
{
    FILE *ADfile, *AITmpFile;
    char AlexDirPath[MAXPATH], DirPWS[MAXPATH];
    char AlexInfo[MAXPATH], AlexInfoTmp[MAXPATH], AlexInfoOld[MAXPATH], AlexInfoNew[MAXPATH]; 
    char line[MAXPATH];
    char  Tokens[MAXTOKENS][MAXTOKENLEN];
    int   Result, HaveNow, NumTokens, TypeOfDir;
    int   NumLines, NumNotFound;  
    struct ParsedDir Current;
    struct timeb Now;
    int Status, DPWSlen, NextHost, AddAnyErrorStatus;
    unsigned int UpdateDate, NewestDate;
    int MakeLinksType;
    int FirstOfTwoPasses, TotalAlexTops, Done, NumUp;

    Log2("In DirToInfo ", DirPath);

    if (LiteStat(DirPath) != ADIR) {
        ToLog(DBERROR, "DirToInfo ERROR BUG did not get a ADIR %s\n", DirPath);
        return(AFAIL);
    }

    InvalidateStatCache=1;

    (void)  strcpy(AlexDirPath, DirPath);
    (void)  strcat(AlexDirPath, SLASHALEXDIR);

    (void)  strcpy(AlexInfo, DirPath);
    (void)  strcat(AlexInfo, SLASHALEXINFO);

    (void)  strcpy(DirPWS, &DirPath[CACHEDIRLEN]);       /* Dir-Path-With-Slash */
    (void)  strcat(DirPWS, "/");
    DPWSlen=strlen(DirPWS);

    (void)  strcpy(AlexInfoOld, AlexInfo);
    (void)  strcat(AlexInfoOld, ".old");

    (void)  strcpy(AlexInfoTmp, AlexInfo);
    (void)  strcat(AlexInfoTmp, ".tmp");

    (void)  strcpy(AlexInfoNew, AlexInfo);
    (void)  strcat(AlexInfoNew, ".new");


    AITmpFile=AFOPEN(AlexInfoTmp, "w"); 
    if (AITmpFile==NULL) {
        error2("DirToInfo could not open", AlexInfoTmp);
        return(AFAIL);
    } else {
        Log2("DirToInfo has done open for sort into ", AlexInfoTmp);
    }

    Status=OutputExtras(AITmpFile, DirPath, UidStr, &NumUp);  /* do this first since they are all . files */
    if ((Status == MAKELINKSTOPARENT) || (Status == MAKELINKSTOALEX)) {
         MakeLinksType=Status;
         Status=AOK;
    } else {
         MakeLinksType=0;
    }

    if (Status == AOK) {
        AddAnyErrorStatus=AddAnyErrorToStream(AITmpFile, DirPath);  
    }
    if ((Status != AOK) || (AddAnyErrorStatus == AFAIL)) {
        ToLog(DBERROR, "DirToInfo ERROR BUG can not OutputExtras/AddAnyError to %s\n", DirPath);
        AFCLOSE(AITmpFile);
        LogT("DirToInfo returning ", Status);
        return(Status);
    }

    NextHost=0;
    Result= AOK;
    NewestDate=0;
    UpdateDate=TimeInSeconds();    /* will use time on .alex.dir if exists */
   
    if (HostPtr == NULL) {
        Log("DirToInfo found NULL HostPtr");                        /* skip many lines of code */
    } else {
        ADfile=AFOPEN(AlexDirPath,"r");
        if (ADfile == NULL) {
            Log2("DirToInfo could not open AlexDirPath", AlexDirPath); /* skip many lines of code */
        } else {
            UpdateDate=MTimeOfFile(AlexDirPath);
            TypeOfDir=DetermineTypeOfDir(ADfile);
            if (fseek(ADfile, (long) 0, 0)) {                           /* go back to start */
                Log("DirToInfo ERROR fseek failed");
                (void) AFCLOSE(ADfile);
                (void) AFCLOSE(AITmpFile);
                return(AFAIL);
            }

            NumLines=0;  
            NumNotFound=0;  
            HaveNow=0;
            line[0]=0;
            if (MakeLinksType == MAKELINKSTOALEX) {
                FirstOfTwoPasses=1;
            } else {
                FirstOfTwoPasses=0;
            }
            TotalAlexTops=0;
            Done=0;
            while (!Done && (Result == AOK)) {
                if (fgets(line, MAXPATH, ADfile) == NULL) {
                    if (FirstOfTwoPasses) {
                        if (TotalAlexTops < 10) {
                            MakeLinksType = 0;       /* have to all should be at least 10 of them */
                        }
                        NumLines=0;  
                        NumNotFound=0;  
                        FirstOfTwoPasses=0;          /* done with first pass */
                        (void) rewind(ADfile);           
                        continue;
                    } else {
                        Done=1;
                        break;
                    }
                }

                NumLines++;
                if ((TypeOfDir == UNKNOWN) && (HasString(line, "not found"))) {
                    NumNotFound++;
                    continue;
                }

                NumTokens=LineToTokens(line, Tokens);

                if (!IsRightType(Tokens, NumTokens, TypeOfDir)) {
                    Log2("DirToInfo is ignoring this line ", line);
                    continue;
                }
    
                if (!HaveNow) {
                    (void) GetRemoteNow(HostPtr, &Now);
                    HaveNow=1;
                }

                ClearParsedDirEntry(&Current);
                Status=TokensToParsedDir(Tokens, NumTokens, TypeOfDir, &Current, &Now, 
                    DirPath, HostPtr->Path);

                if (FirstOfTwoPasses) {
                    if (IsAnAlexTop(Current.Name)) {
                        TotalAlexTops++;
                    }
                } else if ((Status==AOK) && NotForbiddenName(Current.Name)){
                    if (MakeLinksType != 0) {
                        ModifyCurrentToLink(&Current, MakeLinksType, NumUp);
                    }
                    NextHost=OutputHostsBefore(AITmpFile, NextHost, DirPWS, DPWSlen, Current.Name);
                    OneLineOut(AITmpFile, &Current);
                    if (NewestDate < Current.Date) {
                        NewestDate=Current.Date;
                    }
                }
            }
            if ((NumNotFound == 1) && (NumLines==1)) {    /* XXXX might be old if NotFound > 1 */
                if (!ParentHasNewAlexDir(DirPath)) {
                    Result=AOLD;
                }
            }
            if (NumLines==0) {
                Log2("ERROR Could not read from ", AlexDirPath);
            }
            (void) AFCLOSE(ADfile);
        }                                                       /* end of big skip if null ADfile */
    }                                                           /* end of big skip for hosts only */
   

    NextHost=OutputHostsBefore(AITmpFile, NextHost, DirPWS, DPWSlen, "~~~~~"); /* large ascii val */

    SaveInodeNext();                                 /* we may have allocated some new ones */

    Log("DirToInfo about to close tmpfile");
    if (AFCLOSE(AITmpFile) < 0) {                     
        ToLog(DBERROR, "DirToInfo ERROR close of tmpfile failed failed");
        return(AFAIL);
    }

    if (Result != AOK) {
        LogT("DirToInfo returning early with ", Result);
        return(Result);
    }

    if (LiteStat(AlexInfo) != AFILE) {            /* if an old then we will use it but if not */
        if (LiteStat(AlexInfoOld) !=AFILE) {
            (void) strcpy(AlexInfoOld, "/dev/null");  /* then we will use /dev/null               */
        } 
    } else {
        (void) strcpy(AlexInfoOld, AlexInfo);     /* careful now with two of same name        */
    }

    Log2("DirToInfo     AlexInfoOld", AlexInfoOld);
    LogT("DirToInfo    intermediate Result", Result);  /* must be AOK */

    Result=SortFile(AlexInfoTmp);

    if (Result == AOK) {
        int PartsInHostName;

        if (HostPtr == NULL) {
            PartsInHostName = 0;
        } else {
            PartsInHostName = HostPtr->PartsInHostName;
        }

        LogN("DirToInfo PartsInHostName= ", PartsInHostName);

        Status=FixInodesAndRemoveOld(AlexInfoOld, AlexInfoTmp, AlexInfoNew, UidStr, 
                      UpdateDate, NewestDate, TypeOfDir, PartsInHostName, AddAnyErrorStatus);
        if (Status != AOK) {
            ToLog(DBERROR, "DirToInfo  ERROR  could not FixInodes %s\n", AlexInfo); 
            (void) unlink(AlexInfoTmp);                         
            Result=AFAIL;
        } else {
            Log("DirToInfo  think we have a good .alex.info.new"); 
            Result=AOK; 

            Status=unlink(AlexInfoTmp);                         
            if (Status != AOK) {
                ToLog(DBERROR, "DirToInfo  ERROR  could not unlink %s\n", AlexInfoTmp); 
            }
 
            if(rename(AlexInfoNew, AlexInfo)) {      /* last is   mv .alex.info.new .alex.info */
                error2("DirToInfo ERROR could not rename .alex.info.new to ", AlexInfo);
                Result=AFAIL;
            }
        }    
    }

    InvalidateStatCache=1;

    LogT("DirToInfo is returning ", Result);
    return(Result);
}








