
/* Copyright (c) 1992 Vincent Cate
 * All Rights Reserved.
 *
 * Permission to use and modify this software and its documentation
 * is hereby granted, provided that both the copyright notice and this
 * permission notice appear in all copies of the software, derivative works
 * or modified versions, and any portions thereof, and that both notices
 * appear in supporting documentation.  This software or any derivate works
 * may not be sold or distributed without prior written approval from
 * Vincent Cate.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND VINCENT CATE DISCLAIMS ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL
 * VINCENT CATE BE LIABLE FOR ANY SPECIAL, DIRECT, INDIRECT, OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Users of this software agree to return to Vincent Cate any improvements
 * or extensions that they make and grant Vincent Cate the rights to
 * redistribute these changes.
 *
 */



/* Exports:
 *    AlexInfoLStat()      -   sort of like an lstat but uses a .alex.info
 *    AlexInfoInOpen()     -   Use macro ALEXINFOINOPEN
 *    AlexInfoInNext()     -   read lines from a .alex.info
 *    AlexInfoCompatStat() - silly old compatability routine
 */

#include "alexincs.h"
#include "alex.h"

#define MAXSECSSINCECHECK    3600*24*60          /* 2 months max without checking */

#define MAXTOKENS  11
#define MAXTLEN   200 


int CallAlexOnDirForFile(FullPath, UidStr)
char *FullPath, *UidStr;
{
    char DirPath[MAXPATH];

    PathToDir(FullPath, DirPath);
 
    return(CheckFileCache(DirPath, UidStr));
}

int ClearActiveAlexInfo(AAIPtr)
struct ActiveAlexInfo *AAIPtr;
{
    if (AAIPtr == NULL) {
        Log("ClearActiveAlexInfo got a NULL pointer");
        return(AFAIL);
    }

    AAIPtr->File=NULL;                                          /* clear */
    AAIPtr->HasAlexError=0;
    AAIPtr->HasOldAlexError=0;
    AAIPtr->PartsInHostName=0;
    AAIPtr->SizeOk=0;
    AAIPtr->Version=0;
    AAIPtr->UpdateDate=0;
    AAIPtr->NewestDate=0;

    return(AOK);
}


/* Recursion means calling CheckFileCache
 */
extern int AlexInfoInOpen(AlexInfoPath, AAIPtr, UidStr, RecursionOk, srcfile, srcline)
char *AlexInfoPath;
struct ActiveAlexInfo *AAIPtr;
char   *UidStr;
int    RecursionOk;
char   *srcfile;
int    srcline;
{
    struct stat buf;
    int Status;
    char *NameOfFileToOpen;

    Status=ClearActiveAlexInfo(AAIPtr);
    if (Status != AOK) {
        return(AFAIL);
    }

    if (streql(AlexInfoPath, ROOTALEXINFOSTR) || streql(AlexInfoPath, "/.alex.info")) {
        Log2("AlexInfoInOpen had ", AlexInfoPath);
        NameOfFileToOpen=ROOTALEXINFO;
    } else {
        NameOfFileToOpen=AlexInfoPath;
    } 

    Log2("AlexInfoInOpen with ", NameOfFileToOpen);

    (void) strcpy(AAIPtr->OpenAlexInfoPath, NameOfFileToOpen);


#ifdef DEVNULLBROKEN                              /* just define this if /dev/null does not work */
    if (streql(AlexInfoPath, "/dev/null")) {
        Log("AlexInfoInOpen with /dev/null done ");
        return(AOK);                              /* reads will fail which is all we need */
    }
#endif

    if (lstat(NameOfFileToOpen, &buf) != 0) {                   /* try lstat */
        Log2("\n\n AlexInfoInOpen did not find this .alex.info ", NameOfFileToOpen);
        if (!RecursionOk) {
            return(AFAIL);
        } else {
            Status=CallAlexOnDirForFile(NameOfFileToOpen, UidStr);
            if (Status==AWORKING) {
                LogT("AlexInfoInOpen done ", Status);
                return(Status);
            }
            if (lstat(NameOfFileToOpen, &buf) != 0) {           /* try lstat again */
                return(AFAIL);
            } else {
                Log2("\n\n AlexInfoInOpen got Alex to make ", NameOfFileToOpen);
            }
        }
    }
                                                                /* have a good stat now */
    if ((buf.st_mtime < MINTIMEOKALEXINFO) && RecursionOk) {
        Log2("\n\n AlexInfoInOpen had old .alex.info - calling Alex", NameOfFileToOpen);
        Status=CallAlexOnDirForFile(NameOfFileToOpen, UidStr);
        if (Status != AOK) {
            Return(Status);
        }
        if (lstat(NameOfFileToOpen, &buf) != 0) {
            return(AFAIL);
        }
    }

    Log2("AlexInfoInOpen has done stat of ", NameOfFileToOpen);

    AAIPtr->File=afopen(NameOfFileToOpen,"r", srcfile, srcline);  /* caller resposible for close */
    if (AAIPtr->File==NULL) {
        Log2("AlexInfoInOpen could not open ", NameOfFileToOpen);
        return(AFAIL);
    }

    AAIPtr->Version=ALEXINFOVERSION;         /* default to current version */
    Log("AlexInfoInOpen done AOK");
    return(AOK);
}

/* for unfsd we sort of emulating opendir
 */
extern int AlexOpenDir(DirPath, AAIPtr, UidStr)
char *DirPath;
struct ActiveAlexInfo *AAIPtr;
char *UidStr;
{
    int Result;

    char AlexInfoPath[MAXPATH];

    (void) strcpy(AlexInfoPath, DirPath);
    (void) SimplifyPath(AlexInfoPath);
    if (streql(AlexInfoPath, "/")) {
        (void) strcat(AlexInfoPath, ALEXINFO);
    } else {
        (void) strcat(AlexInfoPath, SLASHALEXINFO);
    }

    Result=ALEXINFOINOPEN(AlexInfoPath, AAIPtr, UidStr, RECURSIONOK);

    LogT("AlexOpenDir done ", Result); 
    return(Result);
}

extern int AlexInfoInClose(AAIPtr)
struct ActiveAlexInfo *AAIPtr;
{
    int Result;

    if (AAIPtr->File == NULL) {
        Result=AOK;                       /* might be done by AlexInfoInNext in error cases */
    } else {
        if (AFCLOSE(AAIPtr->File) == 0) {
            Result=AOK;
        } else {
            Result=AFAIL;
        }
        AAIPtr->File=NULL;
    }
   
    LogT("AlexInfoInClose returning ", Result); 
    return(Result);
}



int ParseOneLine(Line, Current, Version)
char *Line;
struct ParsedDir *Current;
int Version;
{
    char Tokens[MAXTOKENS][MAXTLEN];
    int NumTokens, NumScanned, NoLinkSize, T;
    int Result, GoodType, ShouldHaveSymLink;

    NumTokens=LineToTokens(Line, Tokens);

#define MAXTOKENINPARSEDLINE 6

    if (NumTokens < (MAXTOKENINPARSEDLINE -1)) {
        ToLog(DBERROR, "ParseOneLine ERROR BUG not enough tokens %d while working on %s\n", 
                        NumTokens, Line);
        return(AFAIL);
    }

    switch (Version) {
       case 3:
       case 4:                       /* version 3 and 4 are the same in data part */
            NumScanned=1;  (void) strcpy(Current->Name, Tokens[0]);
            if (StringToUnsigned(Tokens[1],  (unsigned *) &Current->Type) == AOK) NumScanned++;
            if (StringToUnsigned(Tokens[2],  &Current->Size) == AOK) NumScanned++;
            if (StringToUnsigned(Tokens[3],  &Current->Date) == AOK) NumScanned++;
            if (StringToUnsigned(Tokens[4],  &Current->Inode) == AOK) NumScanned++;
            if (NumTokens==MAXTOKENINPARSEDLINE) {
                 NumScanned ++; (void) strcpy(Current->SymLink, Tokens[MAXTOKENINPARSEDLINE -1]);
#ifdef EXECUTABLE
                /* if the symlink is "-1", it's faked and we should mark the file executable */
                if (streql(Current->SymLink,"-1")) {
                     Current->Executable = 1;
                     strcpy(Current->SymLink,"");
                     --NumScanned;
                }
#endif
            }
            NoLinkSize=MAXTOKENINPARSEDLINE -1;
            break;

       default:
            ToLog(DBERROR, "ParseOneLine ERROR BUG unknown Version %d\n", Version);
            return(AFAIL);
    }

    T=Current->Type;
    ShouldHaveSymLink = (T==ALINK) && (T=HOSTALIAS);
    GoodType = (T==AFILE) || (T==ADIR) || (T==ALINK) || (T==AHOST) || 
               (T==HOSTALIAS) || (T==AERRORMESSAGE) || (T==ADOMAIN) ;

    if (GoodType &&                                                         /* if type is ok    */
        ( ((NumScanned==NoLinkSize)   && (!ShouldHaveSymLink)) ||           /* and right # tokens */
          ((NumScanned==NoLinkSize+1) &&  (ShouldHaveSymLink))   )  ) {
        Result=AOK;                                                         /* then all is well  */
    } else {
        ToLog(DBERROR, "ParseOneLine ERROR BUG NumTokens=%d NumScanned= %d, Type= %s\n", 
                       NumTokens, NumScanned, ATypeToString(Current->Type));
        Result=AFAIL;
    }

    return(Result);
}



/*  Read next directory item from AAIPtr 
 *  Returns AFAIL if does not work.
 */
extern int AlexInfoInNext(AAIPtr, Current)
struct ActiveAlexInfo *AAIPtr;
struct ParsedDir *Current;
{
    char line[MAXPATH+MAXPATH];
    int NeedALine, Status, Result;
    double Now, Age, OkSlop, SecsSinceCheck;

    if (AAIPtr->File == NULL) {
        Log("AlexInfoInNext ERROR called with no open file");
        return(AFAIL);
    }

    ClearParsedDirEntry(Current);

    NeedALine=1;
    Result=AFAIL;
    while (NeedALine) {
        NeedALine=0;
        if(fgets(line, MAXPATH+MAXPATH, AAIPtr->File ) == NULL) {
            return(AEOF);                                           /* that all folks */
        }

        if (line[0] == METACHAR) {
            NeedALine=1;                                            /* not a data line */
            Status=AOK;
            switch (line[1]) {
                case VERSCHAR:
                    Status=StringToUnsigned(&line[2], (unsigned *) &(AAIPtr->Version)); 
                    break; 

                case DATECHAR:
                    Status=StringToUnsigned(&line[2], (unsigned *) &(AAIPtr->UpdateDate)); 
                    break; 

                case NEWCHAR:
                    Status=StringToUnsigned(&line[2], (unsigned *) &(AAIPtr->NewestDate)); 
                    break; 

                case SIZECHAR:
                    Status=StringToUnsigned(&line[2], (unsigned *) &(AAIPtr->SizeOk)); 
                    break; 

                case HOSTPARTSCHAR:
                    Status=StringToUnsigned(&line[2], (unsigned *) &(AAIPtr->PartsInHostName)); 
                    break; 

                case ERRORCHAR:
                    AAIPtr->HasAlexError = 1; 
                    break; 

                default:
                    Status=AFAIL;
                    ToLog(DBERROR, "AlexInfoInNext ERROR BUG unknown meta-data type %c\n", line[1]);
            }
            
            if (Status != AOK) {
                ToLog(DBERROR, "AlexInfoInNext ERROR BUG meta-data problems Status= %s\n", 
                         ATypeToString(Status));
            }
                
        } else if ((strncmp(line, OLDVERSIONTOKEN, strlen(OLDVERSIONTOKEN)) == 0) && 
             (CountOccurances(line, ' ') == 1)) {                   /* one ping only  */
            NeedALine=1;                                            /* in either case ignore */
            if (StringToUnsigned(&line[strlen(OLDVERSIONTOKEN)], 
                                   (unsigned *) &(AAIPtr->Version)) != AOK) {
                ToLog(DBERROR, "AlexInfoInNext ERROR BUG StringToUnsigned of version %s\n", line);
            }
        }
    }
  
    Result=ParseOneLine(line, Current, AAIPtr->Version); 
    Current->PartsInHostName = AAIPtr->PartsInHostName; 
    Current->Stale = 0;                            /* not known to be stale so far */

    if (Result != AOK) { 
        ToLog(DBERROR, "AlexInfoInNext ERROR BUG could not read V= %d Name= %s Line= %s\n", 
                                AAIPtr->Version, Current->Name, line);
    } else {
        Now = TimeInSeconds();
        Age = Now - Current->Date;
        if (Age < 30) {
            Age=30;
        }

        SecsSinceCheck= Now - AAIPtr->UpdateDate;   /* time since since we looked */

        if (SecsSinceCheck < 0) {
            ToLog(DBERROR, "AlexInfoInNext ERROR BUG in future SSC= %d Now= %d UD= %d Age= %d", 
                         (int) SecsSinceCheck, (int) Now, (int) AAIPtr->UpdateDate, (int) Age);
            SecsSinceCheck=1000;
            Current->Stale = 1;                       /* an update might help (or loop) */
        }
   
        OkSlop = Age / OKINCONSISTENCY;  /* reasonable inconsistency of 10 percent */
                                         /* could not update /alex and next level down */

        if ((SecsSinceCheck > OkSlop) || (SecsSinceCheck > MAXSECSSINCECHECK)) {
            Current->Stale = 1;                            /* not within spec for consistent */
            /* Log2("AlexInfoInNext sees a POSSIBLE INCONSISTENCY ", Current->Name); */
        } 

        if ((Current->Type == AERRORMESSAGE) || (AAIPtr->HasAlexError)) {
            AAIPtr->HasAlexError=1;                        /* there is an error      */
            if ((SecsSinceCheck > SECSTILLRETRY) &&
                ((Current->Stale != 1) || (AAIPtr->HasOldAlexError != 1)))  {
                if (SecsSinceWrite(AAIPtr->OpenAlexInfoPath) > SECSTILLRETRY) {
                    Current->Stale = 1;                                /* really astale XXXX  */
                    AAIPtr->HasOldAlexError=1;                         /* time for a retry    */
                    ToLog(DBALL, "KnownStaleDir %s\n", AAIPtr->OpenAlexInfoPath);
                }
            }
        }

        if (AAIPtr->HasOldAlexError) {
            Current->Stale = 1;                            /* all are stale */
        }

        /* LogN("AlexInfoInNext sees SecsSinceCheck ", (int) SecsSinceCheck);
         * LogN("AlexInfoInNext sees OkSlop ", (int) OkSlop); 
         */
    }

    return(Result);
}


extern int SimpleType(Type)
int Type;
{
    int Result;

    switch(Type) {
        case ADOMAIN:
        case AHOST:
        case ADIR:   Result=ADIR; 
                     break;

        case AERRORMESSAGE:
        case AFILE:  Result= AFILE; 
                     break;

        case HOSTALIAS:
        case ALINK:  Result= ALINK;
                     break;

        default:     Result=Type;     
                     break;
   }
   return(Result);
}

#ifdef EXECUTABLE
int ATypeToStatMode(AType,IsExecutable)
int AType;
int IsExecutable;
#else
int ATypeToStatMode(AType)
int AType;
#endif
{
    int Result, Type;

   /* Log2("ATypeToStatMode is called with ", ATypeToString(AType)); */

    Type=SimpleType(AType);

    switch (Type) {
        case ADIR:   Result= S_IFDIR + 0755;
                     break;

        case AFILE:
#ifdef EXECUTABLE
                     if (IsExecutable) {
                         Result= S_IFREG + 0755;
                     } else {
#endif
                         Result= S_IFREG + 0644;
#ifdef EXECUTABLE
                     }
#endif
                     break;

        case ALINK:  Result= S_IFLNK + 0777;
                     break;

        default:     ToLog(DBERROR, "ATypeToStatMode ERROR BUG %d\n", AType);
                     Result= S_IFDIR + 0444;
    }

    return(Result);
}

StandardStat(StatBuf)
struct stat *StatBuf;
{
    StatBuf->st_dev=1;
    StatBuf->st_nlink=1;
    StatBuf->st_rdev=1; 
    StatBuf->st_uid=ALEXUIDVAR; 
    StatBuf->st_gid=ALEXGIDVAR;
    StatBuf->st_blksize=1024; 
/*    StatBuf->st_spare1=0;
    StatBuf->st_spare2=0;
    StatBuf->st_spare3=0;
    StatBuf->st_spare4[0]=0;
    StatBuf->st_spare4[1]=0;
 */
}

int ParsedDirToStat(Current, StatBuf)
struct ParsedDir *Current;
struct stat *StatBuf;
{


    if (Current->Inode != 0) {
        StatBuf->st_ino=Current->Inode;
    } else {
        ToLog(DBERROR, "ParsedDirToStat ERROR BUG why does this happen? %d\n", (int) InodeNext);
        StatBuf->st_ino=InodeNext++;
    }

#ifdef EXECUTABLE
    StatBuf->st_mode=ATypeToStatMode(Current->Type,Current->Executable);
#else
    StatBuf->st_mode=ATypeToStatMode(Current->Type);
#endif

    StatBuf->st_size=Current->Size; 
    StatBuf->st_blocks= 1 + (Current->Size / 512);

    StatBuf->st_atime=Current->Date;
    StatBuf->st_mtime=Current->Date;
    StatBuf->st_ctime=Current->Date;

    StandardStat(StatBuf);
}

int InvalidateStatCache;


/*  It is important that for the case where stats are in order that
 *  we end up only reading the .alex.info once.  Cache open file
 *  and position in that file.
 *
 *  Recursion means calling Alex
 */
extern int AlexInfoLStat(FullPathArg, Result, UidStr, RecursionOk)
char *FullPathArg;
struct ParsedDir *Result;
char   *UidStr;
int    RecursionOk;
{
    char   FullPath[MAXPATH];
    static char CurrentOpenAlexInfo[MAXPATH];
    static char KnownBadAlexInfo[MAXPATH];
    static struct ActiveAlexInfo SAAI;              /* Static AAI           */
    struct ActiveAlexInfo AAI;                      /* our own AAI           */
    static struct ParsedDir Current;
    static int Status=AFAIL;                        /* status of Current now */
    static int SomethingOpen=0;
    char AlexInfoPath[MAXPATH], FileName[MAXPATH];
    struct stat s2;                                 

    AlexAssert(FullPathArg != NULL);
    AlexAssert(FullPathArg[0] != 0);

    strcpy(FullPath, FullPathArg);
    (void) SimplifyPath(FullPath);

    PathToDir(FullPathArg, AlexInfoPath);
    (void) SimplifyPath(AlexInfoPath);
    (void) strcat(AlexInfoPath, SLASHALEXINFO);

    Log2("AlexInfoLStat with ", AlexInfoPath);
    LogN(UidStr, RecursionOk);

    (void) PathToFileName(FullPath, FileName);

    if ((!InvalidateStatCache) && (streql(AlexInfoPath, KnownBadAlexInfo))) {
        Log2("AlexInfoLStat  using info failure cache ", FullPath);
        return(AFAIL);
    } 

    if ((!InvalidateStatCache) && streql(AlexInfoPath, CurrentOpenAlexInfo) &&
        (Status==AOK) && (streql(FileName, Current.Name)) && 
        Current.SizeTrusted &&  SomethingOpen) {          /* VMS ones can change on us */
        Log2("AlexInfoLStat cache hit", FullPath);
        *Result=Current;
        return(Status);
    }


    if (FileName[0] == 0) {
        Log("AlexInfoLStat NULL filename - adding .");
        strcat(FileName, ".");
    }

    if (InvalidateStatCache) {
        if (SomethingOpen) {
            (void) AlexInfoInClose(&SAAI);
            SomethingOpen=0;
        }
        KnownBadAlexInfo[0]=0;
        CurrentOpenAlexInfo[0]=0;

        InvalidateStatCache=0;                      /* we are making an entry not using cache */
    }

    (void) ClearParsedDirEntry(&Current);

    Status=AOK;
    if ((!SomethingOpen) || (!streql(AlexInfoPath, CurrentOpenAlexInfo))) {
        CurrentOpenAlexInfo[0]=0;
        if (Status == AOK) {   
            Status=ALEXINFOINOPEN(AlexInfoPath, &AAI, UidStr, RecursionOk);
            if (Status == AWORKING) {                                        /* do the open */
                return(Status);
            }
            if (Status != AOK) {                      
                (void) strcpy(KnownBadAlexInfo, AlexInfoPath);
                (void) AlexInfoInClose(&AAI);                                /* should not need  */
                Status=AFAIL;
            } else { 
                if (SomethingOpen) {
                    Status=AlexInfoInClose(&SAAI);                           
                    SomethingOpen=0;
                }
                Status=AOK;
                SAAI=AAI;                                                    /* copy to static   */
                SomethingOpen=1;                                             /* new active one */
                (void) strcpy(CurrentOpenAlexInfo, AlexInfoPath);
            }
        }
    }

    if (Status==AOK) {                                   /* if AOK then AAI has right file open */
        Status=AlexInfoInNext(&SAAI, &Current);                           /* first try next item */
        if ((Status != AOK) || (!streql(Current.Name, FileName))) {
            Log2("AlexInfoLStat going back to begining of file looking for ", FileName);
            if (fseek(SAAI.File, (long) 0, 0)) {
                Log2("AlexInfoLStat ERROR fseek failed", FileName);
                (void) AlexInfoInClose(&SAAI);
                SomethingOpen=0;
                Status=AFAIL;                                              /* going back failed  */
            } else {
                Status=AOK;
                while((Status==AOK) && !streql(Current.Name, FileName)) {  /* goes at least once */
                    Status=AlexInfoInNext(&SAAI, &Current);                /* get new Current    */
                }
            }
        }
    }

    if ((Status != AOK) || !streql(Current.Name, FileName)) {            /* if still not there */
        Log2("AlexInfoLStat found No such file as ", FileName);          /* no such file */
        Result->PartsInHostName=SAAI.PartsInHostName;                    /* what host we checked */
        (void) AlexInfoInClose(&SAAI);
        SomethingOpen=0;
        Status=NOSUCHFILE;
        LogT("AlexInfoLStat ", Status);
        return(Status);
    }


    if ( ! SAAI.SizeOk) {                                           /* if could not parse sizes */
        if (lstat(FullPath, &s2) == 0) {                            /* if we have file in cache */
            Current.Size=s2.st_size;                                /* use size of one in cache */
        }
        Current.SizeTrusted=0;
    } else {
        Current.SizeTrusted=1;                                 /* size comes from good .alex.dir */
    }


    *Result=Current;                                                /* note that Status == AOK */

    ToLog(DBALL, "AlexInfoLStat %s %s %d\n", FullPath, ATypeToString(Current.Type), Current.Inode); 
    return(AOK);
}



/* For compatability we fill in a regular stat struct 
 * For incompatability we use Alex types not regular return codes for lstat 
 *
 * Recursion means calling Alex
 */
extern int AlexInfoCompatStat(FullPath, StatBuf, Current, UidStr, RecursionOk)
char        *FullPath;
struct stat *StatBuf;
struct ParsedDir *Current;
char          *UidStr;
int          RecursionOk;
{
    int  Status;
    struct ParsedDir CurrentBuf;
    char LocalPath[MAXPATH], LastOfPath[MAXPATH];

    Log2("AlexInfoCompatStat", FullPath);

    if (UidStr==NULL) {
        UidStr=LastUidStr;
    }

    if (Current == NULL) {
        Current = &CurrentBuf;
    }

    (void) strcpy(LocalPath, FullPath);
    Status=SimplifyPath(LocalPath);
    PathToFileName(LocalPath, LastOfPath);

    if (streql(LastOfPath, ALEXUPDATE)) {
        if (!RecursionOk) {
            ToLog(DBERROR, "AlexInfoCompatStat ERROR BUG recursion needed for .alex.update %s\n",
                                                             LocalPath);
        }
        Status=CheckFileCache(LocalPath, UidStr);                 
        if (Status == AWORKING) {
            return(Status);
        } else {
            return(AFAIL);
        }
    }

    Status=AlexInfoLStat(LocalPath, Current, UidStr, RecursionOk);
    if ((Status != AWORKING) && (Status != NOSUCHFILE)) {             /* hosts vs nosuchfile XXX */
        if ((Status != AOK) && RecursionOk) {
            Status=CallAlexOnDirForFile(LocalPath, UidStr);           /* maybe its just new */
            if (Status==AOK) {
                Status=AlexInfoLStat(LocalPath, Current, UidStr, RecursionOk);
            }
        }

        if (Status == AOK) {
            LogN("AlexInfoCompatStat has current inode ", (int) Current->Inode);
            if ((debugLog != NULL) && (DEBUGLEVEL >= 10)) {
                OneLineOut(debugLog, Current);
            }
            (void) ParsedDirToStat(Current, StatBuf);
            LogN("AlexInfoCompatStat has stat inode ", (int) StatBuf->st_ino);
        } else if (Status != NOSUCHFILE) {
            Log2("AlexInfoCompatStat error Could not deal with ", FullPath);
            if ((Status != AWORKING) && RecursionOk) {
                Status=CheckFileCache(LocalPath, UidStr);
                if (Status==AOK) {
                    Status=AlexInfoLStat(LocalPath, Current, UidStr, RecursionOk);
                }
            }
        }
    }
    

    if (Status == AOK) {
        LogN("AlexInfoCompatStat returning statbuf with inode ", (int) StatBuf->st_ino);
        errno=0;
    } else {
        LogT("AlexInfoCompatStat is returning ", Status);
    }

    
    return(Status);
}






