/* -*- C -*-
 *
 * Program:	ximap
 * File:        filebrowser.c -- Gets a file name.  Currently, directories 
 *                               are displayed in whatever emphatic font 
 *                               the List widget is using.  The entries 
 *                               are currently unsorted, which is bad, and 
 *                               little error checking is done wrt permissions.
 *
 * Author:	Kevin Brock
 *	        Symbolic Systems Resources Group
 *		Stanford University
 *              MSOB x241
 *		Stanford, CA 94305
 *		Internet: brock@CAMIS.Stanford.Edu
 *
 * Date:	07 September 1992
 *
 * Idosynchrasies - This uses the Xb library and XtNdoubleClick.
 *    The simulation of double-click is done by always calling
 *    the single-click callback first followed by the double-click
 *    callback if appropriate. 
 *
 * Copyright 1992 by The Leland Stanford Junior University.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * above copyright notices and this permission notice appear in supporting
 * documentation, and that the name of The Leland Stanford Junior University 
 * not be used in advertising or publicity pertaining to distribution of the 
 * software without specific, written prior permission.  This software is made 
 * available "as is", and
 * THE LELAND STANFORD JUNIOR UNIVERSITY DISCLAIMS ALL WARRANTIES, EXPRESS OR 
 * IMPLIED, WITH REGARD TO THIS SOFTWARE, INCLUDING WITHOUT LIMITATION ALL IMPLIED 
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE, AND IN NO EVENT 
 * SHALL THE LELAND STANFORD JUNIOR UNIVERSITY BE LIABLE FOR ANY SPECIAL, INDIRECT 
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, 
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, TORT (INCLUDING NEGLIGENCE) 
 * OR STRICT LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE 
 * OF THIS SOFTWARE.
 *
 */
#include <stdio.h>
#include <fcntl.h>
#include <malloc.h>
#include <string.h>

#include <sys/types.h>
#include <sys/dir.h>

/*
 * For systems with no getdents call we supply our own simulation */
#ifdef USELOCALGETDENTS
#include "dirlib/sys.dirent.h"
#else
#include <sys/dirent.h>
#endif

#include <sys/param.h>
#include <sys/stat.h>

#include <Client/osdep.h>
#include <Client/mail.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Label.h>

#include <Xb/Browser.h>

#include "buttons.h"
#include "filebrowser.h"
#include "textutil.h"
#include "mailcap.h"
#include "util.h"

int sys_nerr;
extern char *sys_errlist[];
extern int errno;

typedef struct string_list
{
    char *string;
    unsigned int state;
    struct string_list *next;
} StringList;

typedef struct file_browser
{
    Widget shell;
    Widget panes;
    Widget browser;
    Widget form;
    Widget text;
    Widget buttons;
    Widget path_text;

    char currentDirectory[MAXPATHLEN + 1];
    int  currentDirectoryFD;
    char path[MAXPATHLEN + 1];
    int  resetRoot;
    int   goodFile;
    char *goodFileName;
    
    int dirOnly;
} FileBrowser;

typedef struct file_browser_list
{
    FileBrowser *fb;
    struct file_browser_list *next;
} FBList;

static void fbOk();
static void fbOkAct();
static void fbCancel();
static void fbCancelAct();
static void fbSelect();
static void updateText();
static void updatePath();

static XtActionsRec fbActs[] =
{{ "fb_cancel", fbCancelAct },
 { "fb_ok",     fbOkAct },
};

ButtonStruct fbButtons[] = 
{{"Cancel", fbCancel, NULL, NEVER},
 {"Ok",     fbOk,     NULL, NEVER},
   NULL};

static char* scopy();
static int   changeDirectory();
static XbListSelection* getDirectoryListing();
static void  makeFileBrowserWidget();
static FileBrowser* getFileBrowser();

static FBList *fbl= NULL;

extern XtAppContext app_context;

void addFbActions(ac)
     XtAppContext ac;
{

    XtAppAddActions( ac, fbActs, XtNumber(fbActs));

}
/* char* getFilename(char*)
 *
 * Returns the filename selected by the user.  NULL if there
 *   was no file selected.
 *
 * Takes one argument:  The initial path from which to display
 *   the files.
 *
 */
char* getFilename( w, path, directoriesOnly, inwidget )
     Widget  w;
     char   *path;
     int directoriesOnly;
     int inwidget;
{
    XEvent event;
    char oldPathname[MAXPATHLEN + 1];
    char *ret = NULL;

    FBList *fblt = NULL;
    FBList *fbltemp = NULL;
    FileBrowser *fb = (FileBrowser*) malloc (sizeof(FileBrowser));
    fb->goodFile = 0;
    /* We only look for directory names - DO NOT change this option. Things
     * will break badly if you do */
    fb->dirOnly = directoriesOnly;
    fb->currentDirectoryFD = -1;

    /* If there's no path, start from root directory */
    if(!path)
	path = scopy(ROOTDIR);

    /* Save current directory so we can return */
    if (!getwd(oldPathname)) {
      mm_log("getFilename: system error calling getwd", WARN);
      return ret;
    }
    
    /* Go to starting point, set globals and open directory file descriptor */
    changeDirectory(fb, path);
    /* Initialize the full path string */
    strcpy(fb->path, fb->currentDirectory);
    strcat(fb->path, "/");
    fb->resetRoot = 0;
	  
    makeFileBrowserWidget(w, fb, inwidget);
    /*
     * fb->goodFile < 0 if above call failed */
    if(!fb->goodFile)
    {
	FBList *temp = (FBList *) malloc (sizeof(FBList));
	temp->next = fbl;
	temp->fb = fb;
	fbl = temp;
    }

    while(!fb->goodFile)
    {
	XtAppNextEvent(app_context, &event);
	XtDispatchEvent( &event );
    }

    /*
     * Here we take care of possible a possible fbList with
     * multiple entries since this code is reentrant */
    fbltemp = fbl;
    if(fbltemp->fb != fb)
    {
	while(fbltemp->next->fb != fb)
	    fbltemp = fbltemp->next;
	fblt = fbltemp->next;
	fbltemp->next = fbltemp->next->next;
	free(fblt);
	fblt = NULL;
    }
    else
    {
	fbl = fbltemp->next;
	free(fbltemp);
	fbltemp = NULL;
    }
	
    /* goodFile = -1 is a 'cancel' */
    if(fb->goodFile > 0) {
	ret = scopy(fb->goodFileName);
	free(fb->goodFileName);
	fb->goodFileName = 0;
    }
    /* Return to original directory before we leave...*/
    chdir(oldPathname);
    /* ... and clean up miscellaneous garbage */
    if (fb->currentDirectoryFD >= 0)
      close(fb->currentDirectoryFD);
    fb->currentDirectory[0] = '\0';
    fb->goodFile = 0;

    if(!inwidget)
    {
	XtPopdown(fb->shell);
	XtDestroyWidget(fb->shell);
    }
    else
    {
	XtDestroyWidget(fb->buttons);
	XtDestroyWidget(fb->browser);
	XtDestroyWidget(fb->text);
	XtDestroyWidget(fb->form);
	XtDestroyWidget(fb->path_text);
    }

    free(fb);
    return(ret);
}

static char* scopy(s)
     char* s;
{
    char *ret = NULL;
    if(s)
    {
	ret = malloc(strlen(s) + 1);
	strcpy(ret,s);
    }
    return(ret);
}

/* Returns TRUE/FALSE (1/0) for success and
 * failure respectively
 */
static int changeDirectory(fb, path)
     FileBrowser *fb;
     char* path;
{
  char str[256];

    if(!path) 
	return(0);
    
    if(strlen(path) <= MAXPATHLEN)
    {
	strcpy(fb->currentDirectory, path);
	if(fb->currentDirectoryFD >= 0)
	    close(fb->currentDirectoryFD);

	fb->currentDirectoryFD = open(fb->currentDirectory, O_RDONLY, 0);
	
	if(fb->currentDirectoryFD < 0) {
	  char *errstr;

	  errstr = (char *)strerror(errno);
	  sprintf(str,"Open error: dir %s", fb->currentDirectory);
	  mm_log(str, WARN);
	  sprintf(str, "  %s", errstr);
	  mm_log(str, WARN);
	  return(0);
	}

	if(chdir(fb->currentDirectory))
	    return(0);


	if (!getwd(fb->currentDirectory)) {
	  mm_log("changeDirectory: system error calling getwd", WARN);
	  return(0);
	}

	return(1);
    }
    
    return(0);
}

static int openAndConnectToDirectory(fb, path)
     FileBrowser *fb;
     char* path;
{
  char str[256];

  if(!path) 
    return(0);
    
  if(strlen(path) <= MAXPATHLEN)
    {
      strcpy(fb->currentDirectory, path);
      if(fb->currentDirectoryFD >= 0)
	close(fb->currentDirectoryFD);

      fb->currentDirectoryFD = open(fb->currentDirectory, O_RDONLY, 0);
      
      if(fb->currentDirectoryFD < 0) {
	char *errstr;
	
	errstr = (char *)strerror(errno);
	sprintf(str,"Open error: dir %s", fb->currentDirectory);
	mm_log(str, WARN);
	sprintf(str, "  %s", errstr);
	mm_log(str, WARN);
	return(0);
      } 

      /* Make this our current directory */
      if(chdir(fb->currentDirectory) == 0)
	return(1);
    }
  return(0);
}

/* 
 * Returns the directory listing of currentDirectory in
 * a format suitable for putting into a Browser widget..
 *
 */

#ifdef USELOCALGETDENTS

extern int _getdents();
int (*our_getdents)() = _getdents;

#else

extern int getdents();
int (*our_getdents)() = getdents;
#define DIRBLKSIZ 0

#endif

static XbListSelection* getDirectoryListing(fb)
     FileBrowser* fb;
{
    struct stat sb;
    XbListSelection *ret = NULL;

    if( fb->currentDirectoryFD >= 0 )
    {
	int nbytes = 0;
	int i = 0;
	StringList *slp = NULL;
	StringList *temp = NULL;
	int dirent_len;

	/* Make a scratch buffer bug enough for one entry */
	char *scratchEntry;
	ret = (XbListSelection*) malloc (sizeof(XbListSelection));
	if (!ret) {
	  mm_log("malloc call failed in getDirectoryListing", WARN);
	  return NULL;
	}
	ret->size = 0;
	/*
	 * For Ultrix we are using _getdents and a special library.
	 * Also, insure there are enough bytes to collect data.
	 * 
	 *    Bill Yeager - 15 Oct 92 */
	dirent_len = sizeof(struct dirent) + (2*DIRBLKSIZ);
	scratchEntry = malloc(dirent_len);
	if (!scratchEntry) {
	  mm_log("malloc call failed in getDirectoryListing", WARN);
	  return NULL;
	}
	while((nbytes = (*our_getdents)(fb->currentDirectoryFD,
					scratchEntry,
					dirent_len)) > 0)
	{
	    int totalbytes = 0;
	    char *cp = scratchEntry;

	    while(nbytes > totalbytes)
	    {
		struct dirent *dp= (struct dirent*)cp;
		
		if (stat(dp->d_name, &sb) == 0) {
		
		  /* Forget "." and "..".
		   *   Remember strcmp returns 0 if compare succeeds */
		  if (strcmp(dp->d_name, ".") && strcmp(dp->d_name, "..")) 
		    if(!fb->dirOnly || S_ISDIR(sb.st_mode))
		      {
			temp = (StringList*) malloc (sizeof(StringList));
			temp->string = scopy(dp->d_name);
			temp->next = slp;
			temp->state = (S_ISDIR(sb.st_mode) ? BOLD : NORMAL);
			slp = temp;
			ret->size++;
		      }
	        }
		totalbytes += dp->d_reclen;
		cp += dp->d_reclen;
	    }
	}
	
	if (!(i = ret->size)) {		/* nothing returned */
	  free(scratchEntry);
	  free(ret);
	  return NULL;
	}

	ret->entries = (XbListEntry*) malloc (ret->size*sizeof(XbListEntry));
	while(i--)
	{
	    ret->entries[i].entry = scopy(slp->string);
	    ret->entries[i].index = i;
	    ret->entries[i].invert = 0;
	    ret->entries[i].bold = slp->state;
	    
	    temp = slp;
	    slp = slp->next;
	    free(temp->string);
	    free(temp);
	}
	free(scratchEntry);
    }
    return(ret);
}

static void  makeFileBrowserWidget(w, fb, inwidget)
     Widget w;
     FileBrowser*fb;
     int inwidget;
{
    Arg warg[16];
    int n = 0;
    char host[HNAMELEN];
    Widget tmp, lab;
    XbListSelection *filelist = NULL;
    XFontStruct *font;

    fb->shell = NULL;
    fb->panes = NULL;
    if(!inwidget)
    {
	fb->shell = XtCreatePopupShell("fb_shell", topLevelShellWidgetClass,
				       w, warg, n); n = 0;

	XtSetArg(warg[n], XtNwidth, 350); n++;
	fb->panes = XtCreateManagedWidget("fb_panes", panedWidgetClass,
					  fb->shell, warg, n);
    }
    else
    {
	fb->panes = w;
    }
    n = 0;
    XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    fb->buttons = XtCreateManagedWidget("fb_buttons", boxWidgetClass,
					fb->panes,  warg, n); n = 0;

    /* For the file name - place label and edit widgets in a box*/
    XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    fb->form = XtCreateManagedWidget("fb_form", formWidgetClass,
				     fb->panes,  warg, n); n = 0;

    /* Label - in form */
    XtSetArg(warg[n], XtNborderWidth, 0); n++;
    XtSetArg(warg[n], XtNjustify, XtJustifyLeft); n++;
    XtSetArg(warg[n], XtNlabel , "filename"); n++;
    tmp =
      XtCreateManagedWidget("Filename", labelWidgetClass, 
			    fb->form, warg, n); n = 0;
    /* text widget - in form */
    XtSetArg(warg[n], XtNfromHoriz, tmp); n++;
    XtSetArg(warg[n], XtNborderWidth, 0); n++;
    XtSetArg(warg[n], XtNeditType, XawtextEdit); n++;
    XtSetArg(warg[n], XtNdisplayCaret, True); n++;
    fb->text = XtCreateManagedWidget("fb_text", asciiTextWidgetClass, 
				    fb->form, warg, n); n = 0;	


    if(filelist = getDirectoryListing(fb))
    {
	XtSetArg(warg[n], XtNinitialViewHt, 10); n++;
	XtSetArg(warg[n], XtNlistPosition, 0); n++;
	XtSetArg(warg[n], XtNlist, filelist); n++;
	XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
	fb->browser = XtCreateManagedWidget("fb_browser", 
					    XbbrowserWidgetClass,
					    fb->panes, warg, n); n = 0;

	/* NOTE that a double-click will coerce the single-click
         * callback to be called BEFORE the double-clicker */
	XtAddCallback( fb->browser, XtNsingleClick, updateText, fb );
	XtAddCallback( fb->browser, XtNdoubleClick, fbSelect, fb );
    }
    else
    {
	fb->goodFile = -1;
	return;
    }

    /* Label qui introduit le list. Met ca dans une forme*/
    XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    XtSetArg(warg[n], XtNskipAdjust, TRUE); n++;
    tmp = XtCreateManagedWidget("fb_tmp_form", formWidgetClass,
				fb->panes,  warg, n); n = 0;
    /* Voici le label */
    XtSetArg(warg[n], XtNborderWidth, 0); n++;
    XtSetArg(warg[n], XtNjustify, XtJustifyLeft); n++;
    lab = XtCreateManagedWidget("Selected Directory on", labelWidgetClass, 
				tmp, warg, n); n = 0;
    /* The host name */
    getQualifiedHostname(host, HNAMELEN);
    XtSetArg(warg[n], XtNfromVert, lab); n++;
    XtSetArg(warg[n], XtNborderWidth, 0); n++;
    XtSetArg(warg[n], XtNjustify, XtJustifyCenter); n++;
    XtSetArg(warg[n], XtNlabel , host); n++;
    XtSetArg(warg[n], XtNskipAdjust, TRUE); n++;
    XtCreateManagedWidget("fb_hostname", labelWidgetClass, 
			  tmp, warg, n); n = 0;

    /* Add a text box for the full path name */
    XtSetArg(warg[n], XtNborderWidth, 0); n++;
    XtSetArg(warg[n], XtNshowGrip, FALSE); n++;
    XtSetArg(warg[n], XtNskipAdjust, TRUE); n++;
    fb->path_text = XtCreateManagedWidget("fb_path_text",
					  labelWidgetClass, 
					  fb->panes, warg, n); n = 0;	
    /* adjouter le texte */
    updatePath(fb);			/* Now READONLY */
    
    createButtons(fb->buttons, 
		  fb,
		  fbButtons);
    
    if(!inwidget)
    {
	XtRealizeWidget(fb->shell);
	PositionPopupShell(fb->shell, (XEvent *)NULL, fb->shell);
	XtPopup(fb->shell, XtGrabNone);
    }
    
    XtSetKeyboardFocus(fb->shell, fb->text );
}

static FileBrowser* getFileBrowser(w)
     Widget w;
{
    FBList *temp = fbl;
    while (temp->fb->text != w)
	temp = temp->next;

    if(temp)
	return(temp->fb);
	    
    return(NULL);
}

static void fbOk(w, fb, e)
     Widget       w;
     FileBrowser *fb;
     XEvent      *e;
{
    char *filename = (char*) XbTextCopyBuffer( fb->text );
    if(filename)
    {
	char temp[MAXPATHLEN+1];
	sprintf(temp, "%s%s", fb->path, filename);
	fb->goodFileName = scopy(temp);
	fb->goodFile = 1;
    }
}

static void fbOkAct(w,e)
     Widget w;
     XEvent *e;
{
    FileBrowser *fb = getFileBrowser(w);
    if (fb == NULL)
      return;
    fbOk(w,fb,e);
}

static void fbCancel(w, fb, e)
     Widget          w;
     FileBrowser     *fb;
     XEvent      *e;
{
    fb->goodFile = -1;
}

static void fbCancelAct(w,e)
     Widget w;
     XEvent *e;
{
    FileBrowser *fb = getFileBrowser(w);
    if (fb == NULL)
      return;
    fbCancel(w,fb,e);
}

/*
 * Call to see if a file or directory */
#define ISDIRECTORY 1
#define ISFILE 2
#define JE_NE_SAIS_PAS 3
int isAdirectory(pathname)
     char *pathname;
{
    struct stat sb;

    if(stat(pathname, &sb))
      {
	/* There's an error here... */
	mm_log("fbSelect: stat error", WARN);
	mm_log(pathname, WARN);
	return JE_NE_SAIS_PAS;
      }
    /* it MUST be a dossier - see getDirectoryListing */
    if (S_ISDIR(sb.st_mode)) 
      return ISDIRECTORY;
    else
      return ISFILE;
}
    
/*
 * We enter here after having both entered and exited
 * the single-click callback below us. Thus, fb->path
 * is already correct BUT we are still displaying the
 * directory, fb->currentDirectory. */
static void fbSelect(w, fb, map)
     Widget       w;
     FileBrowser *fb;
     XbListSelection *map;
{
    XbListSelection *temp = NULL;

    if(map->size != 1)
	return;

    if (!openAndConnectToDirectory(fb, fb->path))
      mm_log("fbSelect: open/connect dir err", WARN);
    else 
      if (temp = getDirectoryListing(fb)) {
	XbBrowserChangeItems(fb->browser, temp);
	fb->resetRoot = 0;	/* we're at a new root */
      }
}


/*
 * Remember that if a double-click occured, then the function below is called
 * before the double-clicker above */
static void updateText( w, fb, map )
     Widget       w;
     FileBrowser *fb;
     XbListSelection *map;
{
    if ( map->size == 1 )
    {
	char ct[MAXPATHLEN + 1], c;
	char *txtptr= NULL;
	int len;
	XbListSelectionPtr temp = XbBrowserCurrentHighlight(fb->browser);

	if (!temp)
	  return;

	/* See if already appended - this may be a different 
	 * directory selection, but at the same root directory
         * level. If already appended, then we need to change
         * "root/dir/" back to "root/" for the append. */
	if (fb->resetRoot) {
	  int len= strlen(fb->path);
	  
	  /* NULL terminating "/" */
	  fb->path[len - 1] = '\0';
	  /* Find previous "/" */
	  txtptr = (char *)rindex(fb->path, '/');
	  if (txtptr) {
	    c = *(txtptr + 1);		/* save it */
	    *(txtptr + 1) = '\0';
	  }
	}
	/* APPEND the new subdirectory name */
	strcpy(ct, fb->path);
	strcat(ct, temp->entries[0].entry);

	switch (isAdirectory(ct)) {
	case ISDIRECTORY:
	  strcpy(fb->path, ct);
	  strcat(fb->path, "/");
	  fb->resetRoot = 1;		/* Need to reset root next time */
	  XbTextClearBuffer(fb->text);	/* Since file not valid */
	  len = strlen(fb->path);
	  break;
	case ISFILE:
	  /* Write filename into text area */
	  XbTextClearBuffer(fb->text);
	  XbTextAppend(fb->text, temp->entries[0].entry);
	  fb->resetRoot = 0;		/* path OK. */
	  break;
	default:
	  break;
	}
	/* WRITE to the text widget */
	updatePath(fb);
	XbListFreeSelection(temp);
	/* NOTE that we did not change directories. We stay at
         * the root */
      } else
	return;
}
/*
 * Update the Read Only text widget */
/* static char le_space[MAXPATHLEN + 1]; */

static void updatePath(fb)
     FileBrowser *fb;
{
  Arg warg[16];
  int n= 0, i;

  if (!fb)
    return;

  XtSetArg(warg[n], XtNlabel, (char *)&fb->path[0]); n++;
  XtSetValues(fb->path_text, warg, n);
}

