/*
 * db.c
 *
 * Single-key in-memory database manager -- access by open hashing.
 *
 */

 /*
  * Newsclip(TM) Library Source Code.
  * Copyright 1989 Looking Glass Software Limited.  All Rights Reserved.
  * Unless otherwise licenced, the only authorized use of this source
  * code is compilation into a binary of the newsclip library for the
  * use of licenced Newsclip customers.  Minor source code modifications
  * are allowed.
  * Use of this code for a short term evaluation of the product, as defined
  * in the associated file, 'Licence', is permitted.
  */

#define static

#include "nl.h"

/* Activity codes for the hashing routines: */

#define HA_FIND		1	/* Locate key in table.			  */
#define HA_ADD		2	/* Add key in table, if not present.	  */
#define HA_DEL		3	/* Delete key in table, if present.	  */

int guess_tbl_size AC((unsigned int));
dbrec *access_db AC((dbptr, int, char *, int));
dbrec *ante_db AC((dbptr, int));
dbrec *pred_db AC((dbptr, int));
unsigned int hash_key_fn AC((char *));

#define check_valid(x)	if( !(x) ) error( "db -- nil database pointer")

/*
 * Some definitions controlling the gathering of statistics.
 */
#ifdef STATS_DB
long Naccesses, Ncollisions;
# define StatAccess()		Naccesses++
# define StatCollision()	Ncollisions++
# define InitStats()		Naccesses = Ncollisions = 0L;
#else
# define StatAccess()
# define StatCollision()
# define InitStats()
#endif /*STATS_DB*/


/* init_db() initializes a database with the given attributes of estimated
 * size,  size of contained data elements and other controls, such as whether
 * an access time is to be attached with every record.  A valid pointer to
 * the constructed database structure is returned, or NULL on error. */

dbptr
init_db( est_size, data_size )
unsigned int est_size;		/* Estimated size of the database.        */
unsigned int data_size;		/* Size of data items attached to keys.   */
{
	dbptr dbvar;		/* New database, created.		  */
	unsigned tbl_size;	/* Actual table size to be used.	  */

	dbvar = (dbptr) perm_alloc( sizeof(db_t) );
	dbvar->flags = 0;
	dbvar->item_size = data_size;
	dbvar->tbl_size = tbl_size = guess_tbl_size( est_size );
	dbvar->table = (dbrec **) perm_alloc( sizeof(dbrec *)*tbl_size );
	dbvar->n_elements = 0;
	zero( dbvar->table, sizeof(dbrec *)*tbl_size );

	InitStats();

	return( dbvar );
}

/* free_database() frees all of the memory resources
 * required by the given database.  Zero is returned. */

int
free_db( dbvar )
dbptr dbvar;
{
	register dbrec *buck, *obuck;
	
	check_valid( dbvar );

	for( buck = first_rec( dbvar ); buck; buck = obuck ) {
		obuck = next_rec( dbvar, buck );
		if( !(buck->flags & RF_STATIC) ) {
			/* Free the key memory if it is
			 * not flagged as static memory. */
			perm_free( buck->key );
			}
		perm_free( buck );
		}

	/* Free the database maintenance structures themselves. */
	perm_free( dbvar->table );
	perm_free( dbvar );

	return( 0 );
}

/* add_rec() inserts a record with the given key into the specified
 * database, according to the actions implied by the given flag argument. */

dbrec *
add_rec( dbvar, key, flags )
dbptr dbvar;
char *key;
int flags;
{
	return( access_db( dbvar, HA_ADD, key, flags ) );
}

/* del_rec() deletes the record with the given key from the specified database.
 * If the record was found and deleted OK, zero is returned; otherwise, one. */

int
del_rec( dbvar, key )
dbptr dbvar;
char *key;
{
	return( access_db( dbvar, HA_DEL, key, 0 ) ? 0 : 1 );
}

/* get_rec() returns a pointer to the data field for the record corresponding
 * to the given key in the specified database, or NULL if the record wasn't
 * found */

dbrec *
get_rec( dbvar, key )
dbptr dbvar;
char *key;
{
	return( dbvar ? access_db( dbvar, HA_FIND, key, 0 ) : (dbrec *) NULL );
}

/* first_rec() returns the first record in the database (that is, the head
 * of the linked list of database records as they exist in the database). */

dbrec *
first_rec( dbvar )
dbptr dbvar;
{
	return( dbvar ? ante_db( dbvar, -1 ) : (dbrec *) NULL );
}


/**************************************************************************
 *
 * Routines for accessing the database information, using open hashing.
 *
 **************************************************************************/

/* access_db() accesses the given database, performing the given action
 * with the supplied data.  This routine represents the common code
 * required for adding, deleting and retrieving database entries. */

static dbrec *
access_db( dbvar, action, key, flags )
dbptr dbvar;			/* Hash table on which to operate. */
int action;			/* Action to perform in hash table. */
char *key;			/* Key to search for or to store the data. */
int flags;			/* Various control flags. */
{
	unsigned int hash_num;	/* Computed hash number. */
	unsigned int buck_num;	/* Computed bucket number. */
	int found = 0;		/* Indicates whether key was found. */
	dbrec *buck, *obuck;	/* Current and last bucket in search. */
	int depth = 0;		/* Depth of the bucket search. */

	check_valid( dbvar );

	hash_num = hash_key_fn( key );
	buck_num = hash_num % dbvar->tbl_size;

	StatAccess();

	if( buck = dbvar->table[buck_num] ) do {
		/* Search the hash chain and look for our key.
		 * If it seems necessary, a pre-screen based on
 		 * hash number can be added in this step. */
		if( !strcmp( buck->key, key ) ) {
			found++;
			break;
			}
		obuck = buck;
		buck = buck->next;
		depth++;
		StatCollision();
		} while( !(obuck->flags & RF_LAST) );

	switch( action ) {
	    case HA_ADD:
		if( found ) {
			/* Return either NULL or the located data record,
			 * depending on the AR_NEWONLY flag setting. */
			return( flags & AR_NEWONLY ? (dbrec *) NULL : buck );
			}
		else {
			/* Allocate and zero memory for a new record. */
			buck = (dbrec *) perm_alloc( dbvar->item_size );
			zero( buck, dbvar->item_size );

			if( flags & AR_NOALLOC ) {
				/* Just copy the pointer to the static key. */
				buck->key = key;
				buck->flags |= RF_STATIC;
				}
			else {
				/* Allocate and copy the key. */
				buck->key = perm_alloc( 1 + strlen(key) );
				strcpy( buck->key, key );
				}

			buck->flags |= RF_LAST;

			/* Link the new item into the bucket list, as
			 * either the first item in a new list, or as
			 * the final item on an existing one. */
			if( depth ) {
				buck->next = obuck->next;
				obuck->next = buck;
				obuck->flags &= ~RF_LAST;
				}
			else {
				dbvar->table[buck_num] = buck;
				if( obuck = pred_db( dbvar, buck_num ) ) {
					buck->next = obuck->next;
					obuck->next = buck;
					}
				else
					buck->next = ante_db( dbvar, buck_num );
				}

			dbvar->n_elements++;
			dirty_db(dbvar);
			return( buck );
			}
		/*NOTREACHED*/
		break;
	    case HA_FIND:
		/* Simply return the located record or NULL. */
		return( found ? buck : (dbrec *) NULL );
		/*NOTREACHED*/
		break;
	    case HA_DEL:
		if( !found )
			return( (dbrec *) NULL );

		/* Excise the item from the list of buckets -- either from
		 * the head of the list, or from an internal position.  */

		if( depth ) {
			obuck->next = buck->next;
			if( buck->flags & RF_LAST )
				obuck->flags |= RF_LAST;
			}
		else {
			dbvar->table[buck_num] = (buck->flags & RF_LAST) ?
						 (dbrec *) NULL : buck->next;
			if( obuck = pred_db( dbvar, buck_num ) )
				obuck->next = buck->next;
			}

		if( !(buck->flags & RF_STATIC) )
			perm_free( buck->key );

		perm_free( buck );

		dbvar->n_elements--;

		/* This return value can only be tested against NULL. */
		return( buck );
		/*NOTREACHED*/
		break;
	    default:
		error( "db -- illegal action code" );
		break;
	    }
}

/* hash_key_fn() defines the hash function for a given ASCIIZ key. */

static unsigned int
hash_key_fn( key )
char *key;			/* Key on which to hash.		*/
{
	unsigned int fn = 0;	/* Result of the hash function.		*/

	while( *key )
		fn = fn*37 ^ (*key++ - ' ');

	return( fn % 1048583 );
}

/* pred_db() skips _backward_ through the hash table bucket array,
 * looking for a hash chain; a pointer to the _final_ entry in the chain
 * is returned if one is found, otherwise NULL is the result. */

static dbrec *
pred_db( dbvar, num )
dbptr dbvar;
int num;
{
	dbrec *buck;

	for( --num; num >= 0; num-- )
		if( dbvar->table[num] )
			break;

	if( num >= 0 ) {
		buck = dbvar->table[num];
		for( ; !(buck->flags & RF_LAST); buck = buck->next )
			;
		return( buck );
		}

	return( (dbrec *) NULL );
}

/* ante_db() skips _forward_ through the hash table array, looking
 * for a hash chain; if found, a pointer to the _first_ entry on the
 * chain is returned, otherwise, NULL is the result. */

static dbrec *
ante_db( dbvar, num )
dbptr dbvar;
int num;
{
	int size;

	for( size = dbvar->tbl_size, num++ ; num < size; num++ )
		if( dbvar->table[num] )
			return( dbvar->table[num] );

	return( (dbrec *) NULL );
}

#define MAX_ONE_PER_BUCKET   200  /* If estimate is less than this, use 1... */
#define DEF_RECS_PER_BUCKET    3  /* ...otherwise, use 3 records/hash bucket. */

typedef struct {
	unsigned int estimate;
	unsigned int prime;
	} ptabent;

ptabent primetab[] =  {	{ 51200, 51203 },
			{ 25600, 25601 },
			{ 12800, 12809 },
			{  6400,  6421 },
			{  3200,  3203 },
			{  1600,  1601 },
			{   800,   809 },
			{   400,   401 },
			{   200,   211 },
			{   100,   101 },
			{    50,    53 },
			{     0,     0 } };

/* guess_tbl_size() attempts to determine a good table size, given an
 * estimate of the number of records to be allocated within the database. */

static int
guess_tbl_size( est_size )
unsigned int est_size;
{
	int r_per_b = 1;		/* Number of records per table entry. */
	int i, tsize;

	if( est_size > MAX_ONE_PER_BUCKET )
		r_per_b = DEF_RECS_PER_BUCKET;

	est_size /= r_per_b;

	for( i = 1, tsize = sizeof(primetab)/sizeof(ptabent); i < tsize; i++ ) {
		if( primetab[i].estimate <= est_size )
			break;
		}
	return( primetab[i-1].prime );
}
