/*
 * mknewsrc.c
 * 
 * mknewsrc -- reads the active file and creates a .newsrc file containing
 * all newsgroups found in the file matching the default argument patterns,
 * with all active news articles marked as unread.   If the +n option is
 * specified, newsgroups are also required to be found in the user's
 * existing .newsrc file, sought for in the DOTDIR directory.
 *
 * The created .newsrc format file, is written to stdout, and coupled with
 * the newsclip software, is useful for searching all news on the machine.
 *
 * Copyright 1989 Looking Glass Software Limited.
 * Use of this code for a short term evaluation of the product, as defined
 * in the associated file, 'Licence', is permitted.
 *
 */

#include <stdio.h>
#include <string.h>
#include "sysdefs.h"
#include "rei.h"
#include "db.h"

extern char *newsrcname;	/* Full path and name of the user's .newsrc */
extern char *dotdir;		/* Name of the user's DOTDIR */
extern char *news_lib_dir;	/* Name of the news directory */
#define ACTIVE	"active"	/* Name of the active file in news directory */
#define NEWSRC	".newsrc"	/* Name of the user newsrc file in his DOTDIR */

char *Prgname;			/* Program invocation name. */

int  Percentage = 100;		/* Percentage of articles to place as uread. */
char *ActiveName;		/* Full name and path of the active file */
FILE *ActiveFile;		/* File descriptor for the active file */
char *Newsrc = (char *) NULL;	/* Name and path of newsrc file to be used */
FILE *NewsrcFile;		/* File descriptor for the newsrc file */
char  UseNewsrc = 0;		/* Flag corresponding to setting of +n option */

rxp_type *Rxps = (rxp_type *) NULL;	/* Ptr to array of compiled REs */

#define DB_FORMAT_R "%255s %ld %ld %c\n"   /* Read format for active file. */
#define N_DB_FIELDS	  4		   /* 4 data fields in active file. */
#define MAX_NAMELEN	255		   /* 255 max bytes in group names. */
#define MAX_LINELEN     256		   /* .25K max line length. */

char  ReadBuffer[MAX_LINELEN];	/* Generic buffer used when reading files. */

#define GROUP_PAT   "^[^ \t:!,<]+:"  /* Pattern representing valid newsgroup */
				     /* names in the user's newsrc file. */

typedef struct _db {
	char *key;			/* Name of the newsgroup */
	struct _db *next;
	unsigned char flags;
	char status;			/* Status in active file */
	long lownum;			/* Lowest acticle */
	long highnum;			/* Highest acticle */
	} act_rec;

extern void init_whoami AC(( void ));
extern char *perm_alloc AC(( int ));
extern char *getenv AC(( char * ));

void  open_files AC(( void ));
void  close_files AC(( void ));
char *make_fullname AC(( char *, char * ));
void  do_process AC(( void ));
void  drive_newsrc AC(( FILE *, FILE * ));
void  drive_active AC(( FILE * ));
void  try_group AC(( char *, long, long ));
char *allocstring AC(( char * ));
dbptr read_database AC(( FILE * ));

int
main( argc, argv )
int argc;
char *argv[];
{
	int i, isplus, nrexps = 0;
	char *argstr, *argline;

	if( !(Prgname = strrchr( argv[0], '/' )) )
		Prgname = argv[0];
	else
		Prgname++;

	if( argc > 1 ) {
		/* If command line group patterns have been specified,
		 * allocate space for and and compile them. */
		Rxps = (rxp_type *) perm_alloc( sizeof(rxp_type)*argc );

		for( i = 1; i < argc; i++ ) {
			argline = argv[i];
			if( argstr = strchr( argline, '=' ) ) {
				if( !*++argstr )
					error(
			 "missing argument for %c= option", argline[0] );
				switch( argline[0] ) {
				    case 'p':
					Percentage = atoi( argstr );
					break;
				    case 'n':
					Newsrc = allocstring( argstr );
					UseNewsrc = 1;
					break;
				    default:
					error( "illegal option %s\n", argline );
				    }
				}
			else if( (isplus = argline[0] == '+') ||
				  argline[0] == '-' ) {
				switch( argline[1] ) {
				    case 'n':
					UseNewsrc = isplus;
					break;
				    default:
					error( "illegal option %s\n", argline );
				    }
				}
			else {
				/* Untagged options are assumed to be patterns
				 * for regular expression compilation. */
				if( !(Rxps[nrexps++] = REG_COMP_P( argv[i] )) )
					/* An error was detected in the RE;
					 * since regcomp already issued a
					 * warning, so just leave quietly. */
					exit( 2 );
				}
			}

		if( nrexps ) {
			/* If any regular expressions were found, terminate
			 * the list of expressions with a null pointer. */
			Rxps[nrexps] = (rxp_type) NULL;
			}
		else {
			/* No regular expressions were specified on the command
			 * line, so free the allocated memory, and indicate
			 * that all groups are to be accepted. */
			perm_free( Rxps );
			Rxps = (rxp_type *) NULL;
			}

		if( Percentage > 100 )
			Percentage = 100;
		else if( Percentage < 0 )
			Percentage = 0;
		}

	open_files();		/* Open the necessary input/output files. */

	if( UseNewsrc )
		drive_newsrc( ActiveFile, NewsrcFile );
	else
		drive_active( ActiveFile );

	close_files();		/* Close the files and clean up. */

	return( 0 );
}

void
open_files()
{
	init_whoami();

	ActiveName = make_fullname( news_lib_dir, ACTIVE );

	if( !(ActiveFile = fopen( ActiveName, "r" )) )
		error( "cannot open active file \"%s\"", ActiveName );

	if( UseNewsrc ) {
		if( !Newsrc )
			Newsrc = newsrcname;
		if( !(NewsrcFile = fopen( Newsrc, "r" )) )
			error( "cannot open newsrc file \"%s\"", Newsrc );
		}
}

char *
make_fullname( dir, name )
char *dir, *name;
{
	int nlen, dlen, add_slash = 0;
	char *fname;

	dlen = strlen( dir );
	if( dir[dlen] != '/' )
		add_slash++;
	nlen = strlen( name ) + dlen + add_slash + 1;
	fname = (char *) perm_alloc( nlen*sizeof(char) );
	strcpy( fname, dir );
	if( add_slash )
		fname[dlen++] = '/';
	strcpy( fname + dlen, name );

	return( fname );
}

void
close_files()
{
	fclose( ActiveFile );
	perm_free( ActiveName );

	if( UseNewsrc ) {
		fclose( NewsrcFile );
		perm_free( Newsrc );
		}
}

/*
 * If in "read .newsrc" mode, then the entire active file is read into
 * a database for subsequent searches, and the user's newsrc file is
 * opened, and read line by line.  If the line matches any of the
 * optionally specified RE patterns,  then the active file information
 * is looked up in the database and used to create the new newsrc record.
 *
 * If not in "read newsrc" mode, then the active file is incrementally
 * read, and each group matched against the user specified RE patterns,
 * if any.
 */

void
drive_active( afptr )
FILE *afptr;		/* File descriptor for the active file. */
{
	char *groupname = ReadBuffer;
	long lownum, highnum;
	char status;
	int i, fstatus, accept;

	do {
		if( N_DB_FIELDS == (fstatus = fscanf( afptr, DB_FORMAT_R,
		      groupname, &highnum, &lownum, &status )) ) {
			try_group( groupname, lownum, highnum );
			}
		else if( EOF != fstatus ) {
			int ch;

			warning( 1, "format error in active file" );
			while( (ch = getc( afptr )) != '\n' ) {
				if( ch == EOF ) {
					fstatus = EOF;
					break;
					}
				}
			}
		} while( EOF != fstatus );
}

void
drive_newsrc( afptr, nfptr )
FILE *afptr;		/* File descriptor for the active file. */
FILE *nfptr;		/* File descriptor for the user's newsrc file. */
{
	dbptr actdb;
	char *groupname = ReadBuffer;
	char *ptr;
	act_rec *item;
	rxp_type rxp;

	actdb = read_database( afptr );
	rxp = REG_COMP_P( GROUP_PAT );

	while( ptr = fgets( groupname, MAX_LINELEN, nfptr ) ) {
		if( !REG_EXEC( rxp, ptr ) ) {
			/* Uncomment the following line to output the line,
			 * if desired.  Note that the line will be truncated
			 * at MAX_LINELEN characters, so it would likely be
			 * advisable to increase MAX_LINELEN as well. */

			/* Copy the unrecognized line to the output stream. */
			/* fputs( groupname, stdout ); */
			continue;
			}

		*(rxp->endp[0]-1) = '\0'; /* Null terminate newsgroup name */
					  /* (by stomping on trailing ":") */

		if( !(item = (act_rec *) get_rec( actdb, groupname )) ) {
			warning( 2, "Ignoring bogus newsgroup \"%s\"", groupname );
			continue;
			}
		try_group( groupname, item->lownum, item->highnum );
		}

	REG_FREE_P( rxp );
}

void
try_group( name, lownum, highnum )
char *name;
long lownum, highnum;
{
	int accept, i;

	if( !Rxps )
		accept = 1;
	else {
		accept = 0;
		for( i = 0; Rxps[i]; i++ )
			if( REG_EXEC( Rxps[i], name ) ) {
				accept = 1;
				break;
				}
		}

	if( accept ) {
		if( highnum < lownum ) {
			/* A quick sanity check.  Perhaps a
			 * warning should be issued here? */
			highnum = lownum;
			warning( 1, "\"%s\": bogus values in active file", name );
			}

		if( Percentage == 100 )
			/* 100% of articles are to be marked as
			 * unread, so take one less than low value. */
			lownum--;

		else if( Percentage == 0 )
			/* All articles are to be marked as
			 * read, so make limit the high value. */
			lownum = highnum;

		else
			/* Oh, well -- have to do some actual calculations
			 * to compute the number of articles. */
			lownum += (long) (0.5 + (highnum - lownum)*
				  ((double)(100 - Percentage)/100));

		printf( "%s:", name );
		if( lownum > 0 )
			printf( " 1-%ld", lownum );
		putchar( '\n' );
		}
}

dbptr
read_database( fptr )
FILE *fptr;			/* Stream containing the active file. */
{
	dbptr dbvar;		/* Hash table used for this database.	*/
	int fstatus;		/* File status indicator.		*/
	char *strbuff = ReadBuffer;  /* Buffer for string entries.	*/
	long low, high;
	char status;
	act_rec *item;

	dbvar = init_db( 450, sizeof(act_rec) );

	do {
		/* If required, allocate new memory for
		 * the next record from the database. */

		if( N_DB_FIELDS == (fstatus = fscanf( fptr, DB_FORMAT_R,
		      strbuff, &high, &low, &status )) ) {
			if( item = (act_rec *)
			    add_rec( dbvar, &strbuff[0], AR_CREATE ) ) {
				item->lownum = low;
				item->highnum = high;
				item->status = status;
				}
			else
				warning( 1, "Error adding database record" );
			}
		else if( EOF != fstatus ) {
			int ch;

			warning( 1, "Format error in active file" );
			while( (ch = getc( fptr )) != '\n' ) {
				if( ch == EOF ) {
					fstatus = EOF;
					break;
					}
				}
			}
		} while( EOF != fstatus );

	return( (dbptr) dbvar );
}

error( ptr, a1, a2, a3, a4, a5, a6 )
char *ptr;
int a1, a2, a3, a4, a5, a6;
{
	fprintf( stderr, "%s: ", Prgname );
	fprintf( stderr, ptr, a1, a2, a3, a4, a5, a6 );
	fputc( '\n', stderr );
	exit( 1 );
}

int warning_level = 1;

warning( level, ptr, a1, a2, a3, a4, a5, a6 )
int level;		/* give warnings below a certain level */
char *ptr;
int a1, a2, a3, a4, a5, a6;
{
	if( level <= warning_level ) {
		fprintf( stderr, "%s: ", Prgname );
		fprintf( stderr, ptr, a1, a2, a3, a4, a5, a6 );
		fputc( '\n', stderr );
		}
}

char *
allocstring( str )
char *str;
{
	char *res;

	res = perm_alloc( strlen(str) + 1 );
	strcpy( res, str );

	return( res );
}

/* Zero out a region of memory.  Your system may have a routine
 * around to do this faster.  If so, remove this and use your own. */

zero( mem, bytes )
char *mem;
int bytes;
{
	register int *p;
	register char *cp;
	int words, extra;

	words = bytes/sizeof(int);
	extra = bytes - sizeof(int)*words;

	p = (int *)mem;
	while( words-- )
		*p++ = 0;
	cp = (char *)p;
	while( extra-- )
		*cp++ = 0;
}
