

/*
 * Database routines and externals for access by the user program
 */
 /*
  * Newsclip(TM) Library Source Code.
  * Copyright 1989 Looking Glass Software Limited.  All Rights Reserved.
  * Unless otherwise licenced, the only authorized use of this source
  * code is compilation into a binary of the newsclip library for the
  * use of licenced Newsclip customers.  Minor source code modifications
  * are allowed.
  * Use of this code for a short term evaluation of the product, as defined
  * in the associated file, 'Licence', is permitted.
  */


#include "nl.h"
#include <ctype.h>
/* The routine to read in a database from a file */

/*
 * Good databases that we have written have a size field at the start.
 * They contain lines that have the field value, the access time and the
 * key, comma delimited.
 * We allow databases with no size field, and assume they aren't that big.
 * We also allow plain old string records.  In this case we give them a
 * value of one and an access time of now.
 * This will all get corrected with the database is written out.
 */

dbptr
read_database( fname )
char *fname;
{
	char buf[MAX_LLEN];
	dbptr ret;
	FILE *dbfile;
	int size;		/* predicted size for DB */
	int value;		/* value of key */
	long actime;		/* access time of key */
	extern long time_now;	/* current time of day */
	userdb *rec;		/* database record to add */

	dbfilename( buf, fname );
	dbfile = fopen( buf, "r" );
	if( !dbfile ) {
		ret = init_db( 30, sizeof(userdb) );
		warning( 3, "Database file %s not found.\n", fname );
		return ret;
		}
	if( fgets( buf, sizeof(buf), dbfile ) ) {
		if( strncmp( buf, "NCDBSIZE=", 9 ) == 0 ) {
			size = atoi(buf+9);
			size = min( size, 20 );
			}
		 else {
			/* no size information in this database */
			size = 30;
			rewind( dbfile );
			}
		ret = init_db( size, sizeof(userdb) );

		/* scan the file so long as there are valid lines */
		while( fgets(buf, sizeof(buf), dbfile ) ) {
			int len;
			char *key;

			len = strlen( buf );
			if( len > 0 ) {
				if( buf[len-1] == '\n' )
					buf[len-1] = 0;
				 else {
					/* throw away data to end of line */
					int c;
					while( ( c = getc(dbfile) ) != '\n' &&
								c != EOF )
							;
					}
				}
			if( buf[0] == '>' && (isdigit(buf[1]) || buf[1]=='-')&&
				sscanf( buf, ">%d,%ld,", &value, &actime ) ==2){

				/* go to after the second comma */
				key = strchr( strchr( buf, ',' ) + 1, ',' ) + 1;
				}
			 else {
				key = buf;
				value = 1;
				actime = time_now;
				}

			rec = (userdb *)add_rec( ret, key, AR_CREATE );
			rec->intval = value;
			rec->access_date = actime;
			}
		}
	 else {
		/* empty file */
		ret = init_db( 30, sizeof( userdb ) );
		}
	fclose( dbfile );
	return ret;
}

/* Write out a database to a file in our format */
/* This does not free or clear the database */

dbptr
write_database( db, str, oldest )
dbptr db;			/* the database to write */
char *str;			/* the filename to write to */
datehold oldest;		/* the oldest record to keep */
{
	FILE *dbfile;
	char fnbuf[MAX_FNAME];
	userdb *rec;
	
	dbfilename( fnbuf, str );
	/* if the file is not there and the database is empty, do not write */
	if( (db == 0 || size_db(db) == 0 ) && access( fnbuf, 0 ) != 0 )
		return;
	dbfile = fopen( fnbuf, "w" );
	if( !dbfile ) {
		warning( 1, "Could not write database %s\n", fnbuf );
		/* Create directories?? */
		return;
		}

	if( db == (dbptr)0 ) {
		fclose( dbfile );
		return;
		}


	/* we have a problem if a key contains a newline */

	fprintf( dbfile, "NCDBSIZE=%ld\n", size_db( db ) );
	for( rec = (userdb *)first_rec(db); rec; rec = next_rec(db,rec) ) {
		if( rec->access_date >= oldest )
			if( fprintf( dbfile, ">%d,%ld,%s\n", rec->intval,
					(long)rec->access_date, rec->name )< 0){
				warning( 1, "Error writing database %s\n", str );
				break;
				}
		}
	fclose( dbfile );
}

/* Create an fresh, empty database for the user */

dbptr
fresh_database(size)
int size;			/* estimate for database size */
{
	return init_db( size, sizeof(userdb) );
}

/* free up the memory used by a database */

free_database( db )
dbptr db;
{
	free_db( db );
}

/* array index routine for lvalue in database */

userdb *
db_create( db, key )
dbptr db;			/* database to create or reference in */
char *key;			/* name of key */
{
	userdb *rec;
	extern long time_now;

	rec = (userdb *)add_rec( db, key, AR_CREATE );
	rec->access_date = time_now;
	return rec;
}

/* array index routine for rvalue in database */

int
db_lookup( db, key )
dbptr db;
char *key;
{
	register userdb *rec;
	extern long time_now;

	rec = (userdb *)get_rec( db, key );
	if( rec ) {
		rec->access_date = time_now;
		return rec->intval;
		}
	 else
		return 0;
}

/* For "for( xxx in yyy )" loops -- this returns the first record in a
 * user's database.
 */

userdb *
ufirst_rec( db )
dbptr db;
{
	return (userdb *)first_rec(db);
}

/* delete element from user database */

db_delete( db, str )
dbptr db;
char *str;
{
	del_rec( db, str );
}

/* Does a file exist?  Returns true if it does */

exists( dbfile )
char * dbfile;
{
	char fbuf[MAX_FNAME];
	dbfilename( fbuf, dbfile );
	return access( fbuf, 0 ) == 0;
}

/* Create a filename, mapping special characters into

   ~	(followed by slash) User's home directory
   ~.	User's 'dot' directory
   ~n	current newsgroup name, with dots changed to slashes
   ~N	current newsgroup in a probably unique single filename form
   ~d	newsgroup in "dir_newsgroup" with dots changed to slashes
   ~D	"dir_newsgroup" in a probably unique single filename form
   ~s	News spool directory
   ~l	News lib directory
   ~u	Userid
   ~p	PID


   For example, if you're 'brad' in the group rec.humor, then:
	"~/News/~n/badusers" expands to "/u/brad/News/rec/humor/badusers",
	the name of a probable 'kill file' database.

 */

dbfilename( fnbuf, fname )
char *fnbuf;		/* place to put the filename */
char *fname;		/* database file name */
{
	register char *inp, *outp;	/* pointers for copying over name */
	extern newsgroup main_newsgroup, dir_newsgroup;
	extern char *dotdir, *news_spool_dir, *news_lib_dir;
	extern char *userid, *homedir;

	inp = fname;
	outp = fnbuf;
	while( *inp ) {
		if( *inp == '~' ) switch( *++inp ) {
			case 0:	/* end of string */
				*outp = '~';
				break;
			case '~':	/* copy tilde and skip it */
				*outp = *inp++;
				break;
			case '.':
				strcpy( outp, dotdir );
				outp += strlen( outp );
				inp++;
				break;
			case '/':	/* just a tilde on its own */
				strcpy( outp, homedir );
				outp += strlen( outp );
				break;
			case 'n':
			case 'd':
				outp += outngf( outp, *inp == 'd'
					? dir_newsgroup : main_newsgroup );
				inp++;
				break;
			case 'N':
			case 'D':
				outp += probably_unique( outp, *inp == 'D'
					? dir_newsgroup : main_newsgroup );
				inp++;
				break;

			case 's':
				strcpy( outp, news_spool_dir );
				outp += strlen( outp );
				inp++;
				break;
			case 'l':
				strcpy( outp, news_lib_dir );
				outp += strlen( outp );
				inp++;
				break;
			case 'p':
				sprintf( outp, "%d", getpid() );
				outp += strlen( outp );
				inp++;
				break;
			case 'u':
				strcpy( outp, userid );
				outp += strlen( outp );
				inp++;
				break;
			}
		 else
			*outp++ = *inp++;
		}
	*outp = 0;
}

/* Stores a newsgroup name into a string, mapping dots to slashes.
   Returns the length of the string */

int
outngf( str, group )
char *str;			/* string to store onto */
newsgroup group;
{
	register char *gname;
	int count;

	count = 0;
	for( gname = ngn(group); *gname; count++, gname++ )
		*str++ = *gname == '.' ? '/' : *gname;
	*str = 0;
	return count;
}

/* Function to make a probably unique single filename from a newsgroup */

int
probably_unique( str, group )
char *str;			/* string to store onto */
newsgroup group;		/* group to place in name */
{
	int csum;		/* checksum for name */
	int len;		/* length of newsgroup name */
	int bytes_left;		/* bytes left in which to store name */
	char *name, *p;		/* name and pointers into it */
	int lastdot;		/* did we do a dot last char? */
	int bytes_out;

	name = ngn(group);
	len = strlen( name );
	/* put in full name if it fits */
	if( len <= MAX_FNAME_LEN ) {
		strcpy( str, name );
		return len;
		}
	/* checksum name */
	for( csum = 0, p = name; *p; p++ )
		csum += *p;
	/* take mode 62 */
	csum %= 62;
	/* output checksum letter from 0..9A..Za..z */
	*str++ = csum < 36 ? (csum >= 10 ?'A'+csum-10: '0'+csum) : 'a'+csum-36;

	bytes_out = 1;
	bytes_left = MAX_FNAME_LEN - bytes_out;
	lastdot = TRUE;
	for( p = name; *p && bytes_left > 0; p++, len-- ) {
		if( len <= bytes_left ) {
			strcpy( str, p );
			return bytes_out+len;
			}
		if( lastdot ) {
			*str++ = toupper( *p );
			bytes_out++;
			bytes_left--;
			}
		lastdot = *p == '.';
		}
	*str = 0;
	return bytes_out;
}
