/***************************************************************************/
/* FUNCTION: int prog_title()
/*
/* DESCRIPTION:
/*    Print program title information to standard out.
/*
/* USAGE:
/*    error_ret = prog_title(pname,pver,dev_month,dev_year,pstat,ustr);
/*
/* ARGUMENTS:
/*    char            *pname     (in) : program name (20 chars max)
/*    float           pver       (in) : version number [0.0..99.0]
/*    unsigned short  dev_month  (in) : month of development [1..12]
/*    unsigned short  dev_year   (in) : year of development [1988..1999]
/*    char            pstat      (in) : program status [P,D,R]
/*                                    :    P - Production Code
/*                                    :    D - Development Code
/*                                    :    R - Research Code
/*    char            *ustr      (in) : program string (40 chars max)
/*
/* RETURNS: (int)
/*    0 : function call completed successfully
/*    1 : illegal program name length
/*    2 : illegal value for pver
/*    3 : illegal value for dev_month
/*    4 : illegal value for dev_year
/*    5 : illegal value for pstat
/*    6 : illegal program string length
/*
/* COMMENTS:
/*
/* EXAMPLE:
/*    err_ret = prog_title("test",1.1,10,1990,'D','copyright 1990');
/*    The above statement would print the following:
/*
/*   +-------------------------------------------------------------------+
/*   |   test,  version: 1.1 date: Oct, 1990                          [D]|
/*   |   copyright 1990, NCSA, Champaign, IL                             |
/*   +-------------------------------------------------------------------+
/*
/* SEE ALSO:
/*    prog_opts(), opt_loc().
/*
/* INFO:
/*    Author          : Ray Idaszak
/*    Date            : November 3, 1988
/*    Last modified   : May 7, 1990
/*    email           : rayi@ncsa.uiuc.edu
/***************************************************************************/

#include "rtep.h"

int prog_title(pname,pver,dev_month,dev_year,pstat,ustr)
char *pname;
float pver;
unsigned short dev_month,dev_year;
char pstat;
char *ustr;
{
	int i,str_len,bar_len,m1,err_ret;
	int max_len = 80;

	char ch;
	char month[4],*months,*m;
	char *bar,*pstr;
	
	months = "JanFebMarAprMayJunJulAugSepOctNovDec";
	str_len = strlen(months);

	err_ret = 0;

	str_len = strlen(pname);
	if ((str_len < 2) || (str_len > 20)){
		err_ret = 1;
		goto error;
	}

	if ((pver < 0.0) || (pver > 99.9)){
		err_ret = 2;
		goto error;
	}

	if ((dev_month < 1) || (dev_month > 12)) {
		err_ret = 3;
		goto error;
	}

	if ((dev_year < 1988) || (dev_year > 1999)) {
		err_ret = 4;
		goto error;
	}

	if ((pstat != 'P') && (pstat != 'D') && (pstat != 'R')){
		err_ret = 5;
		goto error;
	}

	str_len = strlen(ustr);
	if ((str_len < 1) || (str_len > 40)){
		err_ret = 6;
		goto error;
	}

	m1 = (dev_month * 3) - 3;

	m = &months[m1];
	month[0] = *m++;
	month[1] = *m++;
	month[2] = *m;
	month[3] = '\0';
	
	pstr = (char *)malloc(max_len);
	if (!pstr) {
		printf("memory request for pstr failed\n");
		exit(1);
	}

/*  
     +-------------------------------------------------------------------+
     |   pname,  version: pver date: dev_month, dev_year          [pstat]|
     |   ustr, NCSA, Champaign, IL                                       |
     +-------------------------------------------------------------------+
*/

	printf("\n");

     	bar = "    +-------------------------------------------------------------------+";
	bar_len = strlen(bar);
	printf("%s\n",bar);

	sprintf(pstr,"    |   %s,  version: %3.2f, date: %s, %d",pname,pver,month,dev_year);
	str_len = strlen(pstr);

	for (i=str_len; i<bar_len-4; i++){
		pstr[i]=' ';
	}
	pstr[bar_len-4] = '[';
	pstr[bar_len-3] = pstat;
	pstr[bar_len-2] = ']';
	pstr[bar_len-1] = '|';
	pstr[bar_len] = '\0';
	printf("%s\n",pstr);

	sprintf(pstr,"    |   %s, NCSA, Urbana-Champaign, IL",ustr);
	str_len = strlen(pstr);

	for (i=str_len; i<bar_len-1; i++){
		pstr[i]=' ';
	}
	pstr[bar_len-1] = '|';
	pstr[bar_len] = '\0';
	printf("%s\n",pstr);

	printf("%s\n",bar);

	printf("\n");

	free(pstr);

error:
#if RTE_PRERR
	err_msg("prog_title",err_ret);
#endif
	return (err_ret);
}

