.TH FTNCHEK 1L "January 1993"
.\"======================================================================
.SH "NAME"
ftnchek \- Fortran program checker
.SH "SYNOPSIS"
.B ftnchek
[
.BI \-array= num
]
[
.B \-[no]calltree
]
[
.BI \-columns= num
]
.if n .ti +5
[
.BI \-common= num
]
[
.B \-[no]declare
]
.if t .ti +.5i
[
.B \-[no]division
]
.if n .ti +5
[
.B \-[no]extern
]
[
.B \-[no]f77
]
[
.B \-[no]help
]
.if n .ti +5
[
.B \-[no]hollerith
]
.if t .ti +.5i
[
.BI \-include= str
]
[
.B \-[no]library
]
.if n .ti +5
[
.B \-[no]linebreak
]
[
.B \-[no]list
]
[
.BI \-[no]novice
]
.if n .ti +5
.if t .ti +.5i
[
.BI \-output= str
]
[
.B \-[no]portability
]
[
.B \-[no]pretty
]
.if n .ti +5
[
.B \-[no]project
]
[
.B \-[no]pure
]
.if t .ti +.5i
[
.B \-[no]sixchar
]
.if n .ti +5
[
.B \-[no]symtab
]
[
.B \-[no]truncation
]
[
.BI \-usage= num
]
.if t .ti +.5i
.if n .ti +5
[
.B \-[no]verbose
]
[
.I files ...
]
.if t .ti +.5i
.SH "INTRODUCTION"
.PP
.B Ftnchek
(short for Fortran checker)
is designed to detect certain errors in a
Fortran program that a compiler usually does not.
.B Ftnchek
is not primarily
intended to detect syntax errors.  Its purpose is to assist the user in finding
semantic errors.  Semantic errors are legal in the Fortran language but are
wasteful or may cause incorrect operation.  For example, variables which are
never used may indicate some omission in the program; uninitialized variables
contain garbage which may cause incorrect results to be calculated; and
variables which are not declared may not have the intended type.
.B Ftnchek
is
intended to assist users in the debugging of their Fortran program.  It is not
intended to catch all syntax errors.  This is the function of the compiler.
Prior to using
.BR Ftnchek ,
the user should verify that the program compiles
correctly.
.PP
This document first summarizes how to invoke
.BR Ftnchek .
That section should be read before beginning to use
.BR Ftnchek .
Later sections describe
.BR Ftnchek 's
options in more detail, give an example of its use, and explain
how to interpret the output.  The final sections mention the limitations
and known bugs in
.BR Ftnchek .
.SH "INVOKING FTNCHEK"
.B Ftnchek
is
invoked through a command of the form:
.sp 1
.ce 1
\fC$ ftnchek [-option -option ...] filename [filename ...]\fP
.PP
The brackets indicate something which is optional.  The brackets
themselves are not actually typed.  Here options are command-line
switches or settings, which control the operation of the program and the
amount of information that will be printed out.  If no option is
specified, the default action is to print error messages, warnings,
and informational messages, but not the program listing or symbol tables.
.PP
Each option begins with
the '\-'
character.
(On VAX/VMS or MS-DOS systems you may use either '/'
or '\-'.)
The options are described at greater length in the next section.
.PP
.B Ftnchek
options fall into two categories: switches, which are either
true or false, and settings, which have a numeric or string value.  The
name of a switch is prefixed by 'no' to turn it off:
e.g.
.B \-nopure
would turn off the warnings about impure functions. The 'no' prefix
can also be used with numeric settings, having the effect of turning
off the corresponding warnings.
Only the first 3
characters of an option name (not counting
the '\-')
need be provided.
.PP
The switches which
.B Ftnchek
currently recognizes are:
.TP
.B \-calltree
Print tree of subprogram call hierarchy.  Default = no.
.TP
.B \-declare
Print a list of all identifiers whose datatype is not
explicitly declared.  Default = no.
.TP
.B \-division
Warn wherever division is done (except division by a
constant).  Default = no.
.TP
.B \-extern
Warn if external subprograms which are invoked
are never defined.  Default = yes.
.TP
.B \-f77
Warn about extensions to the Fortran 77 standard.  Default = no.
.TP
.B \-help
Print command summary.  Default = no.
.TP
.B \-hollerith
Warn about hollerith constants under
.B \-port
option.  Default = yes.
.TP
.B \-library
Begin library mode: do not warn if subprograms
in file are defined but never used.  Default = no.
.TP
.B \-linebreak
Treat linebreaks in continued statements as space.
Default = yes.
.TP
.B \-list
Print source listing of program.  Default = no.
.TP
.B \-novice
Give warnings suitable for novice users.
.TP
.B \-portability
Warn about non-portable usages.  Default = no.
.TP
.B \-pretty
Give certain messages related to appearance of source
code.    Default = yes.
.TP
.B \-project
Create project file (see explanation below).  Default =
no.
.TP
.B \-pure
Assume functions have no side effects.  Default = yes.
.TP
.B \-sixchar
List any variable names which clash at 6 characters length.
Default = no.
.TP
.B \-symtab
Print out symbol table.  Default = no.
.TP
.B \-truncation
Check for possible truncation errors.  Default = yes.
.TP
.B \-verbose
Produce full amount of output.  Default = yes.
.PP
There are six settings:
.TP
.BI \-array= n
Set level of strictness in checking array arguments of
subprograms.  Min is 0 (least checking).  Max is 3 (most checking).
Default = 3.
.TP
.BI \-columns= n
Set maximum line length to \fIn\fP columns. (Beyond
this is ignored.)  Max is 132.  Default = 72.
.TP
.BI \-common= n
Set level of strictness in checking \fCCOMMON\fP blocks.
Min is 0 (no checking).  Max is 3 (must be identical).  Default = 3.
.TP
.BI \-include= path
Define a directory to search for include files.
Cumulative.
.TP
.BI \-output= filename
Send output to the given file.
Default is to send output to the screen. (Default filename extension
is \fI.\|lis\fP).
.TP
.BI \-usage= n
Control warnings about unused variables, etc.  Min is 0
(no checking).  Default = 3 (most checking).
.PP
When more than one option is used, they should be separated by a blank
space, except on systems such as VMS where options begin with slash ( / ).
No blank spaces may be placed around the equals sign ( = ) in a setting.
\fBFtnchek "?"\fP
will produce a list of all options and settings.
.PP
When giving a name of an input file, the extension is optional.  If no
extension is given,
.B Ftnchek
will first look for a project file
with extension \fI.\|prj\fP, and will use that if it exists.  If not, then
.B Ftnchek
will look for a Fortran source file with the
extension \fI.\|for\fP for VMS systems, \fI.\|f\fP for Unix systems.
More than one file name can be given to
.BR Ftnchek ,
and it will process the
modules in all files as if they were in a single file.
.PP
If no filename is given,
.B Ftnchek
will read input from the
standard input.
.SH "FTNCHEK OPTIONS"
This section provides a more detailed discussion of
.B Ftnchek
command-line options.  Options and filenames may be interspersed on a
command line.  Each option remains in effect from the point it is
encountered until it is overridden by a later option.  Thus for
example, the listing may be suppressed for some files and not for
others.
.PP
The option names in the following list are in alphabetical order.


.TP
.BI \-array= num
Controls warnings about
mismatches between actual and dummy subprogram array arguments.  (An actual
argument is an argument passed to the subprogram by the caller; a
dummy argument is an argument received by the subprogram.)  Default = 3.
The warnings which can be turned off are for constructions that might
legitimately be used by a knowledgeable programmer, but that often
indicate programming errors.
.IP
The meanings of the setting values are as follows:
.RS
.TP \w'1.'u+2n
0:
only the warnings noted below.
.TP \w'1.'u+2n
1:
give warnings if the arguments differ in their number of dimensions,
or if the actual argument is an array element while the dummy argument
is a whole array.
.TP \w'1.'u+2n
2:
give warnings if the arguments are both arrays, but differ in size.
.TP \w'1.'u+2n
3:
give both types of warnings.
.RE
.IP
Note: no warning is ever given if the actual argument is an array
element while the dummy argument is a scalar variable, and a warning
is always given regardless of this setting if the actual argument
is an array while the dummy argument is a scalar variable, or if the
actual argument is a scalar variable or expression while the dummy
argument is an array.
Variable-dimensioned arrays match any array size.

.TP
.B \-calltree
Causes
.B Ftnchek
to print out the call structure of
the complete program in the form of a tree.  The tree is printed out
starting from the main program, which is listed on the first line at
the left margin.  Then on the following lines, each routine called by
the main program is listed, indented a few spaces, followed by the
subtree starting at that routine.  Default = no.
.IP
If a routine is called by more than one other routine,
its call subtree is printed only the first time it is encountered.
Later calls give only the routine name and the notice "(see above)".
.IP
Note that the call tree will be incomplete if any of the input files
are project files that were created in
.B \-library
mode.  See the
discussion of project files below.
.IP
Technical points: Each list of routines called by a given routine is
printed in alphabetical order.  If no main program is found, a report
to that effect is printed out, and no call tree is printed.  If
multiple main programs are found, the call tree of each is printed
separately.
.IP
Now that
.B Ftnchek
recognizes the call tree structure of a program,
its checking behavior is somewhat altered from previous versions,
which checked the calls of every routine by every other
routine, regardless of whether those routines could ever actually be
invoked at run time.  Now, if a file is read with the
.B \-library
flag in effect, the calls made by a routine in that file
will be checked only if the calling routine is in the main
program's call tree.
Likewise, \fCCOMMON\fP declarations in a
library file will only be checked if the routine is in the call tree.
If the
.B \-library
flag is not set,
.B Ftnchek
will check all
inter-module calls and all common declarations, as it did formerly.
(If there is no main program anywhere in the set of files that
.B Ftnchek
has read, so that there is no call tree, then library
routines will be checked if they are called by any routine in the
complete set of files.)

.TP
.BI \-columns= n
Set maximum statement length to \fIn\fP columns.
(Beyond this is ignored.)
This setting is provided to allow checking of programs which may
violate the Fortran standard limit of 72 columns for the length of a
statement.  According to
the standard, all characters past column 72 are ignored.
If this setting is used when the
.B \-f77
option is in effect, a
warning will be given for any overlength lines that are processed.
Max is 132.  Default = 72.

.TP
.BI \-common= n
This setting varies the strictness of
checking of \fCCOMMON\fP blocks.  Default = 3.
.IP
The different levels are:
.RS
.TP \w'1.'u+2n
0:
no checking.
.TP \w'1.'u+2n
1:
in each declaration of a given \fCCOMMON\fP block,
corresponding memory locations must agree in data type.
.TP \w'1.'u+2n
2:
also warn if different declarations
of the same block are not equal in total length.
.TP \w'1.'u+2n
3:
corresponding variables in each declaration of a block must
agree in data type and (if arrays) in size and number of dimensions.
.RE

.TP
.B \-declare
If this flag is set, all identifiers whose datatype is
not declared in each module will be listed.  This flag is useful for
helping to find misspelled variable names, etc.  The same listing will
be given if the module contains an \fCIMPLICIT NONE\fP statement.
Default = no.

.TP
.B \-division
This switch is provided to help users spot potential
division by zero problems.  If this switch is selected, every division
except by a constant will be flagged.  (It is assumed that the user is
intelligent enough not to divide by a constant which is equal to zero!)
Default = no.

.TP
.B \-extern
Causes
.B Ftnchek
to report whether any subprograms
invoked by the program are never defined, or are multiply defined.
Ordinarily, if
.B Ftnchek
is being run on a complete program, each
subprogram other than the intrinsic functions should be defined once
and only once somewhere.  Turn off this switch if you just want to
check a subset of files which form part of a larger complete program,
or to check all at once a number of unrelated files which might each
contain an unnamed main program.  Subprogram arguments will still be
checked for correctness.  Default = yes.

.TP
.B \-f77
Use this flag to catch language extensions which violate
the Fortran 77 standard.  Such extensions may cause your program not
to be portable.  Examples include the use of underscores in variable
names; variable names longer than six characters; statement lines
longer than 72 characters; and nonstandard statements such as the
\fCDO\fP ... \fCENDDO\fP structure.
.B Ftnchek
does not report on
the use of lowercase letters.  Default=no.

.TP
.B \-help
This command is identical in function to
the \fB"?"\fP
argument, and is provided simply as a convenience for those systems in
which the question mark has special meaning to the command
interpreter.  Default = no.

.TP
.B \-hollerith
Hollerith constants (other than within format
specifications) are a source of possible portability problems, so when
the
.B \-portability
flag is set, warnings about them will be
produced.  If your program uses many hollerith constants, these
warnings can obscure other more serious warnings.  So you can set this
flag to "no" to suppress the warnings about holleriths.  This flag has
no effect when the
.B \-portability
flag is turned off.
Default = yes.

.TP
.BI \-include= path
Specifies a directory to be searched for files
specified by \fCINCLUDE\fP statements.  Unlike other command-line
options, this setting is cumulative; that is, if it is given more than
once on the command line, all the directories so specified are placed
on a list that will be searched in the same order as they are given.
The order in which
.B Ftnchek
searches for a file to be included is:
the current directory; the directory specified by
environment
variable \fCFTNCHEK_INCLUDE\fP
if any;
the directories specified by any
.B \-include
options;
the directory specified by environment variable \fCINCLUDE\fP;
and finally in
a standard systemwide directory
(\fC/usr/include\fP for Unix, \fCSYS$LIBRARY\fP for VMS,
and \fC\\include\fP
for MSDOS).

.TP
.B \-library
This switch is used when a number of subprograms are
contained in a file, but not all of them are used by the application.
Normally,
.B Ftnchek
warns you if any subprograms are defined but
never used.  This switch will suppress these warnings.  Default = no.

.TP
.B \-linebreak
Normally, when scanning a statement which is
continued onto the next line,
.B Ftnchek
treats the end of the line
as a space.  This behavior is the same as for Pascal and C, and also
corresponds to how humans normally would read and write programs.
However, occasionally one would like to use
.B Ftnchek
to check a
program in which identifiers and keywords are split across lines, for
instance programs which are produced using a preprocessor.  Choosing
the option
.B \-nolinebreak
will cause
.B Ftnchek
to skip over the
end of line and also any leading space on the continuation line (from
the continuation mark up to the first nonspace character).  Default =
yes, i.e. treat linebreaks as space.  Default = no.
.IP
Note that in nolinebreak mode, if token pairs requiring intervening
space (for instance, \fCGOTO 100\fP) are separated only by a
linebreak, they will be rejoined.
.IP
Also, tokens requiring more than one character of lookahead for the
resolution of ambiguities must not be split across lines.
In particular, a complex constant may not be split across a line.

.TP
.B \-list
Specifies that a listing of the Fortran program is to be
printed out with line numbers.  If
.B Ftnchek
detects an error, the
error message follows the program line with a caret
( ^ )
specifying the location of the error.  If no source listing was
requested,
.B Ftnchek
will still print out any line containing an
error, to aid the user in determining where the error occurred.
Default = no.

.TP
.B \-novice
This flag is intended to provide additional helpful
output for beginners.  At this time, the only extra message it
provides is a comment that any function that is used but not defined
anywhere might be an array which the user forgot to declare in
a \fCDIMENSION\fP statement (since the syntax of an array reference is
the same as that of a function reference).  Default = yes.
.IP
In earlier versions of
.B Ftnchek
, this option could take on various
numerical values, as a way of controlling various classes of warnings.
These warnings are now controlled individually by their own flags.
Novice level 1 is now handled by the
.B \-array
flag; level 2 has
been eliminated; level 3 is equivalent now to setting
.B \-novice
to
yes; level 4 is handled by the
.B \-impure
flag.

.TP
.BI \-output= filename
This setting is provided for convenience on
systems which do not allow easy redirection of output from programs.
When this setting is given, the output which normally appears on the
screen will be sent instead to the named file.  Note, however, that
operational errors of
.B Ftnchek
itself (e.g. out of space or cannot
open file) will still be sent to the screen.  The extension for the
filename is optional, and if no extension is given, the
extension \fI.\|lis\fP will be used.

.TP
.B \-portability
.B Ftnchek
will give warnings for a variety of
non-portable usages.  These include the use of tabs except in comments
or inside strings, the use of hollerith constants, and the
equivalencing of variables of different data types.  This option does
not produce warnings for violations of the Fortran 77 standard, which
may also cause portability problems.  To catch those, use
the
.B \-f77
option.  Default = no.

.TP
.B \-pretty
Controls certain messages related to the appearance of
the source code.  These warn about things that might in some cases be
deceptive to the reader.  At present, the only warning that is
controlled by this flag refers to comments that are interspersed among
the continuation lines of a statement.  Default = yes.

.TP
.B \-project
.B Ftnchek
will create a project file from each
source file that is input while this flag is in effect.  The project
file will be given the same name as the input file, but with the
extension \fI.\|f\fP or \fI.\|for\fP replaced by \fI.\|prj\fP.
(If input is from standard input, the project file is named
.IR ftnchek.\|prj .)
Default = no.
.IP
A project file contains a summary of information from the source file,
for use in checking agreement
among \fCFUNCTION\fP, \fCSUBROUTINE\fP, and \fCCOMMON\fP
block usages in other files.  It
allows incremental checking, which saves time whenever you have a
large set of files containing shared subroutines, most of which seldom
change.  You can run
.B Ftnchek
once on each file with the
.B \-project
flag set, creating the project files.  Usually you
would also set the
.B \-library
and
.B \-noextern
flags at this
time, to suppress messages relating to consistency with other files.
Only error messages pertaining to each file by itself will be printed
at this time.  Thereafter, run
.B Ftnchek
without these flags on all
the project files together, to check consistency among the different
files.  All messages internal to the individual files will now be
omitted.  Only when a file is altered will a new project file need to
be made for it.
.IP
Project files contain only information needed for checking
agreement between files.  This means that a project file is of no use if all
modules of the complete program are contained in a single file.
.IP
Ordinarily, project files should be created with the
.B \-library
flag in effect.  In this mode,
the information saved in the project file consists of all subprogram
declarations, all subprogram invocations not resolved by declarations in the
same file, and one instance of each \fCCOMMON\fP
block declaration.  This is the minimum amount of information needed
to check agreement between files.  Of course, this means that the calling
hierarchy among routines defined within the file is lost.  Normally
the loss of this information is unimportant.  If you wish to retain
this information for some reason, you can create the project file with
the
.B \-library
flag turned off.  In this mode,
.B Ftnchek
saves, besides
the information listed above, one invocation of each subprogram by
any other subprogram in the same file, and all common block
declarations.  This means that the project file will be larger than
necessary, and that when it is read in,
.B Ftnchek
may repeat some
inter-module checks that it already did when the project file was created.
.IP
Because of the loss of information entailed by creating a project file
with the
.B \-library
flag in effect, whenever that project file is read
in later, it will be treated as a library file regardless of the
current setting of the
.B \-library
flag.  On the other hand, a project
file created with library mode turned off can be read in later in either
mode.
.IP
Naturally, when the
.B \-project
flag is set,
.B Ftnchek
will not
read project files as input.
.IP
Here is an example of how to use the Unix
.B make
utility to
automatically create a new project file each time the corresponding
source file is altered, and to check the set of files for consistency.
The example assumes that a macro \fCOBJS\fP has been defined which
lists all the names of object files to be linked together to form the
complete executable program.
.nf
\fC
    # tell make what a project file suffix is
    .SUFFIXES: .prj

    # tell make how to create a .prj file from a .f file
    .f.prj:
            ftnchek -project -noextern -library $<

    # set up macro PRJS containing project filenames
    PRJS= $(OBJS:.o=.prj)

    # "make check" will check everything that has been changed.
    check: $(PRJS)
            ftnchek $(PRJS)
\fP
.fi

.TP
.B \-pure
Assume functions are "pure", i.e., they will not have
side effects by modifying their arguments or variables in a common
block.  When this flag is in effect,
.B Ftnchek
will base its
determination of set and used status of the actual arguments on the
assumption that arguments passed to a function are not altered.  It
will also issue a warning if a function is found to modify any of its
arguments or any common variables.
Default=yes.
.IP
When this flag is turned off, actual arguments passed to functions
will be handled the same way as actual arguments passed to
subroutines.  This means that
.B Ftnchek
will assume that arguments
may be modified by the functions.  No warnings will be given if a
function is found to have side effects.  Because stricter checking is
possible if functions are assumed to be pure, you should turn this flag
off only if your program actually uses functions with side effects.

.TP
.B \-sixchar
One of the goals of the
.B Ftnchek
program is to
help users to write portable Fortran programs.  One potential source
of nonportability is the use of variable names that are longer than
six characters.  Some compilers just ignore the extra characters.
This behavior could potentially lead to two different variables being
considered as the same.  For instance, variables named \fCAVERAGECOST\fP
and \fCAVERAGEPRICE\fP are the same in the first six characters.  If
you wish to catch such possible conflicts, use this flag.  Default =
no.

.TP
.B \-symtab
A symbol table will be printed out for each module,
listing all identifiers mentioned in the module.  This table gives the
name of each variable, its datatype, and the number of dimensions for
arrays.  An asterisk (*) indicates that the variable has been
implicitly typed, rather than being named in an explicit type
declaration statement.  The table also lists all subprograms invoked
by the module, all \fCCOMMON\fP blocks declared, etc.  Default = no.

.TP
.B \-truncation
Warn about possible truncation (or roundoff)
errors.  Most of
these are related to integer arithmetic.  The warnings enabled when
this flag is in effect are: conversion of a complex or double
precision value to single precision; conversion of any real type to
integer; use of the result of integer division where a real result
seems intended (namely as an exponent, or if the quotient is later
converted to real); division in an integer constant expression that
yields a result of zero; exponentiation of an integer by a negative
integer (which yields zero unless the base integer is 1 in magnitude);
and use of a non-integer array subscript, DO index or DO loop bounds.
Default=yes.
.IP
Note: warnings about truncating type conversions are given only when
the conversion is done automatically, i.e.
by an assignment statement.  If intrinsic
functions such as \fCINT\fP are used to perform the conversion, no warning
is given.


.TP
.BI \-usage= n
Warn about unused or possible uninitialized variables.
Default=3.
.IP
The meanings of the setting values are as follows:
.RS
.TP \w'1.'u+2n
0:
no warnings.
.TP \w'1.'u+2n
1:
warn if variables are (or may be) used before they are set.
.TP \w'1.'u+2n
2:
warn if variables are declared or set but never used.
.TP \w'1.'u+2n
3:
give both types of warnings.
.RE
.IP
Sometimes
.B Ftnchek
makes a mistake about
these warnings.  Usually it errs on the side of giving a warning where no
problem exists, but in rare cases it may fail to warn where the
problem does exist.  See the section on bugs for examples.  If
variables are equivalenced, the rule used by
.B Ftnchek
is that a
reference to any variable implies the same reference to all variables
it is equivalenced to.  For arrays, the rule is that a reference to
any array element is treated as a reference to all elements of the array.

.TP
.B \-verbose
This option is on by default.  Turning it off reduces
the amount of output relating to normal operation, so that error
messages are more apparent.  This option is provided for the
convenience of users who are checking large suites of files.  The
eliminated output includes the names of project files, and the message
reporting that no syntax errors were found.  (Some of this output is
turned back on by the
.B \-list
and
.B \-symtab
options.)  Default
= yes.
.SH "CHANGING THE DEFAULTS"
.B Ftnchek
includes a mechanism for changing the
default values of all options
by defining environment variables.  When
.B Ftnchek
starts up,
it looks in its environment for any variables whose names are composed by
prefixing the string
\fCFTNCHEK_\fP
onto the uppercased version of the option
name.  If such a
variable is found, its value is used to specify the default for the
corresponding switch or setting.  In the case of settings (for example,
the novice level) the value of the environment variable is read as the
default setting value.  In the case of switches, the default switch will
be taken as true or yes unless the environment variable has the value
\fC0\fP or \fCNO\fP.  Of course,
command-line options will override these defaults the same way as they
override the built-in defaults.
.PP
Note that the environment variable name must be constructed with the
full-length option name, which must be in uppercase.  For example, to
make
.B Ftnchek
print a source listing by default, set the environment
variable
\fCFTNCHEK_LIST\fP
to \fC1\fP or \fCYES\fP or anything other than \fC0\fP or \fCNO\fP.
The names
\fCFTNCHEK_LIS\fP
(not the full option name) or
\fCftnchek_list\fP
(lower case) would not be recognized.
.PP
Here are some examples of how to set environment variables on various
systems.  For simplicity, all the examples set the default
.B \-list
switch to
.B \-YES
.
.PP
.fc > @
.if n .ta 30 +35
.if t .ta 2i +2i
>1. Unix, Bourne shell:  >\fC$ FTNCHEK_LIST=YES\fP
>                        >\fC$ export FTNCHEK_LIST\fP

>2. Unix, C shell:       >\fC% setenv FTNCHEK_LIST YES\fP

>3. VAX/VMS:             >\fC$ DEFINE FTNCHEK_LIST YES\fP

>4. MSDOS:               >\fC$ SET FTNCHEK_LIST=YES\fP
.fc

.SH "AN EXAMPLE"
The following simple Fortran program illustrates the messages given by
.BR Ftnchek .
The program is intended to accept an array of test scores
and then compute the average for the series.
.PP
.nf
\fC
C       AUTHORS: MIKE MYERS AND LUCIA SPAGNUOLO
C       DATE:    MAY 8, 1989

C       Variables:
C               SCORE -> an array of test scores
C               SUM ->   sum of the test scores
C               COUNT -> counter of scores read in
C               I ->     loop counter

        REAL FUNCTION COMPAV(SCORE,COUNT)
            INTEGER SUM,COUNT,J,SCORE(5)

            DO 30 I = 1,COUNT
                SUM = SUM + SCORE(I)
30          CONTINUE
            COMPAV = SUM/COUNT
        END


        PROGRAM AVENUM
C
C                       MAIN PROGRAM
C
C       AUTHOR:   LOIS BIGBIE
C       DATE:     MAY 15, 1990
C
C       Variables:
C               MAXNOS -> maximum number of input values
C               NUMS    -> an array of numbers
C               COUNT   -> exact number of input values
C               AVG     -> average returned by COMPAV
C               I       -> loop counter
C

            PARAMETER(MAXNOS=5)
            INTEGER I, COUNT
            REAL NUMS(MAXNOS), AVG
            COUNT = 0
            DO 80 I = 1,MAXNOS
                READ (5,*,END=100) NUMS(I)
                COUNT = COUNT + 1
80          CONTINUE
100         AVG = COMPAV(NUMS, COUNT)
        END
\fP
.fi
.PP
The compiler gives no error messages when this program is compiled.
Yet here is what happens when it is run:
.PP
.nf
\fC
$ run average
70
90
85
<EOF>
$
\fP
.fi
.PP
What happened?  Why didn't the program do anything?
The following is the output from
.B Ftnchek
when it is used to debug the above
program:

.PP
.nf
\fC

$ ftnchek -list -symtab average

FTNCHEK Version 2.6 January 1993

File average.f:

      1 C       AUTHORS: MIKE MYERS AND LUCIA SPAGNUOLO
      2 C       DATE:    MAY 8, 1989
      3
      4 C       Variables:
      5 C               SCORE -> an array of test scores
      6 C               SUM ->   sum of the test scores
      7 C               COUNT -> counter of scores read in
      8 C               I ->     loop counter
      9
     10         REAL FUNCTION COMPAV(SCORE,COUNT)
     11             INTEGER SUM,COUNT,J,SCORE(5)
     12
     13             DO 30 I = 1,COUNT
     14                 SUM = SUM + SCORE(I)
     15 30          CONTINUE
     16             COMPAV = SUM/COUNT
                           ^
Warning near line 16 col 20: integer quotient expr converted to real
     17         END
     18

Module COMPAV: func: real

Variables:

      Name Type Dims     Name Type Dims     Name Type Dims     Name Type Dims
    COMPAV real         COUNT intg             I intg*            J intg
     SCORE intg  1        SUM intg

* Variable not declared. Type has been implicitly defined.


Variables declared but never referenced in module COMPAV:
         J

Variables may be used before set in module COMPAV:
       SUM


     19
     20         PROGRAM AVENUM
     21 C
     22 C                       MAIN PROGRAM
     23 C
     24 C       AUTHOR:   LOIS BIGBIE
     25 C       DATE:     MAY 15, 1990
     26 C
     27 C       Variables:
     28 C               MAXNOS -> maximum number of input values
     29 C               NUMS    -> an array of numbers
     30 C               COUNT   -> exact number of input values
     31 C               AVG     -> average returned by COMPAV
     32 C               I       -> loop counter
     33 C
     34
     35             PARAMETER(MAXNOS=5)
     36             INTEGER I, COUNT
     37             REAL NUMS(MAXNOS), AVG
     38             COUNT = 0
     39             DO 80 I = 1,MAXNOS
     40                 READ (5,*,END=100) NUMS(I)
     41                 COUNT = COUNT + 1
     42 80          CONTINUE
     43 100         AVG = COMPAV(NUMS, COUNT)
     44         END

Module AVENUM: prog

External subprograms referenced:

    COMPAV: real*

Variables:

      Name Type Dims     Name Type Dims     Name Type Dims     Name Type Dims
       AVG real         COUNT intg             I intg        MAXNOS intg*
      NUMS real  1

* Variable not declared. Type has been implicitly defined.


Variables set but never used in module AVENUM:
       AVG



 0 syntax errors detected in file average.f
 1 warning issued in file average.f

Subprogram COMPAV:  argument data type mismatch
  at position 1:
        Dummy type intg  in module COMPAV line 10 file average.f
        Actual type real  in module AVENUM line 43 file average.f
\fP
.fi
.PP
According to
.BR Ftnchek ,
the program contains variables which may be
used before they are assigned an initial value, and variables which
are not needed.
.B Ftnchek
also warns the user that an integer
quotient has been converted to a real. This may assist the user in
catching an unintended roundoff error.  Since the
.B \-symtab
flag
was given,
.B Ftnchek
prints out a table containing identifiers from
the local module and their corresponding datatype and number of
dimensions.  Finally,
.B Ftnchek
warns that the function is not used
with the proper type of arguments.
.PP
With
.BR Ftnchek 's
help, we can debug the program.  We can see that there
were the following errors:
.TP \w'1.'u+2n
1.
\fCSUM\fP and \fCCOUNT\fP
should have been converted to real before doing the division.
.TP \w'1.'u+2n
2.
\fCSUM\fP should have been initialized to 0 before entering the loop.
.TP \w'1.'u+2n
3.
\fCAVG\fP was never printed out after being calculated.
.TP \w'1.'u+2n
4.
\fCNUMS\fP should have been declared \fCINTEGER\fP
instead of \fCREAL\fP.
.PP
We also see that \fCI\fP, not \fCJ\fP, should have been declared
\fCINTEGER\fP in function \fCCOMPAV\fP. Also, \fCMAXNOS\fP was not
declared as \fCINTEGER\fP, nor \fCCOMPAV\fP as \fCREAL\fP, in
program \fCAVENUM\fP.  These are not errors, but they may indicate
carelessness.  As it happened, the default type of these variables
coincided with the intended type.
.PP
 Here is the corrected program, and its output when run:
.PP
.nf
\fC
C       AUTHORS: MIKE MYERS AND LUCIA SPAGNUOLO
C       DATE:    MAY 8, 1989
C
C       Variables:
C               SCORE -> an array of test scores
C               SUM ->   sum of the test scores
C               COUNT -> counter of scores read in
C               I ->     loop counter
C
       REAL FUNCTION COMPAV(SCORE,COUNT)
            INTEGER SUM,COUNT,I,SCORE(5)
C
            SUM = 0
            DO 30 I = 1,COUNT
                SUM = SUM + SCORE(I)
30          CONTINUE
            COMPAV = FLOAT(SUM)/FLOAT(COUNT)
        END
C
C
        PROGRAM AVENUM
C
C                       MAIN PROGRAM
C
C       AUTHOR:   LOIS BIGBIE
C       DATE:     MAY 15, 1990
C
C       Variables:
C               MAXNOS -> maximum number of input values
C               NUMS    -> an array of numbers
C               COUNT   -> exact number of input values
C               AVG     -> average returned by COMPAV
C               I       -> loop counter
C
C
            INTEGER MAXNOS
            PARAMETER(MAXNOS=5)
            INTEGER I, NUMS(MAXNOS), COUNT
            REAL AVG,COMPAV
            COUNT = 0
            DO 80 I = 1,MAXNOS
                READ (5,*,END=100) NUMS(I)
                COUNT = COUNT + 1
80          CONTINUE
100         AVG = COMPAV(NUMS, COUNT)
            WRITE(6,*) 'AVERAGE =',AVG
        END

$ run average
70
90
85
<EOF>
AVERAGE =   81.66666
$
\fP
.fi
.PP
With
.BR Ftnchek 's
help, our program is a success!

.SH "INTERPRETING THE OUTPUT"
.B Ftnchek
will print out four main types of messages.  They are
portability warnings, other warnings, informational messages, and syntax
errors.  Portability warnings specify nonstandard usages that may not be
accepted by other compilers.  Other warning messages report potential
errors that are not normally flagged by a compiler.  Informational
messages consist of warnings which may assist the user in the debugging
of their Fortran program.
.PP
Syntax errors are violations of the Fortran language.  The user
should have already eliminated any that are flagged by the Fortran compiler.
.PP
.B Ftnchek
does not detect all syntax errors.  Generally,
.B Ftnchek
only does
as much local syntactic error checking as is necessary in order for it to
work properly.
.PP
If
.B Ftnchek
gives you a syntax error message when the compiler does
not, it may be because your program contains an extension to
standard Fortran which is accepted by the compiler but not by
.BR Ftnchek .
On a VAX/VMS system, you can use the compiler option \fC/STANDARD\fP
to cause the compiler to accept only standard Fortran.
On most Unix or Unix-like systems, this can be accomplished by
setting the flag
.B \-ansi.
.PP
Most error messages are self-explanatory.  Those which need a brief
explanation are listed below.  Please note that any error messages which
begin with "oops"
refer to technical conditions and indicate bugs in
.B Ftnchek
or that its resources have been exceeded.
.PP
The following messages warn about possible portability problems,
including common but nonstandard usages:

.TP
.I "Nonstandard format item"
.B Ftnchek
will flag nonstandard items in a \fCFORMAT\fP statement
which may not be compatible with other systems.
Controlled by
.B \-f77
option.

.TP
.I "Characters past 72 columns"
A statement has been read which has nonblank characters past column 72.
Standard Fortran ignores all text in those columns, but many compilers do
not.  Thus the program may be treated differently by different compilers.
Controlled by
.B \-f77
option and
.B \-columns
setting.

.TP
.I "Warning: file contains tabs.  May not be portable."
.B Ftnchek
expands tabs to be equivalent to spaces up to the next column
which is a multiple of 8.  Some compilers treat tabs differently, and
also it is possible that files sent by electronic mail will have the tabs
converted to blanks in some way.  Therefore files containing tabs may not
be compiled correctly after being transferred.
.B Ftnchek
does not give
this message if tabs only occur within comments or strings.
Controlled by
.B \-portability
option.

.TP
.I "Nonstandard type usage in expression"
The program contains an operation such as a logical operation between
integers, which is not standard, and may not be acceptable to some
compilers.  Controlled by
.B \-f77
option.

.TP
.I "Common block has mixed character and non-character variables"
The ANSI standard requires that if any variable in a \fCCOMMON\fP
block is of type \fCCHARACTER\fP, then all other variables in the
same \fCCOMMON\fP block must also be of type \fCCHARACTER\fP.
Controlled by
.B \-portability
option.

.TP
.I "Common block has long data type following short data type"
Some compilers require that if a \fCCOMMON\fP block contains mixed
data types, all long types (namely \fCDOUBLE PRECISION\fP and
\fCCOMPLEX\fP) must precede all short types (namely \fCINTEGER\fP,
\fCREAL\fP, etc.)
Controlled by
.B \-portability
option.

.TP
.I "Unknown intrinsic function"
This message warns the user that a name declared in an
\fCINTRINSIC\fP statement is unknown to
.BR Ftnchek .
Probably it is
a nonstandard intrinsic function, and so the program will not be
portable.  The function will be treated by
.B Ftnchek
as a
user-defined function.
This warning is not controlled by the
.B \-portability
option, since
it affects
.BR Ftnchek 's
analysis of the program.

.TP
.I "Identifiers which are not unique in first six chars"
Warns that two identifiers which are longer than 6 characters do not
differ in first 6 characters.  This is for portability: they may not
be considered distinct by some compilers.
Controlled by
.B \-sixchar
option.

.PP
The following messages warn of possible errors (bugs) that could cause
incorrect operation of the program:

.TP
.I "Integer quotient expr converted to real"
The quotient of two integers results in an integer type result, in
which the fractional part is dropped.  If such an integer expression
involving division is later converted to a real datatype, it may be that
a real type division had been intended.
Controlled by
.B \-truncation
option.

.TP
.I "Integer quotient expr used in exponent"
Similarly, if the quotient of two integers
is used as an exponent, it is quite likely that a real type division was
intended.
Controlled by
.B \-truncation
option.

.TP
.I "Real truncated to intg"
.B Ftnchek
has detected an assignment statement which has a real
expression on the right, but an integer variable on the left.  The
fractional part of the real value will be lost.  If you explicitly
convert the real expression to integer using the \fCINT\fP or
\fCNINT\fP intrinsic function, no warning will be printed.  A similar
message is printed if a double precision expression is assigned to a
single precision variable, etc.
Controlled by
.B \-truncation
option.

.TP
.I "Subscript is not integer"
Since array subscripts are normally integer quantities, the use of a
non-integer expression here may signal an error.
Controlled by
.B \-truncation
option.

.TP
.I "Non-integer DO loop bounds"
This warning is only given when the \fCDO\fP index and bounds are
non-integer.
Use of non-integer quantities in a \fCDO\fP statement may cause
unexpected errors, or different results on different machines, due to
roundoff effects.
Controlled by
.B \-truncation
option.

.TP
.I "DO index is not integer"
This warning is only given when the \fCDO\fP bounds are integer, but
the \fCDO\fP index is not.  It may indicate a failure to declare the
index to be an integer.
Controlled by
.B \-truncation
option.

.TP
.I "Possible division by zero"
This message is printed out wherever division is done (except division
by a constant).  Use it to
help locate a runtime division by zero problem.
Controlled by
.B \-division
option.

.TP
.I "NAME not set when RETURN encountered"
The way that functions in Fortran return a value is by assigning the
value to the name of the function.  This message indicates that the
function was not assigned a value before the point where a
\fCRETURN\fP statement was found.  Therefore it is possible that the
function could return an undefined value.

.TP
.I "Variables used before set"
This message indicates that an identifier is used to compute a value
prior to its initialization.  Such usage may lead to an incorrect value
being computed, since its initial value is not controlled.
Given for
.B \-usage
setting 1 or 3.

.TP
.I "Variables may be used before set"
Similar to used before set except that
.B Ftnchek
is not able to
determine its status with certainty.
.B Ftnchek
assumes a variable
may be used before set if the first usage of the variable occurs prior
in the program text to its assignment.
Given for
.B \-usage
setting 1 or 3.

.TP
.I "Subprogram NAME:  varying length argument lists:"
An inconsistency has been found between the number of dummy arguments
(parameters) a subprogram has and the number of actual arguments given it
in an invocation.
.B Ftnchek
keeps track of all invocations of subprograms
(\fCCALL\fP
statements and expressions using functions) and compares them with
the definitions of the subprograms elsewhere in the source code.  The
Fortran compiler normally does not catch this type of error.

.TP
.I "Subprogram NAME: argument data type mismatch at position n"
The subprogram's
\fIn\fP-th actual argument (in the \fCCALL\fP or the
usage of a function) differs in datatype from the \fIn\fP-th dummy
argument (in the \fCSUBROUTINE\fP or \fCFUNCTION\fP
declaration).  For instance, if the user defines a
subprogram by
.nf
\fC
     SUBROUTINE SUBA(X)
     REAL X
\fP
.fi
.IP
and elsewhere invokes \fCSUBA\fP by
.nf
\fC
     CALL SUBA(2)
\fP
.fi
.IP
.B Ftnchek
will detect the error.  The reason here
is that the number 2 is integer, not real.  The user should have said
.nf
\fC
     CALL SUBA(2.0)
\fP
.fi
.IP
When checking an argument which is a subprogram,
.B Ftnchek
must be
able to determine whether it is a function or a subroutine.  The rules
used by
.B Ftnchek
to do this are as follows: If the subprogram,
besides being passed as an actual argument, is also invoked directly
elsewhere in the same module, then its type is determined by that
usage.  If not, then if the name of the subprogram does not appear in
an explicit type declaration, it is assumed to be a subroutine; if it
is explicitly typed it is taken as a function.  Therefore, subroutines
passed as actual arguments need only be declared by an \fCEXTERNAL\fP
statement in the calling module, whereas functions must also be
explicitly typed in order to avoid generating this error message.


.TP
.I "Subprogram invoked inconsistently"
Here the mismatch is between the datatype of the subprogram itself as
used and as defined. For instance, if the user declares
.nf
\fC
     INTEGER FUNCTION COUNT(A)
\fP
.fi
.IP
and invokes \fCCOUNT\fP in another module as
.nf
\fC
     N = COUNT(A)
\fP
.fi
.IP
without declaring its datatype, it will default to real type, based on
the first letter of its name.  The calling module should have included
the declaration
.nf
\fC
     INTEGER COUNT
\fP
.fi
.IP

.TP
.I "Subprogram NAME: argument usage mismatch"
.B Ftnchek
detects a possible conflict between the way a subprogram uses
an argument and the way in which the argument is supplied to the
subprogram.  The conflict can be one of two types, as outlined below.

.TP
.I "Dummy arg is modified, Actual arg is const or expr"
A dummy argument is an argument as named in a \fCSUBROUTINE\fP or
\fCFUNCTION\fP statement and used within the subprogram.  An actual
argument is an argument as passed to a subroutine or function by the
caller.
.B Ftnchek
is saying that a dummy argument is modified by
the subprogram, i.e. its value will be changed in the calling module.
The corresponding actual argument should not be a constant or
expression, but rather a variable or array element which can be
legitimately assigned to.
Given for
.B \-usage
setting 1 or 3.

.TP
.I "Dummy arg used before set, Actual arg not set"
Here a dummy argument may be used in the subprogram
before having a value assigned to it by the subprogram.  The
corresponding actual argument should have a value assigned to it by the
caller prior to invoking the subprogram.
Given for
.B \-usage
setting 1 or 3.

.TP
.I "Common block NAME: varying length"
A \fCCOMMON\fP block declared in different subprograms has different
numbers of variables in it in different declarations.  This is not
necessarily an error, but it may indicate that a variable is missing
from the list.  Note that according to the Fortran 77 standard,
it is an error for named common blocks (but not blank common) to differ
in length in declarations in different modules.
Given for
.B \-common
setting 2 or 3.

.TP
.I "Common block NAME: data type mismatch at position n"
The \fIn\fP-th variable in the \fCCOMMON\fP block differs in data type
in two different declarations of the \fCCOMMON\fP block.  By default
(common strictness level 3),
.B Ftnchek
is very picky about
\fCCOMMON\fP blocks: the variables listed in them must match exactly
by data type and array dimensions.  That is, the legal pair of
declarations in different modules:
.nf
\fC
     COMMON /COM1/ A,B
\fP
.fi
.IP
and
.nf
\fC
    COMMON /COM1/ A(2)
\fP
.fi
.IP
will cause
.B Ftnchek
to give warnings at strictness level 3.  These
two declarations are legal in Fortran since they both declare two real
variables.  At strictness level 1 or 2, no warning would be given in
this example, but the warning would
be given if there were a data type mismatch, for instance, if \fCB\fP
were declared \fCINTEGER\fP.
Controlled by
.B \-common
setting.

.PP
The following messages refer to local syntax errors:

.TP
.I "Syntax error"
This means that the parser, which analyzes the Fortran
program into expressions,
statements, etc., has been unable to find a valid interpretation for some
portion of a statement in the program.  If the compiler does not report a
syntax error at the same place, the most common explanations are: (1) use
of a reserved word as an array or character variable (see Table 2 in the
section entitled "Limitations and Extensions"), or (2) use of an
extension to ANSI standard Fortran that is not recognized by
.BR Ftnchek .
.IP
NOTE: This message means that the affected statement is not interpreted.
Therefore, it is possible that
.BR Ftnchek 's
subsequent processing will be in error, if it depends on any matters
affected by this statement (type declarations, etc.).

.TP
.I "No path to this statement"
.B Ftnchek
will detect statements which are ignored or by-passed because
there is no foreseeable route to the statement.  For example, an
unnumbered statement (a statement without a statement label), occurring
immediately after a \fCGOTO\fP
statement, cannot possibly be executed.

.TP
.I "Statement out of order."
.B Ftnchek
will detect statements that are out of the
sequence specified for ANSI standard Fortran-77.  Table 1 illustrates the
allowed sequence of statements in the Fortran language.  Statements which are
out of order are nonetheless interpreted by
.BR Ftnchek ,
to prevent "cascades" of error messages.

.IP
.nf
.if n .ne 11
.if t .ne 1.75i
\fC
--------------------------------------------------------
                  |               | implicit
                  |  parameter    |---------------------
                  |               | other specification
        format    |---------------|---------------------
         and      |               | statement-function
        entry     |  data         |---------------------
                  |               | executable
--------------------------------------------------------

                      \fRTable 1\fP
\fP
.fi
.PP
The following messages are informational messages, which probably do
not indicate bugs, but may indicate carelessness or oversights during
modification of a program.

.TP
.I "Continuation follows comment or blank line"
.B Ftnchek
issues this warning message to alert the user that a
continuation of a statement is interspersed with comments, making it easy
to overlook.
Controlled by
.B \-pretty
option.

.TP
.I "Declared but never referenced"
Detects any identifiers that were declared in your program but were
never used, either to be assigned a value or to have their value
accessed.  Variables in \fCCOMMON\fP are excluded.
Given for
.B \-usage
setting 2 or 3.

.TP
.I "Variables set but never used"
.B Ftnchek
will notify the user when a variable has been assigned a
value, but the variable is not otherwise used in the program.  Usually
this results from an oversight.
Given for
.B \-usage
setting 2 or 3.

.TP
.I "Type has been implicitly defined"
.B Ftnchek
will flag all identifiers that are not explicitly typed
and will show the datatype that was assigned through implicit typing.
This provides support for users who wish to declare all variables as
is required in Pascal or some other languages.  This message is
printed only when the
.B \-symtab
option is in effect.
Alternatively, use the
.B \-declare
flag if you want to get a list of all
undeclared variables.

.TP
.I "Possibly it is an array which was not declared"
This message refers to a function invocation or to an argument type
mismatch, for which the possibility exists that what appears to be a
function is actually meant to be an array.  If the programmer forgot to
dimension an array, references to the array will be interpreted as
function invocations.  This message will be suppressed if the name in
question appears in an \fCEXTERNAL\fP or \fCINTRINSIC\fP statement.
Controlled by the
.B \-novice
option.

.SH "LIMITATIONS AND EXTENSIONS"

.B Ftnchek
accepts ANSI standard Fortran-77 programs with the following
exceptions:
.TP
.B Restrictions:
.B Ftnchek
is sensitive to blank spaces.  This encourages the user to
use good programming style.  The rules are similar to Pascal or C where a
blank space is required between identifiers or keywords and not allowed
inside identifiers or keywords.
Any keywords which
occur in pairs may be written as either one or two words,
e.g. \fCELSE IF\fP or \fCELSEIF\fP.
Unlike Pascal and C,
.B Ftnchek
allows blanks inside numeric constants, except within the exponent
part of E and D form numbers.  Also, if the
.B \-nolinebreak
option is selected, the end of line in continued statements is ignored.
.IP
Complex constants are subject to a special restriction:  they
may not be split across lines, even in
.B \-nolinebreak
mode.
.IP
The dummy arguments in statement functions are treated like ordinary
variables of the program.  That is, their scope is the entire module, not
just the statement function definition.
.IP
Some keywords and identifiers are partially reserved.  See Table 2 for details.
.PP
.IP
The following keywords may be freely used as variables:
.nf
\fC
ASSIGN          BLOCK           BLOCKDATA       BYTE
CALL            CHARACTER       COMMON          COMPLEX
CONTINUE        DIMENSION       DO              DOUBLE
DOUBLEPRECISION ELSE            END             ENDDO
ENDIF           ENTRY           EXTERNAL        FILE
FUNCTION        GO              IMPLICIT        INCLUDE
INTEGER         INTRINSIC       LOGICAL         NAMELIST
PAUSE           PRECISION       PROGRAM         REAL
SAVE            STOP            SUBROUTINE      THEN
TO
\fP
.fi
.IP
.IP
The following keywords may be used in scalar contexts only, for example,
not as arrays or as character variables used in substring expressions.
.nf
\fC
ACCEPT          BACKSPACE       CLOSE           DATA
DOWHILE         ELSEIF          ENDFILE         EQUIVALENCE
FORMAT          GOTO            IF              INQUIRE
OPEN            PARAMETER       PRINT           READ
RETURN          REWIND          TYPE            WRITE
WHILE
                      \fRTable 2\fP
\fP
.fi
.IP
.TP
.B Extensions:
Tabs are permitted, and translated into equivalent blanks which correspond
to tab stops every 8 columns.  The standard does not recognize tabs.
Note that some compilers allow tabs, but treat them differently.
.IP
Lower case characters are permitted, and are converted internally to
uppercase except in strings.  The standard specifies upper case only,
except in comments and strings.
.IP
Hollerith constants are permitted, in accordance with the ANSI
Manual, appendix C.  They should not be used in expressions, or confused
with datatype \fCCHARACTER\fP.
.IP
Statements may be longer than 72 columns provided that the setting
.B \-column
was used to increase the limit.  According to the standard, all
text from columns 73 through 80 is ignored, and no line may be longer
than 80 columns.
.IP
Variable names may be longer than six characters.  The standard specifies
six as the maximum.
.IP
Variable names may contain underscores, which are treated the same
as alphabetic letters.  The VMS version of
.B Ftnchek
also allows dollar signs in variable names, but not as the initial
character.
.IP
The \fCDO\fP ...  \fCENDDO\fP control structure is permitted.  The
syntax which is recognized is according to either of the following two
forms:
.in +5
\fCDO\fP [\fIlabel\fP [\fC,\fP]] \fIvar\fP \fC=\fP
\fIexpr\fP \fC,\fP \fIexpr\fP [\fC,\fP \fIexpr\fP]
.br
 ...
.br
\fCEND DO\fP
.in -5
or
.in +5
\fCDO\fP [\fIlabel\fP [\fC,\fP]] \fCWHILE (\fP \fIexpr\fP \fC)\fP
.br
 ...
.br
\fCEND DO\fP
.in -5
where square brackets indicate optional elements.
.IP
The \fCACCEPT\fP and \fCTYPE\fP statements (for terminal I/O) are
permitted, with the same syntax as \fCPRINT\fP.
.IP
Statements may have any number of continuation lines.  The standard
allows a maximum of 19.
.IP
Inline comments, beginning with an exclamation mark, are
permitted.
.IP
\fCNAMELIST\fP I/O is supported.  The syntax is the same as in VAX/VMS
or IBM Fortran.
.IP
The \fCIMPLICIT NONE\fP statement is supported.  The meaning of this
statement is that all variables must have their data types explicitly
declared.  Rather than flag the occurrences of such variables with
syntax error messages,
.B Ftnchek
waits till the end of the module,
and then prints out a list of all undeclared variables.
.IP
Data types \fCINTEGER\fP, \fCREAL\fP, \fCCOMPLEX\fP, and
\fCLOGICAL\fP are allowed to have an optional length specification in
type declarations.  For instance, \fCREAL*8\fP means an 8-byte
floating point data type.  The \fCREAL*8\fP datatype is interpreted by
.B Ftnchek
as equivalent to \fCDOUBLE PRECISION\fP.
.B Ftnchek
ignores length specifications on all other types.  The standard allows
a length specification only for \fCCHARACTER\fP data.
.IP
.B Ftnchek
permits the \fCINCLUDE\fP statement, which causes
inclusion of the text of the given file.  The syntax is
.in +5
\fCINCLUDE '\fP\fIfilename\fP\fC'\fP
.in -5
.IP
When compiled for VMS,
.B Ftnchek
will assume a default
extension of \fI.\|for\fP if no filename extension is given.
Also for compatibility with VMS, the VMS version allows the qualifier
\fC[NO]LIST\fP
following the filename, to control the listing of the included file.
There is no support for including VMS text modules.
.IP
At this time, diagnostic output relating to items contained in include
files is minimal.  Only information about the location in the include
file is given.  There is no traceback giving the parent file(s),
although usually this can be inferred from the context.
.SH "NEW FEATURES"
Here are the changes from Version 2.5 to Version 2.6:
.TP \w'1.'u+2n
1.
The following bugs in Version 2.5 were fixed:
Overflow of large integer parameter values.
Inline comment character mistaken in difficult contexts.
Unnamed \fCBLOCK DATA\fP modules treated as main programs.
\fCDATA\fP implied-do statements sometimes parsed incorrectly.
Size of variable-dimensioned arrays sometimes calculated incorrectly.
Documented bug number 1 in the previous documentation, which
caused a used-before-set warning if a function modifies an argument, has
been fixed.  The warning is now controlled by the
.B \-impure
option.
.TP \w'1.'u+2n
2.
New features:
Support for NAMELIST I/O.
Directories to be searched for include files can be specified.
All keyword pairs are now accepted in both split or single-word form.
\fCPARAMETER\fP definitions may contain intrinsic functions.
Cross-module checking within library files is now limited to modules
in the call tree.
\fCBYTE\fP data type is accepted (treated as INTEGER).
VMS and MS-DOS versions no longer require command-line flags having the "/"
prefix to be separated by space.
.TP \w'1.'u+2n
3.
New command-line flags added and existing flags modified
for better control of error and warning reporting.  Affected flags are:
.BR \-array ,
.BR \-calltree ,
.BR \-help ,
.BR \-hollerith ,
.BR \-novice ,
.BR \-pretty ,
.BR \-pure ,
.BR \-trunc ,
.BR \-usage ,
and the
.B \-no
prefix to turn functions off.  See documentation sections for full
explanation of these flags.
.TP \w'1.'u+2n
4.
New warnings: if array subscript or DO index is non-integer;
if constant value of 0 results from integer division or exponentiation;
and if data type of expression in logical or arithmetic \fCIF\fP
statement is improper.
The warning of "variable declared but not used" is now suppressed when the
declaration is in an include file.
A warning is now given under the
.B \-f77
option if the standard
limit of 19 continuation lines is exceeded.
.PP
Here are the changes from Version 2.4 to Version 2.5:
.TP \w'1.'u+2n
1.
The name was changed from
.B Forchek
to
.BR Ftnchek ,
to avoid confusion with a
similar program named
.B Forcheck
,
developed earlier at Leiden University.
.TP \w'1.'u+2n
2.
Some bugs were fixed:
Version 2.4 incorrectly processed \fCDO\fP index variable
names beginning with D or E.
It did not support the \fC+kP\fP format descriptor.
The VMS version failed to accept the
\fCNOSPANBLOCKS\fP, \fCREADONLY\fP or \fCSHARED\fP keywords in \fCOPEN\fP
statements.
Also, a couple of error messages were improved.
.IP
.SH "BUGS"
.B Ftnchek
still has much room for improvement.
Your feedback is appreciated.  We want to know about any bugs you notice.
Bugs include not only cases in which
.B Ftnchek
issues an error message
where no error exists, but also if
.B Ftnchek
fails to issue a warning when
it ought to.  Note, however, that
.B Ftnchek
is not intended to catch all
syntax errors.  Also, it is not considered a bug for a variable to be
reported as used before set, if the reason is that the usage of the
variable occurs prior in the text to where the variable is set.  For
instance, this could occur when a \fCGOTO\fP causes execution to loop backward
to some previously skipped statements.
.B Ftnchek
does not analyze the
program flow, but assumes that statements occurring earlier in the text
are executed before the following ones.
.PP
We especially want to know if
.B Ftnchek
crashes for any reason.  It is
not supposed to crash, even on programs with syntax errors.  Suggestions
are welcomed for additional features which you would find useful.  Tell
us if any of
.BR Ftnchek 's
messages are incomprehensible.  Comments on the
readability and accuracy of this document are also welcome.
.PP
You may also suggest support for additional extensions to the
Fortran language.  These will be included only if it is felt that the
extensions are sufficiently widely accepted by compilers.
.PP
If you find a bug in
.BR Ftnchek ,
first consult the list of known bugs
below to see if it has already been reported.  Also check the section
entitled "Limitations and Extensions" above for restrictions
that could be causing the problem.  If you do not find the problem
documented in either place, then send a report including
.TP \w'1.'u+2n
1.
The operating system and CPU type on which
.B Ftnchek
is running.
.TP \w'1.'u+2n
2.
The version of
.BR Ftnchek .
.TP \w'1.'u+2n
3.
A brief description of the bug.
.TP \w'1.'u+2n
4.
If possible, a small sample program showing the bug.
.PP
.br
The report should be sent to either of the following addresses:
.PP
.nf
\fC
      MONIOT@FORDMULC.BITNET
      moniot@mary.fordham.edu
\fP
.fi
.IP
.PP
.PP
Highest priority will be given to bugs which cause
.B Ftnchek
to crash.
Bugs involving incorrect warnings or error messages may take longer to fix.
.PP
Certain problems that arise when checking large programs can be fixed
by increasing the sizes of the data areas in
.B Ftnchek
.  (These
problems are generally signaled by error messages beginning with "Oops".)
The simplest way to increase the table sizes is by
recompiling
.B Ftnchek
with the \fCLARGE_MACHINE\fP macro name
defined.  Consult the makefile for the method of doing this.
.PP
 The following is a list of known bugs.
.TP \w'1.'u+2n
1.
Bug: Used-before-set message is suppressed for any variable which
is used as the loop index in an implied-do loop, even if it was in
fact used before being set in some earlier statement.
For example, consider \fCJ\fP in the statement
.IP
.nf
\fC
      WRITE(5,*) (A(J), J=1,10)
\fP
.fi
.IP
.IP
Here
.B Ftnchek
parses the I/O expression,
\fCA(J)\fP, where \fCJ\fP is used, before it parses the implied loop
where \fCJ\fP is set.  Normally this would cause
.B Ftnchek
to report a spurious used-before-set warning for \fCJ\fP.
Since this report is usually in error and occurs fairly commonly,
.B Ftnchek
suppresses the warning for \fCJ\fP altogether.
.IP
Prognosis: A future version of
.B Ftnchek
is planned which will handle
implied-do loops correctly.
.TP \w'1.'u+2n
2.
Bug: Variables used (not as arguments) in statement-function
subprograms do not have their usage status updated when the statement
function is invoked.
.IP
Prognosis: To be fixed in a future version of
.BR Ftnchek .
.TP \w'1.'u+2n
3.
Bug: Length declarations of character variables are not
correctly handled in \fCCOMMON\fP block checking.  Nonstandard length
declarations of other data types, except \fCREAL*8\fP, are also not
handled correctly.
.IP
Prognosis: We hope to fix this soon, possibly in the next release.

.SH "CONCLUSION"
.page
.send toc .blank 2
.hl 1 Conclusion
.b 2
.p
.B Ftnchek
was designed by Dr. Robert Moniot, professor at Fordham
University, College at Lincoln Center.  During the academic year of
1988-1989, Michael Myers and Lucia Spagnuolo developed the program to
perform the variable usage checks.  During the following year it was
augmented by Lois Bigbie to check subprogram arguments and
\fCCOMMON\fP block
declarations.  Brian Downing assisted with the implementation of the
\fCINCLUDE\fP statement.
Additional features will be added as time permits.
.PP
We would like to thank Markus Draxler of the University of Stuttgart,
Greg Flint of Purdue University,
Phil Sterne of Lawrence Livermore National Laboratory,
and
Warren J. Wiscombe of NASA Goddard
for reporting some bugs in Versions 2.1 and 2.2.
We also thank
John Amor of the University of British Columbia,
Daniel P. Giesy of NASA Langley Research Center,
Hugh Nicholas of the Pittsburgh Supercomputing Center,
Dan Severance of Yale University,
and
Larry Weissman of the University of Washington
for suggesting some improvements.
Nelson H. F. Beebe of the University of Utah kindly helped with the
documentation, and pointed out several bugs in Version 2.3.
Reg Clemens of the Air Force Phillips Lab in Albuquerque and Fritz
Keinert of Iowa State University helped debug Version 2.4.
We also thank Jack Dongarra for putting
.B Ftnchek
into the Netlib
library of publicly available software.
.PP
For further information, you may contact Dr. Robert Moniot
at either of the following network addresses:
.PP
.nf
\fC
      MONIOT@FORDMULC.BITNET
      moniot@mary.fordham.edu
\fP
.fi
.IP
.PP
This document is named
.IR ftnchek.man .
.PP
The
.B Ftnchek
program can be obtained
by sending the message
.lt
\fCsend ftnchek from fortran\fP
.el

.br
to the Internet address:
.lt
\fCnetlib@ornl.gov\fP .
.el

Installation requires a C compiler for your computer.
.\"==============================[The End]==============================


