//-------------------------------------------------------------------------
// The Game of Life - A Screen Saver
//
// Written by Larry Salomon, Jr. with the intent of disseminating yet more
// PM source for the purpose of teaching those who are still struggling
// and need a bit of guidance.
//
// The link step of the build process requires the library CMN32.LIB,
// which can be found in the common.zip package also on cdrom.com
// (in the /pub/os2/all/program directory, I believe).  For more inform-
// ation on the common.zip package, see the Programming Reference (.INF
// format) contained in that ZIP file.
//
// Any questions, comments, or concerns can be sent via email to me.  My
// address is os2man@panix.com
//-------------------------------------------------------------------------
#define INCL_DOSMISC
#define INCL_DOSPROCESS
#define INCL_GPIBITMAPS
#define INCL_GPILOGCOLORTABLE
#define INCL_GPIPRIMITIVES
#define INCL_WINBUTTONS
#define INCL_WINDIALOGS
#define INCL_WINENTRYFIELDS
#define INCL_WINFRAMEMGR
#define INCL_WINHELP
#define INCL_WININPUT
#define INCL_WINPALETTE
#define INCL_WINPOINTERS
#define INCL_WINSHELLDATA
#define INCL_WINSTDSLIDER
#define INCL_WINSYS
#define INCL_WINTIMER
#define INCL_WINWINDOWMGR
#include "defs.h"
#include <stdlib.h>
#include <string.h>
#include <time.h>

//-------------------------------------------------------------------------
// The colors to be used.  The age of a square (in ticks) is mod'd with
// the maximum number of colors to get an index into the palette, which
// is based on this array.
//-------------------------------------------------------------------------
static ULONG aulColors[MAX_COLORS]={
   0x00FFFF00L,
   0x00EEFF11L,
   0x00DDFF22L,
   0x00CCFF33L,
   0x00BBFF44L,
   0x00AAFF55L,
   0x0099FF66L,
   0x0088FF77L,
   0x0077FF88L,
   0x0066FF99L,
   0x0055FFAAL,
   0x0044FFBBL,
   0x0033FFCCL,
   0x0022FFDDL,
   0x0011FFEEL,
   0x0000FFFFL,
   0x0011EEFFL,
   0x0022DDFFL,
   0x0033CCFFL,
   0x0044BBFFL,
   0x0055AAFFL,
   0x006699FFL,
   0x007788FFL,
   0x008877FFL,
   0x009966FFL,
   0x00AA55FFL,
   0x00BB44FFL,
   0x00CC33FFL,
   0x00DD22FFL,
   0x00EE11FFL,
   0x00FF00FFL,
   0x00FF11EEL,
   0x00FF22DDL,
   0x00FF33CCL,
   0x00FF44BBL,
   0x00FF55AAL,
   0x00FF6699L,
   0x00FF7788L,
   0x00FF8877L,
   0x00FF9966L,
   0x00FFAA55L,
   0x00FFBB44L,
   0x00FFCC33L,
   0x00FFDD22L,
   0x00FFEE11L,
   0x00000000L
};

MRESULT EXPENTRY saverWndProc(HWND hwndWnd,
                              ULONG ulMsg,
                              MPARAM mpParm1,
                              MPARAM mpParm2)
//-------------------------------------------------------------------------
// Saver window procedure
//-------------------------------------------------------------------------
{
   PSAVERDATA psdData;

   psdData=WinQueryWindowPtr(hwndWnd,0);

   switch (ulMsg) {
   case WM_CREATE:
      {
         SIZEL szlPaint;
         LONG lPalette;
         BITMAPINFOHEADER bmihInfo;
         ULONG ulChanged;
         ULONG ulSzBuf;

         //----------------------------------------------------------------
         // Allocate and initialize the instance data
         //----------------------------------------------------------------
         psdData=calloc(1,sizeof(SAVERDATA));
         if (psdData==NULL) {
            return MRFROMSHORT(TRUE);
         } /* endif */

         WinSetWindowPtr(hwndWnd,0,psdData);

         psdData->usSzStruct=sizeof(SAVERDATA);
         psdData->habAnchor=WinQueryAnchorBlock(hwndWnd);
         psdData->hwndOwner=WinQueryWindow(hwndWnd,QW_OWNER);

         szlPaint.cx=0;
         szlPaint.cy=0;

         psdData->hpsPaint=GpiCreatePS(psdData->habAnchor,
                                       WinOpenWindowDC(hwndWnd),
                                       &szlPaint,
                                       PU_PELS|GPIT_NORMAL|GPIA_ASSOC);
         if (psdData->hpsPaint==NULLHANDLE) {
            free(psdData);
            return MRFROMSHORT(TRUE);
         } /* endif */

         DevQueryCaps(WinQueryWindowDC(hwndWnd),
                      CAPS_ADDITIONAL_GRAPHICS,
                      1,
                      &lPalette);
         if ((lPalette & CAPS_PALETTE_MANAGER)!=0) {
            psdData->hpPalette=GpiCreatePalette(psdData->habAnchor,
                                                0,
                                                LCOLF_CONSECRGB,
                                                MAX_COLORS,
                                                aulColors);
            if (psdData->hpPalette==NULLHANDLE) {
               GpiDestroyPS(psdData->hpsPaint);
               free(psdData);
               return MRFROMSHORT(TRUE);
            } /* endif */

            GpiSelectPalette(psdData->hpsPaint,psdData->hpPalette);
            WinRealizePalette(hwndWnd,psdData->hpsPaint,&ulChanged);
         } else {
            psdData->hpPalette=NULLHANDLE;

            GpiCreateLogColorTable(psdData->hpsPaint,
                                   0,
                                   LCOLF_RGB,
                                   0,
                                   MAX_COLORS,
                                   (PLONG)aulColors);
         } /* endif */

         psdData->hbmSpace=GpiLoadBitmap(psdData->hpsPaint,
                                         NULLHANDLE,
                                         BMP_SPACE,
                                         0,
                                         0);

         GpiQueryBitmapParameters(psdData->hbmSpace,&bmihInfo);
         psdData->szlSpace.cx=bmihInfo.cx;
         psdData->szlSpace.cy=bmihInfo.cy;

         psdData->szlBoard.cx=WinQuerySysValue(HWND_DESKTOP,SV_CXSCREEN)/
                                 psdData->szlSpace.cx;
         psdData->szlBoard.cy=WinQuerySysValue(HWND_DESKTOP,SV_CYSCREEN)/
                                 psdData->szlSpace.cy;

         psdData->psBoard=calloc(psdData->szlBoard.cx*psdData->szlBoard.cy,
                                 sizeof(SHORT));
         if (psdData->psBoard==NULL) {
            GpiDeleteBitmap(psdData->hbmSpace);

            if (psdData->hpPalette!=NULLHANDLE) {
               GpiDeletePalette(psdData->hpPalette);
            } /* endif */

            GpiDestroyPS(psdData->hpsPaint);
            free(psdData);
            return MRFROMSHORT(TRUE);
         } /* endif */

         psdData->psNew=calloc(psdData->szlBoard.cx*psdData->szlBoard.cy,
                               sizeof(SHORT));
         if (psdData->psNew==NULL) {
            GpiDeleteBitmap(psdData->hbmSpace);

            if (psdData->hpPalette!=NULLHANDLE) {
               GpiDeletePalette(psdData->hpPalette);
            } /* endif */

            GpiDestroyPS(psdData->hpsPaint);
            free(psdData->psBoard);
            free(psdData);
            return MRFROMSHORT(TRUE);
         } /* endif */

         //----------------------------------------------------------------
         // If we can't find the settings info in the .INI file, initialize
         // them to the defaults
         //----------------------------------------------------------------
         ulSzBuf=sizeof(SETTINGS);

         if (!PrfQueryProfileData(HINI_USERPROFILE,
                                  PRF_APPNAME,
                                  PRF_SETTINGS,
                                  &psdData->sSettings,
                                  &ulSzBuf) ||
             (psdData->sSettings.usSzStruct!=sizeof(SETTINGS))) {
            psdData->sSettings.usSzStruct=sizeof(SETTINGS);
            psdData->sSettings.ulSaveTime=60;
            psdData->sSettings.bDisable=FALSE;
         } /* endif */

         psdData->ulSeconds=0;

         psdData->bPatternsLoaded=FALSE;
         psdData->usNumPatterns=0;

         _beginthread((PFNASYNC)loadPatternsThread,
                      NULL,
                      0x4000,
                      (PVOID)psdData);

         WinStartTimer(psdData->habAnchor,
                       hwndWnd,
                       TID_SECONDS,
                       1000);

         //----------------------------------------------------------------
         // Tell the DLL to install itself
         //----------------------------------------------------------------
         LifeInit(hwndWnd,&psdData->phdHook);
      }
      break;
   case WM_DESTROY:
      {
         SHORT sIndex;

         LifeTerm();
         WinStopTimer(psdData->habAnchor,hwndWnd,TID_SECONDS);
         WinStopTimer(psdData->habAnchor,hwndWnd,TID_LIFECLICKS);

         if (psdData->hpPalette!=NULLHANDLE) {
            GpiDeletePalette(psdData->hpPalette);
         } /* endif */

         GpiDeleteBitmap(psdData->hbmSpace);
         GpiDestroyPS(psdData->hpsPaint);
         free(psdData->psBoard);
         free(psdData->psNew);

         for (sIndex=0; sIndex<psdData->usNumPatterns; sIndex++) {
            free(psdData->appPatterns[sIndex]);
         } /* endfor */

         free(psdData);
      }
      break;
   case WM_SIZE:
      psdData->szlBoard.cx=SHORT1FROMMP(mpParm2)/psdData->szlSpace.cx;
      psdData->szlBoard.cy=SHORT2FROMMP(mpParm2)/psdData->szlSpace.cy;
      break;
   case WM_TIMER:
      switch (SHORT1FROMMP(mpParm1)) {
      case TID_LIFECLICKS:
         {
            USHORT usX;
            USHORT usY;
            USHORT usNum;
            RECTL rclSpace;

            psdData->ulNumTicks++;

            //-------------------------------------------------------------
            // If we've exceeded the lifespan of this universe, create a
            // new one, else if it's time for the roamer to appear, create
            // it.
            //-------------------------------------------------------------
            if (psdData->ulNumTicks==psdData->ulMaxTicks) {
               initBoard(psdData);
               WinInvalidateRect(hwndWnd,NULL,FALSE);
               WinUpdateWindow(hwndWnd);
            } else
            if (psdData->ulNumTicks==psdData->ulRoamer) {
               initRoamer(psdData);
            } /* endif */

            //-------------------------------------------------------------
            // For every space in the universe, check the number of neighbors
            // and update it accordingly.
            //-------------------------------------------------------------
            for (usX=0; usX<psdData->szlBoard.cx; usX++) {
               for (usY=0; usY<psdData->szlBoard.cy; usY++) {
                  usNum=queryNumNeighbors(psdData,usX,usY);

                  switch (usNum) {
                  case 0:
                  case 1:
                  case 4:
                  case 5:
                  case 6:
                  case 7:
                  case 8:
                     psdData->psNew[BCOORD(psdData,usX,usY)]=-1;
                     break;
                  case 2:
                     if (psdData->psBoard[BCOORD(psdData,usX,usY)]!=-1) {
                        psdData->psNew[BCOORD(psdData,usX,usY)]=
                           psdData->psBoard[BCOORD(psdData,usX,usY)]+1;
                     } else {
                        psdData->psNew[BCOORD(psdData,usX,usY)]=-1;
                     } /* endif */
                     break;
                  case 3:
                     psdData->psNew[BCOORD(psdData,usX,usY)]=
                        psdData->psBoard[BCOORD(psdData,usX,usY)]+1;
                     break;
                  default:
                     break;
                  } /* endswitch */

                  querySpaceRect(hwndWnd,usX,usY,&rclSpace);

                  if ((psdData->psNew[BCOORD(psdData,usX,usY)])==-1) {
                     //----------------------------------------------------
                     // Note that we black out the rectangle only if it wasn't
                     // empty already (otherwise, we would be filling a
                     // rectangle that was already black == lost performance.
                     //
                     // We can't duplicate this optimization in WM_PAINT since
                     // we are invalidated *only* when the window is first
                     // shown or the universe is re-gen'd and we *have* to
                     // paint it black then.
                     //----------------------------------------------------
                     if ((psdData->psBoard[BCOORD(psdData,usX,usY)])!=-1) {
                        if (psdData->hpPalette!=NULLHANDLE) {
                           WinFillRect(psdData->hpsPaint,
                                       &rclSpace,
                                       MAX_COLORS-1);
                        } else {
                           WinFillRect(psdData->hpsPaint,
                                       &rclSpace,
                                       aulColors[MAX_COLORS-1]);
                        } /* endif */
                     } /* endif */
                  } else {
                     if (psdData->hpPalette!=NULLHANDLE) {
                        WinDrawBitmap(psdData->hpsPaint,
                                      psdData->hbmSpace,
                                      NULL,
                                      (PPOINTL)&rclSpace.xLeft,
                                      MAX_COLORS-1,
                                      psdData->psNew[BCOORD(psdData,usX,usY)]%
                                         (MAX_COLORS-1),
                                      DBM_NORMAL);
                     } else {
                        WinDrawBitmap(psdData->hpsPaint,
                                      psdData->hbmSpace,
                                      NULL,
                                      (PPOINTL)&rclSpace.xLeft,
                                      aulColors[psdData->psNew
                                         [BCOORD(psdData,usX,usY)]%
                                         (MAX_COLORS-1)],
                                      aulColors[MAX_COLORS-1],
                                      DBM_NORMAL);
                     } /* endif */
                  } /* endif */
               } /* endfor */
            } /* endfor */

            memcpy(psdData->psBoard,
                   psdData->psNew,
                   psdData->szlBoard.cx*psdData->szlBoard.cy*sizeof(SHORT));
         }
         break;
      case TID_SECONDS:
         {
            USHORT usSelGlobal;
            USHORT usSelLocal;
            PGINFOSEG _Seg16 pgisGlobal;
            ULONG ulCx;
            ULONG ulCy;

            if (!psdData->bPatternsLoaded) {
               psdData->ulSeconds=0;
               return MRFROMSHORT(FALSE);
            } /* endif */

            //-------------------------------------------------------------
            // Talk about some hocus-pocus to call a 16-bit routine!!!
            // Get the current session id and if it is not PM (id==1),
            // don't monitor the inactivity.
            //-------------------------------------------------------------
            DOS16GETINFOSEG((PSEL _Seg16)&usSelGlobal,(PSEL _Seg16)&usSelLocal);
            pgisGlobal=(PGINFOSEG _Seg16)MAKEULONG(0,usSelGlobal);

            if ((pgisGlobal->sgCurrent!=1) || (psdData->sSettings.bDisable)) {
               psdData->ulSeconds=0;
               return MRFROMSHORT(FALSE);
            } /* endif */

            psdData->ulSeconds++;

            //-------------------------------------------------------------
            // If you've been inactive for too long, check your heart. :)
            // Seriously, if the number of seconds of inactivity has
            // elapsed, start the saver.
            //-------------------------------------------------------------
            if (psdData->ulSeconds>=psdData->sSettings.ulSaveTime) {
               initBoard(psdData);

               WinShowPointer(HWND_DESKTOP,FALSE);

               ulCx=WinQuerySysValue(HWND_DESKTOP,SV_CXSCREEN);
               ulCy=WinQuerySysValue(HWND_DESKTOP,SV_CYSCREEN);

               //----------------------------------------------------------
               // What a kludge!  If we don't set the focus to ourselves,
               // certain applications might paint over us (i.e. the
               // system clock).
               //----------------------------------------------------------
               psdData->hwndFocus=WinQueryFocus(HWND_DESKTOP);
               WinSetFocus(HWND_DESKTOP,psdData->hwndOwner);

               WinSetWindowPos(hwndWnd,
                               NULLHANDLE,
                               0,
                               0,
                               ulCx,
                               ulCy,
                               SWP_MOVE|SWP_SIZE|SWP_SHOW);

               //----------------------------------------------------------
               // Stop the inactivity timer and start the universe timer
               //----------------------------------------------------------
               WinStopTimer(psdData->habAnchor,
                            hwndWnd,
                            TID_SECONDS);

               WinStartTimer(psdData->habAnchor,
                             hwndWnd,
                             TID_LIFECLICKS,
                             TM_TIMEOUT);

               //----------------------------------------------------------
               // The usMouseKludge field is used because if you are a
               // hidden window and then you show yourself, if the mouse
               // is over you when you appear you get a pair of WM_MOUSEMOVE
               // messages.  We don't want the saver to appear and then
               // disappear again, so set this to the number of WM_MOUSEMOVEs
               // to ignore so the hook doesn't notify us.
               //----------------------------------------------------------
               psdData->phdHook->usMouseKludge=2;
               psdData->phdHook->bSaving=TRUE;
            } /* endif */
         }
         break;
      default:
         return WinDefWindowProc(hwndWnd,ulMsg,mpParm1,mpParm2);
      } /* endswitch */
      break;
   case WM_PAINT:
      {
         RECTL rclPaint;
         RECTL rclWnd;
         USHORT usX;
         USHORT usY;
         RECTL rclSpace;

         //----------------------------------------------------------------
         // Note that we should never receive this message EXCEPT when we
         // regen the universe since the WM_TIMER code paints the
         // updated board directly.
         //----------------------------------------------------------------
         WinBeginPaint(hwndWnd,psdData->hpsPaint,&rclPaint);

         //----------------------------------------------------------------
         // Paint the sections of the window that are not covered by a
         // space black (aulColors[MAX_COLORS-1]==0x00000000, RGB).
         //----------------------------------------------------------------
         WinQueryWindowRect(hwndWnd,&rclWnd);
         rclWnd.xLeft=psdData->szlBoard.cx*psdData->szlSpace.cx;

         if (psdData->hpPalette!=NULLHANDLE) {
            WinFillRect(psdData->hpsPaint,&rclWnd,MAX_COLORS-1);
         } else {
            WinFillRect(psdData->hpsPaint,&rclWnd,aulColors[MAX_COLORS-1]);
         } /* endif */

         WinQueryWindowRect(hwndWnd,&rclWnd);
         rclWnd.yTop-=psdData->szlBoard.cy*psdData->szlSpace.cy;

         if (psdData->hpPalette!=NULLHANDLE) {
            WinFillRect(psdData->hpsPaint,&rclWnd,MAX_COLORS-1);
         } else {
            WinFillRect(psdData->hpsPaint,&rclWnd,aulColors[MAX_COLORS-1]);
         } /* endif */

         rclPaint.xLeft=rclPaint.xLeft/psdData->szlSpace.cx;
         rclPaint.xRight=rclPaint.xRight/psdData->szlSpace.cx+1;
         rclPaint.yBottom=(rclWnd.yTop-rclPaint.yBottom)/psdData->szlSpace.cy;
         rclPaint.yTop=(rclWnd.yTop-rclPaint.yTop)/psdData->szlSpace.cy+1;

         for (usX=0; usX<psdData->szlBoard.cx; usX++) {
            for (usY=0; usY<psdData->szlBoard.cy; usY++) {
               querySpaceRect(hwndWnd,usX,usY,&rclSpace);

               if ((psdData->psBoard[BCOORD(psdData,usX,usY)])==-1) {
                  if (psdData->hpPalette!=NULLHANDLE) {
                     WinFillRect(psdData->hpsPaint,&rclSpace,MAX_COLORS-1);
                  } else {
                     WinFillRect(psdData->hpsPaint,
                                 &rclSpace,
                                 aulColors[MAX_COLORS-1]);
                  } /* endif */
               } else {
                  if (psdData->hpPalette!=NULLHANDLE) {
                     WinDrawBitmap(psdData->hpsPaint,
                                   psdData->hbmSpace,
                                   NULL,
                                   (PPOINTL)&rclSpace.xLeft,
                                   MAX_COLORS-1,
                                   psdData->psBoard[BCOORD(psdData,usX,usY)]%
                                      (MAX_COLORS-1),
                                   DBM_NORMAL);
                  } else {
                     WinDrawBitmap(psdData->hpsPaint,
                                   psdData->hbmSpace,
                                   NULL,
                                   (PPOINTL)&rclSpace.xLeft,
                                   aulColors[psdData->psBoard
                                      [BCOORD(psdData,usX,usY)]%
                                      (MAX_COLORS-1)],
                                   aulColors[MAX_COLORS-1],
                                   DBM_NORMAL);
                  } /* endif */
               } /* endif */
            } /* endfor */
         } /* endfor */

         WinEndPaint(psdData->hpsPaint);
      }
      break;
   case LFM_INPUT:
      if (psdData->phdHook->bSaving) {
         //----------------------------------------------------------------
         // If we're in saver mode, stop it.
         //----------------------------------------------------------------
         WinStopTimer(psdData->habAnchor,
                      hwndWnd,
                      TID_LIFECLICKS);
         psdData->phdHook->bSaving=FALSE;

         WinSetFocus(HWND_DESKTOP,psdData->hwndFocus);

         WinSetWindowPos(hwndWnd,
                         NULLHANDLE,
                         0,
                         0,
                         0,
                         0,
                         SWP_HIDE);

         WinShowPointer(HWND_DESKTOP,TRUE);
      } /* endif */

      //-------------------------------------------------------------------
      // Reset the inactivity count and restart the timer.
      //-------------------------------------------------------------------
      psdData->ulSeconds=0;
      WinStartTimer(psdData->habAnchor,
                    hwndWnd,
                    TID_SECONDS,
                    1000);
      break;
   default:
      return WinDefWindowProc(hwndWnd,ulMsg,mpParm1,mpParm2);
   } /* endswitch */

   return MRFROMSHORT(FALSE);
}

MRESULT EXPENTRY settingsDlgProc(HWND hwndWnd,
                                 ULONG ulMsg,
                                 MPARAM mpParm1,
                                 MPARAM mpParm2)
{
   PCLIENTDATA pcdData;

   pcdData=WinQueryWindowPtr(hwndWnd,0);

   switch (ulMsg) {
   case WM_INITDLG:
      {
         HELPINIT hiInit;
         CHAR achBuf[256];

         srand(time(NULL));

         //----------------------------------------------------------------
         // Allocate and initialize the instance data
         //----------------------------------------------------------------
         pcdData=calloc(1,sizeof(CLIENTDATA));
         if (pcdData==NULL) {
            WinAlarm(HWND_DESKTOP,WA_ERROR);
            WinDismissDlg(hwndWnd,FALSE);
            return MRFROMSHORT(FALSE);
         } /* endif */

         WinSetWindowPtr(hwndWnd,0,pcdData);

         pcdData->usSzStruct=sizeof(CLIENTDATA);
         pcdData->habAnchor=WinQueryAnchorBlock(hwndWnd);

         pcdData->hwndSaver=WinCreateWindow(HWND_DESKTOP,
                                            CLS_SAVER,
                                            "",
                                            0,
                                            0,
                                            0,
                                            0,
                                            0,
                                            hwndWnd,
                                            HWND_TOP,
                                            WND_SAVER,
                                            NULL,
                                            NULL);
         if (pcdData->hwndSaver==NULLHANDLE) {
            free(pcdData);
            WinAlarm(HWND_DESKTOP,WA_ERROR);
            WinDismissDlg(hwndWnd,FALSE);
            return MRFROMSHORT(FALSE);
         } /* endif */

         pcdData->psdData=WinQueryWindowPtr(pcdData->hwndSaver,0);

         //----------------------------------------------------------------
         // Create the help instance
         //----------------------------------------------------------------
         hiInit.cb=sizeof(HELPINIT);
         hiInit.ulReturnCode=0L;
         hiInit.pszTutorialName=NULL;
         hiInit.phtHelpTable=(PHELPTABLE)MAKEULONG(HELP_SETTINGS,0xFFFF);
         hiInit.hmodHelpTableModule=NULLHANDLE;
         hiInit.hmodAccelActionBarModule=NULLHANDLE;
         hiInit.idAccelTable=0;
         hiInit.idActionBar=0;

         WinLoadString(pcdData->habAnchor,
                       NULLHANDLE,
                       STR_HELPTITLE,
                       sizeof(achBuf),
                       achBuf);
         hiInit.pszHelpWindowTitle=achBuf;

         hiInit.fShowPanelId=CMIC_HIDE_PANEL_ID;
         hiInit.pszHelpLibraryName="LIFE.HLP";

         pcdData->hwndHelp=WinCreateHelpInstance(pcdData->habAnchor,&hiInit);
         if (pcdData->hwndHelp!=NULLHANDLE) {
            if (hiInit.ulReturnCode==0) {
               WinAssociateHelpInstance(pcdData->hwndHelp,hwndWnd);
            } else {
               WinDestroyHelpInstance(pcdData->hwndHelp);
               pcdData->hwndHelp=NULLHANDLE;
            } /* endif */
         } /* endif */

         if (pcdData->hwndHelp==NULLHANDLE) {
            WinEnableWindow(WinWindowFromID(hwndWnd,DST_PB_HELP),FALSE);
         } /* endif */

         pcdData->hpIcon=WinLoadPointer(HWND_DESKTOP,NULLHANDLE,DLG_SETTINGS);
         if (pcdData->hpIcon==NULLHANDLE) {
            WinDestroyWindow(pcdData->hwndSaver);
            WinDestroyHelpInstance(pcdData->hwndHelp);
            free(pcdData);
            WinAlarm(HWND_DESKTOP,WA_ERROR);
            WinDismissDlg(hwndWnd,FALSE);
            return MRFROMSHORT(FALSE);
         } /* endif */

         WinSendDlgItemMsg(hwndWnd,
                           DST_SL_TIMEOUT,
                           SLM_SETTICKSIZE,
                           MPFROM2SHORT(SMA_SETALLTICKS,5),
                           0);

         WinSendMsg(hwndWnd,WM_COMMAND,MPFROMSHORT(DST_PB_UNDO),0);

         //----------------------------------------------------------------
         // Since the initial position of any slider is 0, if the current
         // setting also evaluates to position 0, the slider will not
         // change the value and thus we will not get a WM_CONTROL message.
         // So, send the message anyways so that the entryfield will get
         // updated.  Also, since no changes have been made, disable the
         // Undo button.
         //----------------------------------------------------------------
         WinSendMsg(hwndWnd,
                    WM_CONTROL,
                    MPFROM2SHORT(DST_SL_TIMEOUT,SLN_CHANGE),
                    0);
         WinEnableWindow(WinWindowFromID(hwndWnd,DST_PB_UNDO),FALSE);

         CmnWinCenterWindow(hwndWnd);
      }
      break;
   case WM_DESTROY:
      WinDestroyWindow(pcdData->hwndSaver);
      WinDestroyHelpInstance(pcdData->hwndHelp);
      WinDestroyPointer(pcdData->hpIcon);
      free(pcdData);
      break;
   case WM_MINMAXFRAME:
      {
         PSWP psControl;
         HENUM heEnum;
         HWND hwndChild;

         psControl=(PSWP)PVOIDFROMMP(mpParm1);

         heEnum=WinBeginEnumWindows(hwndWnd);

         hwndChild=WinGetNextWindow(heEnum);
         while (hwndChild!=NULLHANDLE) {
            if ((psControl->fl & SWP_MINIMIZE)!=0) {
               WinShowWindow(hwndChild,FALSE);
            } else
            if ((psControl->fl & (SWP_RESTORE | SWP_MAXIMIZE))!=0) {
               WinShowWindow(hwndChild,TRUE);
            } /* endif */

            hwndChild=WinGetNextWindow(heEnum);
         } /* endwhile */

         WinEndEnumWindows(heEnum);
      }
      break;
   case WM_PAINT:
      {
         HPS hpsPaint;

         if ((WinQueryWindowULong(hwndWnd,QWL_STYLE) & WS_MINIMIZED)==0) {
            return WinDefDlgProc(hwndWnd,ulMsg,mpParm1,mpParm2);
         } /* endif */

         hpsPaint=WinBeginPaint(hwndWnd,NULLHANDLE,NULL);
         WinDrawPointer(hpsPaint,
                        WinQuerySysValue(HWND_DESKTOP,SV_CXSIZEBORDER),
                        WinQuerySysValue(HWND_DESKTOP,SV_CYSIZEBORDER),
                        pcdData->hpIcon,DP_NORMAL);
         WinEndPaint(hpsPaint);
      }
      break;
   case WM_MOUSEMOVE:
      {
         HPOINTER hpPtr;

         if (!pcdData->psdData->bPatternsLoaded) {
            hpPtr=WinQuerySysPointer(HWND_DESKTOP,SPTR_WAIT,FALSE);
            WinSetPointer(HWND_DESKTOP,hpPtr);
            return MRFROMSHORT(TRUE);
         } else {
            return WinDefDlgProc(hwndWnd,ulMsg,mpParm1,mpParm2);
         } /* endif */
      }
   case WM_CONTROLPOINTER:
      {
         HPOINTER hpPtr;

         if (!pcdData->psdData->bPatternsLoaded) {
            hpPtr=WinQuerySysPointer(HWND_DESKTOP,SPTR_WAIT,FALSE);
            return MRFROMLONG(hpPtr);
         } else {
            return WinDefDlgProc(hwndWnd,ulMsg,mpParm1,mpParm2);
         } /* endif */
      }
   case WM_CONTROL:
      switch (SHORT1FROMMP(mpParm1)) {
      case DST_CB_DISABLE:
         switch (SHORT2FROMMP(mpParm1)) {
         case BN_CLICKED:
         case BN_DBLCLICKED:
            WinEnableWindow(WinWindowFromID(hwndWnd,DST_PB_UNDO),TRUE);
            break;
         default:
            return WinDefDlgProc(hwndWnd,ulMsg,mpParm1,mpParm2);
         } /* endswitch */
         break;
      case DST_SL_TIMEOUT:
         switch (SHORT2FROMMP(mpParm1)) {
         case SLN_CHANGE:
         case SLN_SLIDERTRACK:
            {
               ULONG ulPos;
               CHAR achText[32];

               //----------------------------------------------------------
               // The user has changed the slider position, so update the
               // entryfield to reflect the current value
               //----------------------------------------------------------
               ulPos=LONGFROMMR(WinSendDlgItemMsg(hwndWnd,
                                                  DST_SL_TIMEOUT,
                                                  SLM_QUERYSLIDERINFO,
                                                  MPFROM2SHORT(SMA_SLIDERARMPOSITION,
                                                               SMA_INCREMENTVALUE),
                                                  0));
               ulPos=ulPos*15+15;

               sprintf(achText,"%ld:%02ld",ulPos/60,ulPos%60);
               WinSetDlgItemText(hwndWnd,DST_EF_TIMEOUT,achText);

               WinEnableWindow(WinWindowFromID(hwndWnd,DST_PB_UNDO),TRUE);
            }
            break;
         default:
            return WinDefDlgProc(hwndWnd,ulMsg,mpParm1,mpParm2);
         } /* endswitch */
         break;
      default:
         return WinDefDlgProc(hwndWnd,ulMsg,mpParm1,mpParm2);
      } /* endswitch */
      break;
   case WM_COMMAND:
      switch (SHORT1FROMMP(mpParm1)) {
      case DST_PB_SET:
      case DID_OK:
      case DST_PB_APPLY:
         {
            ULONG ulPos;

            //-------------------------------------------------------------
            // Query the values and update the saver instance data.  If
            // "Set" was specified, write to the .INI file.
            //-------------------------------------------------------------
            ulPos=LONGFROMMR(WinSendDlgItemMsg(hwndWnd,
                                               DST_SL_TIMEOUT,
                                               SLM_QUERYSLIDERINFO,
                                               MPFROM2SHORT(SMA_SLIDERARMPOSITION,
                                                            SMA_INCREMENTVALUE),
                                               0));
            ulPos=ulPos*15+15;

            pcdData->psdData->sSettings.ulSaveTime=ulPos;
            pcdData->psdData->ulSeconds=0;

            pcdData->psdData->sSettings.bDisable=
               SHORT1FROMMR(WinSendDlgItemMsg(hwndWnd,
                                              DST_CB_DISABLE,
                                              BM_QUERYCHECK,
                                              0,
                                              0));

            if (SHORT1FROMMP(mpParm1)!=DST_PB_APPLY) {
               PrfWriteProfileData(HINI_USERPROFILE,
                                   PRF_APPNAME,
                                   PRF_SETTINGS,
                                   &pcdData->psdData->sSettings,
                                   (ULONG)pcdData->psdData->usSzStruct);
            } /* endif */

            WinEnableWindow(WinWindowFromID(hwndWnd,DST_PB_UNDO),FALSE);
         }
         break;
      case DST_PB_UNDO:
         //----------------------------------------------------------------
         // Revert back to the last "Set" or "Apply"
         //----------------------------------------------------------------
         WinSendDlgItemMsg(hwndWnd,
                           DST_SL_TIMEOUT,
                           SLM_SETSLIDERINFO,
                           MPFROM2SHORT(SMA_SLIDERARMPOSITION,
                                           SMA_INCREMENTVALUE),
                           MPFROMLONG((pcdData->psdData->sSettings.ulSaveTime-
                                        15)/15));
         WinSendDlgItemMsg(hwndWnd,
                           DST_CB_DISABLE,
                           BM_SETCHECK,
                           MPFROMSHORT(pcdData->psdData->sSettings.bDisable),
                           0);
         WinEnableWindow(WinWindowFromID(hwndWnd,DST_PB_UNDO),FALSE);
         break;
      case DST_PB_HELP:
         WinSendMsg(pcdData->hwndHelp,HM_EXT_HELP,0,0);
         break;
      default:
         break;
      } /* endswitch */
      break;
   default:
      return WinDefDlgProc(hwndWnd,ulMsg,mpParm1,mpParm2);
   } /* endswitch */

   return MRFROMSHORT(FALSE);
}

INT main(VOID)
{
   HAB habAnchor;
   HMQ hmqQueue;

   habAnchor=WinInitialize(0);
   hmqQueue=WinCreateMsgQueue(habAnchor,0);

   //----------------------------------------------------------------------
   // Register the classes of the various types of windows we use
   //----------------------------------------------------------------------
   WinRegisterClass(habAnchor,
                    CLS_SAVER,
                    saverWndProc,
                    CS_SIZEREDRAW,
                    sizeof(PVOID));

   WinDlgBox(HWND_DESKTOP,
             HWND_DESKTOP,
             settingsDlgProc,
             NULLHANDLE,
             DLG_SETTINGS,
             NULL);

   WinDestroyMsgQueue(hmqQueue);
   WinTerminate(habAnchor);
   return 0;
}
