/* SphereState.h - interface to point repulsion simulation class
 *
 * $Id: SphereState.h,v 1.2 96/02/11 21:29:36 leech Exp $
 *
 * Copyright (C) 1996, Jonathan P. Leech
 *
 * This software may be freely copied, modified, and redistributed,
 * provided that this copyright notice is preserved on all copies.
 *
 * There is no warranty or other guarantee of fitness for this software,
 * it is provided solely "as is". Bug reports or fixes may be sent
 * to the author, who may or may not act on them as he desires.
 *
 * You may not include this software in a program or other software product
 * without supplying the source, or without informing the end-user that the
 * source is available for no extra charge.
 *
 * If you modify this software, you should include a notice giving the
 * name of the person performing the modification, the date of modification,
 * and the reason for such modification.
 *
 * $Log:	SphereState.h,v $
 * Revision 1.2  96/02/11  21:29:36  leech
 * Fix potential energy computation and nearest-neighbor computation.
 * Many minor tweaks and reorganization of code.
 * 
 * Revision 1.1  92/04/27  04:25:11  leech
 * Initial revision
 *
 */

#ifndef _jpl_spherestate_h
#define _jpl_spherestate_h

#include "boolean.h"
#include "algebra.h"

// Representation of a single mass point on the sphere.
struct MassPoint {
    double  mass;
    dvector  r;     // position
    dvector  v;     // velocity

    MassPoint() { };
    void randomize(double mass = 1);
};

class SphereState {
public:
    SphereState(unsigned nbody = 4);
   ~SphereState();

    // Current state of bodies
    const MassPoint *Bodies() const { return current; }

    // Accessors
    double  GetSoftening() const;
    double  GetKForce() const;
    double  GetR0() const;
    double  GetDamping() const;
    double  GetTimestep() const;
    double  GetCutoffRadius() const;
    int     GetCriticalDamping() const;
    int     GetNumBodies() const;
    dvector GetPosition(unsigned) const;
    dvector GetVelocity(unsigned) const;

    // Modifiers
    void    SetSoftening(double);
    void    SetKForce(double);
    void    SetR0(double);
    void    SetDamping(double);
    void    SetTimestep(double);
    void    SetCutoffRadius(double);
    void    SetCriticalDamping(int);
    void    SetNumBodies(int);
    void    SetPosition(unsigned, const dvector &);
    void    SetVelocity(unsigned, const dvector &);

    // Functions to drive the simulation
    boolean Timestep();
    ForceLaw
	   *GetForceLaw();

    // Statistics-gathering functions
    double  KineticEnergy();
    double  PotentialEnergy();
    double  MaxVelocity();
    double  ClosestPair();
    double  FurthestPair();
    double  IdealPair();
    int     Interactions();

    // Utility functions
    void    SetDebug(int d);
    int     GetDebug() const;
    boolean ReadBodies(const char *file);
    boolean WriteBodies(const char *file) const;
    double  DefaultTimestep() const;

private:
    void    StepVelocity();
    boolean StepPosition();
    void    InvalidateCache();
    void    ComputeIdealPair();
    void    ComputePairDistances();
    void    AllocateBuffers(int n);
    void    FreeBuffers();

    int
	debug,
	nbody;		// # of bodies to distribute

    // Buffer space
    MassPoint
       *current,    // Current point state
       *next;	    // Updated point state
    double
       *maxdotp;    // Buffer space for ComputePairDistances()
    dvector
       *force;	    // Buffer space for StepVelocity()

    double
	dt,		// Time step
	lambda;		// Velocity damping factor
    double
	cutoffRadius,	// Cutoff radius of force law [0..PI]
	cutoffDotprod;	// cos of maximum separation angle, from cutoffRadius
    ForceLaw
	forceLaw;	// Force law between two points on the sphere

    boolean
	critical_damping;   // True if critical damping is being done

    // Flags for lazy evaluation of cached statistics;
    //	true if quantity is still valid
    boolean
	keComputed,
	peComputed,
	vmaxComputed,
	pairdistComputed;

    // Cached statistics
    double
	ke, pe, vmax,	    // Energies & velocities
	min_pairdist,	    // Minimum near-neighbor distance
	max_pairdist,	    // Maximum near-neighbor distance
	ideal_pairdist;     // Ideal near-neighbor distance
    int
	interactions;	// # of particle-particle interactions in previous timestep
};

#endif /*_jpl_spherestate_h*/
