# ui.tcl - Tk GUI for point repulsion simulation
#
# $Id: ui.tcl,v 1.2 96/02/11 21:33:43 leech Exp $
#
# Copyright (C) 1996, Jonathan P. Leech
#
# This software may be freely copied, modified, and redistributed,
# provided that this copyright notice is preserved on all copies.
#
# There is no warranty or other guarantee of fitness for this software,
# it is provided solely "as is". Bug reports or fixes may be sent
# to the author, who may or may not act on them as he desires.
#
# You may not include this software in a program or other software product
# without supplying the source, or without informing the end-user that the
# source is available for no extra charge.
#
# If you modify this software, you should include a notice giving the
# name of the person performing the modification, the date of modification,
# and the reason for such modification.
#
set __rcslog {
$Log:	ui.tcl,v $
Revision 1.2  96/02/11  21:33:43  leech
Added lots of GUI functionality.

Revision 1.1  96/02/09	16:41:08  leech
Initial revision

}

# Update the point display and status info after something changes
proc points_redisplay { model } {
    points_draw_points $model
    points_status $model
    update
}

# Run or stop simulation
proc points_runstop { model state } {
    global modinfo

    # Old state
    set oldstate $modinfo($model,state)

    # New state
    set modinfo($model,state) $state

    switch $state {
	step {
	    # single-step simulation if not currently running
	    # otherwise do nothing
	    if { ! [string compare $oldstate $state] } {
		points_step $model
	    }
	}
	run {
	    # start running if currently single-stepping
	    # otherwise do nothing
	    if [string compare $oldstate $state] {
		after 1 points_step $model
	    }
	}
    }
}

# Redraw all points on the projected display
proc points_draw_points { model } {
    global modinfo

    # Display canvas with map projection
    set c $modinfo($model,mapwidget)
    set width [$c cget -width]
    set height [$c cget -height]

    set pi 3.14159
    set wscale [expr $width / (2 * $pi)]

    # Update display
    # Incredibly slow Tcl solution, for now

    # Get rid of all canvas objects first
    $c addtag foo all
    $c delete foo

    # Create a whole bunch of new ones
    set n [$model nbody]
    for { set i 0 } { $i < $n } { incr i } {
	set pos [$model pos $i]
	set x [lindex $pos 0]
	set y [lindex $pos 1]
	set z [lindex $pos 2]

	set theta [expr atan2($y,$x)]	    ;# -pi..pi
	set sx [expr ($theta + $pi) * $wscale]

	# canvas y is down, not up
	# might want acos(z)
	set sy [expr (1 - $z) * $height / 2]

	# puts "theta = atan2($y,$x) = $theta; sx = $sx"
	# puts "$i: $sx $sy $sx $sy"
	$c create oval $sx $sy $sx $sy
    }
}

# Timestep simulation
proc points_step { model } {
    global modinfo

    # Run a timestep of simulation
    $model compute

    points_redisplay $model

    # Only queue another step if simulation is running
    if { ! [string compare $modinfo($model,state) run] } {
	after 1 points_step $model
    }
}

# Set damping to critical or user-specified
proc points_damping { model damping } {
    global modinfo
    set widget $modinfo($model,damping,widget)

    switch $damping {
	critical {
	    $model critdamp 1
	    $widget config -state disabled
	}
	user {
	    $model critdamp 0
	    $widget config -state normal
	}
    }
}

# Set timestep value (when changing # of bodies)
#   to automatically computed or user-specified
proc points_timestep_mode { model mode } {
    global modinfo
    set widget $modinfo($model,timestep,widget)

    switch $mode {
	auto {
	    set modinfo($model,timestep_mode) $mode
	    $model timestep [$model idealtimestep]
	    $widget set_value [$model timestep]
	}
	user {
	    set modinfo($model,timestep_mode) $mode
	}
	default {
	    error "Unrecognized timestep mode $mode"
	}
    }
}

# Randomize bodies - perturb everything by
#   a random amount up to half the ideal pair distance.
proc points_randomize { model } {
    set n [$model nbody]
    set scale [$model idealpair]

    for { set i 0 } { $i < $n } { incr i } {
	set pos [$model pos $i]
	set delta [vsub [vector [frand] [frand] [frand]] [vector 0.5 0.5 0.5]]
	set pos [vadd $pos [vsmul $delta $scale]]
	eval $model pos $i $pos
    }

    points_redisplay $model
}

# Set distance function
proc points_distance { model function } {
    $model distfunc $function
    points_redisplay $model
}

# Set falloff function
proc points_falloff { model function } {
    $model falloff $function
    points_redisplay $model
}

# Set # bodies
proc points_nbody { model bodies } {
    global modinfo
    set widget $modinfo($model,timestep,widget)

    $model nbody $bodies

    if { ! [string compare $modinfo($model,timestep_mode) auto] } {
	$model timestep [$model idealtimestep]
	$modinfo($model,timestep,widget) set_value [$model timestep]
    }

    points_redisplay $model
}

# Quit (make sure that 'after' doesn't defeat us)
proc points_quit { model } {
    points_runstop $model step
    after 2
    destroy .
}

# Build gui displaying specified model in specified widget
# user_timestep is 1 if user specified timestep, 0 otherwise
proc points_gui { w model user_timestep } {
    global modinfo

    # Simulation is not running initially
    set modinfo($model,state) step

    # Timestep is selected automatically
    if { $user_timestep } {
	set modinfo($model,timestep_mode) user
    } else {
	set modinfo($model,timestep_mode) auto
    }

    delwin $w
    frame $w

    # Keep track of the widget tree associated with this model
    set modinfo($model,widget) $w

    # Menu bar
    frame $w.mbar -relief raised -border 3

	set win [menubutton $w.mbar.file -text File -menu $w.mbar.file.menu]
	    menu $win.menu
	    $win.menu add command -label "Load Coords" -command "points_load_coords $model"
	    $win.menu add command -label "Save Coords" -command "points_save_coords $model"
	    $win.menu add command -label "Save VRML" -command "points_save_vrml $model"
	    $win.menu add command -label Quit -command "points_quit $model"

	selectmenu $w.mbar.distance "Distance Function" \
	    { arc euclid } \
	    arc "points_distance $model"
	selectmenu $w.mbar.force "Force Law" \
	    { { Linear linear }
	      { Square square }
	      { Lennart-Jones lennart }
	      { Morse morse } } \
	    square "points_falloff $model"

	# Bind menus for traversal
	tk_menuBar $w.mbar $w.mbar.file $w.mbar.distance $w.mbar.force

	pack $w.mbar.file $w.mbar.distance $w.mbar.force -side left

    # Selection panels
    frame $w.control

	set win $w.control
	selector $win.run "Simulation" \
	    { run step } \
	    step "points_runstop $model"
	selector $win.damping "Damping" \
	    { critical user } \
	    critical "points_damping $model"
	selector $win.timestep "Timestep" \
	    { auto user } \
	    $modinfo($model,timestep_mode) "points_timestep_mode $model"
	button $win.randomize -text {Randomize
Points} -command "points_randomize $model"

	pack $win.run $win.damping $win.timestep $win.randomize \
	    -side left -anchor nw -fill x

    # Display area: point projection & convex hull view
    frame $w.display

	set win [frame $w.display.l]
	    frame $win.f -bg red -relief sunken -bd 4
		set modinfo($model,mapwidget) $win.f.view
		canvas $win.f.view -width 360 -height 180 \
		    -bg red -highlightthickness 0
		pack $win.f.view
	    pack $win.f -side top -anchor nw

	    set modinfo($model,statuswidget) $win
	    foreach display {
		ke pe vmax interactions minpair maxpair idealpair ratio
	    } {
		label $win.$display
		pack $win.$display -side top -anchor nw
	    }

	set win [frame $w.display.r]
	    frame $win.f -bg black -relief sunken -bd 4
		set modinfo($model,hullwidget) $win.f.view
		canvas $win.f.view -width 180 -height 180 \
		    -bg black -highlightthickness 0
		pack $win.f.view
	    button $win.update -text "Recompute Hull" \
		-command "points_update_view $model"
	    pack $win.f -side top -anchor nw
	    pack $win.update -side top -anchor n -fill x
	pack $w.display.l $w.display.r -side left -anchor nw -fill x

    # Param window
    set win [frame $w.param]

	Varedit $win.nbody -label "Points" -min 4 -max 100 \
	    -value [$model nbody] -command "points_nbody $model" -integral 1
	pack $win.nbody

	set modinfo($model,damping,widget) $win.damping
	set modinfo($model,timestep,widget) $win.timestep
	foreach scrollbar {
	    { Timestep		    timestep  .01  10	  1 }
	    { "Damping Coefficient" damping   .01  100	  1 }
	    { "Spring Constant"     kforce    .1   10	  1 }
	    { Softening		    softening 0    0.1	  0 }
	    { R0		    r0	      0    3.1415 0 }
	    { "Distance Cutoff"     cutoff    0    3.1415 0 }
	} {
	    set label [lindex $scrollbar 0]
	    set param [lindex $scrollbar 1]
	    set min   [lindex $scrollbar 2]
	    set max   [lindex $scrollbar 3]
	    set log   [lindex $scrollbar 4]

	    Varedit $win.$param -label $label -min $min -max $max \
		-value [$model $param] -command "$model $param"
	    if { $log } {
		$win.$param logarithmic 1
	    }

	    pack $win.$param -fill x
	}

	# Show damping widget as disabled if critical damping enabled
	if [$model critdamp] {
	    points_damping $model critical
	}


    # Pack top-level elements
    pack $w.mbar -side top -fill x
    pack $w.control $w.display $w.param -side top

    pack $w

    # Draw initial state
    points_draw_points $model
    points_status $model

    return $w
}

# Update status display
proc points_status { model } {
    global modinfo

    # Widget containing status display
    set widget $modinfo($model,statuswidget)

    foreach display {
	{ "Kinetic Energy  " ke }
	{ "Potential Energy" pe }
	{ "Maximum Velocity" vmax }
	{ "Interactions    " interactions }
	{ "Min Neighbor    " minpair }
	{ "Max Neighbor    " maxpair }
	{ "Ideal Neighbor  " idealpair }
    } {
	set label [lindex $display 0]
	set param [lindex $display 1]

	$widget.$param config -text "$label [$model $param]"
    }

    set min [$model minpair]
    set ideal [$model idealpair]

    if { $min > 0 } {
	set ratio [expr $ideal / $min]
    } else {
	set ratio "Infinity"
    }

    $widget.ratio config -text "Ratio           $ratio"
}

# Update 3D tesselation
# slow_as_molasses
proc points_update_view { model } {
    global modinfo

    set hull [$model hull]

    # Canvas containing convex hull display
    set c $modinfo($model,hullwidget)
    set height [$c cget -height]
    set width [$c cget -width]

    $c addtag foo all
    $c delete foo

    set scale $height
    if { $width < $scale } {
	set scale $width
    }

    foreach facet $hull {
	set v1 [lindex $facet 0]
	set v2 [lindex $facet 1]
	set v3 [lindex $facet 2]

	set p1 [$model pos $v1]
	set p2 [$model pos $v2]
	set p3 [$model pos $v3]

	set s1 [eval points_project $p1 $scale]
	set s2 [eval points_project $p2 $scale]
	set s3 [eval points_project $p3 $scale]

	eval $c create polygon $s1 $s2 $s3 {-fill "" -outline white}
    }
    update
}

# project onto unit screen, returns 0..scale
proc points_project { x y z scale } {
    set z [expr $z + sqrt(2)]
    return [list [expr $scale * 0.5 * (1 + $x/$z)] \
		 [expr $scale * 0.5 * (1 + $y/$z)]]
    #return [list [expr $scale * 0.5 * (1 + $x)] \
    #		  [expr $scale * 0.5 * (1 + $y)]]
}

# Load simulation state
proc points_load_coords { model } {
    global modinfo

    # Get input file name (should be a dialog)
    set file $modinfo($model,loadfile)

puts "Reading state from $file"
    $model read $file

    # Inform display that # bodies changed & redraw
    points_nbody $model [$model nbody]
}

# Save simulation state in reloadable format
proc points_save_coords { model } {
    global modinfo

    # Get output file name (should be a dialog)
    set file $modinfo($model,savefile)

puts "Writing state to $file"
    $model write $file
}

# Save convex hull of points in VRML format on specified file
proc points_save_vrml { model } {
    global modinfo

    # Get output file name
    set file $modinfo($model,vrmlfile)

puts "Writing VRML to $file"

    set hull [$model hull]
    set fd [open $file w]
    set nbody [$model nbody]

    puts $fd "#VRML V1.0 ascii"
    puts $fd "Separator {"
    puts $fd "\tMaterial { ambientColor 1 1 1 diffuseColor 1 1 1 }"

    # Vertices
    puts $fd "\tCoordinate3 {"
    puts $fd "\t    point \["
    for { set v 0 } { $v < $nbody } { incr v } {
	puts $fd "\t\t[$model pos $v],"
    }
    puts $fd "\t    \]"
    puts $fd "\t}"

    # Polygons
    puts $fd "\tIndexedFaceSet {"
    puts $fd "\t    coordIndex \["
    foreach facet $hull {
	set v1 [lindex $facet 0]
	set v2 [lindex $facet 1]
	set v3 [lindex $facet 2]

	puts $fd "\t\t$v1, $v2, $v3, -1,"
    }
    puts $fd "\t    \]"
    puts $fd "\t}"
    puts $fd "}"

    close $fd
}
