/*-
 * Copyright (c) 2013 The FreeBSD Foundation
 * All rights reserved.
 *
 * This software was developed by Pawel Jakub Dawidek under sponsorship from
 * the FreeBSD Foundation.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD: stable/10/sys/sys/nv_impl.h 292973 2015-12-31 03:28:14Z ngie $
 */

#ifndef	_NV_IMPL_H_
#define	_NV_IMPL_H_

#ifndef	_NVPAIR_T_DECLARED
#define	_NVPAIR_T_DECLARED
struct nvpair;

typedef struct nvpair nvpair_t;
#endif

#define	NV_TYPE_NVLIST_UP		255

#define	NV_TYPE_FIRST		NV_TYPE_NULL
#define	NV_TYPE_LAST		NV_TYPE_BINARY

#define	NV_FLAG_BIG_ENDIAN		0x80

#ifdef _KERNEL
#define	nv_malloc(size)			malloc((size), M_NVLIST, M_NOWAIT)
#define	nv_calloc(n, size)		malloc((n) * (size), M_NVLIST, \
					    M_NOWAIT | M_ZERO)
#define	nv_realloc(buf, size)		realloc((buf), (size), M_NVLIST, \
					    M_NOWAIT)
#define	nv_free(buf)			free((buf), M_NVLIST)
#define	nv_strdup(buf)			strdup((buf), M_NVLIST)
#define	nv_vasprintf(ptr, ...)		vasprintf(ptr, M_NVLIST, __VA_ARGS__)

#define	SAVE_ERRNO(var)			((void)(var))
#define	RESTORE_ERRNO(var)		((void)(var))

#define	ERRNO_OR_DEFAULT(default)	(default)

#else

#define	nv_malloc(size)			malloc((size))
#define	nv_calloc(n, size)		calloc((n), (size))
#define	nv_realloc(buf, size)		realloc((buf), (size))
#define	nv_free(buf)			free((buf))
#define	nv_strdup(buf)			strdup((buf))
#define	nv_vasprintf(ptr, ...)		vasprintf(ptr, __VA_ARGS__)

#define	SAVE_ERRNO(var) 		(var) = errno
#define	RESTORE_ERRNO(var) 		errno = (var)

#define	ERRNO_OR_DEFAULT(default)	(errno == 0 ? (default) : errno)

#endif

int	*nvlist_descriptors(const nvlist_t *nvl, size_t *nitemsp);
size_t	 nvlist_ndescriptors(const nvlist_t *nvl);

nvpair_t *nvlist_first_nvpair(const nvlist_t *nvl);
nvpair_t *nvlist_next_nvpair(const nvlist_t *nvl, const nvpair_t *nvp);
nvpair_t *nvlist_prev_nvpair(const nvlist_t *nvl, const nvpair_t *nvp);

void nvlist_add_nvpair(nvlist_t *nvl, const nvpair_t *nvp);

void nvlist_move_nvpair(nvlist_t *nvl, nvpair_t *nvp);

void nvlist_set_parent(nvlist_t *nvl, nvpair_t *parent);

const nvpair_t *nvlist_get_nvpair(const nvlist_t *nvl, const char *name);

nvpair_t *nvlist_take_nvpair(nvlist_t *nvl, const char *name);

/* Function removes the given nvpair from the nvlist. */
void nvlist_remove_nvpair(nvlist_t *nvl, nvpair_t *nvp);

void nvlist_free_nvpair(nvlist_t *nvl, nvpair_t *nvp);

int nvpair_type(const nvpair_t *nvp);
const char *nvpair_name(const nvpair_t *nvp);

nvpair_t *nvpair_clone(const nvpair_t *nvp);

nvpair_t *nvpair_create_null(const char *name);
nvpair_t *nvpair_create_bool(const char *name, bool value);
nvpair_t *nvpair_create_number(const char *name, uint64_t value);
nvpair_t *nvpair_create_string(const char *name, const char *value);
nvpair_t *nvpair_create_stringf(const char *name, const char *valuefmt, ...) __printflike(2, 3);
nvpair_t *nvpair_create_stringv(const char *name, const char *valuefmt, va_list valueap) __printflike(2, 0);
nvpair_t *nvpair_create_nvlist(const char *name, const nvlist_t *value);
nvpair_t *nvpair_create_descriptor(const char *name, int value);
nvpair_t *nvpair_create_binary(const char *name, const void *value, size_t size);

nvpair_t *nvpair_move_string(const char *name, char *value);
nvpair_t *nvpair_move_nvlist(const char *name, nvlist_t *value);
nvpair_t *nvpair_move_descriptor(const char *name, int value);
nvpair_t *nvpair_move_binary(const char *name, void *value, size_t size);

bool		 nvpair_get_bool(const nvpair_t *nvp);
uint64_t	 nvpair_get_number(const nvpair_t *nvp);
const char	*nvpair_get_string(const nvpair_t *nvp);
const nvlist_t	*nvpair_get_nvlist(const nvpair_t *nvp);
int		 nvpair_get_descriptor(const nvpair_t *nvp);
const void	*nvpair_get_binary(const nvpair_t *nvp, size_t *sizep);

void nvpair_free(nvpair_t *nvp);

#endif	/* !_NV_IMPL_H_ */
