//
// selection.h
//
// (C) 2001-20004 Stefan Kebekus
// Distributed under the GPL

#ifndef selection_h
#define selection_h

#include <qstring.h>

#include "pageNumber.h"


/* The --very simple-- class textSelection represents text that the
   user has selected on a certain page of a document. The class stores

   the page number,

   two Q_INT32 numbers, 'selectedTextStart' and 'selectedTextEnd' that
   should hold indiced referring to the start and the end of the
   selected text in the appropriate documentPage's textLinkList

   a QString, 'selectedText' that holds the text that is selected

   these values can be stored, retrieved, it can be checked if the
   selection is empty, and signals are emitted when the values change
 */

class textSelection : public QObject
{
  Q_OBJECT

 public:
  // Constructs an empty selection with an invalid page number, and an
  // empty text string. The values 'selectedTextStart' and
  // 'selectedTextEnd' are set to 0. None of the signals pageChanged()
  // or selectionIsNotEmpty() is emitted at construction time
  textSelection(void) {page = PageNumber::invalidPage; selectedText = QString::null;}
  
  // Use this method to set the data that is described above. Note
  // that the consistency of 'pageNr' 'selectedTextStart' and
  // 'selectedTextEnd' are not checked; it is entirely of the program
  // that calls this method to make sure that it sets reasonable
  // values; it is not even checked if selectedTextStart <=
  // selectedTextEnd! The signals pageChanged() and
  // selectionIsNotEmpty() are emitted if appropriate.
  void          set(PageNumber pageNr, Q_INT32 selectedTextStart, Q_INT32 selectedTextEnd, const QString &text);
  
  // This method can be used to find out if the selection is empty,
  // i.e. if the page number is invalid or the selected text is empty.
  bool          isEmpty(void) const {return (page.isInvalid() || selectedText.isEmpty());}

  // Method used to retrieve the data described above
  Q_INT32       getSelectedTextStart() const {return selectedTextStart;}

  // Method used to retrieve the data described above
  Q_INT32       getSelectedTextEnd() const {return selectedTextEnd;}

  // Method used to retrieve the data described above
  const QString &getSelectedText() const {return selectedText;}

  // Method used to retrieve the data described above
  PageNumber    getPageNumber() const {return page;}

 public slots:
  // If the selection is not empty, this method copies the text to the
  // system clipboard. If the selection is empty, nothing is done.
  void          copyText(void) const;
   
  // Clears the selection. The selection is guaranteed to be empty
  // afterwards. The signals pageChanged() and selectionIsNotEmpty()
  // are emitted if appropriate.
  void          clear(void);

 signals:
  // This signal is emmitted when the methods 'set' or 'clear' are
  // called, and the content of page number field changes
  void          pageChanged(void);

  // This signal is emmitted when the methods 'set' or 'clear' are
  // called, and the content of the selection changes from 'empty' to
  // 'not empty' or otherwise
  void          selectionIsNotEmpty(bool isNotEmpty);

 private:
  // Described above
  PageNumber    page;

  // Described above
  Q_INT32       selectedTextStart, selectedTextEnd;

  // Described above
  QString       selectedText;
};

#endif
