/*******************************************************************************

  Copyright(c) 2002 - 2005 Promise Technology, Inc. All rights reserved.
  
  cam_swap.c - CAM Enclosure functions

  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the Free 
  Software Foundation; either version 2 of the License, or (at your option) 
  any later version.
  
  This program is distributed in the hope that it will be useful, but WITHOUT 
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
  more details.
  
  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc., 59 
  Temple Place - Suite 330, Boston, MA  02111-1307, USA.
  
  The full GNU General Public License is included in this distribution in the
  file called LICENSE.
  
  Contact Information:
  Promise Technology, Inc.
  <support@promise.com.tw>	[TAIWAN]
  <support@promise.com>		[U.S.A]
  <support-china@promise.com>	[CHINA]

*******************************************************************************/

#include "cam_con.h"
#include "cam_def.h"
#include "cam_imp.h"
#include "cam_macr.h"
#include "cam_gb.h"
#include "cam_exp.h"

U8 EnclosureEnterState(PENCLOSURE_TYPE pBox);
void EnclosureExitState(PENCLOSURE_TYPE pBox);
void Get_Box_Status(PATA_TASK pATATask);
U16 ReadWinBondID(U32 BaseAddress );
void SetupFanDivisor(U32 BaseAddress, U8 Divisor);
U32 ReadFanStatus(U32 BaseAddress );
U16 ReadTemperature(U32 BaseAddress);
U32 ReadVoltage5V(U32 BaseAddress);
U32 ReadVoltage12V(U32 BaseAddress);
void (*Chk_BOX_Function[])(void);

U8 Get_Enclosure_Datas(U8	bCh)
{
	U32	BaseAddress = ATABaseAddr[bCh];
	PENCLOSURE_TYPE pBox = &gBOXConfig[bCh];
	PENCLOSURE_STATUS pStatus = &gBOXStatus[bCh];	
	U8	bData;
	U8	Insertstate=0;
		
	if (EnclosureEnterState(pBox) == TRUE ) {
		
		if ( pBox->BOXConfig & MODE_SF )
		{
			bData = ReadIndexUchar(ATABaseAddr[bCh], iNDEXIDEDeviceHead);
			pStatus->PowerState = ( bData & bit(3) )? PowerON : PowerOFF;
			Insertstate = (U8)(bData & bit(1));
		}
		/* only when PowerOn of box, FAN RPM is available*/
		if (pStatus->PowerState == PowerON)
			pStatus->FANStatus = ReadFanStatus(BaseAddress );
		else
			pStatus->FANStatus = 0;
		
		/* only when box inserted, temp is available*/
		if (!Insertstate)
			pStatus->TEMPStatus = ReadTemperature(BaseAddress);	
		else
			pStatus->TEMPStatus = 0;
		
		pStatus->RemPeriod = pBox->ChkPeriod;
		pStatus->VOL5V = ReadVoltage5V(BaseAddress);	
		pStatus->VOL12V = ReadVoltage12V(BaseAddress);
		EnclosureExitState(pBox);
		return TRUE;
	}
	return FALSE;
}

U8 CAM_EnumEnclosure(U8 bCh, PENCLOSURE_TYPE pBox)
{
	if (gChnlConfig[bCh].bCh == camCHNL_NOT_FOUND)
		return 0xFF;
		
	if (gChnlConfig[bCh].ChnlType == CHNL_SATA )
		return 0xFF;
			
	if ( gBOXConfig[bCh].BOXType != camUNKNOWNBOX ) {
		*pBox =  *(&gBOXConfig[bCh]);
		return(pBox->BOXID);
	}
	
	/* first time to enumerate the enclosure */
	gBOXConfig[bCh].BOXID = bCh;
	gBOXConfig[bCh].BOXConfig = 0;
	if (EnclosureEnterState(&gBOXConfig[bCh]) == TRUE ) {
		EnclosureExitState(&gBOXConfig[bCh]);		

		/* default color is amber */
		CAM_CtrlLED(bCh, LED_Amber);
		
		/* setup relative parameters */
		EnclosureEnterState(&gBOXConfig[bCh]); /* get in by NCS way */
		/* read ID of sensor IC (Winbond H/W Monitoring IC) */
		/* if no find such IC, return number of sensor = 0 */
		if ( ReadWinBondID(ATABaseAddr[bCh]) == WINBONDID) {
			SetupFanDivisor(ATABaseAddr[bCh], DEF_FANDIV );	
		}
		else
		{
			gBOXConfig[bCh].BOXConfig &= ~WITH_SENSOR;
			gBOXConfig[bCh].NumofSensor = 0;
		}
		EnclosureExitState(&gBOXConfig[bCh]);		
		
		/* set up monitor timer for enclusure */
		gBOXConfig[bCh].ChkPeriod = DEF_CHKSTATETIME;
	
		/* get first status of swap box */
		if ( gBOXConfig[bCh].NumofSensor ) {
			gBOXStatus[bCh].NeedCheck = TRUE;
			boxATATask[bCh].bCh = bCh;
			Get_Enclosure_Datas(bCh);
			if (!gBOXStatus[bCh].TEMPStatus) {
				gBOXConfig[bCh].BOXConfig &= ~WITH_SENSOR;
				gBOXConfig[bCh].NumofSensor = 0;
			}
		}

		BOX_Timeout_TimerID[bCh] = camTimer(CAMTimeoutTimes[bCh/MAX_ATA_CHNL].BOXCommand, Chk_BOX_Function[bCh]);
	}
	else {
		gBOXConfig[bCh].BOXID = 0xFF;	/* no box */
		gBOXConfig[bCh].BOXType = camNOBOX;
	}
	
	*pBox =  *(&gBOXConfig[bCh]);
	return(pBox->BOXID);

}

void Get_Box_Status(PATA_TASK pATATask)
{
	U8	bCh = pATATask->bCh;
	U8	bData;
	U32	BaseAddress = ATABaseAddr[bCh];
	PENCLOSURE_TYPE pBox = &gBOXConfig[bCh];
	PENCLOSURE_STATUS pStatus = &gBOXStatus[bCh];
	
	if( pATATask->ATAStatus & ATA_ABORT ) {
		BOX_Timeout_TimerID[bCh] = camTimer(CAMTimeoutTimes[bCh/MAX_ATA_CHNL].BOXCommand, Chk_BOX_Function[bCh]);		
		return;		/* command is aborted.*/
	}
		
	if (EnclosureEnterState(pBox) == TRUE ) 
	{
		pStatus->NeedCheck = FALSE;


		/* Is disk replaced ? */
		bData = ReadIndexUchar(BaseAddress, iNDEXIDEDeviceHead);
		bData &= 0x0F;
		if ( bData == 0x0C ) {
			pStatus->NeedCheck = TRUE;
			gDrvConfig[bCh][0].DevFlag = DEV_NOTFOUND;
			EnclosureExitState(pBox);
			camCheckDriveStatus(bCh, 0, NEW_DRIVE);	
			WriteIndexUchar(BaseAddress, iNDEXIDEDeviceHead, (U8)(bData&0xFB));	/* clear repd bit */
		}
		else if (( bData & 0x03 ) && (gDrvConfig[bCh][0].DevFlag & DEV_EXIST)) {
			pStatus->NeedCheck = TRUE;
			gDrvConfig[bCh][0].DevFlag = DEV_NOTFOUND;
			EnclosureExitState(pBox);
			camCheckDriveStatus(bCh, 0, NO_DRIVE);	
		}
		else if ( !(gDrvConfig[bCh][0].DevFlag & DEV_EXIST) && ( bData == 0x08 ) ) {
			pStatus->NeedCheck = TRUE;
			EnclosureExitState(pBox);
			camCheckDriveStatus(bCh, 0, NEW_DRIVE);	
		}
		
		/* Need to get box status ? */
		if ( pBox->BOXConfig & WITH_SENSOR) {
			if ( ( pStatus->RemPeriod == 0 )
		  	|| ( --pStatus->RemPeriod == 0 ) ) 
		  	{
				EnclosureExitState(pBox);
		 		Get_Enclosure_Datas(bCh);
			}
			else
				EnclosureExitState(pBox);
		}
		else
			EnclosureExitState(pBox);
			
		BOX_Timeout_TimerID[bCh] = camTimer(CAMTimeoutTimes[bCh/MAX_ATA_CHNL].BOXCommand, Chk_BOX_Function[bCh]);
		
	}
	else {
		gBOXStatus[bCh].PowerState = PowerOFF; /* can't enter state again due to power off */
		if ( gDrvConfig[bCh][0].DevFlag & DEV_EXIST ){
			gDrvConfig[bCh][0].DevFlag = DEV_NOTFOUND;
			camCheckDriveStatus(bCh, 0, NO_DRIVE);
		}		
		BOX_Timeout_TimerID[bCh] = camTimer(CAMTimeoutTimes[bCh/MAX_ATA_CHNL].BOXCommand, Chk_BOX_Function[bCh]);
	}
	return;
}

U8 CAM_Get_Enclosure_Status(U8 bID, PENCLOSURE_STATUS pBoxStatus)
{
	U8	bCh = bID & 0x0F;
	PENCLOSURE_TYPE pBox =&gBOXConfig[bCh];
	
	if ((pBox->BOXType == SuperSwap) && (pBox->BOXConfig & MODE_SF ))
	{
		*pBoxStatus = *(&gBOXStatus[bCh]);
					
		return(camSUCCESS);
	}
	else
		return(camFAIL);
}

void Set_Box_LED(PATA_TASK pATATask)
{
	U32  bColor;
	U8	bData;
	U8	bCh = pATATask->bCh;
	PENCLOSURE_TYPE	pBox = &gBOXConfig[bCh];

	if( pATATask->ATAStatus & ATA_ABORT ) {
		return;		/* command is aborted.*/
	}
	
	bColor=(U32)pATATask->pTaskExtension;
	EnclosureEnterState(pBox);
	bData = ReadIndexUchar(ATABaseAddr[bCh], iNDEXIDESectorCount);
	bData &= ~LED_MASK;
	bData |= ((U8)bColor & LED_MASK);
	WriteIndexUchar(ATABaseAddr[bCh], iNDEXIDESectorCount, bData );
	EnclosureExitState(pBox);
	gBOXStatus[bCh].LEDColor = (U8)bColor;	/* keep current color */
	
	return;
}

void CAM_CtrlLED(U8 bID, U8 bColor)
{
	U8	bCh = bID & 0x0F;
	PATA_TASK pATATask = &boxLEDATATask[bCh];
	
	if ((gChnlConfig[bCh].bCh == camCHNL_NOT_FOUND) || (gBOXConfig[bCh].BOXType != SuperSwap))
		return;
	
	if ( gBOXStatus[bCh].LEDColor == bColor )
		return;
		
	pATATask->bCh = bCh;
	pATATask->bID = 0;
	pATATask->ATACmdFlag = SWAPBOX_CMD;
	pATATask->pTaskExtension = (PVOID) (U32)bColor;
	pATATask->callback = Set_Box_LED;		
	CAM_SubmitATA (pATATask);
	return;
}

void CAM_Enclosure_PowerOn(U8 bID)
{
	U8	bData;
	U8	bCh = bID & 0x0F;
	PENCLOSURE_TYPE	pBox = &gBOXConfig[bCh];

	if ((gChnlConfig[bCh].bCh == camCHNL_NOT_FOUND) || (gBOXConfig[bCh].BOXType != SuperSwap))
		return;
		
	/* at any time, if want to EnclosureEnterState or EnclosureExitState, must make sure that there is 
	     no command be running.*/
	stopATAModule(bCh);
	
	EnclosureEnterState(pBox);
	bData = ReadIndexUchar(ATABaseAddr[bCh], iNDEXIDEDeviceHead);
	
	if ( pBox->BOXConfig & MODE_SF )
	{
		if (!(bData & bit(0)) && !(bData & bit(3))) {
			WriteIndexUchar(ATABaseAddr[bCh], iNDEXIDESectorCount, 0x07 );
			camStallExecution(1000);/* wait 10 seconds */
		}
		
		bData = ReadIndexUchar(ATABaseAddr[bCh], iNDEXIDEDeviceHead);
		gBOXStatus[bCh].PowerState = ( bData & bit(3) )? PowerON : PowerOFF;
	}

	EnclosureExitState(pBox);

	return;
}

void CAM_Enclosure_PowerOff(U8 bID)
{
	U8	bData;
	U8	bCh = bID & 0x0F;
	PENCLOSURE_TYPE	pBox = &gBOXConfig[bCh];

	if ((gChnlConfig[bCh].bCh == camCHNL_NOT_FOUND) || (gBOXConfig[bCh].BOXType != SuperSwap))
		return;
		
	/* at any time, if want to EnclosureEnterState or EnclosureExitState, must make sure that there is 
	     no command be running.*/
	stopATAModule(bCh);
	
	EnclosureEnterState(pBox);
	bData = ReadIndexUchar(ATABaseAddr[bCh], iNDEXIDEDeviceHead);
	
	if ( pBox->BOXConfig & MODE_SF )
	{
		bData = ReadIndexUchar(ATABaseAddr[bCh], iNDEXIDEDeviceHead);
		bData &= ~bit(3);
		WriteIndexUchar(ATABaseAddr[bCh], iNDEXIDESectorCount, bData);
		gBOXStatus[bCh].PowerState =  PowerOFF;
	}
	EnclosureExitState(pBox);	
	return;
}

void Check_Enclosure_PowerState(U8 bID)
{
	U8	bData;
	U8	bCh = bID & 0x0F;
	PENCLOSURE_TYPE	pBox = &gBOXConfig[bCh];
	
	if ( EnclosureEnterState(pBox)== TRUE ) {
		bData = ReadIndexUchar(ATABaseAddr[bCh], iNDEXIDEDeviceHead);
	
		if ( pBox->BOXConfig & MODE_SF )
		{
			bData = ReadIndexUchar(ATABaseAddr[bCh], iNDEXIDEDeviceHead);
			gBOXStatus[bCh].PowerState = ( bData & bit(3) )? PowerON : PowerOFF;
		}

		EnclosureExitState(pBox);
	}
	else
		gBOXStatus[bCh].PowerState = PowerOFF; 
		
	return;
}

U8 EnclosureEnterState(PENCLOSURE_TYPE pBox)
{
	U32	BaseAddress = ATABaseAddr[pBox->BOXID & 0x0F];
	U8	bData, btmp=0;
	
	if ( pBox->BOXConfig & MODE_SF )
	{
		bData = ReadIndexUchar( BaseAddress, (iNDEXIDECICR+1));
		bData |= 0x01;	/* disable CSN */
		WriteIndexUchar( BaseAddress, (iNDEXIDECICR+1), bData);
		
		/* Is the box really in SF mode ? */
		bData = ReadIndexUchar(BaseAddress, iNDEXIDECylinderHigh);
		bData &= 0x0F;
		WriteIndexUchar( BaseAddress, iNDEXIDECylinderHigh, bData);
		bData = ReadIndexUchar(BaseAddress, iNDEXIDECylinderHigh);		
		if ( bData == 0x03 )
			return(TRUE);
			
		bData = ReadIndexUchar( BaseAddress, (iNDEXIDECICR+1));
		bData &= ~0x01;	/* re-enable CS */
		WriteIndexUchar( BaseAddress, (iNDEXIDECICR+1), bData);
		
		/* we need to set 199 to SF mode again */
	}
	
	/* try to enter state first time */
	bData=0xE0;
	WriteIndexUchar(BaseAddress, iNDEXIDEDeviceHead, bData);		/* to make sure it isn't in state 6 now */
	ReadIndexUchar(BaseAddress, iNDEXIDEStatus);
	bData = 0x80;
	WriteIndexUchar(BaseAddress, iNDEXIDESectorCount, bData );		/* make a maker */
	
	/* enter state */	
	bData=0xE0;
	WriteIndexUchar(BaseAddress, iNDEXIDEDeviceHead, bData );
	WriteIndexUchar(BaseAddress, iNDEXIDEDeviceHead, bData );
	WriteIndexUchar(BaseAddress, iNDEXIDEDeviceHead, bData );
	ReadIndexUchar(BaseAddress, iNDEXIDESectorCount);
	ReadIndexUchar(BaseAddress, iNDEXIDEDeviceHead);
	
	bData = ReadIndexUchar(BaseAddress, iNDEXIDESectorCount);
	if( bData & 0x80 )
		return(FALSE);		/* no box */
	
	btmp = bData | 0x04;	/* PWR bit of SF mode ! */
	bData |= 0x03;	/* amber color  */
	WriteIndexUchar(BaseAddress, iNDEXIDESectorCount, bData );
	bData = 0;
	WriteIndexUchar(BaseAddress, iNDEXIDESectorNumber, bData );    /* for floating bus */
	bData = ReadIndexUchar(BaseAddress, iNDEXIDESectorCount);
	if( (bData & 0x03) != 0x03 )
		return(FALSE);		/* no box */

	bData = btmp & 0xDF; /* clear REP bit*/
	WriteIndexUchar(BaseAddress, iNDEXIDESectorCount, bData );	/* restore to orignal status */
	WriteIndexUchar(BaseAddress, iNDEXIDESectorNumber, 0x03 );	/* restore to orignal status */

	/* Is the box already in SF mode ? (after a warm boot )*/
	bData = ReadIndexUchar(BaseAddress, iNDEXIDECylinderHigh);
	if ( bData & 0x02 ) {
		pBox->BOXConfig = MODE_SF + WITH_SENSOR + WITH_FAN + WITH_KEY;
		pBox->BOXType = SuperSwap;
		pBox->NumofBay = 1;
		pBox->NumofFAN = 1;
		pBox->NumofSensor = 1;
		return(TRUE);
	}
	
	/* according hardware pins to set config */	
	bData = ReadIndexUchar(BaseAddress, iNDEXIDEDeviceHead);
	pBox->BOXType = SuperSwap;
	if (!(bData & bit(2))) {
		pBox->BOXConfig |= WITH_SENSOR;
		pBox->NumofSensor = 1;
	}
			
	if (!(bData & bit(3))) {
		pBox->BOXConfig |= WITH_FAN;
		pBox->NumofFAN = 1;
	}
			
	if (!(bData & bit(6)))
		pBox->BOXConfig |= WITH_KEY;	
				
	/* can the box be SF mode ? */
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderHigh, 0x03 );
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderHigh, 0x10 ); /* Index 1 of extened registers */
	bData = ReadIndexUchar(BaseAddress, iNDEXIDECylinderHigh);
	if(bData & 0x10) {
		pBox->BOXConfig |=  MODE_SF;		
		WriteIndexUchar(BaseAddress, iNDEXIDECylinderLow, 0x00 ); /* set power on delay Time of SF mode to 0 */
		return(TRUE);
	}
	else
		return(FALSE);
	
}

void EnclosureExitState(PENCLOSURE_TYPE pBox)
{
	U8  bData;
	U32	BaseAddress = ATABaseAddr[pBox->BOXID & 0x0F];
	
	if ( pBox->BOXConfig & MODE_SF )
	{
		bData = ReadIndexUchar( BaseAddress, (iNDEXIDECICR+1));
		bData &= 0xFE;	/* enable CSN */
		WriteIndexUchar( BaseAddress, (iNDEXIDECICR+1), bData);
	}
	
	bData=0xE0;
	WriteIndexUchar(BaseAddress, iNDEXIDEDeviceHead, bData);
	ReadIndexUchar(BaseAddress, iNDEXIDEStatus);
	
	return;
}



/* write value into WinBond Register, input Data value and Register Index */
void WriteWReg(U32 BaseAddress, U8 Index, U8 Data )
{
	U8	status;
	U16	i;
	
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderHigh, 0xB0);	/* write index and data */
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderLow, Data);	/* index 0Bh */		
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderLow, Index);	/* index 0Ch */	
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderLow, 0x5A);	/* index 0Dh */
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderLow, 0x03);	/* index 0Eh - write 3 bytes */
	for ( i = 0; i < 100; i++ ) {
		status = ReadIndexUchar(BaseAddress, iNDEXIDECylinderLow);		/* index 0Fh */
		if ( status & 0x03 )
			break;	/* Error or Done */
		camStallExecution(1);
	}
}

void ReadWReg(U32 BaseAddress, U8 Index, PU8 Data )
{
	U8	status;
	U16	i;

	WriteIndexUchar(BaseAddress, iNDEXIDECylinderHigh, 0xC0);	/* write index */
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderLow, Index);	/* index 0Ch */	
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderLow, 0x5A);	/* index 0Dh */
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderLow, 0x02);	/* index 0Eh - write 2 bytes */
	for ( i = 0; i < 100; i++ ) {
		status = ReadIndexUchar(BaseAddress, iNDEXIDECylinderLow);		/* index 0Fh */
		if ( status & 0x03 )
			break;	/* Error or Done */
		camStallExecution(1);
	}
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderHigh, 0xD0);	/* write index */	
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderLow, 0x5B);	/* index 0Dh */
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderLow, 0x11);	/* index 0Eh */
	for ( i = 0; i < 100; i++ ) {
		status = ReadIndexUchar(BaseAddress, iNDEXIDECylinderLow);		/* index 0Fh */
		if ( status & 0x03 )
			break;	/* Error or Done */
		camStallExecution(1);
	}
	WriteIndexUchar(BaseAddress, iNDEXIDECylinderHigh, 0xD0);	/* write index */	
	*Data = ReadIndexUchar(BaseAddress, iNDEXIDECylinderLow);
	
}


U16 ReadWinBondID(U32 BaseAddress )
{
    U8	low, high;

    /* set CR4E bit7=0 */
    WriteWReg(BaseAddress,0x4E, 0x00);
    /*read vendor ID low byte */
    ReadWReg(BaseAddress,0x4F, (PU8)&low );

    /* set CR4E bit7=1 */
    WriteWReg(BaseAddress,0x4E, 0x80 );
    /* read vendor ID high byte */
    ReadWReg(BaseAddress,0x4F, (PU8)&high );
    return ((high << 8 ) | low);

}

/* setup fan divisor to 010b, which gives a min RPM = 1.35e06/(255 * 2^2) = 1323
 * this is done by setting bit2 = Reg 5D, bit5 = 0
 *                         bit1 = Reg 47, bit5 = 1
 *                         bit0 = Reg 46, bit4 = 0
 */
void SetupFanDivisor(U32 BaseAddress, U8 Divisor)
{
	U8 fd;
	
	ReadWReg(BaseAddress, 0x5D, (PU8)&fd);
	fd &= ~0x20;	/* bit2 of divisor is bit 5 of reg5D */
	fd |= (Divisor & 0x04 ) << 3;
	WriteWReg(BaseAddress, 0x5D, fd);
	ReadWReg(BaseAddress, 0x47, (PU8)&fd);
	fd &= ~0x30;	/* bit1,0 of divisor is bit 4, 5 of reg47 */
	fd |= (Divisor & 0x03 ) << 4;	/* bit1,0 of divisor is bit 4, 5 of reg47 */
	WriteWReg(BaseAddress, 0x47, fd);
	return;
}

U32 ReadFanStatus(U32 BaseAddress )
{
    U8	fd, fd2;	/* FAN1DIV */
    U32	divisor;
    U8	fan1cnt;	/* FAN1COUNT */
    const U32 Dividend = 0x149970; /* 1.35 x 10^6 */
	U32	rpm=0;

    ReadWReg(BaseAddress,0x5D, (PU8)&fd);
    fd &= 0x20;		/* get bit5 - FAN1DIV bit2 */
    fd = fd >> 3;	/* move to bit2 */

    ReadWReg(BaseAddress,0x47, (PU8)&fd2);
    fd2 &= 0x30;	/* get bit5,4 - FAN1DIV bit1,0 */
    fd2 = fd2 >> 4;	/* move to bit1,0 */

    fd |= fd2;
    divisor = 1 << fd;

    ReadWReg(BaseAddress,0x28, (PU8)&fan1cnt);

	if (!fan1cnt * divisor)
        rpm = 0;
	else   
		rpm = Dividend / (fan1cnt * divisor);

	return rpm;
}

/* read Temp from WinBond, in deg C */
/*   this is temp1 of the IC */

U16 ReadTemperature(U32 BaseAddress)
{
	U8  msb, lsb;
	U16 temp, mag;
	int sign;

	/* select Bank 1 */
    WriteWReg(BaseAddress, 0x4E, 0x01);
	/* read MSB */
	ReadWReg(BaseAddress, 0x50, (PU8)&msb);
	/* read LSB */
	ReadWReg(BaseAddress, 0x51, (PU8)&lsb);
    /* msb has 8 bits, lsb has 1 bit (temp1 = 9 bit value) */
    temp = ((U16)msb << 1) | ((U16)lsb >> 7);
    /* this value is 9 bit 2's complement */
    /* resolutin is 0.5 deg, range is -128 to 127.5 */
    sign = (temp & 0x100) ? -1 : 1;
    mag = (temp + 1) & 0x1ff;
    temp = mag * sign;
	/* restore to bank 0 */
	WriteWReg(BaseAddress, 0x4E, 0x00);

	return temp;
}

/* Read 5V Voltage Level. The LSB is 0.016 V=16 mV
 * this gives 8 bit = 4.096 V max voltage. The serial
 * resistor is used to drop the 5V level to 4.096V
*/
U32 ReadVoltage5V(U32 BaseAddress)
{
    U8 vbase=16; /* 8-bit ADC has a 16mv LSB */
    U32 voltage;
    U8 regval;
    U8 SR1=34, SR2=50;	/* Vin is divided into R2/(R1+R2), to fit */
                          	/* with 0-4096. SR=Serial Resistor	*/

    ReadWReg(BaseAddress,0x23, (PU8)&regval);
    voltage = regval * vbase;
    voltage =  voltage  * (SR1 + SR2) / SR2;

   return voltage;
}

U32 ReadVoltage12V(U32 BaseAddress)
{
    U8 vbase=16; /* 8-bit ADC has a 16mv LSB */
    U32 voltage;
    U8 regval;
    U8 R1=28, R2=10;		/* Vin is divided into R2/(R1+R2), to fit */
							/* with 0-4096 */

    ReadWReg(BaseAddress, 0x24, (PU8)&regval);
    voltage = regval * vbase;
    voltage = voltage * (R1 + R2)/R2;

    return voltage;
}



void Chk_BOX(U8 bCh)
{
	PATA_TASK pATATask = &boxATATask[bCh];

	BOX_Timeout_TimerID[bCh]=ZERO_NULL;	
	
	pATATask->bCh = bCh;
	pATATask->bID = 0;
	pATATask->ATACmdFlag = SWAPBOX_CMD;
	pATATask->callback = Get_Box_Status;		
	CAM_SubmitATA (pATATask);
	return;

}

#if (MAX_ADAPTER_NUM>0)
void Chk_BOX0(void)
{
	Chk_BOX(0x00);
	return;
}

void Chk_BOX1(void)
{
	Chk_BOX(0x01);
	return;
}

void Chk_BOX2(void)
{
	Chk_BOX(0x02);
	return;
}

void Chk_BOX3(void)
{
	Chk_BOX(0x03);
	return;
}
#endif
#if (MAX_ADAPTER_NUM>1)
void Chk_BOX4(void)
{
	Chk_BOX(0x04);
	return;
}

void Chk_BOX5(void)
{
	Chk_BOX(0x05);
	return;
}

void Chk_BOX6(void)
{
	Chk_BOX(0x06);
	return;
}

void Chk_BOX7(void)
{
	Chk_BOX(0x07);
	return;
}
#endif
#if (MAX_ADAPTER_NUM>2)
void Chk_BOX8(void)
{
	Chk_BOX(0x08);
	return;
}

void Chk_BOX9(void)
{
	Chk_BOX(0x09);
	return;
}

void Chk_BOXA(void)
{
	Chk_BOX(0x0A);
	return;
}

void Chk_BOXB(void)
{
	Chk_BOX(0x0B);
	return;
}
#endif
#if (MAX_ADAPTER_NUM>3)
void Chk_BOXC(void)
{
	Chk_BOX(0x0C);
	return;
}

void Chk_BOXD(void)
{
	Chk_BOX(0x0D);
	return;
}

void Chk_BOXE(void)
{
	Chk_BOX(0x0E);
	return;
}

void Chk_BOXF(void)
{
	Chk_BOX(0x0F);
	return;
}
#endif

void (*Chk_BOX_Function[])(void) = {

#if (MAX_ADAPTER_NUM>0)
  Chk_BOX0,
  Chk_BOX1,
  Chk_BOX2,
  Chk_BOX3,
#endif
#if (MAX_ADAPTER_NUM>1)
  Chk_BOX4,
  Chk_BOX5,
  Chk_BOX6,
  Chk_BOX7,
#endif
#if (MAX_ADAPTER_NUM>2)
  Chk_BOX8,
  Chk_BOX9,
  Chk_BOXA,
  Chk_BOXB,
#endif  
#if (MAX_ADAPTER_NUM>3)
  Chk_BOXC,
  Chk_BOXD,
  Chk_BOXE,
  Chk_BOXF,  
#endif  
};
