/* sony programmable I/O control device (SPIC) 
   driver for VAIO

   developed on C1VJ(crusoe)
   Takaya Kinjo <t-kinjo@tc4.so-net.ne.jp> 2000/11/14

   Copyright (C) Takaya Kinjo 2000
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#define MODULE
#define __KERNEL__

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <linux/string.h>
#include <linux/pci.h>

#include <asm/uaccess.h>

#include <asm/io.h>
//#include <sys/io.h>


#define SPIC_PCI_VENDOR 0x8086
#define SPIC_PCI_DEVICE 0x7113

#define SPI_IRQ_PORT 0x8034
#define SPI_IRQ_SHIFT 22

#define SPI_BASE 0x50

#define SPI_G10A (SPI_BASE+0x14)
#define SPI_G10L (SPI_BASE+0x16)

#define SPIC_PORT1 0x10a0
#define SPIC_PORT2 0x10a4

#define SET_CPU_SPEED_300MHz_LONGRUN    1
#define SET_CPU_SPEED_600MHz            2

static unsigned char spic_bus, spic_devfn;

void k_usleep( unsigned long usec )
{
  current->state = TASK_INTERRUPTIBLE;
  schedule_timeout(HZ * usec / 1000000);
}

static void OUTB(u8 v, int port)
{
  outb(v, port);
}

static u8 INB(int port)
{
  k_usleep(10);
  return inb(port);
}

/* initialise the SPIC - this comes from the AML code in the ACPI bios */
static void spic_srs( u16 port1, u16 port2, u8 irq)
{
  u8 v;
  u16 v2;

  pcibios_write_config_word( spic_bus, spic_devfn, SPI_G10A, port1 );
  pcibios_read_config_byte( spic_bus, spic_devfn, SPI_G10L, &v);
  v = (v & 0xF0) | (port1 ^ port2);
  pcibios_write_config_byte( spic_bus, spic_devfn, SPI_G10L, v );

  v2 = inw(SPI_IRQ_PORT);
  v2 &= ~(0x3 << SPI_IRQ_SHIFT);
  v2 |= (irq << SPI_IRQ_SHIFT);
  outw(v2, SPI_IRQ_PORT);

  pcibios_read_config_byte( spic_bus, spic_devfn, SPI_G10L, &v);
  v = (v & 0x1F) | 0xC0;
  pcibios_write_config_byte( spic_bus, spic_devfn, SPI_G10L, v );
}

/* disable the SPIC - this comes from the AML code in the ACPI bios */
static void spic_dis(void)
{
  u8 v1;
  u16 v;

  pcibios_read_config_byte( spic_bus, spic_devfn, SPI_G10L, &v1 );
  pcibios_write_config_byte( spic_bus, spic_devfn, SPI_G10L, v1 & 0x3F );

  v = inw(SPI_IRQ_PORT);
  v |= (0x3 << SPI_IRQ_SHIFT);
  outw(v, SPI_IRQ_PORT);
}

static void spic_settle(void)
{
  while (INB(SPIC_PORT2) & 2) k_usleep(1);
}

static u8 spic_call1(u8 dev)
{
  u8 v1, v2;
  spic_settle();

  OUTB(dev, SPIC_PORT2);
  v1 = INB(SPIC_PORT2);
  v2 = INB(SPIC_PORT1);
  return v2;
}

static u8 spic_call2(u8 dev, u8 fn)
{
  u8 v1;

  while (INB(SPIC_PORT2) & 2) ;
  outb(dev, SPIC_PORT2);

  while (INB(SPIC_PORT2) & 2) ;
  outb(fn, SPIC_PORT1);

  v1 = INB(SPIC_PORT1);
  return v1;
}

int spic_jogger_turned(void)
{
  u8 v1, v2;
  v1 = inb(SPIC_PORT1);
  v2 = inb(SPIC_PORT2);
  if ((v2 & 0x10) == 0 || v1 == 0) return 0;
  /* the following clears the event bits */
  spic_call2(0x81, 0xff);
  return (signed char)v1;
}

static int devmajor=60;
static char *devname="spic";
MODULE_PARM(devmajor, "i");
MODULE_PARM(devname, "s");

static int spicdriver_open(struct inode * inode, struct file * file)
{
  MOD_INC_USE_COUNT;
  return 0;
}

static int spicdriver_close(struct inode * inode, struct file * file)
{
  MOD_DEC_USE_COUNT;
  return 0;
}

static int spicdriver_read(struct file * file, 
			   char * buff, size_t count, loff_t *pos)
{
  int num_bytes;
  int jog;
  jog = spic_jogger_turned();
  num_bytes = sizeof(int);
  copy_to_user(buff, (char *) &jog, num_bytes );
  return num_bytes;
}

static int spic_ioctl(struct inode *inode, struct file *file,
                     unsigned int cmd, unsigned long arg)
{
  switch(cmd) {
  case SET_CPU_SPEED_300MHz_LONGRUN:
    pcibios_write_config_byte( 0, 0, 0xa8, 0x11);
    printk("spic: Crusoe 300MHz longrun mode.\n");
    break;

  case SET_CPU_SPEED_600MHz:
    pcibios_write_config_byte( 0, 0, 0xa8, 0x0e);
    printk("spic: Crusoe 600MHz mode.\n");
    break;

  default:
    return -ENOIOCTLCMD;
  }
  return 0;
}

static struct file_operations spicdriver_fops = {
  read:    spicdriver_read,  // ssize_t read(struct file *, char *, size_t, loff_t *)
  ioctl:   spic_ioctl,
  open:    spicdriver_open,  // int     open(struct inode *, struct file *)
  release: spicdriver_close, // int     release(struct inode *, struct file *)
};
int init_module(void)
{
  int i;
  struct pci_dev *pdev=NULL;

  printk("SONY Programmable Interface Controller Driver ver1.0 ");
  printk("<t-kinjo@tc4.so-net.ne.jp>\n");

  /* register character device */
  if(register_chrdev(devmajor,devname,&spicdriver_fops))
    {
      printk("device registration error\n");
      return -EBUSY;
    }
  /* find spic PCI device */
  for(i=0;;i++)
    {
      pdev=pci_find_device(SPIC_PCI_VENDOR, SPIC_PCI_DEVICE, pdev);
      if(!pdev)
	break; //  no more driver
      spic_bus = pdev->bus->number;
      spic_devfn = pdev->devfn;
      printk("%s: found spic device on Bus:%d Device:%d ",
	     devname, spic_bus, spic_devfn );
    }
  if(i==0) 
    { 
      printk("There is no spic device.\n" );
      return 1;
    }
  /* init spic */
  spic_srs( SPIC_PORT1, SPIC_PORT2, 0x3);
  spic_call1(0x82);
  spic_call2(0x81, 0xff);
  spic_call1(0x92);

  printk("... initialized.\n");

  return 0;
}

void cleanup_module(void)
{
  if (unregister_chrdev(devmajor,devname)) 
    {
      printk ("unregister_chrdev failed\n");
    }
  spic_dis();
  printk( "%s: removed.\n", devname );
};



