#define UwsConfg_c


// **********************************************************************
// * uwsConfg.c                                                         *
// *                                                                    *
// * Written by Matthew DeLoera (deloera@us.ibm.com)                    *
// * (C) Copyright IBM Corporation, 1998-2001                           *
// *                                                                    *
// *  This software may be used and distributed according to the terms  *
// *  of the GNU Public License, incorporated herein by reference.      *
// **********************************************************************

// **********************************************************************
// * uwsConfg.c - This source file implements all primary driver entry  *
// *              points, and is thought of as the "main" source file   *
// *              for the IBM Advanced System Management Processor      *
// *              device driver.                                        *
// *                                                                    *
// * Exported Functions                                                 *
// * ------------------                                                 *
// * sysSpinit       - main initialization routine, also sends initial  *
// *                   ASM commands to ASM hardware                     *
// * RemoveNode     - called at driver unloading to de-allocate         *
// *                   resources and to send an OS Shutting Down        *
// *                   message to the SP hardware                       *
// *                                                                    *
// * Changes                                                            *
// * -------                                                            *
// * 01/29/2001 - Cleaned up for open source release.                   *
// * ----------------------------------------------------------------   *
// * 05/09/2002 - ia64 compatability changes. fwt                       *
// * ----------------------------------------------------------------   *
// * 06/27/2002 - Added multi node support       // PHR_MULTI_4         *
// **********************************************************************

// ************
// * INCLUDES *
// ************
#include "uwscmd.h"

// ***********************
// * EXTERNAL PROTOTYPES *
// ***********************
extern unsigned long READ_REGISTER_ULONG(void *x);
extern void          WRITE_REGISTER_ULONG(unsigned long *x, unsigned long y);

extern int           sysSpProcessIoctl(volatile int cmd, volatile PDD_IOCTL_INFO pDDIoctlInfo, volatile unsigned int PID, volatile int callerFlag, PDRIVER_INFO pDriverNode);

extern int           sysSp_ioctl(struct inode *pInode, struct file *pFile, unsigned int uiArg1, unsigned long ulArg2);
extern int           sysSp_open (struct inode *pInode, struct file *pFile);
extern int           sysSp_close(struct inode *pInode, struct file *pFile);

extern void          sysSp_pciIntrHandler(int iRq, void *pDev_id, struct pt_regs *regs);
extern int           sysSpGetPciInfo(PDRIVER_INFO pDriverNode);

extern unsigned int  sysSpCountEagles(void);

#ifndef IA64
  extern void        sysSp_isaIntrHandler(int iRq, void *pDev_id, struct pt_regs *regs);
  extern void        sysSpGetIsaInfo(void);
  extern int         sysSpASR(void);
#endif


// ********************
// * LOCAL PROTOTYPES *
// ********************
int  sysSpinit(int IRQoverride);
void sysSpDisableInterrupts(PDRIVER_INFO pDriverNode, unsigned long templong);
void sysSpRestoreInterrupts(PDRIVER_INFO pDriverNode, unsigned long templong);
void sysSpEnableInterrupts(PDRIVER_INFO pDriverNode);
void RemoveNode(PDRIVER_INFO pDriverNode);

void InitIsaHW(void);
int  GetPrimaryAsmType(void); 
int  InitDriverNode(PDRIVER_INFO pDriverNode); 
int  InstallIntHndlr(PDRIVER_INFO pDriverNode);
int  InitRemMouse(PDRIVER_INFO pDriverNode);
int  SendInitMsgs(PDRIVER_INFO pDriverNode);

// ***********
// * DEFINES *
// ***********
#define DD_SPINSTALL_SUCCESS              0
#define DD_SPINSTALL_INSTPGM_NOT_FOUND    1
#define DD_SPINSTALL_VPD_NOT_FOUND        2


// ***********************
// * GLOBAL DECLARATIONS *
// ***********************
PDRIVER_INFO  pSpNodeArray;
PDRIVER_INFO  pSpPrimaryNode;         

struct pci_dev *PciDev;

int ShuttingDown;
int NumSpNodes;

struct file_operations sysSp_fops = {  NULL,                  // LSEEK
                                       NULL,                  // READ 
                                       NULL,                  // WRITE 
                                       NULL,                  // READDIR  
                                       NULL,                  // SELECT  
                                       ioctl: sysSp_ioctl,    // IOCTL
                                       NULL,                  // MMAP
                                       open: sysSp_open,      // OPEN
                                       release: sysSp_close   // RELEASE
                                    };

// ****************************************************************************
// * sysSpinit - This function performs all driver and device initialization. *
// *             It determines the presence of the SP, initialize the driver, *
// *             allocates all resources, hooks the IRQ vector, and prepares  *
// *             the driver extension.                                        *
// *                                                                          *
// * Returns:                                                                 *
// *      0       - successful                                                *
// *      EIO     - I/O error                                                 *
// *      ENXIO   - no such device or address                                 *
// *      EAGAIN  - try again                                                 *
// *      ENOMEM  - out of memory                                             *
// *      EFAULT  - bad address                                               *
// *      EBUSY   - device busy                                               *
// *      ENODEV  - no such device                                            *
// *      ENOBUFS - no buffer space available                                 *
// ****************************************************************************
int sysSpinit(int IRQoverride)
{
   int HardwareType;
   int i, rc; 
   int SPNodeArraySize;

   PDRIVER_INFO pDriverNode;

   // INDICATE WHETHER ANY GLOBAL DEBUG FLAGS WERE SET
   if (DBG_FUNCENTEXT || DBG_PCIDEBUG || DBG_TIMER || DBG_LOCKS       ||
       DBG_BLOCKS     || DBG_HB       || DBG_DEBUG || DBG_MULTI       ||
       DBG_QUEUES     || DBG_IOCTL    || DBG_INIT  || DBG_CMDTIMEOUT  ||
       DBG_SHUTDOWN   || DBG_EVENTS   || DBG_IRQ   || DBG_ISADEBUG    ||
       DBG_SLIM       || DBG_DDINST   || DBG_PHR   || DBG_MOUSE)
   {
        printk(KERN_CRIT "ibmasm: DBG_INFO - Debug messages enabled.\n");
   }

   // INITIALIZE GLOBALS
   NumSpNodes   = 0;
   ShuttingDown = 0;
   PciDev       = NULL;
   pDriverNode  = NULL;

   // GET THE PRIMARY ASM HW TYPE
   HardwareType = ASMTYPE_UNKNOWN;
   HardwareType = GetPrimaryAsmType();

   switch ( HardwareType )
   {
      case ASMTYPE_UNKNOWN:
      {
         printk(KERN_CRIT "ibmasm: Failed to find required ASM hardware. Aborting\n");
         return -ENODEV;
      }

#ifndef IA64  
      case ASMTYPE_ASR:     
      case ASMTYPE_PEARL:   
      case ASMTYPE_JASPER:   
      {
         if (DBG_INIT)
         {
            if (HardwareType == ASMTYPE_ASR)
            {
              printk(KERN_CRIT "ibmasm: DBG_INIT - Hardware Case = ASMTYPE_ASR. \n");
            }
            else if (HardwareType == ASMTYPE_JASPER) 
            {
              printk(KERN_CRIT "ibmasm: DBG_INIT - Hardware Case = ASMTYPE_JASPER. \n");
            }
            else 
            {
              printk(KERN_CRIT "ibmasm: DBG_INIT - Hardware Case = ASMTYPE_PEARL. \n");
            }
         }
         
         NumSpNodes=1;

         // ALLOCATE AND INITIALIZE THE PRIMARY DEVICE DRIVER EXTENSION
         pSpPrimaryNode = NULL;
         pSpPrimaryNode = (PDRIVER_INFO)kmalloc(sizeof(DRIVER_INFO), GFP_KERNEL); // PHR_MULTI_1

         if ( pSpPrimaryNode == NULL )
         {
            printk(KERN_CRIT "ERROR - Failed to allocate primary device driver extension.\n");
            return -ENOMEM;
         }

         memset(pSpPrimaryNode, 0, sizeof(DRIVER_INFO));

         pSpPrimaryNode->AsmType = HardwareType;

         // IF WE'RE IN ASR MODE, FOLLOW AN ALTERNATE EXECUTION PATH
         return sysSpASR();
      }
      case ASMTYPE_RANGER:
      {
         if (DBG_INIT)
         {
            printk(KERN_CRIT "ibmasm: DBG_INIT - Hardware Case = ASMTYPE_RANGER");
         }

         NumSpNodes=1;

         // ALLOCATE AND INITIALIZE THE PRIMARY DEVICE DRIVER EXTENSION
         pSpPrimaryNode = NULL;
         pSpPrimaryNode = (PDRIVER_INFO)kmalloc(sizeof(DRIVER_INFO), GFP_KERNEL); // PHR_MULTI_1

         if ( pSpPrimaryNode == NULL )
         {
            printk(KERN_CRIT "ERROR - : Failed to allocate primary device driver extension.\n");
            return -ENOMEM;
         }

         memset(pSpPrimaryNode, 0, sizeof(DRIVER_INFO));

         pSpNodeArray = pSpPrimaryNode;

         pSpPrimaryNode->AsmType = HardwareType;

         // SET UP THE ISA HW CONFIGURATION
         sysSpGetIsaInfo();

         // INITIALIZE THE DEVICE DRIVER NODE
         rc = InitDriverNode(pSpPrimaryNode);

         if (rc != RC_SUCCESS)
         {
             printk(KERN_CRIT "ERROR - Failed to initialize primary device driver node. Aborting.\n");
             RemoveNode(pSpPrimaryNode);
             return -ENODEV;
         }

         // INSTALL THE ISA INTERRUPT HANDLER
         InstallIntHndlr(pSpPrimaryNode);

         if (rc != RC_SUCCESS)
         {
            printk(KERN_CRIT "ERROR - Failed to install the ISA interrupt handler. Aborting.\n");
            RemoveNode(pSpPrimaryNode);
            return -ENODEV;
         }

         // INITIALIZE ISA H/W
         InitIsaHW();

         // SEND THE INITIALIZATION MESSAGES TO THE SP
         rc = SendInitMsgs(pSpPrimaryNode);

         if (rc != RC_SUCCESS)
         {
            printk(KERN_CRIT "ERROR - Initialization messages sent to the ASM device failed . Aborting.\n");
            RemoveNode(pSpPrimaryNode);
            return -ENOMEM;
         }

         // INDICATE THAT THE SP IS PRESENT 
         pSpPrimaryNode->devPresent = TRUE;

         break;
      }  // end case ASMTYPE_RANGER 
#endif   // end #ifndef IA64      
      case ASMTYPE_WISEMAN:
      case ASMTYPE_EAGLE:
      {
         if (DBG_INIT)
         {
            printk(KERN_CRIT "ibmasm: DBG_INIT - Hardware Case = ASMTYPE_EAGLE. \n");
         }

         // ENSURE PCI SUPPORT IS IN THE KERNEL
         #ifndef CONFIG_PCI
            printk(KERN_CRIT "ERROR: PCI support is not available in this kernel! Aborting.\n");
            return -ENODEV; 
         #endif
     
         
         // GET THE NUMBER OF MULTI NODE CAPABLE SPs PRESENT  
         NumSpNodes = sysSpCountEagles();

         SPNodeArraySize = ( (sizeof(DRIVER_INFO)) * NumSpNodes);

         // ALLOCATE AN ARRAY OF DRIVER EXTENSIONS
         pSpNodeArray = NULL;
         pSpNodeArray = (PDRIVER_INFO)kmalloc(SPNodeArraySize, GFP_KERNEL);

         if (!pSpNodeArray)
         {
              printk(KERN_CRIT "ibmasm: sysSpinit() Failed to allocate driver extension. Aborting\n");
              return -ENOMEM;
         }

         pSpPrimaryNode = pSpNodeArray;       

         for ( i = 0; i < NumSpNodes; i++ )
         {
            pDriverNode = pSpNodeArray + i;

            memset(pDriverNode, 0, sizeof(DRIVER_INFO));

            // SET HARDWARE TYPE 
            pDriverNode->AsmType = HardwareType;

            // INITIALIZE PCI CONFIGURATION
            rc = sysSpGetPciInfo(pDriverNode);

            if (rc != RC_SUCCESS)
            {
               if (DBG_INIT)
               {
                  printk(KERN_CRIT "ibmasm: DBG_INIT - Failed to determine PCI configuration info. Aborting.\n");
               }

               HardwareType = ASMTYPE_UNKNOWN;

               rc = ASMTYPE_UNKNOWN; 

               break; 
            }

            // INITIALIZE DRIVER NODE
            rc = InitDriverNode(pDriverNode);

            if (rc != RC_SUCCESS)
            {
               printk(KERN_CRIT "ERROR - Failed to initialize device driver node %i. Aborting.\n", i);
               RemoveNode(pDriverNode);
               return -ENODEV;
            }

            // INSTALL THE PCI INTERRUPT HANDLER
            InstallIntHndlr(pDriverNode);

            if (rc != RC_SUCCESS)
            {
               printk(KERN_CRIT "ERROR - Failed to install the PCI interrupt handler. Aborting.\n");
               RemoveNode(pDriverNode);
               return -ENODEV;
            }

            // INDICATE THAT THE SP IS PRESENT 
            pDriverNode->devPresent = TRUE;

            // FOR EAGLES DISABLE COM PORT INTERRUPTS AND INITIALIZE REMOTE MOUSE SUPPORT
            if ( pDriverNode->AsmType == ASMTYPE_EAGLE )
            {
                SCOUT_DISABLE_COM_INTERRUPTS(pDriverNode->vBaseAddr + SCOUT_INT_STATUS_REGISTER);

                rc = InitRemMouse(pDriverNode);

                if ( rc != RC_SUCCESS )
                {
                   printk(KERN_CRIT "ERROR - Failed to initialize remote mouse support. Aborting.\n");
                   RemoveNode(pDriverNode);
                   return -ENOMEM;
                }
            }

            // SEND THE INITIALIZATION MESSAGES TO THE SP
            rc = SendInitMsgs(pDriverNode);

            if (rc != RC_SUCCESS)
            {
               printk(KERN_CRIT "ERROR - Initialization messages sent to the ASM device failed . Aborting.\n");
               RemoveNode(pDriverNode);
               return -ENODEV;
            }

            // INDICATE THAT THE SP IS PRESENT 
            pDriverNode->devPresent = TRUE;

         }  // end for NumNodes

         break;

      }     // end case ASMTYPE_EAGLE
      default:
      {
         printk(KERN_CRIT "ibmasm: Failed to find required ASM hardware. Aborting\n");
         
         //RemoveNode(pDriverNode);
         return -ENODEV;
      }
   }        // end of HardwareType switch

   return 0;
}

// *********************************************************************
// * sysSPDisableInterrupts() - This is a generic macro to disable the *
// *                            SP interrupts. (Eagle)                 *
// *                                                                   *
// * Parameters:                                                       *
// *      pDriverNode                                                  *
// *      templong                                                     *
// *********************************************************************
void sysSpDisableInterrupts(PDRIVER_INFO pDriverNode, unsigned long templong)
{
   if (DBG_INIT)
   {
      printk(KERN_CRIT "ibmasm: DBG_INIT - Disabling all SP PCI interrupts.\n");
   }

   if (pDriverNode->AsmType == ASMTYPE_EAGLE)
   {
      SCOUT_DISABLE_ALL_SP_INTERRUPTS(pDriverNode->vBaseAddr, templong);
   }
   #ifndef IA64     // PHR_180930
   else
   {
      DISABLE_ALL_SP_INTERRUPTS(pDriverNode->ioAddr, templong);
   }
   #endif
   
   return;
}


// *********************************************************************
// * sysSPRestoreInterrupts() - This is a generic macro to restore the *
// *                            SP interrupts. (Eagle)                 *
// *                                                                   *
// * Parameters:                                                       *
// *      pDriverNode                                                  *
// *      templong                                                     *
// *********************************************************************
void sysSpRestoreInterrupts(PDRIVER_INFO pDriverNode, unsigned long templong)
{
   if (DBG_INIT)
   {
      printk(KERN_CRIT "ibmasm: DBG_INIT - Restoring all SP PCI interrupts.\n");
   }

   if ( pDriverNode->AsmType == ASMTYPE_EAGLE )
   {
      SCOUT_RESTORE_SP_INTERRUPTS(pDriverNode->vBaseAddr, templong);
   }
   #ifndef IA64     // PHR_180930
   else 
   {
      RESTORE_SP_INTERRUPTS(pDriverNode->ioAddr, templong);
   }
   #endif

   return;
}

// *********************************************************************
// * sysSPEnableInterrupts() - This is a generic macro to enable the   *
// *                           SP interrupts. (Eagle)                  *
// *                                                                   *
// * Parameters:                                                       *
// *      pDriverNode                                                  *
// *      templong                                                     *
// *********************************************************************
void sysSpEnableInterrupts(PDRIVER_INFO pDriverNode)
{
   if (DBG_INIT)
   {
       printk(KERN_CRIT "ibmasm: DBG_INIT - Enabling all SP PCI interrupts.\n");
   }

   if ( pDriverNode->AsmType == ASMTYPE_EAGLE )
   {
        SCOUT_ENABLE_ALL_SP_INTERRUPTS(pDriverNode->vBaseAddr);
   }
   #ifndef IA64     // PHR_180930
   else
   {
         ENABLE_ALL_SP_INTERRUPTS(pDriverNode->ioAddr); 
   }
   #endif
   return;
}

// *************************************************************************
// * RemoveNode - This is called by sysSPinit when a fatal error occurs   * 
// *               during driver initialization. Basically marks device as *
// *               non-existant, and de-allocates all memory resources.    *
// *               Called both when undoing sysSpinit's work when driver   *
// *               init fails, and when driver module is unloaded.         *
// *                                                                       *
// * Parameters:                                                           *
// *                                                                       *
// *************************************************************************
void RemoveNode(PDRIVER_INFO pDriverNode)
{
   int i;

   // ONLY CONTINUE IF THE REQUIRED NODES EXIST
   if ( ( !pSpPrimaryNode) || (!pSpNodeArray) || (!pDriverNode) )
   {
       if (DBG_SHUTDOWN)
       {
          printk(KERN_CRIT "ibmasm: DBG_SHUTDOWN: Invalid Node pointer. Aborting removal of node\n");
       }

       return;            
   }

   // MARK DRIVER EXTENSION AS DEVICE NOT BEING PRESENT
   pDriverNode->devPresent = FALSE;

   // FREE EVENT BUFFERS
   for (i = 0; i < EV_BUFFERS; i++)
   {
      if (pDriverNode->EvRcvd[i].pEvBuffer != NULL)
      {
           kfree(pDriverNode->EvRcvd[i].pEvBuffer);
           pDriverNode->EvRcvd[i].pEvBuffer = NULL;
      }
   }

   // FREE HEART BEAT DATA BUFFER         // PHR_MULTI_2
   if (pDriverNode->pHB_Block->pBuffer)
   {
      kfree(pDriverNode->pHB_Block->pBuffer);                                                                             
      pDriverNode->pHB_Block->pBuffer = NULL;

      if (DBG_HB)
      {
         printk(KERN_CRIT "DBG_HB: Freeing heart beat data buffer\n");
      }
   }

   // FREE HEART BEAT COMMAND BLOCK       // PHR_MULTI_2
   if (pDriverNode->pHB_Block)
   {
      kfree(pDriverNode->pHB_Block);                                                                             
      pDriverNode->pHB_Block = NULL;

      if (DBG_HB)
      {
         printk(KERN_CRIT "DBG_HB: Freeing heart beat command block\n");
      }
   }

   // FREE THE RESPONSE BUFFER
   if (pDriverNode->pSpResponseBuffer)
   {
       kfree(pDriverNode->pSpResponseBuffer);
       pDriverNode->pSpResponseBuffer = NULL;
   }

   // FREE REMOTE MOUSE/KEYBOARD DATA BUFER
   if (pDriverNode->pRemoteDataStart)                     
   {                                                      
       kfree(pDriverNode->pRemoteDataStart);              
       pDriverNode->pRemoteDataStart = NULL;              
   }

   // FREE THE IRQ 
   free_irq((unsigned int)pDriverNode->intVector, pDriverNode); 

   // FREE THE NODE ARRAY IF THIS REQUEST IS TO REMOVE THE PRIMARY NODE (0)
   if ( pDriverNode == pSpPrimaryNode )
   {
      if ( pSpNodeArray )
      {
         kfree( pSpNodeArray );
         pSpNodeArray   = NULL;
         pSpPrimaryNode = NULL;
      }
   }
}


// *********************************************************************
// *********************************************************************
int GetPrimaryAsmType()
{
     unsigned char HardwareCode [10];
     int           HardwareType;

     HardwareType = ASMTYPE_UNKNOWN;   

#ifdef IA64
     HardwareType = ASMTYPE_EAGLE;

     if (DBG_INIT)
     {
        printk(KERN_CRIT "ibmasm: DBG_INIT - IA64 Defined - Eagle expected.\n");
     }


     return HardwareType;
#else             // PHR_180930
     if (DBG_FORCEASR)
     {
        printk(KERN_CRIT "DBG_FORCEASR: Forcing ASR.\n");
        return ASMTYPE_ASR;
     }

     // DETECT HARDWARE (IN CASE MULTIPLE HARDWARE INDICATED - WE FOLLOW A HIERARCHY)
     if (!DDINST_WhichDriver(HardwareCode))
     {
        if (DBG_INIT)
        {
           printk(KERN_CRIT "ibmasm: DBG_INIT - Failed to detect Service Processor hardware. Aborting.\n");
        }

        return ASMTYPE_UNKNOWN;
     }

     switch (HardwareCode[0])
     {
        case SPTYPE_RANGER:
        {
           HardwareType = ASMTYPE_RANGER;

           if (DBG_INIT)
           {
              printk(KERN_CRIT "ibmasm: DBG_INIT - DDINST_WhichDriver() found Ranger.\n"); 
           }

           break;
        }
        case SPTYPE_WISEMAN: 
        {
           HardwareType = ASMTYPE_WISEMAN;
           
           if (DBG_INIT)
           {
              printk(KERN_CRIT "ibmasm: DBG_INIT - DDINST_WhichDriver() found Wiseman.\n"); 
           }

           break;
        }
        case SPTYPE_EAGLE:
        {
           HardwareType = ASMTYPE_EAGLE;

           if (DBG_INIT)
           {
              printk(KERN_CRIT "ibmasm: DBG_INIT - DDINST_WhichDriver() found Eagle.\n"); 
           }

           break;
        }
        case SPTYPE_ASR:
        {
           HardwareType = ASMTYPE_ASR;

           if ( DBG_INIT )
           {
              printk(KERN_CRIT "ibmasm: DBG_INIT - DDINST_WhichDriver() found Topaz ASR.\n"); 
           }
           else if ( DBG_ASR )
           {
              printk(KERN_CRIT "DBG_ASR: DDINST_WhichDriver() found Topaz ASR.\n"); 
           }

           break;
        }
        case SPTYPE_JASPER:
        {
           HardwareType = ASMTYPE_JASPER;

           if ( DBG_INIT )
           {
              printk(KERN_CRIT "ibmasm: DBG_INIT - DDINST_WhichDriver() found Jasper ASR.\n"); 
           }
           else if ( DBG_ASR )
           {
              printk(KERN_CRIT "DBG_ASR: DDINST_WhichDriver() found Jasper ASR.\n"); 
           }

           break;
        }
        case SPTYPE_PEARL:
        {
           HardwareType = ASMTYPE_PEARL;

           if ( DBG_INIT )
           {
              printk(KERN_CRIT "ibmasm: DBG_INIT - DDINST_WhichDriver() found Pearl ASR.\n"); 
           }
           else if ( DBG_ASR )
           {
              printk(KERN_CRIT "DBG_ASR: DDINST_WhichDriver() found Pearl ASR.\n"); 
           }

           break;
        }
        default:
           if (DBG_INIT)
           {
             printk(KERN_CRIT "ibmasm: DBG_INIT - Received invalid hardware code from DDINST_WhichDriver().\n");
           }

           HardwareType = ASMTYPE_UNKNOWN;
     }  // end switch 

     return HardwareType;
#endif  // end #if IA64
}



// *********************************************************************
// *********************************************************************
int InitDriverNode(PDRIVER_INFO pDriverNode)
{
   int rc;
   int i;

   ULONG tempUlong;

   rc        = RC_SUCCESS;
   tempUlong = 0;

   // INITIALIZE PID TABLE - MAXPIDS = 16
   for ( i = 0; i < MAXPIDS; i++ )
   {
       pDriverNode->PIDtable[i].valid = 0;
   }

   // INITIALIZE COMMAND AND EVENT QUEUES
   pDriverNode->EventQueue = NULL;
   pDriverNode->CmdQueue   = NULL;

   // INITIALIZE QUEUE SPINLOCKS 
   pDriverNode->CmdQueueLock   = SPIN_LOCK_UNLOCKED;
   pDriverNode->EventQueueLock = SPIN_LOCK_UNLOCKED;
   pDriverNode->RemoteDataLock = SPIN_LOCK_UNLOCKED;      

   // ALLOCATE AND INITIALIZE EVENT BUFFERS

   for (i = 0; i < EV_BUFFERS; i++)
   {
      pDriverNode->EvRcvd[i].pEvBuffer = NULL;
   }

   for (i = 0; i < EV_BUFFERS; i++)
   {
      pDriverNode->EvRcvd[i].pEvBuffer = (EVRCVD *)kmalloc(EV_BUFFER_SIZE, GFP_KERNEL);

      if (pDriverNode->EvRcvd[i].pEvBuffer == NULL)
      {
           if (DBG_INIT) 
           {
               printk(KERN_CRIT "ibmasm: DBG_INIT - Failed to prepare event buffer #%d. Exiting.\n", i);
           }

           RemoveNode(pDriverNode);

           return RC_SYSTEM_ERROR;
      }

      memset(pDriverNode->EvRcvd[i].pEvBuffer, 0, EV_BUFFER_SIZE);
   }

   // ALLOCATE AND INITIALIZE HEART BEAT COMMAND BUFFER 
   if (DBG_HB)
   {
      printk(KERN_CRIT "DBG_HB: Allocating and initializing heart beat command block\n");
   }

   pDriverNode->pHB_Block = NULL;
   pDriverNode->pHB_Block = (PWORK_BLOCK)kmalloc((sizeof(WORK_BLOCK)), GFP_KERNEL);

   if ( !pDriverNode->pHB_Block )
   {
       printk(KERN_CRIT "ibmasm: ERROR - sysSpinit() failed to allocate required buffer(s). Exiting.\n");

       RemoveNode(pDriverNode);
       return RC_SYSTEM_ERROR;
   }

   memset(pDriverNode->pHB_Block, 0, sizeof(WORK_BLOCK));

   // ALLOCATE AND INITIALIZE HEART BEAT DATA BUFFER 
   pDriverNode->pHB_Block->pBuffer = NULL;
   pDriverNode->pHB_Block->pBuffer = (UCHAR *)kmalloc(HB_BUFFER_SIZE, GFP_KERNEL);

   if ( !pDriverNode->pHB_Block->pBuffer )
   {
       printk(KERN_CRIT "ibmasm: ERROR - sysSpinit() failed to allocate required buffer(s). Exiting.\n");

       RemoveNode(pDriverNode);
       return -ENOMEM;
   }

   memset(pDriverNode->pHB_Block->pBuffer, 0, HB_BUFFER_SIZE);

   if (DBG_HB)
   {
      printk(KERN_CRIT "DBG_HB: Allocating and initializing heart beat data buffer\n");
   }

   // ALLOCATE AND INITIALIZE THE RESPONSE BUFFER 
   pDriverNode->pSpResponseBuffer = NULL;
   pDriverNode->pSpResponseBuffer = (UCHAR *)kmalloc(RESPONSE_BUFFER_SIZE, GFP_KERNEL);

   if ( !pDriverNode->pSpResponseBuffer )
   {
       printk(KERN_CRIT "ibmasm: ERROR - sysSpinit() failed to allocate required buffer(s). Exiting.\n");

       RemoveNode(pDriverNode);
       return -ENOMEM;
   }

   memset(pDriverNode->pSpResponseBuffer, 0, RESPONSE_BUFFER_SIZE );

   // FOR EAGLE OR WISEMAN - TEMPORARILY DISABLE SP INTERRUPTS
   // BEFORE INITIALIZING THE TIMER(S)
   if ( pDriverNode->AsmType == ASMTYPE_EAGLE   || 
        pDriverNode->AsmType == ASMTYPE_WISEMAN )
   {
      sysSpDisableInterrupts(pDriverNode, tempUlong);

      init_timer(&(pDriverNode->CommandTimer));

      sysSpRestoreInterrupts(pDriverNode, tempUlong);

   }
   else  // RANGER - JUST INITIALIZE THE COMMAND TIMER 
   {
      init_timer(&(pDriverNode->CommandTimer));
   }


   return rc;
}

// *********************************************************************
// *********************************************************************
int InstallIntHndlr(PDRIVER_INFO pDriverNode)
{
   int   rc;
   int   iIRQreqrc; 
   ULONG tempUlong;

   rc = RC_SUCCESS;
   tempUlong = 0;

   // INSTALL THE INTERRUPT HANDLER
   if (DBG_INIT) 
   {
       printk(KERN_CRIT "ibmasm: DBG_INIT - Installing interrupt handler at %d.\n", pDriverNode->intVector);
   }

   #ifdef IA64      // PHR_180930
      iIRQreqrc = request_irq((unsigned int)pDriverNode->intVector, sysSp_pciIntrHandler, (SA_INTERRUPT | SA_SHIRQ), "ibmasm", (void*)pDriverNode); 
   #else
      if (pDriverNode->AsmType == ASMTYPE_RANGER) 
      {
          iIRQreqrc = request_irq((unsigned int)pDriverNode->intVector, sysSp_isaIntrHandler, SA_INTERRUPT, "ibmasm", (void*)pDriverNode); 
      }
      else
      {
          iIRQreqrc = request_irq((unsigned int)pDriverNode->intVector, sysSp_pciIntrHandler, (SA_INTERRUPT | SA_SHIRQ), "ibmasm", (void*)pDriverNode); 
      }
   #endif

   //  CHECK TO MAKE SURE THE INTERRUPT HANDLER INSTALLED OK, IF NOT RESTORE 
   //  ANY IRQ BITS THAT WERE POSSIBLY TURNED OFF, DE-REGISTER IRQ, AND EXIT
   if (iIRQreqrc == 0)
   {
      if (DBG_INIT) printk(KERN_CRIT "sysSpinit: Interrupt handler registered.\n");
      disable_irq((int)pDriverNode->intVector);
      enable_irq( (int)pDriverNode->intVector);
   }
   else
   {
      if ( pDriverNode->AsmType == ASMTYPE_EAGLE   || 
           pDriverNode->AsmType == ASMTYPE_WISEMAN )
      {
          sysSpRestoreInterrupts(pDriverNode, tempUlong);
      }

      free_irq((unsigned int)pDriverNode->intVector, pDriverNode); 
      RemoveNode(pDriverNode);
      return RC_SYSTEM_ERROR;
   }
   
   // IF PCI DEVICE, ENSURE THE PCI & Doorbell INTERRUPTS ARE ENABLED
   if ( (pDriverNode->AsmType == ASMTYPE_EAGLE)   ||
        (pDriverNode->AsmType == ASMTYPE_WISEMAN)  ) 
   {
      sysSpEnableInterrupts(pDriverNode);
   }

   return rc;
}

#ifndef IA64
// *********************************************************************
// *********************************************************************
void InitIsaHW(void)
{
   int data34h;                                          

   if (DBG_INIT)
   {
       printk(KERN_CRIT "ibmasm: DBG_INIT - Initialize Slim TX & RX machines for Ranger.\n");
   }

   // INITIALIZE SLIM TX & RX STATE MACHINES
   InitializeSlimReceive(&pSpPrimaryNode->receiveSlim, NULL, 0);
   ResetSlimSend(&pSpPrimaryNode->sendSlim);

   // Read the byte at offset 0x34.  The MSB indiscates which nibble to
   // store the SP IRQ vector in when writing to the IMUX register.
   // The lower six bits are the IMUX address.
   outb(0x34, 0x70);
   data34h = inb(0x71);

   // Determine what Ranger family this driver is connected to:
   // If I/O Port 0xEC reads 0xFF, then this is KIOWA/MOHICAN system
   // and must gate the IMUX accordingly; otherwise, there's nothing
   // to do.
   // Gate the interrupt multiplexer on. This can be one of two values.
   // The most significant bit (MSB) of the data byte read at CMOS
   // address 0x34 indicates how to compute the value.  If this bit
   // is set, then this KIOWA compatible; otherwise, it is MOHICAN
   // compatible.  Another difference is the address of the IMUX 
   // register.  In KIOWA, it is at offset 0x0D.  In MOHICAN, it is
   // at offset 0x0E.  This offset value is also stored in the lower
   // six bits of the byte at CMOS address 0x34.
   if (inb(0xEC) == 0xff)
   {
      if (DBG_INIT) 
      {
         printk(KERN_CRIT "ibmasm: DBG_INIT - Gating IMUX for Kiowa or Mohican.\n");
      }

      PUT_IMUX_SEL((data34h & 0x3f));

      // DETERMINE WHETHER THIS IS A KIOWA-BASED PLATFORM
      if (data34h & 0x80)
      {
        PUT_IMUX_CFG((UCHAR)((GET_IMUX_CFG & IMUX_KIOWA_CFG_MASK) | pSpPrimaryNode->intVector<<4));
      }
      else
      {
        PUT_IMUX_CFG((UCHAR)((GET_IMUX_CFG & IMUX_MOHIC_CFG_MASK) | pSpPrimaryNode->intVector));
      }
   }
   else
   {
      if (DBG_INIT) printk(KERN_CRIT "sysSpinit: Not gating IMUX.\n");
   }
}
#endif

// *********************************************************************
// *********************************************************************
int InitRemMouse(PDRIVER_INFO pDriverNode)
{
   // INITIALIZE THE REMOTE MOUSE/KEYBOARD SUPPORT
   pDriverNode->pRemoteDataBaseAddr = pDriverNode->vBaseAddr + CONDOR_MOUSE_DATA_PCI_OFFSET;

   if (DBG_MOUSE) printk(KERN_CRIT "PHR_DBG: sysSpinit - pRemoteDataBaseAddr = %pc.\n", pDriverNode->pRemoteDataBaseAddr ); 

   pDriverNode->pRemoteDataStart = NULL;
   pDriverNode->pRemoteDataStart = (PREMOTE_INPUT_DATA)kmalloc( MAX_QUEUE_LENGTH, GFP_KERNEL ); 

   if ( !pDriverNode->pRemoteDataStart ) 
   {
       printk(KERN_CRIT "ibmasm: ERROR - sysSpinit() failed to allocate required buffer(s). Exiting.\n");

       RemoveNode(pDriverNode);
       return RC_SYSTEM_ERROR;
   }

   memset(pDriverNode->pRemoteDataStart, 0, MAX_QUEUE_LENGTH);

   pDriverNode->pRemoteDataEnd =(PREMOTE_INPUT_DATA)((PCHAR)(pDriverNode->pRemoteDataStart) + MAX_QUEUE_LENGTH);
   pDriverNode->pRemoteDataIn  = pDriverNode->pRemoteDataStart;
   pDriverNode->pRemoteDataOut = pDriverNode->pRemoteDataStart;

   pDriverNode->RemoteInputCount = 0;

   // SET THE BIT THAT TELLS THE SP THAT THE DRIVER SUPPORTS REMOTE GRAPHICS
   WRITE_REGISTER_ULONG( (PULONG)(pDriverNode->pRemoteDataBaseAddr + CONDOR_MOUSE_ISR_CONTROL_OFFSET), 0);

   return RC_SUCCESS;
}




// *********************************************************************
// *********************************************************************
int SendInitMsgs(PDRIVER_INFO pDriverNode)
{
   DD_IOCTL_INFO  dd_Ioctl_Info;
   PDD_IOCTL_INFO pdd_Ioctl_Info;
   unsigned char *pInitBuffer;

   int rc, i;

   pdd_Ioctl_Info = &dd_Ioctl_Info;

   rc = RC_SUCCESS;

   // ALLOCATE AND INITIALIZE THE INITIALIZATION BUFFER
   pInitBuffer = (unsigned char *)kmalloc((SEND_BUFFER_SIZE), GFP_KERNEL);

   if ( !pInitBuffer  )
   {
       printk(KERN_CRIT "ibmasm: ERROR - sysSpinit() failed to allocate initialization buffer. Exiting.\n");

       RemoveNode(pDriverNode);
       return RC_SYSTEM_ERROR;
   }

   memset(pInitBuffer, 0, SEND_BUFFER_SIZE );

   // Fill the buffer with spaces 
   for (i = 0; i < 27; i++ )
   {
      pInitBuffer[i] = ' ';
   }

   // SEND THE 4.3.5.A (DRIVER VPD) TO THE SP
   if (DBG_INIT) printk(KERN_CRIT "sysSpinit: Sending 4.3.5.A to SP....\n");

   pInitBuffer[0]  = spWrite;    // Command Type   = spWrite
   pInitBuffer[1]  = 4;          // Command Length = 4 bytes (4.3.5.A)
   pInitBuffer[2]  = 16;         // Data Length    = 0016
   pInitBuffer[3]  = 0;              
   pInitBuffer[4]  = 0;          // Status         = 0
   pInitBuffer[5]  = 0;          // Reserved       = 0
   pInitBuffer[6]  = 4;          // Command Byte 0 = 4
   pInitBuffer[7]  = 3;          // Command Byte 1 = 3
   pInitBuffer[8]  = 5;          // Command Byte 2 = 5
   pInitBuffer[9]  = 0x0a;       // Command Byte 3 = A

   pInitBuffer[10] = 'L';        // Start of Data - String = "Lin 1 X.XX      "
   pInitBuffer[11] = 'I';
   pInitBuffer[12] = 'N';
   pInitBuffer[13] = '3';
   pInitBuffer[14] = '2';
   pInitBuffer[15] = ' ';

   #ifdef IA64
     pInitBuffer[13] = '6';
     pInitBuffer[14] = '4';
   #endif

   sprintf(&(pInitBuffer[16]), "%d.%02d", LINUX_DD_MAJOR, LINUX_DD_MINOR);

   pInitBuffer[25] = '\0';

   // Initialize device driver structure interface
   dd_Ioctl_Info.returnCode   = 0;
   dd_Ioctl_Info.bufferLength = 26;
   dd_Ioctl_Info.pBuffer      = pInitBuffer;

   rc = sysSpProcessIoctl(IOCTL_RW, pdd_Ioctl_Info, PID_OS_IS_UP, INTERNAL_IOCTL, pDriverNode);

   if ( rc == 0 ) 
   {
      printk(KERN_CRIT "ibmasm: ERROR - failed to initialize\n");
      rc = RC_SYSTEM_ERROR;
   }
   else  // SEND THE 4.3.6 (SYSTEM STATE - OS IS UP) TO THE SP  
   {
      memset(pInitBuffer, 0, SEND_BUFFER_SIZE );

      if (DBG_INIT) printk(KERN_CRIT "sysSpinit: Sending 4.3.6 to SP....\n");
      
      pInitBuffer[0] = spWrite;  // Command Type   
      pInitBuffer[1] = 3;        // Command Length 
      pInitBuffer[2] = 1;        // Data Length  LSB  
      pInitBuffer[3] = 0;        // Data Length  MSB         
      pInitBuffer[4] = 0;        // Status         
      pInitBuffer[5] = 0;        // Reserved       
      pInitBuffer[6] = 4;        // Command Byte 0 
      pInitBuffer[7] = 3;        // Command Byte 1 
      pInitBuffer[8] = 6;        // Command Byte 2 
      pInitBuffer[9] = 5;        // Data Byte 0    

      // Initialize device driver structure interface
      dd_Ioctl_Info.returnCode   = 0;
      dd_Ioctl_Info.bufferLength = 10;
      dd_Ioctl_Info.pBuffer      = pInitBuffer;

      rc = sysSpProcessIoctl(IOCTL_RW, pdd_Ioctl_Info, PID_OS_IS_UP, INTERNAL_IOCTL, pDriverNode);

      if (rc == 0)
      {
         printk(KERN_CRIT "ibmasm: ERROR - failed to initialize\n");
         rc = RC_SYSTEM_ERROR;
      }
      else
      {
         rc = RC_SUCCESS;
      }
   }

   if (pInitBuffer)
   {
      kfree(pInitBuffer);
   }

   return rc;
}

#undef UwsConfg_c


