/*
 * $Id: vethcfg.c,v 1.17 2004/03/04 14:01:24 yumo Exp $
 * vethcfg -- veth device driver configurator program
 *
 * Author: Yumo (Katsuyuki Yumoto) 2000-2004
 *         yumo@st.rim.or.jp
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#ifdef ETHMAP
#include <ethmap.h>
#endif

#include "vethcfg.h"

void usage(void)
{
	printf("This program must be executed by privilege user\n\n");
	printf("usage:\n");
	printf("  vethcfg ver\n");
	printf("  vethcfg add <physical i/f> <virtual i/f>\n");
	printf("  vethcfg del <physical i/f>\n");
	printf("  vethcfg move <physical i/f> <virtual i/f>\n");
	printf("  vethcfg list\n");
	printf("  vethcfg showagg <virtual i/f>\n");
	printf("  vethcfg showport <physical i/f>\n");
	printf("  vethcfg setagg <virtual i/f> <var> <val>\n");
	printf("         -- actor part --\n");
	printf("      <var>      <val>\n");
	printf("      key       256*n (n: 1-255)\n");
	printf("      distalgo  tcpudp | dasa | rrobin (take care)\n");
	printf("  vethcfg setport <physical i/f> <var> <val>\n");
	printf("            == actor part ==\n");
	printf("      <var>      <val>             description\n");
	printf("      akey      256*n (n: 1-255) -- actor key\n");
	printf("            == actor port status ==\n");
	printf("      aps_ac    1 | 0            -- LACP ACTIVITY\n");
	printf("      aps_to    1 | 0            -- LACP TIMEOUT\n");
	printf("      aps_ag    1 | 0            -- AGGREGATION\n");
	printf("      aps_sy    1 | 0            -- SYNCHRONIZATION\n");
	printf("      aps_co    1 | 0            -- COLLECTING\n");
	printf("      aps_di    1 | 0            -- DISTRIBUTING\n");
	printf("      aps_de    1 | 0            -- DEFAULTED\n");
	printf("      aps_ex    1 | 0            -- EXPIRED\n");

	printf("           == partner part ==\n");
	printf("      <var>      <val>\n");
	printf("      psyspri   0-65535          -- system priority\n");
	printf("      pkey      0-65535          -- key\n");
	printf("      ppnum     0-65535          -- port number\n");
	printf("      pppri     0-65535          -- port priority\n");
	printf("         == partner port status ==\n");
	printf("      pps_ac    1 | 0            -- LACP ACTIVITY\n");
	printf("      pps_to    1 | 0            -- LACP TIMEOUT\n");
	printf("      pps_ag    1 | 0            -- AGGREGATION\n");
	printf("      pps_sy    1 | 0            -- SYNCHRONIZATION\n");
	printf("      pps_co    1 | 0            -- COLLECTING\n");
	printf("      pps_di    1 | 0            -- DISTRIBUTING\n");
	printf("      pps_de    1 | 0            -- DEFAULTED\n");
	printf("      pps_ex    1 | 0            -- EXPIRED\n");
	return;
}

int build_args(char **argv, char *str)
{
	int argc = 0;
	int wf = 0;

	if (str[0] == '#' || str[0] == ';')
		return 0;

	for (; *str; str++){
		switch (*str){
		case ' ':
		case '\t':
		case '\n':
			*str = '\0';
			wf = 0;
			break;
		default:
			if (!wf){
				argv[argc++] = str;
				wf = 1;
			}
		}
	}
	return argc;
}

int veth_config_using_file(char *config_file)
{
	FILE *fp;
	int ret = 0;
	int argc;
	char *argv[10];
	char str[MAXLINE+1];
	int line = 1;

	fp = fopen(config_file, "r");
	for (; fgets(str, MAXLINE, fp); line++){
		argc = build_args(argv+1, str);
		if (argc){
			ret = veth_command(argc+1, argv);
			if (ret){
				fprintf(stderr, "%s: Error in \"%s\" line %d. Aborted.\n",
						PROG_NAME, config_file, line);
				break;
			}
		}
	}
	fclose(fp);
	return ret;
}

int main(int argc, char **argv)
{
	struct stat buf;
	char *config_file;
	int ret;

	ret = 1;
	config_file = strdup(DEFAULT_CONFIG_FILE);
	if (!config_file){
		perror(PROG_NAME);
		goto exit;
	}

	if (argc > 1){
		if (strcmp(argv[1], "-h") == 0 ||
			strcmp(argv[1], "--help") == 0){
			usage();
			ret = 0;
			goto exit;
		} else if (strcmp(argv[1], "-c") == 0){
			if (argc < 3){
				usage();
				goto exit;
			}
			free(config_file);
			config_file = strdup(argv[2]);
			if (!config_file){
				perror(PROG_NAME);
				goto exit;
			}
		} else {
#ifdef ETHMAP
			ethmap_make_map();
			if (!ethmap_count_mapped() && !ethmap_count_unmapped()){
				/* Map is not initialized.
				 * In this case every logical interface name
				 * is same as physical interface name.
				 */
				ret = ethmap_unmap_all();
				if (ret)
					goto exit;
				ret = ethmap_init_map();
				if (ret)
					goto exit;
			}
#endif
			ret = veth_command(argc, argv);
			goto exit;
		}
	}

	if (stat(config_file, &buf)){
		fprintf(stderr, "%s: Cannot open config file \"%s\"\n", PROG_NAME, config_file);
		goto exit;
	}

#ifdef ETHMAP
	ethmap_make_map();
	if (!ethmap_count_mapped() && !ethmap_count_unmapped()){
		/* Map is not initialized.
		 * In this case every logical interface name
		 * is same as physical interface name.
		 */
		ret = ethmap_unmap_all();
		if (ret)
			goto exit;
		ret = ethmap_init_map();
		if (ret)
			goto exit;
	}
#endif
	ret = veth_config_using_file(config_file);

exit:
#ifdef ETHMAP
	ethmap_destroy_map();
#endif
	if (config_file)
		free(config_file);

	return ret;
}

/*
 * Local variables:
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  tab-width: 4
 *  indent-tabs-mode: t
 * End:
 */
