/*
 * display.c -- code to handle display related functions
 * 
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * Written by Sos Pter <sp@osb.hu>, 2002-2003
 */


#include <linux/config.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/types.h>
#include <linux/init.h>
#include <linux/proc_fs.h>

#include <asm/system.h>
#include <asm/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#ifdef OMNIBOOK_STANDALONE
#include "omnibook.h"
#else
#include <linux/omnibook.h>
#endif

#include "ec.h"

static struct proc_dir_entry *proc_display;

int omnibook_get_display(void)
{
	int retval = 0;
#ifdef CONFIG_VT
	u8 sta;
	
	switch (omnibook_ectype) {
	case XE3GF:
		retval = omnibook_ec_read(XE3GF_STA1, &sta);
		if (retval)
			return retval;
		retval = (sta & XE3GF_SHDD_MASK) ? 1 : 0;
		break;
	case XE3GC:
		retval = omnibook_ec_read(XE3GC_STA1, &sta);
		if (retval)
			return retval;
		retval = (sta & XE3GC_CRTI_MASK) ? 1 : 0;
		break;
	case OB500:
	case OB510:
	case OB6000:
	case OB6100:
	case XE4500:
		retval = omnibook_ec_read(OB500_STA1, &sta);
		if (retval)
			return retval;
		retval = (sta & OB500_CRTS_MASK) ? 1 : 0;
		break;
	case OB4150:
		retval = omnibook_ec_read(OB4150_STA2, &sta);
		if (retval)
			return retval;
		retval = (sta & OB4150_CRST_MASK) ? 1 : 0;
		break;
	default:
		printk(KERN_INFO "%s: External display status monitoring is unsupported on this machine.\n", MODULE_NAME);
		retval = -ENODEV;
	}
#endif
	return retval;
}

static int omnibook_proc_display(char *buffer, char **start, off_t off, int count, int *eof, void *data)
{
	int len;
	char *b = buffer;
	u8 dsp;
	char *dsp_str;

	dsp = omnibook_get_display();
	dsp_str = (dsp) ? "present" : "not present";
	
	b += sprintf(b, "External display is %s\n", dsp_str);
	
	len = b - buffer;
	if (len <= off + count)
		*eof = 1;
	*start = buffer + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}

int __init omnibook_display_init(void)
{
#ifdef CONFIG_VT
	mode_t pmode;

	switch (omnibook_ectype) {
	case XE3GF:
	case XE3GC:
	case OB500:
	case OB510:
	case OB6000:
	case OB6100:
	case XE4500:
		pmode = S_IFREG | S_IRUGO;
		proc_display = create_proc_read_entry("display", pmode, omnibook_proc_root, omnibook_proc_display, NULL);
		break;
	default:
		printk(KERN_INFO "%s: External display status monitoring is unsupported on this machine.\n", MODULE_NAME);
		return 0;
	}
	if (! proc_display) {
		printk(KERN_ERR "%s: Unable to create /proc/%s/display.\n", MODULE_NAME, MODULE_NAME);
		return -ENOENT;
	}
	printk(KERN_INFO "%s: External display status monitoring is enabled.\n", MODULE_NAME);
	return 0;
#else
	printk(KERN_ERR "%s: Virtual terminal support is not compiled into your kernel.\n", MODULE_NAME);
	return -ENODEV;
#endif
}

void __exit omnibook_display_cleanup(void)
{
#ifdef CONFIG_VT
	if (proc_display)
		remove_proc_entry("display", omnibook_proc_root);
#endif
}

EXPORT_SYMBOL(omnibook_get_display);

/* End of file */
