/*
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; You may only use version 2 of the License,
	you have no option to use any other version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

        gnome theme-switcher capplet - (c) Jonathan Blandford <jrb@gnome.org>
        xfce4 mcs plugin   - (c) 2002 Olivier Fourdan

 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifdef HAVE_MEMORY_H
#include <memory.h>
#endif
#include <stdio.h>
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <X11/Xlib.h>

#ifdef USE_XF86MISC
#include <X11/extensions/xf86misc.h>
#endif

#ifdef USE_XKB
#include <X11/XKBlib.h>
#endif

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gdk/gdkx.h>

#include <libxfce4mcs/mcs-common.h>
#include <libxfce4mcs/mcs-manager.h>
#include <libxfce4util/libxfce4util.h>
#include <xfce-mcs-manager/manager-plugin.h>
#include <libxfcegui4/libxfcegui4.h>

#include "plugins/gtk_common/gtk_common.h"

#define RCDIR "mcs_settings"
#define OLDRCDIR "settings"
#define CHANNEL1 "settings"
#define RCFILE1 "gtk.xml"
#define CHANNEL2 "keyboard"
#define RCFILE2 "keyboard.xml"
#define PLUGIN_NAME "keyboard"
#define DEFAULT_THEME "Default"

#define ON 1
#define OFF 0
#define ALL -1

#define MAX_ELEMENTS_BEFORE_SCROLLING 6

static void create_channel(McsPlugin * mcs_plugin);
static gboolean write_options(McsPlugin * mcs_plugin);
static void run_dialog(McsPlugin * mcs_plugin);

static gboolean setting_model = FALSE;
static gboolean initial_scroll = TRUE;
static gboolean is_running = FALSE;
static gchar *current_key_theme = NULL;
static gboolean cursor_blink = TRUE;
static int cursor_blink_time = 500;
static gboolean repeat_key = TRUE;
static int repeat_delay = 500;
static int repeat_rate = 30;

#ifdef USE_XF86MISC
static gboolean miscpresent = FALSE;
#endif

#ifdef USE_XKB
static gboolean xkbpresent = FALSE;
#endif

typedef struct _Itf Itf;
struct _Itf
{
    McsPlugin *mcs_plugin;

    GtkWidget *keyboard_dialog;
    GtkWidget *dialog_vbox1;
    GtkWidget *dialog_header;
    GtkWidget *hbox1;
    GtkWidget *frame1;
    GtkWidget *vbox2;
    GtkWidget *hbox2;
    GtkWidget *theme_swindow;
    GtkWidget *theme_treeview;
    GtkWidget *label1;
    GtkWidget *vbox3;
    GtkWidget *frame2;
    GtkWidget *vbox4;
    GtkWidget *checkrepeat;
    GtkWidget *table1;
    GtkWidget *label2;
    GtkWidget *label3;
    GtkWidget *label4;
    GtkWidget *label5;
    GtkWidget *label6;
    GtkWidget *label7;
    GtkWidget *hscale1;
    GtkWidget *hscale2;
    GtkWidget *label8;
    GtkWidget *frame3;
    GtkWidget *vbox5;
    GtkWidget *checkblink;
    GtkWidget *table2;
    GtkWidget *label9;
    GtkWidget *label10;
    GtkWidget *label11;
    GtkWidget *hscale3;
    GtkWidget *label12;
    GtkWidget *hbuttonbox1;
    GtkWidget *helpbutton1;
    GtkWidget *closebutton1;
};

static void
set_repeat(int key, int auto_repeat_mode)
{
    XKeyboardControl values;
    values.auto_repeat_mode = auto_repeat_mode;
    
    gdk_flush();
    gdk_error_trap_push();
    if(key != ALL) {
        values.key = key;
        XChangeKeyboardControl(GDK_DISPLAY(), KBKey | KBAutoRepeatMode, &values);
    }
    else {
        XChangeKeyboardControl(GDK_DISPLAY(), KBAutoRepeatMode, &values);
    }
    gdk_flush();
    gdk_error_trap_pop();
}

#ifdef USE_XF86MISC
static void
set_repeat_rate(int delay, int rate)
{
    XF86MiscKbdSettings values;

    if (miscpresent) {
        gdk_flush();
        gdk_error_trap_push();
    	XF86MiscGetKbdSettings(GDK_DISPLAY(), &values);
	values.delay = delay;
    	values.rate = rate;
    	XF86MiscSetKbdSettings(GDK_DISPLAY(), &values);
        gdk_flush();
        gdk_error_trap_pop();
    }
}
#endif

#ifdef USE_XKB
static void 
xkb_set_repeat_rate(int delay, int interval)
{

    if (xkbpresent) {
        XkbDescPtr xkb = XkbAllocKeyboard();
        if (xkb) {
            gdk_error_trap_push();
            XkbGetControls(GDK_DISPLAY(), XkbRepeatKeysMask, xkb);
            xkb->ctrls->repeat_delay = delay;
            xkb->ctrls->repeat_interval = interval;
            XkbSetControls(GDK_DISPLAY(), XkbRepeatKeysMask, xkb);
            XFree(xkb);
            gdk_flush();
            gdk_error_trap_pop();
        }
        else {
            g_warning ("XkbAllocKeyboard() returned null pointer");
        }
    }
}
#endif

static void
theme_selection_changed(GtkTreeSelection *selection, gpointer data)
{
    GtkTreeModel *model;
    gchar *new_key_theme;
    GtkTreeIter iter;
    McsPlugin *mcs_plugin;

    if(setting_model)
        return;

    mcs_plugin = (McsPlugin *) data;

    if(gtk_tree_selection_get_selected(selection, &model, &iter))
    {
        gtk_tree_model_get(model, &iter, THEME_NAME_COLUMN, &new_key_theme, -1);
    }
    else
    {
        new_key_theme = NULL;
    }

    if(new_key_theme != NULL)
    {
        if(current_key_theme && strcmp(current_key_theme, new_key_theme))
        {
            g_free(current_key_theme);
            current_key_theme = new_key_theme;
            mcs_manager_set_string(mcs_plugin->manager, "Gtk/KeyThemeName", CHANNEL1, current_key_theme);
            mcs_manager_notify(mcs_plugin->manager, CHANNEL1);
            write_options(mcs_plugin);
        }
    }
}

static void read_themes(Itf * dialog)
{
    static GList *gtk_theme_list = NULL;
    GList *list;
    GtkTreeModel *model;
    GtkTreeView *tree_view;
    gint i = 0;
    gboolean current_key_theme_found = FALSE;
    GtkTreeRowReference *row_ref = NULL;

    gtk_theme_list = theme_common_get_list(gtk_theme_list);
    tree_view = GTK_TREE_VIEW(dialog->theme_treeview);
    model = gtk_tree_view_get_model(tree_view);

    setting_model = TRUE;
    gtk_list_store_clear(GTK_LIST_STORE(model));

    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(dialog->theme_swindow), GTK_POLICY_NEVER, GTK_POLICY_NEVER);
    gtk_widget_set_size_request(dialog->theme_swindow, -1, -1);

    for(list = gtk_theme_list; list; list = list->next)
    {
        ThemeInfo *info = list->data;
        GtkTreeIter iter;

        if(!info->has_keybinding)
            continue;

        gtk_list_store_prepend(GTK_LIST_STORE(model), &iter);
        gtk_list_store_set(GTK_LIST_STORE(model), &iter, THEME_NAME_COLUMN, info->name, -1);

        if(strcmp(current_key_theme, info->name) == 0)
        {
            GtkTreePath *path = gtk_tree_model_get_path(model, &iter);
            row_ref = gtk_tree_row_reference_new(model, path);
            gtk_tree_path_free(path);
            current_key_theme_found = TRUE;
        }

        if(i == MAX_ELEMENTS_BEFORE_SCROLLING)
        {
            GtkRequisition rectangle;
            gtk_widget_size_request(GTK_WIDGET(tree_view), &rectangle);
            gtk_widget_set_size_request(dialog->theme_swindow, -1, rectangle.height);
            gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(dialog->theme_swindow), GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
        }
        i++;
    }

    if(!current_key_theme_found)
    {
        GtkTreeIter iter;
        GtkTreePath *path;

        gtk_list_store_prepend(GTK_LIST_STORE(model), &iter);
        gtk_list_store_set(GTK_LIST_STORE(model), &iter, THEME_NAME_COLUMN, current_key_theme, -1);

        path = gtk_tree_model_get_path(model, &iter);
        row_ref = gtk_tree_row_reference_new(model, path);
        gtk_tree_path_free(path);
    }

    if(row_ref)
    {
        GtkTreePath *path;

        path = gtk_tree_row_reference_get_path(row_ref);
        gtk_tree_view_set_cursor(tree_view, path, NULL, FALSE);

        if(initial_scroll)
        {
            gtk_tree_view_scroll_to_cell(tree_view, path, NULL, TRUE, 0.5, 0.0);
            initial_scroll = FALSE;
        }

        gtk_tree_path_free(path);
        gtk_tree_row_reference_free(row_ref);
    }
    setting_model = FALSE;
}

static gint sort_func(GtkTreeModel * model, GtkTreeIter * a, GtkTreeIter * b, gpointer user_data)
{
    gchar *a_str = NULL;
    gchar *b_str = NULL;

    gtk_tree_model_get(model, a, 0, &a_str, -1);
    gtk_tree_model_get(model, b, 0, &b_str, -1);

    if(a_str == NULL)
        a_str = g_strdup("");
    if(b_str == NULL)
        b_str = g_strdup("");

    if(!strcmp(a_str, DEFAULT_THEME))
        return -1;
    if(!strcmp(b_str, DEFAULT_THEME))
        return 1;

    return g_utf8_collate(a_str, b_str);
}

static void cb_dialog_response(GtkWidget * dialog, gint response_id)
{
    if(response_id == GTK_RESPONSE_HELP)
    {
        g_message("HELP: TBD");
    }
    else
    {
        is_running = FALSE;
        gtk_widget_destroy(dialog);
    }
}

static void cb_checkrepeat_changed(GtkWidget * dialog, gpointer user_data)
{
    Itf *itf = (Itf *) user_data;
    McsPlugin *mcs_plugin = itf->mcs_plugin;

    repeat_key = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(itf->checkrepeat));
    gtk_widget_set_sensitive(itf->hscale1, repeat_key);
    gtk_widget_set_sensitive(itf->hscale2, repeat_key);
    set_repeat(ALL, repeat_key ? ON : OFF);

    mcs_manager_set_int(mcs_plugin->manager, "Key/RepeatKey", CHANNEL2, repeat_key ? 1 : 0);
    mcs_manager_notify(mcs_plugin->manager, CHANNEL2);
    write_options(mcs_plugin);
}

static void cb_repeatdelay_changed(GtkWidget * dialog, gpointer user_data)
{
    Itf *itf = (Itf *) user_data;
    McsPlugin *mcs_plugin = itf->mcs_plugin;

    repeat_rate = (int)gtk_range_get_value(GTK_RANGE(itf->hscale1));
    repeat_delay = (int)gtk_range_get_value(GTK_RANGE(itf->hscale2));

    mcs_manager_set_int(mcs_plugin->manager, "Key/RepeatDelay", CHANNEL2, repeat_delay);
    mcs_manager_set_int(mcs_plugin->manager, "Key/RepeatRate", CHANNEL2, repeat_rate);

#ifdef USE_XKB
    xkb_set_repeat_rate(repeat_delay, 1000 / repeat_rate);
#endif

#ifdef USE_XF86MISC
    set_repeat_rate(repeat_delay, repeat_rate);
#endif

    mcs_manager_notify(mcs_plugin->manager, CHANNEL2);
    write_options(mcs_plugin);
}

static void cb_checkblink_changed(GtkWidget * dialog, gpointer user_data)
{
    Itf *itf = (Itf *) user_data;
    McsPlugin *mcs_plugin = itf->mcs_plugin;

    cursor_blink = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(itf->checkblink));
    gtk_widget_set_sensitive(itf->hscale3, cursor_blink);

    mcs_manager_set_int(mcs_plugin->manager, "Net/CursorBlink", CHANNEL1, cursor_blink ? 1 : 0);
    mcs_manager_notify(mcs_plugin->manager, CHANNEL1);
    write_options(mcs_plugin);
}

static void cb_blinktime_changed(GtkWidget * dialog, gpointer user_data)
{
    Itf *itf = (Itf *) user_data;
    McsPlugin *mcs_plugin = itf->mcs_plugin;

    cursor_blink_time = 2600 - (int)gtk_range_get_value(GTK_RANGE(itf->hscale3));
    mcs_manager_set_int(mcs_plugin->manager, "Net/CursorBlinkTime", CHANNEL1, cursor_blink_time);
    mcs_manager_notify(mcs_plugin->manager, CHANNEL1);
    write_options(mcs_plugin);
}

Itf *create_keyboard_dialog(McsPlugin * mcs_plugin)
{
    Itf *dialog;

    dialog = g_new(Itf, 1);

    dialog->mcs_plugin = mcs_plugin;

    dialog->keyboard_dialog = gtk_dialog_new();

    gtk_window_set_icon(GTK_WINDOW(dialog->keyboard_dialog), mcs_plugin->icon);

    gtk_window_set_title(GTK_WINDOW(dialog->keyboard_dialog), _("Keyboard Preferences"));
    gtk_window_set_default_size(GTK_WINDOW(dialog->keyboard_dialog), 440, 200);
    gtk_dialog_set_has_separator(GTK_DIALOG(dialog->keyboard_dialog), FALSE);

    dialog->dialog_vbox1 = GTK_DIALOG(dialog->keyboard_dialog)->vbox;
    gtk_widget_show(dialog->dialog_vbox1);

    dialog->dialog_header = xfce_create_header(mcs_plugin->icon,
                                               _("Keyboard Preferences"));
    gtk_widget_show(dialog->dialog_header);
    gtk_box_pack_start(GTK_BOX(dialog->dialog_vbox1), dialog->dialog_header, FALSE, TRUE, 0);

    dialog->hbox1 = gtk_hbox_new(FALSE, 5);
    gtk_widget_show(dialog->hbox1);
    gtk_box_pack_start(GTK_BOX(dialog->dialog_vbox1), dialog->hbox1, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(dialog->hbox1), 5);

    dialog->frame1 = xfce_framebox_new (_("Keyboard map"), FALSE);
    gtk_widget_show(dialog->frame1);
    gtk_box_pack_start(GTK_BOX(dialog->hbox1), dialog->frame1, FALSE, TRUE, 0);

    dialog->vbox2 = gtk_vbox_new(FALSE, 5);
    gtk_widget_show(dialog->vbox2);
    xfce_framebox_add (XFCE_FRAMEBOX (dialog->frame1), dialog->vbox2);

    dialog->hbox2 = gtk_hbox_new(FALSE, 8);
    gtk_widget_show(dialog->hbox2);
    gtk_box_pack_start(GTK_BOX(dialog->vbox2), dialog->hbox2, TRUE, TRUE, 0);

    dialog->theme_swindow = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(dialog->theme_swindow);
    gtk_box_pack_start(GTK_BOX(dialog->hbox2), dialog->theme_swindow, TRUE, TRUE, 0);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(dialog->theme_swindow), GTK_SHADOW_IN);

    dialog->theme_treeview = gtk_tree_view_new();
    gtk_widget_show(dialog->theme_treeview);
    gtk_container_add(GTK_CONTAINER(dialog->theme_swindow), dialog->theme_treeview);
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(dialog->theme_treeview), FALSE);

    dialog->vbox3 = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(dialog->vbox3);
    gtk_box_pack_start(GTK_BOX(dialog->hbox1), dialog->vbox3, TRUE, TRUE, 0);

    dialog->frame2 = xfce_framebox_new (_("Typing Settings"), TRUE);
    gtk_widget_show(dialog->frame2);
    gtk_box_pack_start(GTK_BOX(dialog->vbox3), dialog->frame2, TRUE, TRUE, 0);

    dialog->vbox4 = gtk_vbox_new(FALSE, 5);
    gtk_widget_show(dialog->vbox4);
    xfce_framebox_add (XFCE_FRAMEBOX (dialog->frame2), dialog->vbox4);

    dialog->checkrepeat = gtk_check_button_new_with_mnemonic(_("Repeat"));
    gtk_widget_show(dialog->checkrepeat);
    gtk_box_pack_start(GTK_BOX(dialog->vbox4), dialog->checkrepeat, FALSE, FALSE, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->checkrepeat), repeat_key);

    dialog->table1 = gtk_table_new(4, 3, FALSE);
    gtk_widget_show(dialog->table1);
    gtk_box_pack_start(GTK_BOX(dialog->vbox4), dialog->table1, TRUE, TRUE, 0);

    dialog->label2 = xfce_create_small_label(_("Short"));
    gtk_widget_show(dialog->label2);
    gtk_table_attach(GTK_TABLE(dialog->table1), dialog->label2, 0, 1, 1, 2, (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 2);
    gtk_label_set_justify(GTK_LABEL(dialog->label2), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(dialog->label2), 1, 0.5);

    dialog->label3 = xfce_create_small_label(_("Long"));
    gtk_widget_show(dialog->label3);
    gtk_table_attach(GTK_TABLE(dialog->table1), dialog->label3, 2, 3, 1, 2, (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 2);
    gtk_label_set_justify(GTK_LABEL(dialog->label3), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(dialog->label3), 0, 0.5);

    dialog->label4 = xfce_create_small_label(_("Slow"));
    gtk_widget_show(dialog->label4);
    gtk_table_attach(GTK_TABLE(dialog->table1), dialog->label4, 0, 1, 3, 4, (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 2);
    gtk_label_set_justify(GTK_LABEL(dialog->label4), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(dialog->label4), 1, 0.5);

    dialog->label5 = xfce_create_small_label(_("Fast"));
    gtk_widget_show(dialog->label5);
    gtk_table_attach(GTK_TABLE(dialog->table1), dialog->label5, 2, 3, 3, 4, (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 2);
    gtk_label_set_justify(GTK_LABEL(dialog->label5), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(dialog->label5), 0, 0.5);

    dialog->label6 = gtk_label_new(_("Delay :"));
    gtk_widget_show(dialog->label6);
    gtk_table_attach(GTK_TABLE(dialog->table1), dialog->label6, 0, 3, 0, 1, (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 2);
    gtk_label_set_justify(GTK_LABEL(dialog->label6), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(dialog->label6), 0, 0.5);

    dialog->label7 = gtk_label_new(_("Speed :"));
    gtk_widget_show(dialog->label7);
    gtk_table_attach(GTK_TABLE(dialog->table1), dialog->label7, 0, 3, 2, 3, (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 2);
    gtk_label_set_justify(GTK_LABEL(dialog->label7), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(dialog->label7), 0, 0.5);

    dialog->hscale1 = gtk_hscale_new(GTK_ADJUSTMENT(gtk_adjustment_new(repeat_rate, 10, 500, 10, 10, 0)));
    gtk_widget_show(dialog->hscale1);
    gtk_table_attach(GTK_TABLE(dialog->table1), dialog->hscale1, 1, 2, 3, 4, (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (GTK_FILL), 0, 2);
    gtk_scale_set_draw_value(GTK_SCALE(dialog->hscale1), FALSE);
    gtk_range_set_update_policy(GTK_RANGE(dialog->hscale1), GTK_UPDATE_DISCONTINUOUS);
    gtk_widget_set_sensitive(dialog->hscale1, repeat_key);

    dialog->hscale2 = gtk_hscale_new(GTK_ADJUSTMENT(gtk_adjustment_new(repeat_delay, 100, 2000, 10, 100, 0)));
    gtk_widget_show(dialog->hscale2);
    gtk_table_attach(GTK_TABLE(dialog->table1), dialog->hscale2, 1, 2, 1, 2, (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (GTK_FILL), 0, 2);
    gtk_scale_set_draw_value(GTK_SCALE(dialog->hscale2), FALSE);
    gtk_range_set_update_policy(GTK_RANGE(dialog->hscale2), GTK_UPDATE_DISCONTINUOUS);
    gtk_widget_set_sensitive(dialog->hscale2, repeat_key);

    dialog->frame3 = xfce_framebox_new (_("Cursor"), TRUE);
    gtk_frame_set_shadow_type(GTK_FRAME(dialog->frame3), GTK_SHADOW_NONE);
    gtk_widget_show(dialog->frame3);
    gtk_box_pack_start(GTK_BOX(dialog->vbox3), dialog->frame3, TRUE, TRUE, 0);

    dialog->vbox5 = gtk_vbox_new(FALSE, 5);
    gtk_widget_show(dialog->vbox5);
    xfce_framebox_add (XFCE_FRAMEBOX (dialog->frame3), dialog->vbox5);

    dialog->checkblink = gtk_check_button_new_with_mnemonic(_("Show blinking"));
    gtk_widget_show(dialog->checkblink);
    gtk_box_pack_start(GTK_BOX(dialog->vbox5), dialog->checkblink, FALSE, FALSE, 0);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->checkblink), cursor_blink);

    dialog->table2 = gtk_table_new(2, 3, FALSE);
    gtk_widget_show(dialog->table2);
    gtk_box_pack_start(GTK_BOX(dialog->vbox5), dialog->table2, TRUE, TRUE, 0);

    dialog->label9 = gtk_label_new(_("Speed :"));
    gtk_widget_show(dialog->label9);
    gtk_table_attach(GTK_TABLE(dialog->table2), dialog->label9, 0, 3, 0, 1, (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 2);
    gtk_label_set_justify(GTK_LABEL(dialog->label9), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(dialog->label9), 0, 0.5);

    dialog->label10 = xfce_create_small_label(_("Slow"));
    gtk_widget_show(dialog->label10);
    gtk_table_attach(GTK_TABLE(dialog->table2), dialog->label10, 0, 1, 1, 2, (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 2);
    gtk_label_set_justify(GTK_LABEL(dialog->label10), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(dialog->label10), 1, 0.5);

    dialog->label11 = xfce_create_small_label(_("Fast"));
    gtk_widget_show(dialog->label11);
    gtk_table_attach(GTK_TABLE(dialog->table2), dialog->label11, 2, 3, 1, 2, (GtkAttachOptions) (GTK_FILL), (GtkAttachOptions) (0), 0, 2);
    gtk_label_set_justify(GTK_LABEL(dialog->label11), GTK_JUSTIFY_LEFT);
    gtk_misc_set_alignment(GTK_MISC(dialog->label11), 0, 0.5);

    dialog->hscale3 = gtk_hscale_new(GTK_ADJUSTMENT(gtk_adjustment_new(CLAMP(2600 - cursor_blink_time, 100, 2500), 100, 2500, 200, 0, 0)));
    gtk_widget_show(dialog->hscale3);
    gtk_table_attach(GTK_TABLE(dialog->table2), dialog->hscale3, 1, 2, 1, 2, (GtkAttachOptions) (GTK_EXPAND | GTK_FILL), (GtkAttachOptions) (GTK_FILL), 0, 2);
    gtk_scale_set_draw_value(GTK_SCALE(dialog->hscale3), FALSE);
    gtk_range_set_update_policy(GTK_RANGE(dialog->hscale3), GTK_UPDATE_DISCONTINUOUS);
    gtk_widget_set_sensitive(dialog->hscale3, cursor_blink);

    dialog->hbuttonbox1 = GTK_DIALOG(dialog->keyboard_dialog)->action_area;
    gtk_widget_show(dialog->hbuttonbox1);
    gtk_button_box_set_layout(GTK_BUTTON_BOX(dialog->hbuttonbox1), GTK_BUTTONBOX_END);

    dialog->helpbutton1 = gtk_button_new_from_stock("gtk-help");
    /* gtk_widget_show (dialog->helpbutton1); */
    gtk_dialog_add_action_widget(GTK_DIALOG(dialog->keyboard_dialog), dialog->helpbutton1, GTK_RESPONSE_HELP);
    GTK_WIDGET_SET_FLAGS(dialog->helpbutton1, GTK_CAN_DEFAULT);

    dialog->closebutton1 = gtk_button_new_from_stock("gtk-close");
    gtk_widget_show(dialog->closebutton1);
    gtk_dialog_add_action_widget(GTK_DIALOG(dialog->keyboard_dialog), dialog->closebutton1, GTK_RESPONSE_CLOSE);
    GTK_WIDGET_SET_FLAGS(dialog->closebutton1, GTK_CAN_DEFAULT);

    gtk_widget_grab_focus(dialog->closebutton1);
    gtk_widget_grab_default(dialog->closebutton1);

    return dialog;
}

static void setup_dialog(Itf *itf)
{
    GtkTreeModel *model;
    GtkTreeSelection *selection;

    gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(itf->theme_treeview), -1, NULL, gtk_cell_renderer_text_new(), "text", THEME_NAME_COLUMN, NULL);
    model = (GtkTreeModel *) gtk_list_store_new(N_COLUMNS, G_TYPE_STRING);
    gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(model), 0, sort_func, NULL, NULL);
    gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(model), 0, GTK_SORT_ASCENDING);
    gtk_tree_view_set_model(GTK_TREE_VIEW(itf->theme_treeview), model);
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(itf->theme_treeview));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_BROWSE);
    g_signal_connect(G_OBJECT(selection), "changed", (GCallback) theme_selection_changed, itf->mcs_plugin);

    read_themes(itf);

    g_signal_connect(G_OBJECT(itf->keyboard_dialog), "response", G_CALLBACK(cb_dialog_response), itf->mcs_plugin);
    g_signal_connect(G_OBJECT(itf->checkrepeat), "toggled", G_CALLBACK(cb_checkrepeat_changed), itf);
    g_signal_connect(G_OBJECT(itf->hscale1), "value_changed", (GCallback) cb_repeatdelay_changed, itf);
    g_signal_connect(G_OBJECT(itf->hscale2), "value_changed", (GCallback) cb_repeatdelay_changed, itf);
    g_signal_connect(G_OBJECT(itf->checkblink), "toggled", G_CALLBACK(cb_checkblink_changed), itf);
    g_signal_connect(G_OBJECT(itf->hscale3), "value_changed", (GCallback) cb_blinktime_changed, itf);

    xfce_gtk_window_center_on_monitor_with_pointer (GTK_WINDOW (itf->keyboard_dialog));
    gtk_widget_show(itf->keyboard_dialog);
}

McsPluginInitResult
mcs_plugin_init(McsPlugin *mcs_plugin)
{
    /* This is required for UTF-8 at least - Please don't remove it */
    xfce_textdomain(GETTEXT_PACKAGE, LOCALEDIR, "UTF-8");

    create_channel(mcs_plugin);
    mcs_plugin->plugin_name = g_strdup(PLUGIN_NAME);
    mcs_plugin->caption = g_strdup(_("Keyboard"));
    mcs_plugin->run_dialog = run_dialog;
    mcs_plugin->icon = xfce_themed_icon_load ("xfce4-keyboard", 48);
    mcs_manager_notify(mcs_plugin->manager, CHANNEL1);

    return MCS_PLUGIN_INIT_OK;
}

static void
create_channel(McsPlugin *mcs_plugin)
{
    McsSetting *setting;
    gchar *rcfile, *path;
#ifdef USE_XF86MISC
    int major, minor;
#endif
#ifdef USE_XKB
    int xkbmajor = XkbMajorVersion, xkbminor = XkbMinorVersion;
    int xkbopcode, xkbevent, xkberror;
#endif
    
    path = g_build_filename ("xfce4", RCDIR, RCFILE1, NULL);
    rcfile = xfce_resource_lookup (XFCE_RESOURCE_CONFIG, path);

    if (!rcfile)
    {
        rcfile = xfce_get_userfile(OLDRCDIR, RCFILE1, NULL);
    }

    if (g_file_test (rcfile, G_FILE_TEST_EXISTS))
    {
        mcs_manager_add_channel_from_file(mcs_plugin->manager, CHANNEL1, 
                                          rcfile);
    }
    else
    {
        mcs_manager_add_channel(mcs_plugin->manager, CHANNEL1);
    }
    
    g_free(path);
    g_free(rcfile);

    path = g_build_filename ("xfce4", RCDIR, RCFILE2, NULL);
    rcfile = xfce_resource_lookup (XFCE_RESOURCE_CONFIG, path);

    if (!rcfile)
    {
        rcfile = xfce_get_userfile(OLDRCDIR, RCFILE2, NULL);
    }

    if (g_file_test (rcfile, G_FILE_TEST_EXISTS))
    {
        mcs_manager_add_channel_from_file(mcs_plugin->manager, CHANNEL2, 
                                          rcfile);
    }
    else
    {
        mcs_manager_add_channel(mcs_plugin->manager, CHANNEL2);
    }
    
    g_free(path);
    g_free(rcfile);


    setting = mcs_manager_setting_lookup(mcs_plugin->manager,
		    "Gtk/KeyThemeName", CHANNEL1);
    if (setting) {
        if(current_key_theme)
            g_free(current_key_theme);

        current_key_theme = g_strdup(setting->data.v_string);
    }
    else {
        if(current_key_theme)
            g_free(current_key_theme);

        current_key_theme = g_strdup(DEFAULT_THEME);

	mcs_manager_set_string(mcs_plugin->manager, "Gtk/KeyThemeName",
			CHANNEL1, current_key_theme);
    }

    setting = mcs_manager_setting_lookup(mcs_plugin->manager,
		    "Net/CursorBlink", CHANNEL1);
    
    if (setting)
        cursor_blink = (setting->data.v_int ? TRUE : FALSE);
    else {
        cursor_blink = TRUE;
        mcs_manager_set_int(mcs_plugin->manager, "Net/CursorBlink",
			CHANNEL1, cursor_blink ? 1 : 0);
    }

    setting = mcs_manager_setting_lookup(mcs_plugin->manager,
		    "Net/CursorBlinkTime", CHANNEL1);
    
    if (setting)
        cursor_blink_time = setting->data.v_int;
    else {
        cursor_blink_time = 500;
        mcs_manager_set_int(mcs_plugin->manager, "Net/CursorBlinkTime",
			CHANNEL1, cursor_blink_time);
    }

    setting = mcs_manager_setting_lookup(mcs_plugin->manager, "Key/RepeatKey",
		    CHANNEL2);

    if (setting)
        repeat_key = (setting->data.v_int ? TRUE : FALSE);
    else {
        repeat_key = TRUE;
        mcs_manager_set_int(mcs_plugin->manager, "Key/RepeatKey", CHANNEL2,
			repeat_key ? 1 : 0);
    }

    set_repeat(ALL, repeat_key ? ON : OFF);

    setting = mcs_manager_setting_lookup(mcs_plugin->manager, "Key/RepeatDelay",
		    CHANNEL2);

    if (setting)
        repeat_delay = setting->data.v_int;
    else {
        repeat_delay = 500;
        mcs_manager_set_int(mcs_plugin->manager, "Key/RepeatDelay", CHANNEL2,
			repeat_delay);
    }

    setting = mcs_manager_setting_lookup(mcs_plugin->manager, "Key/RepeatRate",
		    CHANNEL2);
    if (setting)
        repeat_rate = setting->data.v_int;
    else {
        repeat_rate = 30;
        mcs_manager_set_int(mcs_plugin->manager, "Key/RepeatRate", CHANNEL2,
			repeat_rate);
    }

#ifdef USE_XF86MISC
#ifdef DEBUG
    g_message("Querying XF86Misc extension");
#endif
    if(XF86MiscQueryVersion(GDK_DISPLAY(), &major, &minor)) {
#ifdef DEBUG
        g_message("XF86Misc extension found");
#endif
        miscpresent = TRUE;
        set_repeat_rate(repeat_delay, repeat_rate);
    }
    else {
#ifdef DEBUG
        g_warning("Your X server does not support XF86Misc extension");
#endif
        miscpresent = FALSE;
    }
#else
#ifdef DEBUG
    g_message("This build doesn't include support for XF86Misc extension");
#endif
#endif

#ifdef USE_XKB
#ifdef DEBUG
    g_message("Querying Xkb extension");
#endif
    if(XkbQueryExtension(GDK_DISPLAY(), &xkbopcode, &xkbevent, &xkberror,
			    &xkbmajor, &xkbminor)) {
#ifdef DEBUG
        g_message("Xkb extension found");
#endif
        xkbpresent = TRUE;
        xkb_set_repeat_rate(repeat_delay, 1000 / repeat_rate);
    }
    else {
#ifdef DEBUG
        g_message("Your X server does not support Xkb extension");
#endif
        xkbpresent = FALSE;
    }
#else
#ifdef DEBUG
    g_warning("This build doesn't include support for Xkb extension");
#endif
#endif
    
    write_options (mcs_plugin);
}

static gboolean
write_options(McsPlugin *mcs_plugin)
{
    gboolean result;
    gchar *rcfile, *path;

    path = g_build_filename ("xfce4", RCDIR, RCFILE1, NULL);
    rcfile = xfce_resource_save_location (XFCE_RESOURCE_CONFIG, path, TRUE);
    result = mcs_manager_save_channel_to_file(mcs_plugin->manager, CHANNEL1,
		    rcfile);
    g_free(path);
    g_free(rcfile);

    path = g_build_filename ("xfce4", RCDIR, RCFILE2, NULL);
    rcfile = xfce_resource_save_location (XFCE_RESOURCE_CONFIG, path, TRUE);
    result &= mcs_manager_save_channel_to_file(mcs_plugin->manager, CHANNEL2,
		    rcfile);
    g_free(path);
    g_free(rcfile);

    return(result);
}

static void
run_dialog(McsPlugin *mcs_plugin)
{
    Itf *dialog;

    if (!is_running) {
	is_running = TRUE;

    xfce_textdomain (GETTEXT_PACKAGE, LOCALEDIR, "UTF-8");

	dialog = create_keyboard_dialog(mcs_plugin);
    	setup_dialog(dialog);
    }
}

/* macro defined in manager-plugin.h */
MCS_PLUGIN_CHECK_INIT
