#!/bin/sh

#set -x

fatal_error()
{
    echo "$1" 1>&2
    exit 1
}

debug_echo()
{
    echo "$1" 1>&2
}

verbose_echo()
{
    echo "$1" 1>&2
}

which_tool()
{
    echo "${PATH}" | tr ":" "\n" | while read i ;
    do ls -1ad "${i}/${1}" 2>/dev/null ; done | sort | uniq
}

which_program()
{
    echo "${1}" | tr ":" "\n" | while read i ;
    do 
        which_tool "${i}"
    done | sort | uniq
}

which_xsl()
{
    ls -1ad ${1}/docbook-xsl-stylesheets*/ 2>/dev/null | head -1
}

# Fix HTML generated by the DocBook XSL stylesheets
# In many cases <nsxyz:p> is used instead of <p>, screwing-up
# display (this stuff only works for XHTML)
fix_docbook_html()
{
  sed "s/<ns[0-9]*:p>/<p>/g;s/<\/ns[0-9]*:p>/<\/p>/g"
}

# main
infile="${1}"
outputformat="${2}"
outputfile="${3}"

# xsl processing
case "`uname -s`" in
    FreeBSD)
        PATH="${PATH}:/usr/local/bin:/usr/local/sbin"
        export PATH
        stylesheetbase="/usr/local/share/xsl/"
        ;;
    *)
        stylesheetbase="/usr/share/sgml/docbook/"
        ;;
esac
xsltproc="`which_tool xsltproc`"
stylesheetdir="`which_xsl ${stylesheetbase}`"
docbook2man="`which_tool docbook2man`"
verbose_echo "# Using xsltproc=${xsltproc}."
verbose_echo "# Using stylesheetdir=${stylesheetdir}."
verbose_echo "# Using docbook2man=${docbook2man}."

# Prechecks
[ "${infile}" = "" ]         && fatal_error "$0: No input file."
[ "${outputfile}" = "" ]     && fatal_error "$0: No output file."
[ "${outputformat}" = "" ]   && fatal_error "$0: No format given."
[ ! -r "${infile}" ]         && fatal_error "$0: Input file not found or readable."
[ ! -x "${xsltproc}" ]       && fatal_error "$0: No xsltproc found."
[ ! -r "${stylesheetdir}" ]  && fatal_error "$0: No DocBook/XSL style sheets found."
[ ! -x "${docbook2man}" ]    && fatal_error "$0: No docbook2man found."


# this is hack style to work around the problem that "docbook2man"
# writes lots of files into the current dir
MYTMPDIR="/tmp/docbookconv_${RANDOM}"
MYCURRDIR="${PWD}"

(
  mkdir "${MYTMPDIR}"
  cd "${MYTMPDIR}"

  case "${outputformat}" in
    "html")
        cp "${MYCURRDIR}/${infile}" "${infile}.tmp"
        if [ "${infile}" != "${infile%.sgml}" ] ; then
          verbose_echo "# processing as SGML document"
          ${xsltproc} --docbook ${stylesheetdir}/html/docbook.xsl "${infile}.tmp" | fix_docbook_html >"${MYCURRDIR}/${outputfile}"
        else
          verbose_echo "# processing as XML document"
          ${xsltproc}           ${stylesheetdir}/html/docbook.xsl "${infile}.tmp" | fix_docbook_html >"${MYCURRDIR}/${outputfile}"
        fi
        ;;
    "man")
        if [ "${infile}" != "${infile%.sgml}" ] ; then
          #cp "${MYCURRDIR}/${infile}" "${infile%.sgml}.xml"
          #${docbook2man} --network "${infile%.sgml}.xml"
          cp "${MYCURRDIR}/${infile}" "${infile}.tmp"
          ${docbook2man} --network "${infile}.tmp"
        else
          cp "${MYCURRDIR}/${infile}" "${infile}.tmp"
          ${docbook2man} --network "${infile}.tmp"
        fi
        manfile="$(ls -1 ${infile%.*}.__*)"

        verbose_echo "manfile=${manfile}"
        [ ! -r "${manfile}" ] && fatal_error "$0: manfile not found."
        cp "${manfile}" "${MYCURRDIR}/${outputfile}"
        ;;
    *)
        fatal_error "Unsupported output format ${outputformat}."
        ;;
  esac
)

# EOF.

