/**************************************************************************
 *
 * Copyright 2003 Tungsten Graphics, Inc., Cedar Park, Texas.
 * All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sub license, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT.
 * IN NO EVENT SHALL TUNGSTEN GRAPHICS AND/OR ITS SUPPLIERS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 **************************************************************************/

#include "i915_batch.h"
#include "i915_state_inlines.h"
#include "i915_context.h"
#include "i915_reg.h"
#include "i915_state.h"

#include "util/u_memory.h"
#include "util/u_pack_color.h"


/* State that we have chosen to store in the DYNAMIC segment of the
 * i915 indirect state mechanism.
 *
 * Can't cache these in the way we do the static state, as there is no
 * start/size in the command packet, instead an 'end' value that gets
 * incremented.
 *
 * Additionally, there seems to be a requirement to re-issue the full
 * (active) state every time a 4kb boundary is crossed.
 */

static INLINE void set_dynamic_indirect(struct i915_context *i915,
                                        unsigned offset,
                                        const unsigned *src,
                                        unsigned dwords)
{
   unsigned i;

   if (!memcmp(src, &i915->current.dynamic[offset], dwords * 4))
      return;

   for (i = 0; i < dwords; i++)
      i915->current.dynamic[offset + i] = src[i];

   i915->hardware_dirty |= I915_HW_DYNAMIC;
}



/***********************************************************************
 * Modes4: stencil masks and logicop
 */
static void upload_MODES4(struct i915_context *i915)
{
   unsigned modes4 = 0;

   /* I915_NEW_STENCIL
    */
   modes4 |= i915->depth_stencil->stencil_modes4;

   /* I915_NEW_BLEND
     */
   modes4 |= i915->blend->modes4;

   /* Always, so that we know when state is in-active:
    */
   set_dynamic_indirect(i915,
                        I915_DYNAMIC_MODES4,
                        &modes4,
                        1);
}

const struct i915_tracked_state i915_upload_MODES4 = {
   "MODES4",
   upload_MODES4,
   I915_NEW_BLEND | I915_NEW_DEPTH_STENCIL
};



/***********************************************************************
 */
static void upload_BFO(struct i915_context *i915)
{
   unsigned bfo[2];
   bfo[0] = i915->depth_stencil->bfo[0];
   bfo[1] = i915->depth_stencil->bfo[1];
   /* I don't get it only allowed to set a ref mask when the enable bit is set? */
   if (bfo[0] & BFO_ENABLE_STENCIL_REF) {
      bfo[0] |= i915->stencil_ref.ref_value[1] << BFO_STENCIL_REF_SHIFT;
   }

   set_dynamic_indirect(i915,
                        I915_DYNAMIC_BFO_0,
                        &(bfo[0]),
                        2);
}

const struct i915_tracked_state i915_upload_BFO = {
   "BFO",
   upload_BFO,
   I915_NEW_DEPTH_STENCIL
};



/***********************************************************************
 */
static void upload_BLENDCOLOR(struct i915_context *i915)
{
   unsigned bc[2];

   memset(bc, 0, sizeof(bc));

   /* I915_NEW_BLEND
    */
   {
      const float *color = i915->blend_color.color;

      bc[0] = _3DSTATE_CONST_BLEND_COLOR_CMD;
      bc[1] = pack_ui32_float4(color[0],
                               color[1],
                               color[2],
                               color[3]);
   }

   set_dynamic_indirect(i915,
                        I915_DYNAMIC_BC_0,
                        bc,
                        2);
}

const struct i915_tracked_state i915_upload_BLENDCOLOR = {
   "BLENDCOLOR",
   upload_BLENDCOLOR,
   I915_NEW_BLEND
};



/***********************************************************************
 */
static void upload_IAB(struct i915_context *i915)
{
   unsigned iab = i915->blend->iab;

   set_dynamic_indirect(i915,
                        I915_DYNAMIC_IAB,
                        &iab,
                        1);
}

const struct i915_tracked_state i915_upload_IAB = {
   "IAB",
   upload_IAB,
   I915_NEW_BLEND
};



/***********************************************************************
 */
static void upload_DEPTHSCALE(struct i915_context *i915)
{
   set_dynamic_indirect(i915,
                        I915_DYNAMIC_DEPTHSCALE_0,
                        &(i915->rasterizer->ds[0].u),
                        2);
}

const struct i915_tracked_state i915_upload_DEPTHSCALE = {
   "DEPTHSCALE",
   upload_DEPTHSCALE,
   I915_NEW_RASTERIZER
};



/***********************************************************************
 * Polygon stipple
 *
 * The i915 supports a 4x4 stipple natively, GL wants 32x32.
 * Fortunately stipple is usually a repeating pattern.
 *
 * XXX: does stipple pattern need to be adjusted according to
 * the window position?
 *
 * XXX: possibly need workaround for conform paths test.
 */
static void upload_STIPPLE(struct i915_context *i915)
{
   unsigned st[2];

   st[0] = _3DSTATE_STIPPLE;
   st[1] = 0;

   /* I915_NEW_RASTERIZER
    */
   st[1] |= i915->rasterizer->st;

   /* I915_NEW_STIPPLE
    */
   {
      const ubyte *mask = (const ubyte *)i915->poly_stipple.stipple;
      ubyte p[4];

      p[0] = mask[12] & 0xf;
      p[1] = mask[8] & 0xf;
      p[2] = mask[4] & 0xf;
      p[3] = mask[0] & 0xf;

      /* Not sure what to do about fallbacks, so for now just dont:
       */
      st[1] |= ((p[0] << 0) |
                (p[1] << 4) |
                (p[2] << 8) |
                (p[3] << 12));
   }

   set_dynamic_indirect(i915,
                        I915_DYNAMIC_STP_0,
                        &st[0],
                        2);
}

const struct i915_tracked_state i915_upload_STIPPLE = {
   "STIPPLE",
   upload_STIPPLE,
   I915_NEW_RASTERIZER | I915_NEW_STIPPLE
};



/***********************************************************************
 * Scissor enable
 */
static void upload_SCISSOR_ENABLE( struct i915_context *i915 )
{
   set_dynamic_indirect(i915,
                        I915_DYNAMIC_SC_ENA_0,
                        &(i915->rasterizer->sc[0]),
                        1);
}

const struct i915_tracked_state i915_upload_SCISSOR_ENABLE = {
   "SCISSOR ENABLE",
   upload_SCISSOR_ENABLE,
   I915_NEW_RASTERIZER
};



/***********************************************************************
 * Scissor rect
 */
static void upload_SCISSOR_RECT(struct i915_context *i915)
{
   unsigned x1 = i915->scissor.minx;
   unsigned y1 = i915->scissor.miny;
   unsigned x2 = i915->scissor.maxx;
   unsigned y2 = i915->scissor.maxy;
   unsigned sc[3];

   sc[0] = _3DSTATE_SCISSOR_RECT_0_CMD;
   sc[1] = (y1 << 16) | (x1 & 0xffff);
   sc[2] = (y2 << 16) | (x2 & 0xffff);

   set_dynamic_indirect(i915,
                        I915_DYNAMIC_SC_RECT_0,
                        &sc[0],
                        3);
}

const struct i915_tracked_state i915_upload_SCISSOR_RECT = {
   "SCISSOR RECT",
   upload_SCISSOR_RECT,
   I915_NEW_SCISSOR
};



/***********************************************************************
 */
static const struct i915_tracked_state *atoms[] = {
   &i915_upload_MODES4,
   &i915_upload_BFO,
   &i915_upload_BLENDCOLOR,
   &i915_upload_IAB,
   &i915_upload_DEPTHSCALE,
   &i915_upload_STIPPLE,
   &i915_upload_SCISSOR_ENABLE,
   &i915_upload_SCISSOR_RECT
};

/* These will be dynamic indirect state commands, but for now just end
 * up on the batch buffer with everything else.
 */
static void update_dynamic(struct i915_context *i915)
{
   int i;

   for (i = 0; i < Elements(atoms); i++)
      if (i915->dirty & atoms[i]->dirty)
         atoms[i]->update(i915);
}

struct i915_tracked_state i915_hw_dynamic = {
   "dynamic",
   update_dynamic,
   ~0 /* all state atoms, becuase we do internal checking */
};
