/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 *
 * Copyright (C) 2003 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"

#ifdef HAVE_STDINT_H
#  include <stdint.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>

#include "types.h"
#include "distcc.h"
#include "rpc.h"
#include "io.h"
#include "trace.h"
#include "exitcode.h"
#include "state.h"
#include "tempfile.h"
#include "filename.h"
#include "snprintf.h"
#include "mon.h"


/**
 * @file
 *
 * Plain text monitor program.  Just prints out the state once, or
 * repeatedly, kind of like Linux vmstat.
 */


const char *rs_program_name = "distccmon-text";


static void usage(void)
{
    fprintf(stderr, "usage: distccmon-text [DELAY]\n"
"\n"
"Displays current compilation jobs in text form.\n"
"\n"
"If delay is specified, repeatedly updates after that many (fractional)\n"
"seconds.  Otherwise, runs just once.\n");
}

int main(int argc, char *argv[])
{
    struct dcc_mon_list *list;
    int ret;
    float delay;
    char *end;
    
    dcc_set_trace_from_env();

    if (argc == 1)
        delay = 0.0;
    else if (argc == 2) {
        delay = strtod(argv[1], &end);
        if (*end) {
            usage();
            return 1;
        }
    } else {
        usage();
        return 1;
    }

    /* We might be writing to e.g. a pipe that's being read by some
     * other program, so make sure we're always line buffered. */
    setvbuf (stdout, NULL, _IOLBF, BUFSIZ);

    do {
        if ((ret = dcc_mon_poll(&list)))
            return ret;

        for (; list; list = list->next) {
            /* Assume 80 cols = */
            printf("%6d  %-10.10s  %-30.30s %28.28s\n",
                   list->cpid, list->state, list->file, list->host);
        }

        printf("\n");

        /* XXX: usleep() is probably not very portable */
        usleep(delay * 1000000);
    } while (delay);
    
    return 0;
}
