# Performs generic (non-project specific) validation of Trace File Contents

include(${CMAKE_CURRENT_LIST_DIR}/json.cmake)
include(${CMAKE_CURRENT_LIST_DIR}/verify-snippet.cmake)

function(trace_entry_has_fields trace entry)
  json_has_key("${trace}" "${entry}" cat)
  json_has_key("${trace}" "${entry}" dur)
  json_has_key("${trace}" "${entry}" name)
  json_has_key("${trace}" "${entry}" ph)
  json_has_key("${trace}" "${entry}" pid)
  json_has_key("${trace}" "${entry}" tid)
  json_has_key("${trace}" "${entry}" ts)
  json_has_key("${trace}" "${entry}" args)

  return(PROPAGATE RunCMake_TEST_FAILED ERROR_MESSAGE)
endfunction()

function(trace_valid_entry trace entry)
  string(JSON ph GET "${entry}" ph)
  if (NOT ph STREQUAL "X")
    json_error("${trace}"
      "Invalid event \'${ph}\' (only complete events \'X\' expected)")
  endif()
  string(JSON start GET "${entry}" ts)
  if (start LESS 0)
    json_error("${trace}" "Negative time start: ${start}")
  endif()
  string(JSON duration GET "${entry}" dur)
  if (duration LESS 0)
    json_error("${trace}" "Negative duration: ${duration}")
  endif()
  string(JSON pid GET "${entry}" pid)
  if (NOT pid EQUAL 0)
    json_error("${trace}" "Invalid PID: ${pid}")
  endif()
  string(JSON tid GET "${entry}" tid)
  if (tid LESS 0)
    json_error("${trace}" "Invalid TID: ${tid}")
  endif()

  # Validate "args" as snippet data
  string(JSON args GET "${entry}" args)
  verify_snippet_data("${trace}" "${args}")

  # Check the formation of the "name" based on the snippet data
  string(JSON name GET "${entry}" name)
  if (NOT name)
    json_error("${trace}" "Name is empty: ${entry}")
  endif()
  string(JSON cat GET "${entry}" cat)
  set(expected_name "${cat}")
  if (cat STREQUAL "compile")
    string(JSON source GET "${args}" source)
    string(APPEND expected_name ": ${source}")
  elseif (cat STREQUAL "link")
    string(JSON target GET "${args}" target)
    string(APPEND expected_name ": ${target}")
  elseif (cat STREQUAL "install")
    string(JSON workingDir GET "${args}" workingDir)
    cmake_path(GET workingDir FILENAME lastDirName)
    string(APPEND expected_name ": ${lastDirName}")
  elseif (cat STREQUAL "custom")
    string(JSON command GET "${args}" command)
    string(APPEND expected_name ": ${command}")
  elseif (cat STREQUAL "test")
    string(JSON testName GET "${args}" testName)
    string(APPEND expected_name ": ${testName}")
  endif()
  if (NOT name STREQUAL expected_name)
    json_error("${trace}" "Invalid name: ${name}")
  endif()

  return(PROPAGATE ERROR_MESSAGE RunCMake_TEST_FAILED)
endfunction()

function(verify_trace_entry trace entry)
  trace_entry_has_fields("${trace}" "${entry}")
  trace_valid_entry("${trace}" "${entry}")
  return(PROPAGATE ERROR_MESSAGE RunCMake_TEST_FAILED)
endfunction()

function(verify_trace_file_name index_file trace_file)
  cmake_path(GET trace_file FILENAME trace_filename)
  cmake_path(GET index_file FILENAME index_filename)

  set(timestamp_regex "^(index|trace)-([A-Z0-9\\-]+)\\.json")
  if ("${trace_filename}" MATCHES "${timestamp_regex}")
    set(trace_timestamp "${CMAKE_MATCH_2}")
  else()
    add_error("Unable to parse timestamp from trace file name: \'${trace_filename}\'")
  endif()
  if ("${index_filename}" MATCHES "${timestamp_regex}")
    set(index_timestamp "${CMAKE_MATCH_2}")
  else()
    add_error("Unable to parse timestamp from index file name: \'${index_filename}\'")
  endif()

  if (NOT "${trace_timestamp}" STREQUAL "${index_timestamp}")
    add_error("Trace file timestamp \'${trace_filename}\' does not match the index \'${index_file}\'")
  endif()

  return(PROPAGATE ERROR_MESSAGE RunCMake_TEST_FAILED)
endfunction()
