/* Copyright (C) 2002, 2003 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License version 2 as
   published by the Free Software Foundation.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#define _GNU_SOURCE

#include <pwd.h>
#include <ctype.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <syslog.h>
#include <signal.h>
#include <unistd.h>
#include <getopt.h>
#include <shadow.h>
#include <locale.h>
#include <libintl.h>
#include <sys/stat.h>
#include <sys/resource.h>
#include <security/pam_appl.h>
#include <security/pam_misc.h>

#include "error_codes.h"
#include "getdef.h"
#include "public.h"
#include "main.h"
#include "read-files.h"

#ifndef _
#define _(String) gettext (String)
#endif

#define DAY (24L*3600L)
#define SCALE DAY

/* Print the version information.  */
void
print_version (const char *program)
{
  fprintf (stdout, "%s (%s) %s\n", program, PACKAGE, VERSION);
  fprintf (stdout, gettext ("\
Copyright (C) %s Thorsten Kukuk.\n\
This is free software; see the source for copying conditions.  There is NO\n\
warranty; not even for MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.\n\
"), "2002");
  /* fprintf (stdout, _("Written by %s.\n"), "Thorsten Kukuk"); */
}

static void
print_usage (FILE *stream, const char *program)
{
  fprintf (stream, _("Usage: %s [-m mindays][-M maxdays][-d lastday][-I inactive][-E expiredate][-W warndays] user\n"),
	   program);
  fprintf (stream, _("       %s -l user\n"),
	   program);
}

static void
print_help (const char *program)
{
  print_usage (stdout, program);
  fprintf (stdout, _("%s - change user password expiry information\n\n"),
	   program);
  fputs (_("  -r service     Use nameservice 'service'\n"), stdout);
  fputs (_("  -q, --quiet    Don't be verbose\n"), stdout);
  fputs (_("      --help     Give this help list\n"), stdout);
  fputs (_("  -u, --usage    Give a short usage message\n"), stdout);
  fputs (_("  -v, --version  Print program version\n"), stdout);
  fputs (_("Valid services for -r are: files, nis, nisplus, ldap\n"), stdout);
}

void
print_error (const char *program)
{
  fprintf (stderr,
           _("Try `%s --help' or `%s --usage' for more information.\n"),
           program, program);
}

/* Print the time in a human readable format.  */
static void
print_date (time_t date)
{
#ifdef HAVE_STRFTIME
  struct tm *tp;
  char buf[80];

  tp = gmtime (&date);
  strftime (buf, sizeof buf, "%b %d, %Y", tp);
  puts (buf);
#else
  struct tm *tp;
  char *cp;

  tp = gmtime (&date);
  cp = asctime (tp);
  printf ("%6.6s, %4.4s\n", cp + 4, cp + 20);
#endif
}

/* convert time_t into a readable date string.  */
static char *
date2str (time_t date)
{
  struct tm *tp;
  char buf[12];

  tp = gmtime (&date);
#ifdef HAVE_STRFTIME
  strftime (buf, sizeof (buf), "%Y-%m-%d", tp);
#else
  snprintf (buf, sizeof (buf), "%04d-%02d-%02d",
	    tp->tm_year + 1900, tp->tm_mon + 1, tp->tm_mday);
#endif
  return strdup (buf);
}

/* convert a string to a time_t value.  */
static long int
str2date (const char *str)
{
  struct tm tp;
  char *cp;
  time_t result;

  memset (&tp, 0, sizeof tp);
  cp = strptime (str, "%Y-%m-%d", &tp);
  if (!cp || *cp != '\0')
    return -1;

  result = mktime (&tp);
  if (result == (time_t) -1)
    return -1;

  return (result + (DAY/2)) / DAY;
}

/* Print the current values of the expiration fields.  */
static void
print_shadow_info (user_t *data)
{
  printf (_("Minimum:\t%ld\n"), data->sp.sp_min);
  printf (_("Maximum:\t%ld\n"), data->sp.sp_max);
  printf (_("Warning:\t%ld\n"), data->sp.sp_warn);
  printf (_("Inactive:\t%ld\n"), data->sp.sp_inact);
  printf (_("Last Change:\t\t"));
  if (data->sp.sp_lstchg == 0)
    printf (_("Unknown, password is forced to change at next login\n"));
  else if (data->sp.sp_lstchg < 0)
    printf (_("Never\n"));
  else
    print_date (data->sp.sp_lstchg * SCALE);
  printf (_("Password Expires:\t"));
  if (data->sp.sp_lstchg <= 0 || data->sp.sp_max >= 10000 * (DAY / SCALE)
      || data->sp.sp_max <= 0)
    printf (_("Never\n"));
  else
    print_date (data->sp.sp_lstchg * SCALE + data->sp.sp_max * SCALE);
  printf (_("Password Inactive:\t"));
  if (data->sp.sp_lstchg <= 0 || data->sp.sp_inact <= 0 ||
      data->sp.sp_max >= 10000 * (DAY / SCALE) || data->sp.sp_max <= 0)
    printf (_("Never\n"));
  else
    print_date (data->sp.sp_lstchg * SCALE +
		(data->sp.sp_max + data->sp.sp_inact) * SCALE);
  printf (_("Account Expires:\t"));
  if (data->sp.sp_expire <= 0)
    printf (_("Never\n"));
  else
    print_date (data->sp.sp_expire * SCALE);
}

static int
change_shadow_info (user_t *data)
{
  char *buf, *res, *cp;

  asprintf (&buf, "%ld", data->sp.sp_min);
  res = get_value (buf, _("Minimum Password Age"));
  free (buf);
  if (res == NULL ||
      ((data->spn.sp_min = strtol (res, &cp, 10)) == 0 && *cp) ||
      data->spn.sp_min < -1)
    {
      if (cp && *cp)
	fprintf (stderr, _("Input is no integer value\n"));
      else
	fprintf (stderr, _("Negative numbers are not allowed as input (except -1)\n"));
      return E_FAILURE;
    }
  free (res);

  asprintf (&buf, "%ld", data->sp.sp_max);
  res = get_value (buf, _("Maximum Password Age"));
  free (buf);
  if (res == NULL ||
      ((data->spn.sp_max = strtol (res, &cp, 10)) == 0 && *cp) ||
      data->spn.sp_max < -1)
    {
      if (cp && *cp)
	fprintf (stderr, _("Input is no integer value\n"));
      else
	fprintf (stderr, _("Negative numbers are not allowed as input (except -1)\n"));
      return E_FAILURE;
    }
  free (res);

  asprintf (&buf, "%ld", data->sp.sp_warn);
  res = get_value (buf, _("Password Expiration Warning"));
  free (buf);
  if (res == NULL ||
      ((data->spn.sp_warn = strtol (res, &cp, 10)) == 0 && *cp) ||
      data->spn.sp_warn < -1)
    {
      if (cp && *cp)
	fprintf (stderr, _("Input is no integer value\n"));
      else
	fprintf (stderr, _("Negative numbers are not allowed as input (except -1)\n"));
      return E_FAILURE;
    }
  free (res);

  asprintf (&buf, "%ld", data->sp.sp_inact);
  res = get_value (buf, _("Password Inactive"));
  free (buf);
  if (res == NULL ||
      ((data->spn.sp_inact = strtol (res, &cp, 10)) == 0 && *cp) ||
      data->spn.sp_inact < -1)
    return E_FAILURE;

  buf = date2str (data->sp.sp_lstchg * SCALE);
  res = get_value (buf, _("Last Password Change (YYYY-MM-DD)"));
  free (buf);
  if (res == NULL)
    return E_FAILURE;
  else if (strcmp (res, "1969-12-31") == 0)
    data->sp.sp_lstchg = -1;
  else
    {
      data->spn.sp_lstchg = str2date (res);
      free (res);
      if (data->spn.sp_lstchg == -1)
	{
	  fprintf (stderr, _("Invalid date\n"));
	  return E_FAILURE;
	}
    }

  buf = date2str (data->sp.sp_expire * SCALE);
  res = get_value (buf, _("Account Expiration Date (YYYY-MM-DD)"));
  free (buf);
  if (res == NULL)
    return E_FAILURE;
  else if (strcmp (res, "1969-12-31") == 0)
    data->spn.sp_expire = -1;
  else
    {
      data->spn.sp_expire = str2date (res);
      free (res);
      if (data->spn.sp_expire == -1)
	{
	  fprintf (stderr, _("Invalid date\n"));
	  return E_FAILURE;
	}
    }
  return 0;
}

static void
init_environment (void)
{
  struct rlimit rlim;

  /* Don't create a core file.  */
  rlim.rlim_cur = rlim.rlim_max = 0;
  setrlimit (RLIMIT_CORE, &rlim);

  /* Set all limits to unlimited to avoid to run in any
     problems later.  */
  rlim.rlim_cur = rlim.rlim_max = RLIM_INFINITY;
  setrlimit (RLIMIT_AS, &rlim);
  setrlimit (RLIMIT_CPU, &rlim);
  setrlimit (RLIMIT_DATA, &rlim);
  setrlimit (RLIMIT_FSIZE, &rlim);
  setrlimit (RLIMIT_NOFILE, &rlim);
  setrlimit (RLIMIT_RSS, &rlim);
  setrlimit (RLIMIT_STACK, &rlim);

  /* Ignore all signals which can make trouble later.  */
  signal (SIGALRM, SIG_IGN);
  signal (SIGXFSZ, SIG_IGN);
  signal (SIGHUP, SIG_IGN);
  signal (SIGINT, SIG_IGN);
  signal (SIGPIPE, SIG_IGN);
  signal (SIGQUIT, SIG_IGN);
  signal (SIGTERM, SIG_IGN);
  signal (SIGTSTP, SIG_IGN);
  signal (SIGTTOU, SIG_IGN);

  umask (077);
}

int
main (int argc, char *argv[])
{
  const char *program = "chage";
  uid_t uid = getuid ();
  user_t *pw_data = NULL;
  char *use_service = NULL;
  char *caller_name = NULL;
  char *mindays = NULL, *maxdays = NULL, *lastday = NULL, *inactive = NULL;
  char *expiredate = NULL, *warndays = NULL;
  char *binddn = NULL;
  int interactive = 1;
  int silent = 0;
  int l_flag = 0;

  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);

  openlog (program, LOG_PID, LOG_AUTHPRIV);

  /* Before going any further, raise the ulimit and ignore
     signals.  */
  init_environment ();

  while (1)
    {
      int c;
      int option_index = 0;
      static struct option long_options[] = {
	{"mindays",    required_argument, NULL, 'm' },
	{"maxdays",    required_argument, NULL, 'M' },
	{"lastday",    required_argument, NULL, 'd' },
	{"inactive",   required_argument, NULL, 'I' },
	{"expiredate", required_argument, NULL, 'E' },
	{"warndays",   required_argument, NULL, 'W' },
	{"list",       no_argument,       NULL, 'l' },
	{"binddn",     required_argument, NULL, 'D' },
	{"quiet",      no_argument,       NULL, 'q' },
	{"path",       required_argument, NULL, 'P' },
	{"service",    required_argument, NULL, 'r' },
	{"version",    no_argument,       NULL, 'v' },
	{"usage",      no_argument,       NULL, 'u' },
	{"help",       no_argument,       NULL, '\255' },
	{NULL,         0,                 NULL, '\0'}
      };

      c = getopt_long (argc, argv, "lm:M:d:D:I:E:W:P:r:vuq",
		       long_options, &option_index);
      if (c == (-1))
        break;
      switch (c)
        {
	case 'm':
	  mindays = optarg;
	  interactive = 0;
	  break;
	case 'M':
	  maxdays = optarg;
	  interactive = 0;
	  break;
	case 'd':
	  lastday = optarg;
	  interactive = 0;
	  break;
	case 'I':
	  inactive = optarg;
	  interactive = 0;
	  break;
	case 'E':
	  expiredate = optarg;
	  interactive = 0;
	  break;
	case 'W':
	  warndays = optarg;
	  interactive = 0;
	  break;
	case 'D':
	  binddn = optarg;
	  break;
	case 'l':
	  l_flag = 1;
	  break;
	case 'P':
	  if (uid != 0)
	    {
	      fprintf (stderr,
		       _("Only root is allowed to specify another path\n"));
	      return E_NOPERM;
	    }
	  else
	    files_etc_dir = strdup (optarg);
	  break;
	case 'r':
	  if (use_service != NULL)
	    {
	      print_usage (stderr, program);
	      return E_BAD_ARG;
	    }

	  if (strcasecmp (optarg, "yp") == 0 ||
	      strcasecmp (optarg, "nis") == 0)
	    use_service = "nis";
	  else if (strcasecmp (optarg, "nis+") == 0 ||
		   strcasecmp (optarg, "nisplus") == 0)
	    use_service = "nisplus";
	  else if (strcasecmp (optarg, "files") == 0)
	    use_service = "files";
	  else if (strcasecmp (optarg, "ldap") == 0)
	    use_service = "ldap";
	  else
	    {
	      fprintf (stderr, _("Service \"%s\" not supported\n"), optarg);
	      print_usage (stderr, program);
	      return E_BAD_ARG;
	    }
	  break;
	case 'q':
	  silent = 1;
	  break;
        case '\255':
          print_help (program);
          return 0;
        case 'v':
          print_version (program);
          return 0;
        case 'u':
          print_usage (stdout, program);
          return 0;
        default:
          print_error (program);
          return E_USAGE;
        }
    }

  argc -= optind;
  argv += optind;

  if (argc > 1)
    {
      fprintf (stderr, _("%s: Too many arguments\n"), program);
      print_error (program);
      return E_USAGE;
    }

  if (l_flag && !interactive)
    {
      fprintf (stderr, _("%s: do not include \"l\" with other flags\n"),
	       program);
      print_usage (stderr, program);
      return E_USAGE;
    }
  else
    {
      int buflen = 256;
      char *buffer = alloca (buflen);
      struct passwd resultbuf;
      struct passwd *pw;
      char *arg_user;

      /* Determine our own user name for PAM authentication.  */
      while (getpwuid_r (uid, &resultbuf, buffer, buflen, &pw) != 0
	     && errno == ERANGE)
	{
	  errno = 0;
	  buflen += 256;
	  buffer = alloca (buflen);
	}

      if (!pw)
	{
	  fprintf (stderr, _("%s: Cannot determine your user name.\n"),
		   program);
	  return E_UNKNOWN_USER;
	}

      caller_name = strdupa (pw->pw_name);

#if 0
      /* We change the shadow information for another user, get that
	 data, too.  */
      if (argc == 1)
	{
	  char *user = argv[0];

	  while (getpwnam_r (user, &resultbuf, buffer, buflen, &pw) != 0
		 && errno == ERANGE)
	    {
	      errno = 0;
	      buflen += 256;
	      buffer = alloca (buflen);
	    }
	  if (!pw)
	    {
	      fprintf (stderr, _("%s: Unknown user %s\n"), program, user);
	      return E_UNKNOWN_USER;
	    }
	}
#endif

      /* if we show/modify the data for another user, get the data from
	 this one.  */
      if (argc == 1)
	arg_user = argv[0];
      else
	arg_user = pw->pw_name;

      pw_data = do_getpwnam (arg_user, use_service);
      if (pw_data == NULL || pw_data->service == S_NONE)
	{
	  if (use_service)
	    fprintf (stderr, _("%s: User %s is not known to service %s\n"),
		     program, arg_user, use_service);
	  else
	    fprintf (stderr, _("%s: Unknown user %s\n"), program,
		     arg_user);
	  return E_UNKNOWN_USER;
	}
    }

  /* Only root is allowed to change aging for local users. */
  if (!l_flag && uid && pw_data->service == S_LOCAL)
    {
      fprintf (stderr,
	       _("Only root is allowed to change aging information.\n"));
      free_user_t (pw_data);
      return E_NOPERM;
    }

  /* misuse l_flag: We don't need to extra ask for a password with "-l".  */
  if (do_authentication (program, caller_name, pw_data, !l_flag) != 0)
    return E_NOPERM;

  if (l_flag)
    {
      if (uid != 0 && strcmp (caller_name, pw_data->pw.pw_name) != 0)
	{
	  fprintf (stderr, _("You can only list your own aging information.\n"));
	  return E_NOPERM;
	}

      if (setgid (getgid ()) || setuid (getuid ()))
	{
	  syslog (LOG_ERR, "%s: failed to drop privileges: %s\n",
		  program, strerror (errno));
	  fprintf (stderr, "%s: failed to drop privileges: %s\n",
		   program, strerror (errno));
	  return E_FAILURE;
        }

      if (pw_data->use_shadow)
	print_shadow_info (pw_data);
      else
	fprintf (stdout, _("No aging information available for %s.\n"),
		 pw_data->pw.pw_name);

      return 0;
    }

  if (uid != 0)
    return E_USAGE;

  if (interactive)
    {
      int res;

      if (!silent)
	printf (_("Changing aging information for %s.\n"),
		pw_data->pw.pw_name);

      if ((res = change_shadow_info (pw_data)) != 0)
	{
	  if (!silent)
	    printf (_("Aging information not changed.\n"));
	  return E_FAILURE;
	}
    }
  else
    {
      char *cp;
      int error = 0;

      if (mindays)
	if (((pw_data->spn.sp_min = strtol (mindays, &cp, 10)) == 0 && *cp) ||
	    pw_data->spn.sp_min < -1)
	  ++error;

      if (maxdays)
	if (((pw_data->spn.sp_max = strtol (maxdays, &cp, 10)) == 0 && *cp) ||
	    pw_data->spn.sp_max < -1)
	  ++error;

      if (warndays)
	if (((pw_data->spn.sp_warn = strtol (warndays, &cp, 10)) == 0 && *cp)
	    || pw_data->spn.sp_warn < -1)
	  ++error;

      if (inactive)
	if (((pw_data->spn.sp_inact = strtol (inactive, &cp, 10)) == 0 && *cp)
	    || pw_data->spn.sp_inact < -1)
	  ++error;

      if (lastday)
	{
	  if (strcmp (lastday, "1969-12-31") == 0)
	    pw_data->sp.sp_lstchg = -1;
	  else
	    {
	      pw_data->spn.sp_lstchg = str2date (lastday);
	      if (pw_data->spn.sp_lstchg == -1)
		{
		  if (((pw_data->spn.sp_lstchg =
			strtol (lastday, &cp, 10)) == 0 && *cp) ||
		      pw_data->spn.sp_lstchg < -1)
		    {
		      fprintf (stderr, _("Lastday is no date and no\
 integer value >= -1\n"));
		      ++error;
		    }
		}
	    }
	}

      if (expiredate)
	{
	  if (strcmp (expiredate, "1969-12-31") == 0)
	    pw_data->spn.sp_expire = -1;
	  else
	    {
	      pw_data->spn.sp_expire = str2date (expiredate);
	      if (pw_data->spn.sp_expire == -1)
		{
		  if (((pw_data->spn.sp_expire =
			strtol (expiredate, &cp, 10)) == 0 && *cp) ||
		      pw_data->spn.sp_expire < -1)
		    {
		      fprintf (stderr, _("Expiredate is no date and no\
 integer value >= -1\n"));
		      ++error;
		    }
		}
	    }
	}
      if (error)
	{
	  if (!silent)
	    fprintf (stderr, _("Error while parsing options.\n"));
	  free_user_t (pw_data);
	  return E_BAD_ARG;
	}
    }

  /* we don't need to change the data if there is no change */
  if (pw_data->sp.sp_min == pw_data->spn.sp_min &&
      pw_data->sp.sp_max == pw_data->spn.sp_max &&
      pw_data->sp.sp_warn == pw_data->spn.sp_warn &&
      pw_data->sp.sp_inact == pw_data->spn.sp_inact &&
      pw_data->sp.sp_lstchg == pw_data->spn.sp_lstchg &&
      pw_data->sp.sp_expire == pw_data->spn.sp_expire)
    {
      if (!silent)
	printf (_("Aging information not changed.\n"));
      return 0;
    }
  else
    pw_data->sp_changed = TRUE;

  if (do_setpwnam (pw_data) != 0)
    {
      fprintf (stderr,_( "Error while changing aging information.\n"));
      free_user_t (pw_data);
      return E_FAILURE;
    }
  else
    {
      nscd_flush_cache ("passwd");
      if (!silent)
	printf (_("Aging information changed.\n"));
    }

  free_user_t (pw_data);

  return 0;
}
