/*-
 * customs.c --
 *	A server to inform clients, using RPC, where they can send
 *	processes.
 *
 *	Each machine that wishes to export processes must have
 *	a server running. The first server to start is the
 *	Master Customs Agent (MCA), while everything else is a
 *	Slave Customs Agent (SCA).
 *
 *	The job of the MCA is to track the availability of machines
 *	running SCA's and allocate them to other SCA's as necessary.
 *
 *	An SCA is responsible for sampling its host's status from
 *	time to time to see if it can accept processes from other
 *	machines. This information must be provided to the MCA
 *	at regular intervals.
 *
 *	An SCA can specify which hosts it is willing to serve for
 *	(there must be at least one) and the MCA will only allocate
 *	it to one of the hosts in that list. A host will not be allocated
 *	to an SCA again until all other available hosts which serve
 *	the requesting SCA have been used.
 *
 *	Several procedures are provided to clients of the customs agents.
 *	The CUSTOMSPROC_HOST will return a structure containing the address
 *	of a machine on which the exported process may be run. It also
 *	returns an identifying number which must be passed to the serving
 *	machine before it will execute the process. The MCA does not guarantee
 *	that the serving machine is actually up, though it tries hard to
 *	ensure this. If the machine has a long interval between availability
 *	packets, all bets are off.
 *
 * Copyright (c) 1988, 1989 by the Regents of the University of California
 * Copyright (c) 1988, 1989 by Adam de Boor
 * Copyright (c) 1989 by Berkeley Softworks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any non-commercial purpose
 * and without fee is hereby granted, provided that the above copyright
 * notice appears in all copies.  The University of California,
 * Berkeley Softworks and Adam de Boor make no representations about
 * the suitability of this software for any purpose.  It is provided
 * "as is" without express or implied warranty.
 *
 */
#ifndef lint
static char *rcsid =
"$Id: customs.c,v 1.37 1992/08/01 02:07:48 stolcke Exp $ ICSI (Berkeley)";
#endif lint

#include    <sys/file.h>
#include    <stdio.h>
#include    <sys/ioctl.h>
#include    <strings.h>
#include    <signal.h>
#include    <sys/time.h>
#include    <netdb.h>
#include    <errno.h>
extern int errno;

#include    "sprite.h"
#include    "customsInt.h"
#include    "xmalloc.h"
#include    "xlog.h"

#include    "../patchlevel.h"	/* for global version info */
#include    "version.h"		/* for build version info */

char	    	  	    localhost[MACHINE_NAME_SIZE];
Boolean	    	  	    amMaster;	/* TRUE if this process is MCA */

Boolean	    	  	    verbose = FALSE;

unsigned long	    	    arch;   	/* Machine architecture code */
char	    	  	    *regPacket;
int	    	  	    regPacketLen;
int	    	  	    numClients;
char	    	  	    **clients;
int	    	  	    initARGC;
char    	  	    **initARGV;


struct timeval	retryTimeOut = {
    CUSTOMSINT_RETRY, CUSTOMSINT_URETRY
};

short	    	    	    tcpPort;
int	    	  	    tcpSocket;
short			    udpPort;
int	    	  	    udpSocket;	/* The actual socket to which clients
					 * connect */
struct sockaddr_in	    localAddr;	/* Address of local socket */

static Version_Data version = {
    MAJORVERSION,
    MINORVERSION,
    PATCHLEVEL,
    BUILDNO,
    BUILDDATE,
    BUILDER
};

/*-
 *-----------------------------------------------------------------------
 * MakeRegistrationPacket --
 *	Mangles the local host and argument vector into a packet to
 *	be sent to the master at registration time.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	regPacket is allocated and filled and regPacketLen is altered.
 *
 *-----------------------------------------------------------------------
 */
static void
MakeRegistrationPacket()
{
    register int  i;
    register char **cpp;
    register char *cp;
    int localhostLen;

    localhostLen = (strlen(localhost) + 1 + 3) & ~3;
    regPacketLen = localhostLen + sizeof(long) + sizeof(arch);
    for (i = 0; i < numClients; i++) {
	regPacketLen += strlen(clients[i]) + 1;
    }
    regPacket = (char *)malloc((unsigned)regPacketLen);
    strncpy (regPacket, localhost, localhostLen);
    cp = regPacket + localhostLen;
    *((unsigned long *)cp) = arch;
    cp += sizeof(arch);
    *((long *)cp) = numClients;
    cp += sizeof(long);

    for (cpp = clients, i = numClients; i != 0; i--, cpp++) {
	strcpy(cp, *cpp);
	cp += strlen(cp) + 1;
    }
}

/*-
 *-----------------------------------------------------------------------
 * CustomsPing --
 *	Do-nothing function to handle the CUSTOMS_PING procedure.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
CustomsPing(from, msg, len, data)
    struct sockaddr_in	*from;
    Rpc_Message	  	msg;
    int	    	  	len;
    Rpc_Opaque 	  	data;
{
    Rpc_Return(msg, 0, data);
}

/*-
 *-----------------------------------------------------------------------
 * CustomsHost --
 *	Stub for handling the CUSTOMS_HOST procedure. Issues
 *	CUSTOMS_HOSTINT call to master. While the master could conceivably
 *	call MCA_HostInt, we don't do that b/c the port number for return
 *	will be wrong. If the flags in the data contain EXPORT_USELOCAL,
 *	the availability of the local machine is checked before the call
 *	is issued.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	An ExportPermit is sent in reply.
 *
 *-----------------------------------------------------------------------
 */
static void
CustomsHost(from, msg, len, data)
    struct sockaddr_in	*from;
    Rpc_Message	  	msg;
    int	    	  	len;
    Rpc_Opaque	  	data;
{
    ExportPermit    	permit;
    Host_Data	  	*host = (Host_Data *)data;
    long    	    	rating;	/* Junk variable */

    /*
     * We require this call to be issued from a reserved port to
     * ensure that the uid in the request is accurate.  This will
     * usually mean the client program has to be suid root, so that
     * users cannot write fake clients.
     */
    CustomsReserved("CustomsHost", from, msg);
	
    /*
     * A CustomsHost call must come from the local host to
     * keep people from requesting from a different machine, not
     * that it would work anyway, since the permit published
     * by the MCA wouldn't match...
     */
    if (!Local(from)) {
	Rpc_Error(msg, RPC_ACCESS);
    } if (len != sizeof(Host_Data)) {
	Rpc_Error(msg, RPC_BADARGS);
    } else if (Elect_InProgress() ||
	       ((host->flags & EXPORT_USELOCAL) &&
		Avail_Local (~(AVAIL_IDLE|AVAIL_LOAD), &rating) == 0))
    {
		   
	/*
	 * If we're electing, or being nice and the local machine is
	 * available, barring keyboard idle time and load, of course, then
	 * send a failure message to the client -- the id field is
	 * unimportant in this case...
	 */
	permit.addr.s_addr = htonl(INADDR_ANY);
	Rpc_Return(msg, sizeof(permit), (Rpc_Opaque)&permit);
    } else {
	/*
	 * Ask the master where to go.
	 */
	Rpc_Stat  	rstat;

	rstat = Rpc_Call(udpSocket, &masterAddr, (Rpc_Proc)CUSTOMS_HOSTINT,
			 len, data, sizeof(permit), (Rpc_Opaque)&permit,
			 CUSTOMSINT_NRETRY, &retryTimeOut);
	if (rstat != RPC_SUCCESS) {
	    permit.addr.s_addr = htonl(INADDR_ANY);
	    if (verbose) {
		xlog (XLOG_DEBUG, "CustomsHost: %s",
			Rpc_ErrorMessage(rstat));
	    }
	} else if (verbose) {
	    xlog (XLOG_DEBUG, "CustomsHost: returns %s\n",
		    InetNtoA(permit.addr));
	}
	Rpc_Return(msg, sizeof(permit), (Rpc_Opaque)&permit);
	if (rstat == RPC_TIMEDOUT) {
	    Elect_GetMaster();
	}
    }
}

/*-
 *-----------------------------------------------------------------------
 * CustomsMaster --
 *	Stub to handle CUSTOMS_MASTER procedure. Returns the address of
 *	the current master agent.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The address of the master is sent to the client, or an RPC_TIMEDOUT
 *	error is returned if an election is going on.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
CustomsMaster(from, msg, len, data)
    struct sockaddr_in  *from;
    Rpc_Message		msg;
    int			len;
    Rpc_Opaque 	  	data;
{
    if (Elect_InProgress()) {
	Rpc_Error(msg, RPC_TIMEDOUT);
    } else {
	Rpc_Return(msg, sizeof(masterAddr), (Rpc_Opaque)&masterAddr);
    }
}

/*-
 *-----------------------------------------------------------------------
 * CustomsAbort --
 *	Abort this daemon. Returns nothing. Just exits.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The process exits.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
CustomsAbort(from, msg, len, data)
    struct sockaddr_in  *from;
    Rpc_Message		msg;
    int			len;
    Rpc_Opaque 	  	data;
{
    CustomsReserved("CustomsAbort", from, msg);

    xlog(XLOG_INFO, "Received ABORT message from %d@%s...",
	   ntohs(from->sin_port),
	   InetNtoA(from->sin_addr));

    Rpc_Return(msg, 0, (Rpc_Opaque)0);
    OS_Exit();

    xlog(XLOG_INFO, "Customs EXITING");
    exit(0);
}

/*-
 *-----------------------------------------------------------------------
 * CustomsRestart --
 *	Re-execute ourselves using the original argument vector.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	We reexecute.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
CustomsRestart(from, msg, len, data)
    struct sockaddr_in	*from;
    Rpc_Message	  	msg;
    int	    	  	len;
    Rpc_Opaque	  	data;
{
    int	    	  	i;
    int			fd;
    char		*cmdstr;
    int			cmdlen;

    CustomsReserved("CustomsRestart", from, msg);

    xlog(XLOG_INFO, "Received RESTART from %d@%s ...", ntohs(from->sin_port),
	   InetNtoA(from->sin_addr));

    cmdlen = 0;
    for (i = 0; i < initARGC; i++) {
	cmdlen += strlen(initARGV[i]) + 1;
    }
    cmdstr = malloc(cmdlen);
    cmdstr[0] = '\0';
    for (i = 0; i < initARGC; i++) {
	strcat(cmdstr, initARGV[i]);
	strcat(cmdstr, " ");
    }
    xlog(XLOG_INFO, "executing: %s", cmdstr);

    /*
     * We would like to send the ACK only after successful exec, but without
     * vfork we can't.
     */
    Rpc_Return(msg, 0, (Rpc_Opaque)0);

    /*
     * Close any files the OS module has opened.
     */
    OS_Exit();

    /*
     * Close log.
     */
    xlog_set(NULL, XLOG_NONE);

    /*
     * Make sure stdin and stderr are open to something so we don't close
     * the sockets in our next life.
     */
    fd = open("/dev/null", 0);
    if (fd != 0) {
	dup2(fd, 0);
    }
    if (fd != 2) {
	dup2(fd, 2);
    }
    if (fd > 2) {
	close(fd);
    }

    execvp(initARGV[0],initARGV);

    xlog_set(NULL, XLOG_PREVIOUS);

    perror(initARGV[0]);
    xlog(XLOG_FATAL, "Customs ABORTING");

    exit(1);
}

/*-
 *-----------------------------------------------------------------------
 * CustomsDebug --
 *	Turn on debugging for us and/or the rpc system
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	verbose may be set true, as may rpcDebug.
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
CustomsDebug(from, msg, len, data)
    struct sockaddr_in	*from;
    Rpc_Message	  	msg;
    int	    	  	len;
    int	  	  	*data;
{
    CustomsReserved("CustomsAbort", from, msg);

    Rpc_Debug(*data & DEBUG_RPC);
    verbose = *data & DEBUG_CUSTOMS;

    Rpc_Return(msg, 0, (Rpc_Opaque)NULL);
}

/*-
 *-----------------------------------------------------------------------
 * PrintVersion --
 *	Print version info on stdout
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Ditto.
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
PrintVersion()
{
    xlog(XLOG_DEBUG, "Customs version %d.%d.%d #%d: built by %s on %s",
		version.majorversion,
		version.minorversion,
		version.patchlevel,
		version.buildno,
		version.builddate,
		version.builder);
}
/*-
 *-----------------------------------------------------------------------
 * CustomsVersion --
 *	Return version info for this daemon
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	None.
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
static void
CustomsVersion(from, msg, len, data)
    struct sockaddr_in	*from;
    Rpc_Message	  	msg;
    int	    	  	len;
    int	  	  	*data;
{
    if (verbose)
	PrintVersion();

    Rpc_Return(msg, sizeof(version), (Rpc_Opaque)&version);
}

/*-
 *-----------------------------------------------------------------------
 * Customs_ProcTitle --
 *	Set process title for ps (borrowed from sendmail).
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Clobbers argv of our main procedure so ps(1) will
 *	display the title.
 *
 *-----------------------------------------------------------------------
 */
void
Customs_SetProcTitle(str)
    char *str;
{
#ifdef SETPROCTITLE
    char *p;
    int i;
    static char *lastArgv = NULL;

    if (!lastArgv) {
	lastArgv = initARGV[initARGC - 1] + strlen(initARGV[initARGC - 1]);
    }

    /*
     * Make ps print "(customs)"
     */
    p = initARGV[0];
    *p++ = '-';

    i = strlen(str);
    if (i > lastArgv - p - 2) {
	i = lastArgv - p - 2;
    }
    (void) strncpy(p, str, i);
    p += i;
    while (p < lastArgv)
	*p++ = ' ';
#endif SETPROCTITLE
}

/*-
 *-----------------------------------------------------------------------
 * gettime --
 *	Get a time value from a string. It must be in the form
 *	    mm:ss
 *	where 'm' is a minute's digit and 's' is a second's digit.
 *	seconds may not be greater than 60, for obvious reasons.
 *
 * Results:
 *
 * Side Effects:
 *
 *-----------------------------------------------------------------------
 */
int
gettime (str)
    char    *str;
{
    int	    min,
	    sec;

    for (min = 0;
	 *str != '\0' && *str != ':' ;
	 min = 10 * min + *str++ - '0') {
	     continue;
    }
    if (*str == '\0') {
	sec = min;
	min = 0;
    } else {
	for (sec = 0, str++; *str != '\0'; sec = 10 * sec + *str++ - '0') {
	    continue;
	}
    }
    if (sec >= 60) {
	printf("malformed time (only 60 seconds in a minute)\n");
	exit(1);
    }
    return (min * 60 + sec);
}

static SIGRET CustomsDebugOn() { verbose = TRUE; Rpc_Debug(True); }
static SIGRET CustomsDebugOff() { verbose = FALSE; Rpc_Debug(False); }

static SIGRET
CustomsDown(signo)
    int		signo;
{
    xlog(XLOG_INFO, "Going down on signal %d ...", signo);

    OS_Exit();

    xlog(XLOG_INFO, "Customs EXITING");
    exit(0);
}

/*-
 *-----------------------------------------------------------------------
 * Usage --
 *	Print out the flags we accept and die.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The process exits.
 *
 *-----------------------------------------------------------------------
 */
void
Usage ()
{
    printf ("Usage: customs <options> {host [host ...] | ALL }\n");
    printf ("\t-verbose	Print verbose log messages\n");
    printf ("\t-check <time>	Set availability check interval\n");
    printf ("\t-idle <time> 	Set minimum idle time\n");
    printf ("\t-load <loadavg>	Set maximum load average\n");
    printf ("\t-swap <percent>	Set minimum free swap space\n");
    printf ("\t-jobs <num>  	Set maximum imported jobs\n");
    printf ("\t-proc <num>  	Set minimum free processes\n");
    printf ("\t-cpu <time>  	Set cpu time limit for imported jobs\n");
    printf ("\t-nice <level>  	Set nice increment for imported jobs\n");
    printf ("\t-access <level>	Set user access level\n");
    printf ("\t-evict <time>  	Set eviction delay for imported jobs\n");
    printf ("\t-net <netnum>	Set customs network number\n");
    printf ("\t-arch <code>	Set customs architecture code\n");
    exit(1);
}

/*-
 *-----------------------------------------------------------------------
 * main --
 *	Check the correctness of the arguments, look for an MCA and become
 *	it if it doesn't exist, registering ourselves in the process.
 *	Else register with the MCA and go into SCA mode.
 *
 * Results:
 *	none.
 *
 * Side Effects:
 *	Billions and Billions.
 *
 *-----------------------------------------------------------------------
 */
main (argc, argv)
    int	    	  	argc;
    char    	  	**argv;
{
    Boolean 	  	doLog = TRUE;
    Boolean 	  	debug = FALSE;
    Boolean 	  	version = FALSE;
    Avail_Data	  	criteria;
    int	    	  	checkTime;
    struct servent  	*sep;
    int			i;

    clients = (char **)malloc(sizeof(char *) * argc);
    numClients = 0;

    initARGC = argc;
    initARGV = argv;

    argc--, argv++;
    criteria.changeMask = 0;
    checkTime = 0;

    while (argc > 0) {
	if (strcmp (*argv, "-verbose") == 0) {
	    verbose = TRUE;
	} else if (strcmp (*argv, "-nolog") == 0) {
	    doLog = FALSE; debug = TRUE;
	} else if (strcmp (*argv, "-debug") == 0) {
	    debug = TRUE;
	} else if (strcmp (*argv, "-v") == 0 ||
	         strcmp (*argv, "-version") == 0) {
	    version = TRUE;
	} else if (strcmp (*argv, "-check") == 0) {
	    if (argc > 1) {
		checkTime = gettime(argv[1]);
		argc--;
		argv++;
	    } else {
		printf("-check needs a time as an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-idle") == 0) {
	    if (argc > 1) {
		criteria.idleTime = gettime(argv[1]);
		argc--;
		argv++;
		criteria.changeMask |= AVAIL_IDLE;
	    } else {
		printf("-idle needs a time as an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-swap") == 0) {
	    if (argc > 1) {
		criteria.swapPct = atoi(argv[1]);
		argc--;
		argv++;
		criteria.changeMask |= AVAIL_SWAP;
	    } else {
		printf("-swap needs a percentage as an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-load") == 0) {
	    double	maxLoad, atof();

	    if (argc > 1) {
		maxLoad = atof(argv[1]);
		criteria.loadAvg = (int)(maxLoad * LOADSCALE);
		criteria.changeMask |= AVAIL_LOAD;
		argc--;
		argv++;
	    } else {
		printf("-load needs a load average as an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-jobs") == 0) {
	    if (argc > 1) {
		criteria.imports = atoi(argv[1]);
		argc--;
		argv++;
		criteria.changeMask |= AVAIL_IMPORTS;
	    } else {
		printf("-jobs needs a number of jobs as an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-proc") == 0) {
	    if (argc > 1) {
		criteria.procs = atoi(argv[1]);
		argc--;
		argv++;
		criteria.changeMask |= AVAIL_PROC;
	    } else {
		printf("-proc needs a number of processes as an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-cpu") == 0) {
	    if (argc > 1) {
		criteria.cpuLimit = gettime(argv[1]);
		argc--;
		argv++;
		criteria.changeMask |= AVAIL_CPU;
	    } else {
		printf("-cpu needs a time as an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-nice") == 0) {
	    if (argc > 1) {
		criteria.niceLevel = atoi(argv[1]);
		argc--;
		argv++;
		criteria.changeMask |= AVAIL_NICE;
	    } else {
		printf("-nice needs a nice level number as an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-access") == 0) {
	    if (argc > 1) {
		criteria.checkUser = atoi(argv[1]);
		argc--;
		argv++;
		criteria.changeMask |= AVAIL_CHECK;
	    } else {
		printf("-access needs a level number as an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-evict") == 0) {
	    if (argc > 1) {
		criteria.evictDelay = gettime(argv[1]);
		argc--;
		argv++;
		criteria.changeMask |= AVAIL_EVICT;
	    } else {
		printf("-evict needs a time as an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-net") == 0) {
	    if (argc > 1) {
		elect_Token = atoi(argv[1]);
		argc--;
		argv++;
	    } else {
		printf("-net needs a network number for an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp (*argv, "-arch") == 0) {
	    if (argc > 1) {
		arch = atoi(argv[1]);
		argc--;
		argv++;
	    } else {
		printf("-arch needs an architecture code for an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (strcmp(*argv, "-bias") == 0) {
	    if (argc > 1) {
		avail_Bias = atoi(argv[1]);
		argc--;
		argv++;
	    } else {
		printf("-bias needs a bias value for an argument\n");
		Usage();
		/*NOTREACHED*/
	    }
	} else if (**argv == '-') {
	    printf ("Unknown option %s\n", *argv);
	    Usage();
	    /*NOTREACHED*/
	} else {
	    clients[numClients] = *argv;
	    numClients += 1;
	}
	argc--; argv++;
    }

    if (version) {
	PrintVersion();
	exit(0);
    }

    if (numClients == 0) {
	xlog (XLOG_FATAL, "You must serve at least one host!");
	exit(1);
    }
	
    if (!debug) {
	int t,
	    pid;

	/*
	 * First detach from our parent, then from the tty...
	 */
	pid = fork();
	if (pid == -1) {
	    perror("fork");
	}
	if (pid != 0) {
	    exit(0);
	}

#ifdef SYSV
	setpgrp();
#else
	t = open ("/dev/tty", O_RDWR, 0);
	if (t >= 0) {
	    ioctl (t, TIOCNOTTY, 0);
	    (void) close (t);
	}
#endif

	if (getuid() || geteuid()) {
	    xlog (XLOG_FATAL,
		    "This program must be run as root to execute properly");
	    exit(1);
	}

#ifndef NO_PRIORITY
	/*
	 * Renice the main server so it can respond to calls even if four
	 * of its jobs are running...
	 */
#ifdef SYSV
	if (nice(0) == 0 && nice(-2) != -2) {
	    perror("nice");
	}
#else
	if (setpriority(PRIO_PROCESS, getpid(), -2) < 0) {
	    perror("setpriority");
	}
#endif
#endif
    }
    
    if (gethostname (localhost, MACHINE_NAME_SIZE)) {
	xlog (XLOG_FATAL, "The name of this machine is too long (%d chars max)",
		  MACHINE_NAME_SIZE);
	exit(1);
    }

    if (doLog) {
#ifdef LOG_FACILITY
	/*
	 * Detach from terminal but keep stdout open for compatibility
	 * with the remaining code.
	 */
	freopen ("/dev/null", "a", stdout);

	xlog_set("customs", XLOG_SYSLOG, LOG_FACILITY);
#else /* !LOG_FACILITY */
	char logName[256];

#ifdef LOG_BASE
	char *cp = index (localhost, '.');
	if (cp) {
	    *cp = '\0';
	}
	sprintf (logName, "%s.%s", LOG_BASE, localhost);
	if (cp) {
	    *cp = '.';
	}
#else /* !LOG_BASE */
	sprintf (logName, "%s", LOG_FILE);
#endif /* LOG_BASE */

	freopen (logName, "a", stdout);
	fcntl(fileno(stdout), F_SETFL, FAPPEND);

	xlog_set("customs", stdout);
#endif /* LOG_FACILITY */
    }

    for (i = 3; i > 0; i--) {
	sep = getservbyname(CUSTOMS_SERVICE_NAME, "udp");
	if (sep != NULL) {
	    sleep(2);
	} else {
	    break;
	}
    }
    if (sep == NULL) {
	xlog(XLOG_WARNING, "%s/udp (still) unknown", CUSTOMS_SERVICE_NAME);
	udpPort = DEF_CUSTOMS_UDP_PORT;
    } else {
	udpPort = ntohs(sep->s_port);
    }
    udpSocket = Rpc_UdpCreate(True, udpPort);
    if (udpSocket < 0) {
	perror ("Rpc_UdpCreate");
	exit(1);
    }

    for (i = 3; i > 0; i--) {
	sep = getservbyname(CUSTOMS_SERVICE_NAME, "tcp");
	if (sep != NULL) {
	    sleep(2);
	} else {
	    break;
	}
    }
    if (sep == NULL) {
	xlog(XLOG_WARNING, "%s/tcp (still) unknown", CUSTOMS_SERVICE_NAME);
	tcpPort = DEF_CUSTOMS_TCP_PORT;
    } else {
	tcpPort = ntohs(sep->s_port);
    }
    tcpSocket = Rpc_TcpCreate(True, tcpPort);
    if (tcpSocket < 0) {
	perror("Rpc_TcpCreate");
	exit(1);
    }
    /*
     * Mark both service sockets as close on exec.
     */
    (void)fcntl (udpSocket, F_SETFD, 1);
    (void)fcntl (tcpSocket, F_SETFD, 1);

    /*
     * Register all the servers every agent must handle
     */
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_PING, CustomsPing,
		     Rpc_SwapNull, Rpc_SwapNull, (Rpc_Opaque)0);
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_HOST, CustomsHost,
		     Swap_Host, Swap_ExportPermit, (Rpc_Opaque)0);
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_MASTER, CustomsMaster,
		     Rpc_SwapNull, Swap_SockAddr, (Rpc_Opaque)0);
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_ABORT, CustomsAbort,
		     Rpc_SwapNull, Rpc_SwapNull, (Rpc_Opaque)0);
    Rpc_ServerCreate(tcpSocket, (Rpc_Proc)CUSTOMS_ABORT, CustomsAbort,
		     Rpc_SwapNull, Rpc_SwapNull, (Rpc_Opaque)0);
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_RESTART, CustomsRestart,
		     Rpc_SwapNull, Rpc_SwapNull, (Rpc_Opaque)0);
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_DEBUG, CustomsDebug,
		     Rpc_SwapLong, Rpc_SwapNull, (Rpc_Opaque)0);
    Rpc_ServerCreate(udpSocket, (Rpc_Proc)CUSTOMS_VERSION, CustomsVersion,
		     Rpc_SwapNull, Swap_Version, (Rpc_Opaque)0);
    
    signal(SIGUSR1, CustomsDebugOn);
    signal(SIGUSR2, CustomsDebugOff);

    signal(SIGHUP, CustomsDown);
    signal(SIGINT, CustomsDown);
    signal(SIGTERM, CustomsDown);
    
    /*
     * Close stdin and stderr so the descriptors are reused by the Avail
     * module. Note that if the Avail module changes to use fewer than two
     * descriptors, you should leave one of these things open so the Import
     * module doesn't have to worry about its socket already being in the
     * right place (dup2 (0, 0) would probably not be cool...)
     */
    fclose (stdin);
    fclose (stderr);

    /*
     * XXX: There should be some way to actually share these things so
     * all the perror-type functions would play with the same things.
     * unfortunately, stderr and stdout are macros, so we can only
     * get around things by making stdout be unbuffered...
     */
    setbuf (stdout, (char *)NULL);
    *stderr = *stdout;
	
    /*
     * Find the local address. get_myaddress is actually from Sun RPC
     * functions, but...
     */
    get_myaddress(&localAddr);
    localAddr.sin_port = htons(udpPort);
    xlog (XLOG_INFO, "daemon started: local address = %d@%s",
	    ntohs(localAddr.sin_port),
	    InetNtoA(localAddr.sin_addr));

    MakeRegistrationPacket();

    Log_Init();
    Avail_Init(&criteria, checkTime);
    Import_Init();
    Elect_Init();
    Elect_GetMaster();

    Rpc_Run();
}

