/*-
 * customslib.c --
 *	Front end RPC stubs for clients of the customs daemons.
 *
 * Copyright (c) 1988, 1989 by the Regents of the University of California
 * Copyright (c) 1988, 1989 by Adam de Boor
 * Copyright (c) 1989 by Berkeley Softworks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any non-commercial purpose
 * and without fee is hereby granted, provided that the above copyright
 * notice appears in all copies.  The University of California,
 * Berkeley Softworks and Adam de Boor make no representations about
 * the suitability of this software for any purpose.  It is provided
 * "as is" without express or implied warranty.
 */
#ifndef lint
static char *rcsid =
"$Id: customslib.c,v 1.24 1992/10/05 04:43:11 stolcke Exp $ ICSI (Berkeley)";
#endif lint

#include    <strings.h>
#include    <sys/time.h>
#include    <stdio.h>
#include    <sys/file.h>
#include    <netdb.h>

#include    "customs.h"

int  	  	    	customs_Socket = -1;
struct sockaddr_in   	customs_AgentAddr;
struct timeval	    	customs_RetryTimeOut = {
    CUSTOMS_RETRY, CUSTOMS_URETRY
};
static Rpc_Stat	  	    lastStatus;
static short	    	    udpPort,
			    tcpPort;

/*-
 *-----------------------------------------------------------------------
 * Customs_Init --
 *	Initialize things so these functions may be used.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	A udp socket is opened and customs_AgentAddr is initialized to be
 *	suitable for talking to the local agent.
 *
 *-----------------------------------------------------------------------
 */
void
Customs_Init()
{
    struct servent  *sep;
    int	    	i;

    i = 3;

    do {
	sep = getservbyname(CUSTOMS_SERVICE_NAME, "udp");
    } while (sep == NULL && i-- > 0);

    if (sep == NULL) {
#if 0
	printf("%s/udp unknown\n", CUSTOMS_SERVICE_NAME);
	/*XXX*/
	exit(1);
#else
	udpPort = DEF_CUSTOMS_UDP_PORT;
#endif
    } else {
	udpPort = ntohs(sep->s_port);
    }
    i = 3;
    do {
	sep = getservbyname(CUSTOMS_SERVICE_NAME, "tcp");
    } while (sep == NULL && i-- > 0);

    if (sep == NULL) {
#if 0
	printf("%s/tcp unknown\n", CUSTOMS_SERVICE_NAME);
	/*XXX*/
	exit(1);
#else
	tcpPort = DEF_CUSTOMS_TCP_PORT;
#endif
    } else {
	tcpPort = ntohs(sep->s_port);
    }
    
    /*
     * Try to use a reserved port so we may call CUSTOMS_HOST and 
     * other rpcs that require root priviledge.
     */
    customs_Socket = Rpc_UdpCreate(False, (geteuid() == 0));
    if (customs_Socket < 0) {
	static Boolean printed = False;
	if (!printed) {
	    perror("Rpc_UdpCreate");
	    printed = True;
	}
    }
    else {
	(void)fcntl(customs_Socket, F_SETFD, 1);
    }

    bzero(&customs_AgentAddr, sizeof(customs_AgentAddr));
    customs_AgentAddr.sin_family = AF_INET;
    customs_AgentAddr.sin_port = htons(udpPort);
    customs_AgentAddr.sin_addr.s_addr = htonl(INADDR_LOOPBACK);
}
/*-
 *-----------------------------------------------------------------------
 * Customs_Port --
 *	Get the TCP port number for talking to customs
 *
 * Results:
 *	The port number in host byte order.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
short
Customs_Port()
{
    if (customs_Socket == -1) {
	Customs_Init();
    }
    return tcpPort;
}
/*-
 *-----------------------------------------------------------------------
 * Customs_Ping --
 *	See if the local agent is alive.
 *
 * Results:
 *	RPC_SUCCESS if the agent is alive and responding.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
Rpc_Stat
Customs_Ping()
{
    if (customs_Socket == -1) {
	Customs_Init();
    }
    lastStatus = Rpc_Call(customs_Socket, &customs_AgentAddr,
			  (Rpc_Proc)CUSTOMS_PING,
			  0, (Rpc_Opaque)0,
			  0, (Rpc_Opaque)0,
			  CUSTOMS_NRETRY, &customs_RetryTimeOut);
    return(lastStatus);
}

/*-
 *-----------------------------------------------------------------------
 * Customs_Host --
 *	Request a host for exportation from the local agent. We send the
 *	effective user-id partly because it's in the protocol and
 *	partly because that's what determines file accesses...
 *
 * Results:
 *	The status of the call. If RPC_SUCCESS, permitPtr contains the
 *	agent's response, which may have an addr field of INADDR_ANY. This
 *	indicates that a host could not be allocated.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
Rpc_Stat
Customs_Host(flags, permitPtr)
    short   	  flags;
    ExportPermit  *permitPtr;
{
    Host_Data	  data;
    
    data.uid = geteuid();
    data.flags = flags;

    if (customs_Socket == -1) {
	Customs_Init();
    }
    lastStatus = Rpc_Call(customs_Socket, &customs_AgentAddr,
			  (Rpc_Proc)CUSTOMS_HOST,
			  sizeof(data), (Rpc_Opaque)&data,
			  sizeof(ExportPermit), (Rpc_Opaque)permitPtr,
			  CUSTOMS_NRETRY, &customs_RetryTimeOut);
    return(lastStatus);
}

/*-
 *-----------------------------------------------------------------------
 * Customs_AvailInterval --
 *	Set the interval at which the local agent informs the master of
 *	its availability.
 *
 * Results:
 *	The status of the call.
 *
 * Side Effects:
 *	See above.
 *
 *-----------------------------------------------------------------------
 */
Rpc_Stat
Customs_AvailInterval(interval)
    struct timeval	*interval;
{
    if (customs_Socket == -1) {
	Customs_Init();
    }
    lastStatus = Rpc_Call(customs_Socket, &customs_AgentAddr,
			  (Rpc_Proc)CUSTOMS_AVAILINTV,
			  sizeof(*interval), (Rpc_Opaque)interval,
			  0, (Rpc_Opaque)0,
			  CUSTOMS_NRETRY, &customs_RetryTimeOut);
    return(lastStatus);
}

/*-
 *-----------------------------------------------------------------------
 * Customs_Master --
 *	Find the location of the current master customs agent.
 *
 * Results:
 *	The status of the call. If the call succeeds, the passed
 *	sockaddr_in is filled with address of the current master.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
Rpc_Stat
Customs_Master(masterAddrPtr)
    struct sockaddr_in	*masterAddrPtr;
{
    if (customs_Socket == -1) {
	Customs_Init();
    }
    lastStatus = Rpc_Call(customs_Socket, &customs_AgentAddr,
			  (Rpc_Proc)CUSTOMS_MASTER,
			  0, (Rpc_Opaque)0,
			  sizeof(struct sockaddr_in),
			  (Rpc_Opaque)masterAddrPtr,
			  CUSTOMS_NRETRY, &customs_RetryTimeOut);
    return(lastStatus);
}

/*-
 *-----------------------------------------------------------------------
 * Customs_MasterForHost --
 *	Find the location of the current master customs agent for
 *	another host.
 *
 * Results:
 *	The status of the call. If the call succeeds, the passed
 *	sockaddr_in is filled with address of the current master.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
Rpc_Stat
Customs_MasterForHost(hostAddrPtr, masterAddrPtr)
    struct sockaddr_in	*hostAddrPtr;
    struct sockaddr_in	*masterAddrPtr;
{
    if (customs_Socket == -1) {
	Customs_Init();
    }
    lastStatus = Rpc_Call(customs_Socket, hostAddrPtr,
			  (Rpc_Proc)CUSTOMS_MASTER,
			  0, (Rpc_Opaque)0,
			  sizeof(struct sockaddr_in),
			  (Rpc_Opaque)masterAddrPtr,
			  CUSTOMS_NRETRY, &customs_RetryTimeOut);
    return(lastStatus);
}

/*-
 *-----------------------------------------------------------------------
 * Customs_SetAvail --
 *	Change the availability criteria for the local machine.
 *
 * Results:
 *	The status of the call. If RPC_SUCCESS, criteria is overwritten
 *	with the current/new criteria and criteria->changeMask contains
 *	bits to indicate which, if any, values of the passed criteria
 *	were out-of-bounds.
 *
 * Side Effects:
 *	The criteria are changed if all are acceptable.
 *
 *-----------------------------------------------------------------------
 */
Rpc_Stat
Customs_SetAvail(criteria)
    Avail_Data	  *criteria;
{
    if (customs_Socket == -1) {
	Customs_Init();
    }
    lastStatus = Rpc_Call(customs_Socket, &customs_AgentAddr,
			  (Rpc_Proc)CUSTOMS_SETAVAIL,
			  sizeof(Avail_Data), (Rpc_Opaque)criteria,
			  sizeof(Avail_Data), (Rpc_Opaque)criteria,
			  CUSTOMS_NRETRY, &customs_RetryTimeOut);
    return(lastStatus);
}

/*-
 *-----------------------------------------------------------------------
 * Customs_Info --
 *	Acquire information about the registered hosts from the master
 *	agent at the given address.
 *
 * Results:
 *	The status of the call. If RPC_SUCCESS, the passed buffer is filled
 *	with information about the registered hosts.
 *
 * Side Effects:
 *	Not really.
 *
 *-----------------------------------------------------------------------
 */
Rpc_Stat
Customs_Info(masterAddrPtr, buf)
    struct sockaddr_in	*masterAddrPtr;
    char    	  	buf[MAX_INFO_SIZE];
{
    if (customs_Socket == -1) {
	Customs_Init();
    }
    lastStatus = Rpc_Call(customs_Socket, masterAddrPtr,
			  (Rpc_Proc)CUSTOMS_INFO,
			  0, (Rpc_Opaque)0,
			  MAX_INFO_SIZE, (Rpc_Opaque)buf,
			  CUSTOMS_NRETRY, &customs_RetryTimeOut);
    return(lastStatus);
}

/*-
 *-----------------------------------------------------------------------
 * Customs_PError --
 *	Print error message based on last call.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *  	A message is printed.
 *
 *-----------------------------------------------------------------------
 */
void
Customs_PError(msg)
    char    	  *msg;
{
    fprintf(stderr, "%s: %s\n", msg, Rpc_ErrorMessage(lastStatus));
}

/*-
 *-----------------------------------------------------------------------
 * Customs_MakeWayBill --
 *	Create a WayBill to be passed to the CUSTOMS_IMPORT function.
 *
 * Results:
 *	Length of the buffer.
 *
 * Side Effects:
 *	The passed buffer is overwritten.
 *
 *-----------------------------------------------------------------------
 */
int
Customs_MakeWayBill(permitPtr, cwd, file, argv, environ, port, buf)
    ExportPermit  	*permitPtr; /* Permit for the job */
    char    	  	*cwd;	    /* The current working directory */
    char    	  	*file;	    /* File to execute */
    char    	  	**argv;	    /* Arguments for it */
    char    	  	**environ;  /* Environment in which it should run */
    unsigned short	port;	    /* Port of udp socket for CUSTOMS_EXIT
				     * return RPC call */
    char    	  	*buf;	    /* Place to stuff the information */
{
    register char 	*cp;
    register int  	i;
    register WayBill	*wb;

    wb = (WayBill *)buf;

    /*
     * We compute a deadline for the remote side to start its process.
     * This is to prevent bogus timeouts that could lead to duplicate
     * exports.  To be on the safe side, the dead line is half the caller
     * side RPC timeout.
     */
#ifdef DOUBLECHECK_TIMEOUT
    time(&wb->deadline);
    wb->deadline += CUSTOMS_TIMEOUT/2;
#else
    wb->deadline = 0;
#endif

    /*
     * First the constant information:
     *	    - permit ID
     *	    - return call port
     *	    - real user id
     *	    - effective user id
     *	    - real group id
     *	    - effective group id
     *	    - array of groups process is in
     *	    - file creation mask
     *      - process priority
     *      - resource limits
     */
    wb->id = permitPtr->id;
    wb->port = port;
    wb->ruid = getuid();
    wb->euid = geteuid();
    wb->rgid = getgid();
    wb->egid = getegid();
    wb->ngroups = getgroups(sizeof(wb->groups) / sizeof(int), wb->groups);
    wb->umask = umask(0);
    (void) umask(wb->umask);
#ifdef SYSV
    wb->priority = nice(0);
#else
    wb->priority = getpriority(PRIO_PROCESS,0);
#endif
    /*
     * This could be simplified if we could rely on rlimits being numbered
     * the same in all OSes.
     */
#define INF_RLIMIT(index) \
	wb->rlimits[index].rlim_cur = RLIM_INFINITY; \
	wb->rlimits[index].rlim_max = RLIM_INFINITY
#define GET_RLIMIT(index, resource) \
    if (getrlimit(resource, &wb->rlimits[index]) < 0) { \
	INF_RLIMIT(index); \
    }
#ifdef RLIMIT_CPU
    GET_RLIMIT(0, RLIMIT_CPU);
#else
    INF_RLIMIT(0);
#endif
#ifdef RLIMIT_FSIZE
    GET_RLIMIT(1, RLIMIT_FSIZE);
#ifdef FSIZE_SCALE
    if (wb->rlimits[1].rlim_cur != RLIM_INFINITY)
	wb->rlimits[1].rlim_cur *= FSIZE_SCALE;
    if (wb->rlimits[1].rlim_max != RLIM_INFINITY)
	wb->rlimits[1].rlim_max *= FSIZE_SCALE;
#endif /* FSIZE_SCALE */
#else /* RLIMIT_FSIZE */
    INF_RLIMIT(1);
    {
	/*
	 * Emulate RLIMIT_FILE using System V ulimit(2).
	 */
        int ulim = ulimit(1,0) * 512;
	if (ulim >= 0 && ulim <= RLIM_INFINITY - 512)
	    wb->rlimits[1].rlim_cur = wb->rlimits[1].rlim_max = ulim;
    }
#endif /*RLIMIT_FSIZE */
#ifdef RLIMIT_DATA
    GET_RLIMIT(2, RLIMIT_DATA);
#else
    INF_RLIMIT(2);
#endif
#ifdef RLIMIT_STACK
    GET_RLIMIT(3, RLIMIT_STACK);
#else
    INF_RLIMIT(3);
#endif
#ifdef RLIMIT_CORE
    GET_RLIMIT(4, RLIMIT_CORE);
#else
    INF_RLIMIT(4);
#endif
#ifdef RLIMIT_RSS
    GET_RLIMIT(5, RLIMIT_RSS);
#else
    INF_RLIMIT(5);
#endif
#ifdef RLIMIT_NOFILE
    GET_RLIMIT(6, RLIMIT_NOFILE);
#else
    INF_RLIMIT(6);
#endif

    /*
     * Then the variable-length part:
     *	    - the absolute path of the current working directory
     *	    - the file to execute (needn't be absolute)
     *	    - the number of arguments (stored on a longword boundary)
     *	    - the argument strings
     *	    - the number of environment strings (stored on a 32-bit boundary)
     *	    - the environment strings themselves
     */
    cp = (char *)&wb[1];
    strcpy(cp, cwd);
    cp += strlen(cp);
    *cp++ = '\0';
    strcpy(cp, file);
    cp += strlen(file);
    *cp++ = '\0';
    cp = Customs_Align(cp, char *);

    for (i = 0; argv[i]; i++) {
	;
    }
    *(int *)cp = i;
    cp += sizeof(int);
    for (i = 0; argv[i]; i++) {
	strcpy(cp, argv[i]);
	cp += strlen(cp);
	*cp++ = '\0';
    }
    cp = Customs_Align(cp, char *);
    for (i = 0; environ[i]; i++) {
	;
    }
    *(int *)cp = i;
    cp += sizeof(int);
    for (i = 0; environ[i]; i++) {
	strcpy(cp, environ[i]);
	cp += strlen(cp);
	*cp++ = '\0';
    }
    return (cp - buf);
}

/*-
 *-----------------------------------------------------------------------
 * Customs_RawExport --
 *	Start a job running on another machine, but don't fork an
 *	"export" job to handle it -- just return the tcp socket open
 *	to the remote job, or -1 if the job could not be exported.
 *
 * Results:
 *	socket to remote job if ok. If < 0, value is:
 *	    -100    Couldn't find host
 *	    -101    Couldn't create return socket
 *	    -102    Couldn't get name of return socket
 *	    -104    Remote side refused import
 *	   <-200    -(result+200) gives the return status from the
 *		    CUSTOMS_IMPORT call.
 *	This is hokey, but was done quickly to provide debugging info in
 *	pmake.
 *
 * Side Effects:
 *
 *-----------------------------------------------------------------------
 */
int
Customs_RawExport(file, argv, cwd, flags, retSockPtr, permitPtr)
    char    	  *file;    	    /* File to exec */
    char    	  **argv;   	    /* Arguments to give it */
    char    	  *cwd;	    	    /* Current directory. NULL if not
				     * determined */
    int	    	  flags; 	    /* Flags to pass to Customs_Host */
    int	    	  *retSockPtr;	    /* Socket on which return call should be
				     * made when process exits. If < 0, will
				     * return a socket for the purpose. */
    ExportPermit  *permitPtr;	    /* OUT: permit returned by agent */
{
    extern char   	**environ;  /* Current process environment */
    int	    	  	sock;	    /* TCP socket connecting to exported
				     * process */
    struct sockaddr_in 	importServer;	/* Address of server running our
					 * process */
    char    	  	msg[128];   /* Error message from import server */
    Rpc_Stat	  	rstat;	    /* Return status from RPC calls */
    char    	  	buf[MAX_DATA_SIZE]; /* Buffer for outgoing waybill */
    int	    	  	buflen;	    /* Length of buffer */
    char    	  	loccwd[MAXPATHLEN]; /* Place to stuff cwd if not
					     * given */
    struct timeval	timeout;    /* Timeout for IMPORT request (since it's
				     * TCP, there's a different timeout than
				     * normal Customs calls) */
    u_short 	  	retPort;    /* Port number of return call socket for
				     * import server to return the process'
				     * exit status */
    Boolean 	  	ourSock;    /* True if we allocated the return call
				     * socket */
    ExportPermit  	locPermit;  /* Local permit if the caller isn't
				     * interested */

    if (permitPtr == (ExportPermit *)NULL) {
	permitPtr = &locPermit;
    }
    /*
     * Find out where we may go, first.
     */
    rstat = Customs_Host(flags, permitPtr);
    if (rstat != RPC_SUCCESS) {
	return (-(int)rstat);
    } else if (CUSTOMS_FAIL(&permitPtr->addr)) {
	return(CUSTOMS_NOEXPORT);
    }
    /*
     * We have somewhere to go. Now we need to set up the return-call
     * socket so we can pass its port number to the import server. If the
     * caller already has a socket in mind, ourSock is set False.
     */
    if (*retSockPtr < 0) {
	ourSock = True;
	*retSockPtr = Rpc_UdpCreate(True, 0);
	if (*retSockPtr < 0) {
	    return(CUSTOMS_NORETURN);
	}
    } else {
	ourSock = False;
    }
    
    /*
     * Figure out the port number. If this fails, we can't export...
     */
    buflen = sizeof(importServer);
    if (getsockname(*retSockPtr, (struct sockaddr *)&importServer,
							&buflen) < 0) {
	if (ourSock) {
	    (void) close(*retSockPtr);
	}
	return (CUSTOMS_NONAME);
    }
    retPort = importServer.sin_port;
    
    /*
     * Create the TCP socket for talking to the remote process and set up
     * the address of the remote server for doing the RPC
     */
    sock = Rpc_TcpCreate(False, 0);
    if (sock < 0) {
	if (ourSock) {
	    (void) close(*retSockPtr);
	}
	return(CUSTOMS_NOIOSOCK);
    }
    importServer.sin_family = AF_INET;
    importServer.sin_port = htons(tcpPort);
    importServer.sin_addr = permitPtr->addr;

    /*
     * If they haven't already figured out the current working directory,
     * we have to do it for them.
     */
    if (cwd == (char *)NULL) {
#ifdef SYSV
	if (!getcwd(loccwd, sizeof(loccwd)))
#else
	if (!getwd(loccwd))
#endif
		return(CUSTOMS_NOEXPORT);
	if (Customs_NormPath(loccwd, sizeof(loccwd)) < 0)
		return(CUSTOMS_NOEXPORT);
	cwd = loccwd;
    }

    /*
     * Using all this information, create a WayBill buffer to pass to
     * the server.
     */
    buflen = Customs_MakeWayBill(permitPtr, cwd, file, argv, environ,
				 retPort, buf);
    /*
     * Call the server. We only send one message, since TCP is "reliable".
     * If we don't get a response in CUSTOMS_TIMEOUT seconds, the export failed.
     */
    timeout.tv_sec = CUSTOMS_TIMEOUT;
    timeout.tv_usec = 0;
    rstat = Rpc_Call(sock, &importServer, (Rpc_Proc)CUSTOMS_IMPORT,
		     buflen, (Rpc_Opaque)buf,
		     sizeof(msg), (Rpc_Opaque)msg,
		     1, &timeout);
#ifdef DEBUG_TIMEOUT
    if (rstat == RPC_TIMEDOUT) {
	printf("CUSTOMS_IMPORT to %s timed out after %.2f secs (msg size = %d)\n",
		InetNtoA(*(struct in_addr *)&importServer.sin_addr),
		(float)(timeout.tv_sec + 0.0000001 * timeout.tv_usec),
		buflen);
    }
#endif
    lastStatus = rstat;
    /*
     * There are two ways an IMPORT call may fail -- if the server is down,
     * we'll get a RPC error code. If the server denies permission to export,
     * for some reason, it'll return some message other than "Ok".
     * In both cases, we clean up and return < 0 to indicate failure.
     */
    if (rstat != RPC_SUCCESS) {
	if (ourSock) {
	    (void)close(*retSockPtr);
	}
	(void)close(sock);
	return (-200-(int)rstat);
    } else if (strcmp(msg, "Ok") == 0) {
	return (sock);
    } else {
	fprintf(stderr, "CUSTOMS_IMPORT: %s\n", msg);
	if (ourSock) {
	    (void)close(*retSockPtr);
	}
	(void) close(sock);
	return (CUSTOMS_ERROR);
    }
}


/*-
 *-----------------------------------------------------------------------
 * Customs_NormPath --
 *	Normalize path so that it can be used all over the network.
 *	This function has to be customized according to the local
 *	mounting conventions.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The string passed is modied in place.
 *
 *-----------------------------------------------------------------------
 */
#define TMPMNT	"/tmp_mnt/"
#define NETMNT	"/n/"

#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

int
Customs_NormPath(path, pathlen)
char *path;		/* pointer to path buffer */
int pathlen;		/* size of path buffer */
{
#ifndef ORIG_CWD
	/* we fudge a uniform file name space here under the assumption
	 * we're running Sun automount with /n mounting
	 */
	if ( strncmp(path, TMPMNT, strlen(TMPMNT)) == 0 ) {
		/* strip tmp_mnt path */
		bcopy(path+sizeof(TMPMNT)-2, path,
			strlen(path+sizeof(TMPMNT)-2) + 1);
	}
	else if ( strncmp(path, NETMNT, strlen(NETMNT)) != 0 ) {
		/* prepend /n/hostname path */
		char hname[MAXHOSTNAMELEN];
		char *dot;

		if ( gethostname(hname, sizeof(hname)) == -1 ) {
			fprintf(stderr,"Customs_NormPath: gethostname failed\n" );
			return -1;
		}

		/* strip domain name */
		if ( (dot = index(hname, '.')) != NULL )
			*dot = '\0';

		if ( sizeof(NETMNT)+strlen(hname)+strlen(path) > pathlen ) {
			fprintf(stderr,"Customs_NormPath: normalized path too long\n" );
			return -1;
		}
		bcopy(path, path+sizeof(NETMNT)-1+strlen(hname),
			strlen(path)+1);
		bcopy(NETMNT, path, strlen(NETMNT));
		bcopy(hname, path+strlen(NETMNT), strlen(hname));
	}
#endif /* ! ORIG_CWD */
	return 0;
}
