/*-
 * export.c --
 *	Program to handle the interaction with the customs agent over
 *	an exported process. This should, perhaps, be generalized so
 *	other programs can use it, but...maybe a library...
 *
 * Copyright (c) 1988, 1989 by the Regents of the University of California
 * Copyright (c) 1988, 1989 by Adam de Boor
 * Copyright (c) 1989 by Berkeley Softworks
 *
 * Permission to use, copy, modify, and distribute this
 * software and its documentation for any non-commercial purpose
 * and without fee is hereby granted, provided that the above copyright
 * notice appears in all copies.  The University of California,
 * Berkeley Softworks and Adam de Boor make no representations about
 * the suitability of this software for any purpose.  It is provided
 * "as is" without express or implied warranty.
 */
#ifndef lint
static char *rcsid =
"$Id: export.c,v 1.19 1992/07/31 00:12:31 stolcke Exp $ ICSI (Berkeley)";
#endif lint

#include    <strings.h>
#include    <sys/time.h>
#include    <signal.h>
#include    <stdio.h>
#include    <sys/wait.h>
#include    <sys/ioctl.h>
#include    <errno.h>
extern int errno;
#include    <netdb.h>
#include    <sys/file.h>

#include    "customs.h"

int	    	  	rmt;  	/* Socket to remote process */
int	    	  	ret;   	/* Socket server will call us back on with
				 * exit status */
u_long	    	  	id;    	/* Permit id under which the process is
				 * running */
struct sockaddr_in	server;	/* Address of server running the process */
extern int  	  	customs_Socket;
extern struct timeval	customs_RetryTimeOut;

union wait  	  	status;
int			exitSeen = 0;
void	    	  	DoExit();

/*-
 *-----------------------------------------------------------------------
 * SendSig --
 *	Pass a signal on to a remote process.  This the handler for an
 *	event scheduled by the local signal handler.
 *
 * Results:
 *	False.
 *
 * Side Effects:
 *	An RPC call is made to the import server to deliver the signal.
 *
 *-----------------------------------------------------------------------
 */
Boolean
SendSig(signo)
    int	    signo;	/* The signal number we've received */
{
    Kill_Data	  packet;
    Rpc_Stat	  rpcstat;
    
    packet.id = id;
    packet.signo = signo;

    rpcstat = Rpc_Call (ret, &server, (Rpc_Proc)CUSTOMS_KILL,
		       sizeof(packet), (Rpc_Opaque)&packet,
		       0, (Rpc_Opaque)0,
		       CUSTOMS_NRETRY, &customs_RetryTimeOut);
    if (rpcstat != RPC_SUCCESS) {
	fprintf(stderr, "Customs_Kill(%d): %s\n",
	        signo, Rpc_ErrorMessage(rpcstat));
	if (!exitSeen) {
	    /*
	     * Fake error exit code.
	     */
	    status.w_status = 0;
	    status.w_retcode = 1;
	}
	DoExit();
    }
    return (False);
}

/*-
 *-----------------------------------------------------------------------
 * PassSig --
 *	Catch a signal and pass it along. We only kill ourselves with the
 *	signal when we receive the exit status of the remote process.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	An event is scheduled to forward the signal to the remote process.
 *	An RPC call is made to the import server to deliver the signal.
 *
 *-----------------------------------------------------------------------
 */
static SIGRET
PassSig(signo)
    int	    signo;	/* The signal number we've received */
{
    struct timeval timeout;

    timeout.tv_sec = 0;
    timeout.tv_usec = 0;
    /*
     * We defer the actual signal forwarding to an RPC event to avoid
     * activating Rpc_Wait() within a signal handler.  This could lead
     * to all kinds of non-reentrant procedures to be called.
     */
    (void)Rpc_EventOnce(&timeout, SendSig, (Rpc_Opaque)signo);
}

/*-
 *-----------------------------------------------------------------------
 * Drain --
 *	Wait for the remote socket to become writable again, handling
 *	any output from the remote side in the mean time.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	If 'what' is RPC_WRITABLE, Transfer is reinstalled as the stream
 *	server for stdin...
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
void
Drain(stream, arg, what)
    int	    	  stream;
    Rpc_Opaque 	  arg;
    int	    	  what;
{
    extern void	  Transfer();
    
    if (what & RPC_READABLE) {
	/*
	 * Transfer any data from remote side
	 */
	Transfer(rmt, 1);
    }
    if (what & RPC_WRITABLE) {
	/*
	 * Socket has drained enough, reinstall the regular handlers
	 * for both streams
	 */
	Rpc_Watch(rmt, RPC_READABLE, Transfer, (Rpc_Opaque)1);
	Rpc_Watch(0, RPC_READABLE, Transfer, (Rpc_Opaque)rmt);
    }
}

/*-
 *-----------------------------------------------------------------------
 * Transfer --
 *	Transfer data from one source to another.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	Data are read from source and written to dest.
 *
 *-----------------------------------------------------------------------
 */
void
Transfer (source, dest)
    int	    source;
    int	    dest;
{
    char    buf[BUFSIZ];
    int	    cc, cw;

    do {
	cc = read(source, buf, sizeof(buf));
    }
    while ((cc < 0) && (errno == EINTR));
    if (cc < 0) {
	perror ("read");
	fprintf(stderr, "source = %d, dest = %d\n", source, dest);
	return;
    } else if (cc == 0) {
	if (source == 0) {
	    /*
	     * When we reach the end-of-file for our input, we want the remote
	     * process to reach that state, too, so we half-shutdown our socket
	     * to it.
	     */
	    if (shutdown(rmt, 1) < 0) {
		perror("shutdown");
		exit(3);
	    }
	} else if (exitSeen) {
	    /*
	     * We've gotten an EOF on the socket and customs has already sent
	     * us an exit signal, so perform the exit now.
	     */
	    DoExit();
	}
	Rpc_Ignore(source);
    } else {
	do {
	    cw = write (dest, buf, cc);
	}
	while ((cw < 0) && (errno == EINTR));
	if (cw != cc) {
	    if (errno != EWOULDBLOCK) {
		if (errno == EPIPE) {
		    if (dest == rmt) {
			/*
			 * Connection to remote side was lost. This means that
			 * both the process and the server died, so there's
			 * no point in waiting for the exit status...
			 */
			fprintf(stderr, "*** connection closed.\n");
			if (exitSeen) {
			    DoExit();
			}
		    }
		}
		exit(2);
	    } else {
		/*
		 * If we can't write because it'd block, we must be transfering
		 * from local to remote. In such a case, we ignore further input
		 * from stdin, and wait for the output socket to become writable.
		 * Drain() will reset the handler for 0.
		 */
		Rpc_Ignore(0);
		Rpc_Watch(rmt, RPC_READABLE|RPC_WRITABLE, Drain, (Rpc_Opaque)0);
	    }
	}
    }
}
/*ARGSUSED*/
void
SwapExit(length, data)
    int	    	  length;
    Exit_Data	  *data;
{
    Rpc_SwapLong(sizeof(long), &data->id);
    /*
     * Exit status is not swapped since bit fields are byte-order
     * independent.
     */
}


/*-
 *-----------------------------------------------------------------------
 * Exit --
 *	Handle CUSTOMS_EXIT call from import server. This process doesn't
 *	actually exit until we get an end-of-file on the socket to the
 *	remote side. This allows any error message from the customs agent
 *	to be printed before we exit.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	exitSeen is set true and status is set to the returned status.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
void
Exit(from, msg, len, data)
    struct sockaddr_in	*from;
    Rpc_Message	  	msg;
    int	    	  	len;
    Rpc_Opaque 	  	data;
{
    int	    	  	nb;
    Exit_Data	  	*eVal = (Exit_Data *)data;
    
    status.w_status = eVal->status;
    exitSeen = 1;
    Rpc_Return(msg, 0, (Rpc_Opaque)0);

    while ((ioctl(rmt, FIONREAD, &nb) == 0) && (nb > 0)) {
#ifdef notdef
	fprintf(stderr, "Exit: %d bytes remaining\n", nb);
	fflush(stderr);
#endif /* notdef */
	Transfer(rmt, 1);
    }
    DoExit();
}


/*-
 *-----------------------------------------------------------------------
 * DoExit --
 *	Exit in the same way the remote process did.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The process will change state, either stopping, dying horribly
 *	or just exiting cleanly.
 *
 *-----------------------------------------------------------------------
 */
void
DoExit()
{
    /*
     * Indicate exit is handled to avoid duplication.
     */
    exitSeen = False;

    if (WIFSTOPPED(status)) {
	int oldmask;
	
	signal (status.w_stopsig, SIG_DFL);
	oldmask = sigsetmask(0);
	kill (getpid(), status.w_stopsig);
	(void)sigsetmask(oldmask);
	signal (status.w_stopsig, PassSig);
    } else if (WIFSIGNALED(status)) {
	if (status.w_coredump) {
	    /*
	     * We don't want our core dump messing up the other one,
	     * so we change to a (we hope) non-writable directory before
	     * commiting suicide.
	     */
	    chdir ("/");
	}
	signal (status.w_termsig, SIG_DFL);
	kill (getpid(), status.w_termsig);
    } else {
	exit (status.w_retcode);
    }
}

    
/*-
 *-----------------------------------------------------------------------
 * main --
 *	Usage:
 *	    export -id connection-fd return-fd id
 *	    export <command>
 *
 *	In the first form, the idea is the exporting program will have
 *	contacted the customs agent already and told it what to do. This
 *	program then simply shuffles I/O and passes signals and the exit
 *	status along...
 *
 *	In the second form, this will
 *
 * Results:
 *	The exit status of the remote process.
 *
 * Side Effects:
 *	Well...
 *
 *-----------------------------------------------------------------------
 */
main (argc, argv)
    int	    argc; 	/* Number of arguments */
    char    **argv;	/* The arguments themselves */
{
    ExportPermit  	permit;
    int	    	  	raLen;
    struct servent  	*sep;
    SIGRET		(*oldhandler)();

    if (argc < 2) {
	fprintf(stderr, "Usage:\n\t%s -id <connection-fd> <return-fd> <id>\nor",
		argv[0]);
	fprintf(stderr, "\t%s <command>\n", argv[0]);
	exit(1);
    }

    if (strcmp (argv[1], "-id") == 0) {
	(void)setuid(getuid());
	if (argc < 5) {
	    fprintf(stderr, "Usage:\n\t%s -id <connection-fd> <return-fd> <id>\n",
		    argv[0]);
	    exit(1);
	}
	rmt = atoi (argv[2]);
	ret = atoi(argv[3]);
	(void)sscanf(argv[4], "%x", &id);
	permit.id = 0;
    } else {
	Customs_Init();
	(void)setuid(getuid());
	rmt = Customs_RawExport(argv[1], &argv[1], (char *)NULL, 0,
				&customs_Socket, &permit);
	if (rmt < 0) {
	    fprintf(stderr,"%s: could not export command\n", argv[0]);
	    fflush(stderr);
	    execvp(argv[1], &argv[1]);
	    perror(argv[1]);
	    exit(3);
	} else {
	    ret = customs_Socket;
	    id = permit.id;
	}
    }

    /*
     * Install RPC server for the remote server to return the exit status
     * of the process.
     */
    Rpc_ServerCreate(ret, (Rpc_Proc)CUSTOMS_EXIT, Exit,
		     SwapExit, Rpc_SwapNull, (Rpc_Opaque)0);
    
    /*
     * Print startup messages regardless of tostop
     */
    oldhandler = signal (SIGTTOU, SIG_IGN);

    raLen = sizeof(server);
    if (getpeername (rmt, (struct sockaddr *)&server, &raLen) < 0) {
	perror ("getpeername");
	exit(2);
    }

    if (permit.id != 0) {
	/*
	 * Only do this if we did the exportation ourselves
	 */
	struct hostent *he;
	
	he = gethostbyaddr (&server.sin_addr, sizeof(server.sin_addr),
			    AF_INET);
	if (he == (struct hostent *)NULL) {
	    fprintf(stderr, "Connected to unknown host?\n");
	} else {
	    fprintf(stderr, "*** exported to %s\n", he->h_name);
	    fflush(stderr);
	    strcpy (argv[1], he->h_name);
	}
    }

    sep = getservbyname(CUSTOMS_SERVICE_NAME, "udp");
    if (sep == NULL) {
	fprintf(stderr, "%s/udp unknown\n", CUSTOMS_SERVICE_NAME);
	fprintf(stderr, "trying default port %d\n", DEF_CUSTOMS_UDP_PORT);
    }
    server.sin_port = sep ? sep->s_port : htons(DEF_CUSTOMS_UDP_PORT);

    signal (SIGTTOU, oldhandler);

    signal (SIGHUP, PassSig);
    signal (SIGINT, PassSig);
    signal (SIGQUIT, PassSig);
    signal (SIGTERM, PassSig);
    signal (SIGTSTP, PassSig);
    signal (SIGCONT, PassSig);
    /*
     * SIGTTIN and SIGTTOU are best handled locally by default action
     * and not forwarded to the remote process.  This is because we have no
     * good way of restarting the read/write calls in Tranfer().
     */
#ifdef SIGWINCH
    signal (SIGWINCH, PassSig);
#endif
#ifdef SIGWINDOW
    signal (SIGWINDOW, PassSig);
#endif
#ifdef SIGXCPU
    signal (SIGXCPU, PassSig);
#endif
    signal (SIGUSR1, PassSig);
    signal (SIGUSR2, PassSig);

    signal (SIGPIPE, SIG_IGN);

    /*
     * We want to avoid I/O deadlock, so place the remote socket into
     * non-blocking mode, allowing us to delay transfering our own input until
     * the remote side can handle it while still accepting its output.
     */
    fcntl(rmt, F_SETFL, FNDELAY);

    /*
     * Install Drain as the initial stream server for rmt to make
     * sure it's writable before bothering to read anything from 0...
     */
    Rpc_Watch(rmt, RPC_READABLE|RPC_WRITABLE, Drain, (Rpc_Opaque)0);
    Rpc_Run();
}
