
/****************************************************************
 *
 * QRT2GIF - QRT Ray Trace Image to GIF Conversion
 *
 * Written by David Rowley, Copyright (C) 1989 by David Rowley
 * Michael's copyright message goes ditto for me...
 *
 * NOTE:  This utility assumes that the .RAW file was written out
 *        in INTEL form (on both INTEL and NON-INTEL machines).
 *        QRT writes the xres and yres to the file in this format
 *        always, but the format of the line number in each scan line
 *        was machine dependant, so I modified my version of QRT to
 *        ALWAYS write the line number in Intel format (low byte, 
 *        high byte).  This allows me to ship .RAW files from
 *        machine to machine without worrying about in which format they
 *        were created.
 *
 * Many thanks to Steve Koren, for writing QRT and giving us something
 * to play with, and to Michael Mauldin, for his Heckbert Median Cut
 * code. (The GIF stuff is mine :-))
 *
 * Derived from:
 *
 * fbquant.c: FBM Library 0.83 (Alpha Test)  22-Feb-89  Michael Mauldin
 *
 * Copyright (C) 1989 by Michael Mauldin.  Permission is granted to
 * use this file in whole or in part provided that you do not sell it
 * for profit and that this copyright notice is retained unchanged.
 *
 * fbquant: Convert an RGB color image to mapped color format (color
 *	    quantization step).  Floyd-Steinberg dithering is used
 *	    to reduce color banding.  The quantization used is a
 *	    modification of Heckbert's median cut.
 *
 ****************************************************************/

# include <stdio.h>
#include "bitmap.h"
extern char *malloc();

#ifdef ATARI

/* Need lots of stack -- change to suit your compiler */
long _stksize = 50000L;
/* define RINDEX as one of 'rindex' or 'strrchr', whichever your compiler
 * supports */
#define RINDEX rindex

#else

#define RINDEX strrchr
unsigned _STACK = 50000;

#endif

int cmp_red(), cmp_grn(), cmp_blu(), cmp_cmap();

# define RD 0
# define GR 1
# define BL 2

# define MAXSHRT 32767

/* Masks used to manipulate RGB values */
unsigned int REDMASK;
unsigned int REDSHFT;
unsigned int GRNMASK;
unsigned int GRNSHFT;
unsigned int BLUMASK;
unsigned int BLUSHFT;
unsigned int CUBITS;
unsigned int CUBIGN;
unsigned int CUBSIZ;

/* HARD - CODED  --  Change these values if you want greater color
 * resolution (ie. more bits of RGB used in the sampling)
 */
#define CUBSID 16	/* 2 ^ max( 4, 4, 4 ) */
#define CUBSIZ 4096	/* 2 ^ (4 + 4 + 4) */

char *
safe_malloc( s )
unsigned int s;
{
	char *ptr;

	ptr = (char *)malloc( s );
	if( ptr == (char *)0 ) {
		printf( "error: allocating memory\n" );
		exit(1);
	}
	return ptr;
}

/*
 * Set up the masks for the given bits of R, G and B
 */
calc_bits( r, g, b )
int r, g, b;
{
	r = g = b = 4;

	BLUSHFT = 0;
	GRNSHFT = g;
	REDSHFT = g+b;

	BLUMASK = (1 << b) - 1;
	GRNMASK = (1 << g) - 1;
	GRNMASK <<= b;
	REDMASK = (1 << r) - 1;
	REDMASK <<= (b+g);

	CUBITS = r;
	CUBIGN = 8 - CUBITS;

}

# define GETR(X) (((X) & REDMASK) >> REDSHFT)
# define GETG(X) (((X) & GRNMASK) >> GRNSHFT)
# define GETB(X)  ((X) & BLUMASK)

# define CLRINDEX(R,G,B)			\
	(((R) << REDSHFT) & REDMASK |		\
	 ((G) << GRNSHFT) & GRNMASK |		\
	 ((B)  & BLUMASK))

# define CLRINDEX8(R,G,B)			\
	(((R) << (REDSHFT-CUBIGN)) & REDMASK |	\
	 ((G) << (GRNSHFT-CUBIGN)) & GRNMASK |	\
	 ((B) >> (CUBIGN))  & BLUMASK)

# define GETR8(X) (((X) & REDMASK) >> (REDSHFT-CUBIGN))
# define GETG8(X) (((X) & GRNMASK) >> (GRNSHFT-CUBIGN))
# define GETB8(X) (((X) & BLUMASK) << CUBIGN)

typedef struct cstruct {
	unsigned char rd, gr, bl, indx;
} COLOR;

COLOR *cmap = NULL;

typedef struct pix_struct {
	short cnt;
	short color;
} PIXEL;

int debug=0, verbose=0, colors=256, showcolor=0;

unsigned int ReadInt( fp )
FILE *fp;
{
	unsigned char low;
	unsigned char high;

	low = (unsigned char)(fgetc(fp));
	high = (unsigned char)(fgetc(fp));
	return high * 256 + low;
}

/****************************************************************
 * main
 ****************************************************************/

main (argc, argv)
char *argv[];
{
  int *hist[CUBSIZ];
  char *str;
  char *fname;
  int   floyd = 0;
  int   have_bw = 0;
  int   interlace = 0;
  int   xscale = 1, yscale = 1;

  printf( "\n" );
  printf( "QRT2GIF v1.0  (c) March 1st, 1989 by David Rowley\n" );
  printf( "12 Bit RGB image conversion using modified Heckbert median cut\n" );
  printf( "for use with the QRT ray tracing package (v1.4)\n" );
  printf( "\n" );
  
  calc_bits( 4, 4, 4 );

  if( argc <= 1 ) {
	printf( "USAGE: qrt2gif [options] input.raw\n" );
	printf( "\n" );
	printf( "   -cNNN      produce a GIF image with NNN colors\n" );
	printf( "              (default is 256)\n" );
	printf( "\n" );
	printf( "   -b         Ensure that pure Black and White are\n" );
	printf( "              in the palette used (sometimes useful)\n" );
	printf( "              For example, use -c2 -b -f to create\n" );
	printf( "              a black and white dithered image\n" );
	printf( "\n" );
	printf( "   -xNNN      scale the width up by a factor of NNN\n" );
	printf( "   -yNNN      scale the height up by a factor of NNN\n" );
	printf( "\n" );
	printf( "   -f         use floyd-steinberg dithering\n" );
	printf( "              (default is to just map to closest color\n" );
	printf( "\n" );
	printf( "   -i         make the GIF image interlaced\n" );
	printf( "\n" );
	exit(1);
  }
  	
  /* Get the options */
  while ( argc > 1 ) {
     str = argv[1];
     if( str[0] == '-' ) {
	switch( str[1] ) {
		case 'c':
		case 'C':
			colors = atoi( str+2 );
			break;
		case 'f':
		case 'F':
			floyd = 1;
			break;
		case 'b':
		case 'B':
			have_bw = 1;
			break;
		case 'i':
		case 'I':
			interlace = 1;
			break;
		case 'x':
		case 'X':
			xscale = atoi( str+2 );
			break;
		case 'y':
		case 'Y':
			yscale = atoi( str+2 );
			break;

	}
     } else
	fname = argv[1];
     argv++;
     argc--;
  }

  /* Allocate space for color map */
  cmap = (COLOR *) safe_malloc ((unsigned) colors * sizeof (COLOR));

  /* Build a histogram of color distribution from the input */
  sample_image (fname, hist);

  /* Select 'colors' different colors for the colormap */  
  build_colormap (hist, cmap, colors, have_bw);
  
  /* Use Floyd-Steinberg error dispersion to quantize using the new cmap */
  clr_quantize ( fname, cmap, colors, floyd, interlace, xscale, yscale);
  
}

/****************************************************************
 * sample_image:
 ****************************************************************/

unsigned char r_vals[1024], g_vals[1024], b_vals[1024];

sample_image (fname, hist)
char *fname;
int *hist;
{ register int i;
  FILE *fp;
  int xres, yres;
  long used = 0;
  int lineno;
 
   fp = fopen( fname, "rb" );
   
   if( fp == (FILE *)0 )
   	printf( "error: could not open %s\n", fname ), exit(1);
   
   xres = ReadInt( fp );
   yres = ReadInt( fp );
   
  /* Clear the histogram */
  for (i=0; i<CUBSIZ; i++) hist[i] = 0;

   printf( "Sampling Bitmap (Resolution: %d by %d)\n", xres, yres );
   
   for( ;; ) {
   	lineno = ReadInt( fp );
   	if( lineno < 0 || lineno > yres )
   		break;
   
   	fread( r_vals, 1, xres, fp );
   	fread( g_vals, 1, xres, fp );
   	fread( b_vals, 1, xres, fp );
   
   	for( i=0; i<xres; i++ ) {
                if (++hist[ CLRINDEX8 (r_vals[i]*4,g_vals[i]*4,b_vals[i]*4) ] 
			== 1)
			 used++;
   	}
   
   	if( feof( fp ) )
   		break;
   }
   
   fclose( fp );
   
}

/****************************************************************
 * build_colormap:
 ****************************************************************/

build_colormap (hist, cmap, colors, have_bw)
int *hist;
COLOR *cmap;
int colors;
int have_bw;
{ register int i, k;
  static PIXEL box[CUBSIZ];
  register PIXEL *b;
  int used=0;
  int cols_to_pick;

  /*
   * If we want to include pure black and white in the palette, then
   * we only have to choose 'colors' - 2.
   */
  if( have_bw )
  	cols_to_pick = colors - 2;
  else
  	cols_to_pick = colors;

  /* Build the first box, encompassing all pixels */  
  for (b=box, i=0; i<CUBSIZ; i++)
  { b->color = i;
    k = hist[i];
    b->cnt = (k > MAXSHRT) ? MAXSHRT : k;
    b++;
  }
  
  /* Move all non-zero count colors to the front of the list */
  for (i=0, used=0; i<CUBSIZ; i++)
  { if (box[i].cnt > 0) box[used++] = box[i]; }

  printf( "%d colors used out of a possible %ld\n", (int)used, (long)CUBSIZ );

  /*-------- Special case if we didnt need all colors --------*/
  if (used <= cols_to_pick)
  {
    /* Copy the colors actually found */
    printf( "since only %d colors were used - no need to reduce palette\n", 
    			(int)used );
    for (i=0; i<used; i++)
    { cmap[i].rd = GETR8 (box[i].color);
      cmap[i].gr = GETG8 (box[i].color);
      cmap[i].bl = GETB8 (box[i].color);
    }

    /* Set the rest to WHITE */
    for (; i<cols_to_pick; i++)
    { cmap[i].rd = 255;
      cmap[i].gr = 255;
      cmap[i].bl = 255;
    }
  }
  
  /*-------- Recursively split the color space and assign colors --------*/
  else
  { split_box (box, used, 0, cols_to_pick, cmap); }
  
  /*----------------------------------------------------------------
   * Now arrange the colors in the desired order.  Sun convention is that
   * color 0 is white and color n-1 is black.  We put the rest of the map
   * in lexicographically sorted r,g,b order from 1 to n-2.
   */

  /* Now sort 1..n-2 according to desired ordering */
  qsort (cmap, cols_to_pick, sizeof (* cmap), cmp_cmap);

  /* Make first color white and last color black */
  if( have_bw ) {
  	cmap[colors-2].rd = cmap[colors-2].gr = cmap[colors-2].bl = 255;
  	cmap[colors-1].rd = 0;
	cmap[colors-1].gr = 0;
  	cmap[colors-1].bl = 0;
  }

  /* Set the output indices */
  for (i=0; i<colors; i++) { cmap[i].indx = i; }
}

/****************************************************************
 * split_box: Basic recursive part of Heckberts adaptive partitioning
 *	      algorithm.
 ****************************************************************/

split_box (box, boxlen, clr, numclr, cmap)
PIXEL *box;
int boxlen, clr, numclr;
COLOR *cmap;
{ int maxv[3], minv[3], numv[3];
  int pcnt[3][CUBSID];
  int sbox, snum, split, half, maxdif, dif;
  register PIXEL *top, *bot;
  int topw, botw;
  int red, grn, blu;
  register int i, c;

  /* If numclr exceeds boxlen, we are in trouble */
  if (numclr > boxlen)
  { fprintf (stderr, "boxlen %d is less numclr %d, panic!\n than",
	     boxlen, numclr);
    fflush (stderr);
    abort ();
  }

  /* Base case: only one color, assign the average for this cell */
  if (numclr <= 1)
  { red = box_avg_red (box, boxlen);
    grn = box_avg_grn (box, boxlen);
    blu = box_avg_blu (box, boxlen);
    
    /* Map x to x+4, because the histogram maps values to multiples of 8 */
    cmap[clr].rd = red + 4;
    cmap[clr].gr = grn + 4;
    cmap[clr].bl = blu + 4;
    
    if (debug)
    { fprintf (stderr, "\t\tassigning color %d  <%d,%d,%d>  (%d)\n",
	       clr, cmap[clr].rd, cmap[clr].gr, cmap[clr].bl,
	       box_weight (box, boxlen));
    }
    
    return;
  }

  /* Gather statistics about the boxes contents */
  minv[RD] = minv[GR] = minv[BL] = CUBSID;
  maxv[RD] = maxv[GR] = maxv[BL] = 0;
  numv[RD] = numv[GR] = numv[BL] = 0;
  for (i=0; i<CUBSID; i++) { pcnt[RD][i] = pcnt[GR][i] = pcnt[BL][i] = 0; }
  
  for (i=0; i<boxlen; i++)
  { c = box[i].color;
    red = GETR(c); grn = GETG(c); blu = GETB(c);
    
    if (red < minv[RD]) minv[RD] = red;
    if (red > maxv[RD]) maxv[RD] = red;
    if (grn < minv[GR]) minv[GR] = grn;
    if (grn > maxv[GR]) maxv[GR] = grn;
    if (blu < minv[BL]) minv[BL] = blu;
    if (blu > maxv[BL]) maxv[BL] = blu;
    
    if (++pcnt[RD][red] == 1) numv[RD]++;
    if (++pcnt[GR][grn] == 1) numv[GR]++;
    if (++pcnt[BL][blu] == 1) numv[BL]++;
  }

  /* Special case, boxlen = numclr, just assign each box one color */
  if (boxlen == numclr)
  { for (i=0; i<boxlen; i++)
    { split_box (box+i, 1, clr+i, 1, cmap); }
    return;
  }

  /* Pick a dimension to split */
  split = -1; maxdif = -1;

  if ((dif = (maxv[RD] - minv[RD])) > maxdif) { maxdif = dif; split = RD; }
  if ((dif = (maxv[GR] - minv[GR])) > maxdif) { maxdif = dif; split = GR; }
  if ((dif = (maxv[BL] - minv[BL])) > maxdif) { maxdif = dif; split = BL; }
  
  /* Sort along the chosen dimension */
  switch (split)
  { case RD:	qsort (box, boxlen, sizeof (*box), cmp_red); break;
    case GR:	qsort (box, boxlen, sizeof (*box), cmp_grn); break;
    case BL:	qsort (box, boxlen, sizeof (*box), cmp_blu); break;
    default:	fprintf (stderr, "panic in split_box, split = -1\n");
		fflush (stderr); fflush (stdout); abort ();
  }
  
  /* 
   * Split at the median, but make sure there are at least numclr/2
   * different colors on each side of the split, to avoid wasting
   * colors.
   *
   * Note: need to keep in mind that when the box is large, dont split
   *       too close to one edge.
   */
   
  half = numclr / 2;
  top = box;		bot = box + (boxlen-1);
  topw = top->cnt;	botw = bot->cnt;
  
  /* Set top and bot to point to min/max feasible splits */
  while ((top-box)+1 < half)		{ top++; topw += top->cnt; }
  while ((boxlen-(bot-box)) < half)	{ bot--; botw += bot->cnt; }

  /* Move top and bottom towards each other 1/8 of remaining distance */
  c = (bot-top) / 8;
  for (i=0; i<c; i++)			{ top++; topw += top->cnt; }
  for (i=0; i<c; i++)			{ bot--; botw += bot->cnt; }

  /* Now search for median */
  while (top < bot)
  { if (topw < botw)			{ top++; topw += top->cnt; }
    else				{ bot--; botw += bot->cnt; }
  }

  /* Decide which half gets the midpoint */
  if (topw > botw)			/* Median wants to go with top */
  { sbox = (top-box) + 1;
    snum = numclr - half;
  }
  else					/* Median wants to go with bottom */
  { sbox = (top - box);
    snum = half;
  }
  
  /* Handle boundary conditions with number of colors vs box size */
  if (sbox == 0) sbox++;
  else if (sbox == boxlen) sbox--;

  while (snum > sbox) snum--;
  while (numclr-snum > boxlen-sbox) snum++;

# ifndef OPTIMIZE
  /* Check for boundary condition errors */
  if (snum <= 0 || snum >= numclr)
  { fprintf (stderr, "panic, using zero colors for box\n");
    fflush (stderr);
    abort ();
  }

  if (boxlen-sbox < numclr-snum)
  { fprintf (stderr, "panic, about to used %d boxes for %d colors\n",
	     boxlen-sbox, numclr-snum);
    fflush (stderr);
    abort ();
  }

  if (sbox < snum)
  { fprintf (stderr, "panic, about to used %d boxes for %d colors\n",
	     sbox, snum);
    fflush (stderr);
    abort ();
  }
# endif

  if (debug)
  { int count = numclr, depth = 8;
    while (count > 0) { depth--; count /= 2; }
    for (i=0; i<depth; i++) fprintf (stderr, "  ");
    
    fprintf (stderr, "box [%d..%d|%d] r(%d,%d,%d) g(%d,%d,%d) b(%d,%d,%d) =>",
	     0, boxlen-1, numclr,
	     minv[RD], maxv[RD], numv[RD],
	     minv[GR], maxv[GR], numv[GR],
	     minv[BL], maxv[BL], numv[BL]);
    fprintf (stderr, " %c [%d..%d|%d] [%d..%d|%d]\n",
	     "RGB"[split], 0, sbox-1, snum, sbox, boxlen-1, numclr-snum);
  }

  /* Now recurse and split each sub-box */
  split_box (box,      sbox,          clr,      snum,        cmap);
  split_box (box+sbox, boxlen - sbox, clr+snum, numclr-snum, cmap);
}

/****************************************************************
 * box_weight: Determine the total count of a box
 ****************************************************************/

box_weight (box, boxlen)
register PIXEL *box;
register int boxlen;
{ register int sum = 0, i;

  for (i=0; i<boxlen; i++)
  { sum += box[i].cnt; }
  
  return (sum);
}

/****************************************************************
 * box_avg_red: Determine the average red value [0..255] of a box
 ****************************************************************/

box_avg_red (box, boxlen)
register PIXEL *box;
register int boxlen;
{ register int sum = 0, n = 0, i;

  for (i=0; i<boxlen; i++)
  { sum += GETR8(box[i].color); n++; }
  
  return (sum / n);
}

/****************************************************************
 * box_avg_grn: Determine the average grn value [0..255] of a box
 ****************************************************************/

box_avg_grn (box, boxlen)
register PIXEL *box;
register int boxlen;
{ register int sum = 0, n = 0, i;

  for (i=0; i<boxlen; i++)
  { sum += GETG8(box[i].color); n++; }
  
  return (sum / n);
}

/****************************************************************
 * box_avg_blu: Determine the average blu value [0..255] of a box
 ****************************************************************/

box_avg_blu (box, boxlen)
register PIXEL *box;
register int boxlen;
{ register int sum = 0, n = 0, i;

  for (i=0; i<boxlen; i++)
  { sum += GETB8(box[i].color); n++; }
  
  return (sum / n);
}


/****************************************************************
 * sort by increasing red ( then green and blue )
 ****************************************************************/

cmp_red (a, b)
PIXEL *a, *b;
{ register r;

  if (r = GETR(a->color) - GETR(b->color))
  { return (r); }
  
  if (r = GETG(a->color) - GETG(b->color))
  { return (r); }

  if (r = GETB(a->color) - GETB(b->color))
  { return (r); }
  
  return (0);
}

/****************************************************************
 * sort by increasing green ( then blue and red )
 ****************************************************************/

cmp_grn (a, b)
PIXEL *a, *b;
{ register r;

  if (r = GETG(a->color) - GETG(b->color))
  { return (r); }

  if (r = GETB(a->color) - GETB(b->color))
  { return (r); }
  
  if (r = GETR(a->color) - GETR(b->color))
  { return (r); }
  
  return (0);
}

/****************************************************************
 * sort by increasing blue ( then red and green )
 ****************************************************************/

cmp_blu (a, b)
PIXEL *a, *b;
{ register r;

  if (r = GETB(a->color) - GETB(b->color))
  { return (r); }
  
  if (r = GETR(a->color) - GETR(b->color))
  { return (r); }
  
  if (r = GETG(a->color) - GETG(b->color))
  { return (r); }

  return (0);
}

/****************************************************************
 * clr_quantize: Do Floyd Steinberg quantizing on the image
 ****************************************************************/

clr_quantize (fname, cmap, colors, floyd, interlace, xscale, yscale )
char *fname;
COLOR *cmap;
int colors;
int floyd;
int interlace;
int xscale, yscale;
{
  register int i, j;
  Bitmap *bit;
  int xres, yres;
  int ii, jj;
  FILE *fp;
  int idx;
  int lineno;
  int **cerr;
  int **lerr;
  int rd, gr, bl;
  int rderr, grerr, blerr;
  int **terr;
  char gif_name[80];
  extern char *RINDEX();
  char *ptr;
  
	strcpy( gif_name, fname );
	ptr = RINDEX( gif_name, '.' );
	if( ptr )
		strcpy( ptr, ".gif" );
	else
		strcat( gif_name, ".gif" );
 		
	fp = fopen( fname, "rb" );

	if( fp == (FILE *)0 )
		printf( "error: could not open %s\n", fname ), exit(1);

	xres = ReadInt( fp );
	yres = ReadInt( fp );

	if( xscale > 1 || yscale > 1 )
		printf( "Scaling up to %d by %d\n",
				xres * xscale, yres * yscale );

	xres *= xscale;
	yres *= yscale;

	bit = Bitmap_Create( xres, yres, colors );
	if( bit == (Bitmap *)0 )
		printf( "error: could not allocate bitmap\n" ), exit(1);

	for( i=0; i<colors; i++ ) {
		bit->pal[i].red = cmap[i].rd;
		bit->pal[i].green = cmap[i].gr;
		bit->pal[i].blue = cmap[i].bl;
	}

	if( floyd ) {
  		cerr = (int **) safe_malloc (3 * sizeof (int *));
  		lerr = (int **) safe_malloc (3 * sizeof (int *));
  		cerr[RD] = (int *) safe_malloc ((xres + 2) * sizeof (int));
  		cerr[GR] = (int *) safe_malloc ((xres + 2) * sizeof (int));
  		cerr[BL] = (int *) safe_malloc ((xres + 2) * sizeof (int));
  		lerr[RD] = (int *) safe_malloc ((xres + 2) * sizeof (int));
  		lerr[GR] = (int *) safe_malloc ((xres + 2) * sizeof (int));
  		lerr[BL] = (int *) safe_malloc ((xres + 2) * sizeof (int));
	}

	init_nearest( cmap, colors );

  	/*-------- Clear error vectors --------*/
  	if( floyd )
  		for (i=0; i<xres+2; i++) {
  			cerr[RD][i] = cerr[GR][i] = cerr[BL][i] = 0;
    			lerr[RD][i] = lerr[GR][i] = lerr[BL][i] = 0;
  		}

  	if( floyd )
  		printf( "Applying Floyd-Steinberg dithering...\n\n" );
  	else
  		printf( "Applying color palette mapping...\n\n" );

	for( ;; ) {
		lineno = ReadInt( fp );
		if( lineno < 0 || lineno > yres )
			break;

		if( (lineno+1) % 50 == 0 )
			printf( "%d", (lineno+1) );
		printf( "." );
		fflush( stdout );

		fread( r_vals, 1, xres / xscale, fp );
		fread( g_vals, 1, xres / xscale, fp );
		fread( b_vals, 1, xres / xscale, fp );

		for( j=lineno * yscale; j<(lineno+1)*yscale; j++ ) {
		for( i=0; i<xres; i++ ) {

			rd = r_vals[i / xscale] * 4;
			gr = g_vals[i / xscale] * 4;
			bl = b_vals[i / xscale] * 4;

			if( floyd ) {

      			/* Sum up errors using Floyd-Steinberg weights */
      
      rderr= cerr[RD][i] + 5*lerr[RD][i] + 7*lerr[RD][i+1] + 3*lerr[RD][i+2];
      grerr= cerr[GR][i] + 5*lerr[GR][i] + 7*lerr[GR][i+1] + 3*lerr[GR][i+2];
      blerr= cerr[BL][i] + 5*lerr[BL][i] + 7*lerr[BL][i+1] + 3*lerr[BL][i+2];

      				rderr >>= 4;	/* Divide by 16 */
      				grerr >>= 4;	/* Divide by 16 */
      				blerr >>= 4;	/* Divide by 16 */

      				/* Chose nearest color to adjusted RGB value */
      				rd += rderr; gr += grerr; bl += blerr;
			}

      			idx = nearest (rd, gr, bl, cmap, colors);

			if( floyd ) {

      				/* Compute accumulated error for this pixel */
      				cerr[RD][i+1] = rd - cmap[idx].rd;
      				cerr[GR][i+1] = gr - cmap[idx].gr;
      				cerr[BL][i+1] = bl - cmap[idx].bl;
			}
			Bitmap_Plot( bit, i, j, idx );
		}

   
		if( floyd ) { 
			/* Swap error vectors */
    			terr = lerr; lerr = cerr; cerr = terr;
		}

		}
		if( feof( fp ) )
			break;
	}

	fclose( fp );

	printf( "\n\nWriting '%s' (%d x %d x %d)\n", gif_name,
		xres, yres, colors );

        Bitmap_WriteAsGIF( bit, gif_name, interlace );

}
/****************************************************************
 * sort colormap by decreasing red ( then green and blue )
 ****************************************************************/

cmp_cmap (a, b)
register COLOR *a, *b;
{ register int r;

  if (r = (a->rd - b->rd)) { return (r); }
  if (r = (a->gr - b->gr)) { return (r); }
  if (r = (a->bl - b->bl)) { return (r); }
  
  return (0);
}

/****************************************************************
 * nearest: Choose nearest color
 ****************************************************************/

int cache[CUBSIZ];

#define SQR(X) 	((X) * (X))

init_nearest( cmap, colors )
COLOR *cmap;
int colors;
{
	int i;

	for( i=0; i<CUBSIZ; i++ )
		cache[i] = -1;
}

calc_entry(i, cmap, colors)
int i;
COLOR *cmap;
int colors;
{ 	register int j;
	long diff, bdiff;
	int idx;
	long tmp;
	int r, g, b;


		r = GETR8( i );
		g = GETG8( i );
		b = GETB8( i );

		bdiff = 100000L;
		for( j=0; j<colors; j++ ) {

			diff =  SQR(cmap[j].rd - r);
			diff += SQR(cmap[j].gr - g);
			diff += SQR(cmap[j].bl - b);

			if( diff < bdiff )
				idx = j, bdiff = diff;
		}
		cache[ i ] = idx;
}

int restrict( c )
int c;
{
	if( c < 0 ) c = 0;
	else if( c > 255 ) c = 255;
	return c;
}

nearest (rd, gr, bl, cmap, colors)
int rd, gr, bl;
COLOR *cmap;
int colors;
{
  int cindx;

  rd = restrict( rd );
  gr = restrict( gr );
  bl = restrict( bl );

  /* Find array index in cache */
  cindx = CLRINDEX8 (rd, gr, bl);
  if( cache[cindx] == -1 )
  	calc_entry( cindx, cmap, colors );
  return cache[ cindx ];
}

