/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 * Copyright (C) 1998-1999 Adrian E. Feiguin <adrian@ifir.ifir.edu.ar>
 *
 * GtkSheetEntry widget by Adrian E. Feiguin
 * Based on GtkEntry widget 
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <ctype.h>
#include <string.h>
#ifdef USE_XIM
#include <gdk/gdkx.h>
#endif
#include <gdk/gdkkeysyms.h>
#include <gdk/gdki18n.h>
#include "gtksheetentry.h"
#include <gtk/gtkmain.h>
#include <gtk/gtkselection.h>
#include <gtk/gtksignal.h>

#define MIN_SHENTRY_WIDTH  150
#define DRAW_TIMEOUT     20
#define INNER_BORDER     2

enum {
  INSERT_TEXT,
  DELETE_TEXT,
  CHANGED,
  SET_TEXT,
  ACTIVATE,
  LAST_SIGNAL
};


typedef void (*GtkTextFunction) (GtkSHEntry  *shentry, GdkEventKey *event);
typedef void (*GtkSHEntrySignal1) (GtkObject *object,
				 gpointer   arg1,
				 gint       arg2,
				 gpointer   arg3,
				 gpointer   data);
typedef void (*GtkSHEntrySignal2) (GtkObject *object,
				 gint       arg1,
				 gint       arg2,
				 gpointer   data);
typedef void (*GtkSHEntrySignal3) (GtkObject *object,
				 gpointer   arg1,
				 gpointer   data);



static void gtk_shentry_marshal_signal_1     (GtkObject         *object,
					    GtkSignalFunc      func,
					    gpointer           func_data,
					    GtkArg            *args);
static void gtk_shentry_marshal_signal_2     (GtkObject         *object,
					    GtkSignalFunc      func,
					    gpointer           func_data,
					    GtkArg            *args);
static void gtk_shentry_marshal_signal_3     (GtkObject         *object,
					    GtkSignalFunc      func,
					    gpointer           func_data,
					    GtkArg            *args);


static void gtk_shentry_class_init          (GtkSHEntryClass     *klass);
static void gtk_shentry_init                (GtkSHEntry          *shentry);
static void gtk_shentry_destroy             (GtkObject         *object);
static void gtk_shentry_realize             (GtkWidget         *widget);
static void gtk_shentry_unrealize           (GtkWidget         *widget);
static void gtk_shentry_draw_focus          (GtkWidget         *widget);
static void gtk_shentry_size_request        (GtkWidget         *widget,
					   GtkRequisition    *requisition);
static void gtk_shentry_size_allocate       (GtkWidget         *widget,
					   GtkAllocation     *allocation);
static void gtk_shentry_make_backing_pixmap (GtkSHEntry *shentry,
					   gint width, gint height);
static void gtk_shentry_draw                (GtkWidget         *widget,
					   GdkRectangle      *area);
static gint gtk_shentry_expose              (GtkWidget         *widget,
					   GdkEventExpose    *event);
static gint gtk_shentry_button_press        (GtkWidget         *widget,
					   GdkEventButton    *event);
static gint gtk_shentry_button_release      (GtkWidget         *widget,
					   GdkEventButton    *event);
static gint gtk_shentry_motion_notify       (GtkWidget         *widget,
					   GdkEventMotion    *event);
static gint gtk_shentry_key_press           (GtkWidget         *widget,
					   GdkEventKey       *event);
static gint gtk_shentry_focus_in            (GtkWidget         *widget,
					   GdkEventFocus     *event);
static gint gtk_shentry_focus_out           (GtkWidget         *widget,
					   GdkEventFocus     *event);
static gint gtk_shentry_selection_clear     (GtkWidget         *widget,
					   GdkEventSelection *event);
static void gtk_shentry_selection_handler   (GtkWidget    *widget,
					   GtkSelectionData  *selection_data,
					   gpointer      data);
static void gtk_shentry_selection_received  (GtkWidget         *widget,
					   GtkSelectionData  *selection_data);
static void gtk_shentry_draw_text           (GtkSHEntry          *shentry);
static void gtk_shentry_draw_cursor         (GtkSHEntry          *shentry);
static void gtk_shentry_draw_cursor_on_drawable
					  (GtkSHEntry          *shentry,
					   GdkDrawable       *drawable);
static void gtk_shentry_queue_draw          (GtkSHEntry          *shentry);
static gint gtk_shentry_timer               (gpointer           data);
static gint gtk_shentry_position            (GtkSHEntry          *shentry,
					   gint               x);
       void gtk_shentry_adjust_scroll       (GtkSHEntry          *shentry);
static void gtk_shentry_grow_text           (GtkSHEntry          *shentry);
static void gtk_shentry_insert_text         (GtkSHEntry          *shentry,
					   const gchar       *new_text,
					   gint               new_text_length,
					   gint              *position);
static void gtk_shentry_delete_text         (GtkSHEntry          *shentry,
					   gint               start_pos,
					   gint               end_pos);
static void gtk_real_shentry_insert_text    (GtkSHEntry          *shentry,
					   const gchar       *new_text,
					   gint               new_text_length,
					   gint              *position);
static void gtk_real_shentry_delete_text    (GtkSHEntry          *shentry,
					   gint               start_pos,
					   gint               end_pos);

static gint move_backward_character	  (gchar *str, gint index);
static void gtk_move_forward_character    (GtkSHEntry          *shentry);
static void gtk_move_backward_character   (GtkSHEntry          *shentry);
static void gtk_move_forward_word         (GtkSHEntry          *shentry);
static void gtk_move_backward_word        (GtkSHEntry          *shentry);
static void gtk_move_beginning_of_line    (GtkSHEntry          *shentry);
static void gtk_move_end_of_line          (GtkSHEntry          *shentry);
static void gtk_delete_forward_character  (GtkSHEntry          *shentry);
static void gtk_delete_backward_character (GtkSHEntry          *shentry);
static void gtk_delete_forward_word       (GtkSHEntry          *shentry);
static void gtk_delete_backward_word      (GtkSHEntry          *shentry);
static void gtk_delete_line               (GtkSHEntry          *shentry);
static void gtk_delete_to_line_end        (GtkSHEntry          *shentry);
static void gtk_delete_selection          (GtkSHEntry          *shentry);
static void gtk_select_word               (GtkSHEntry          *shentry);
static void gtk_select_line               (GtkSHEntry          *shentry);
static void gtk_shentry_cut_clipboard     (GtkSHEntry *shentry, 
					 GdkEventKey *event);
static void gtk_shentry_copy_clipboard    (GtkSHEntry *shentry, 
					 GdkEventKey *event);
static void gtk_shentry_paste_clipboard   (GtkSHEntry *shentry, 
					 GdkEventKey *event);

static GtkWidgetClass *parent_class = NULL;
static gint shentry_signals[LAST_SIGNAL] = { 0 };
static GdkAtom ctext_atom = GDK_NONE;
static GdkAtom text_atom = GDK_NONE;
static GdkAtom clipboard_atom = GDK_NONE;

static GtkTextFunction control_keys[26] =
{
  (GtkTextFunction)gtk_move_beginning_of_line,    /* a */
  (GtkTextFunction)gtk_move_backward_character,   /* b */
  gtk_shentry_copy_clipboard,                       /* c */
  (GtkTextFunction)gtk_delete_forward_character,  /* d */
  (GtkTextFunction)gtk_move_end_of_line,          /* e */
  (GtkTextFunction)gtk_move_forward_character,    /* f */
  NULL,                                           /* g */
  (GtkTextFunction)gtk_delete_backward_character, /* h */
  NULL,                                           /* i */
  NULL,                                           /* j */
  (GtkTextFunction)gtk_delete_to_line_end,        /* k */
  NULL,                                           /* l */
  NULL,                                           /* m */
  NULL,                                           /* n */
  NULL,                                           /* o */
  NULL,                                           /* p */
  NULL,                                           /* q */
  NULL,                                           /* r */
  NULL,                                           /* s */
  NULL,                                           /* t */
  (GtkTextFunction)gtk_delete_line,               /* u */
  gtk_shentry_paste_clipboard,                      /* v */
  (GtkTextFunction)gtk_delete_backward_word,      /* w */
  gtk_shentry_cut_clipboard,                        /* x */
  NULL,                                           /* y */
  NULL,                                           /* z */
};

static GtkTextFunction alt_keys[26] =
{
  NULL,                                           /* a */
  (GtkTextFunction)gtk_move_backward_word,        /* b */
  NULL,                                           /* c */
  (GtkTextFunction)gtk_delete_forward_word,       /* d */
  NULL,                                           /* e */
  (GtkTextFunction)gtk_move_forward_word,         /* f */
  NULL,                                           /* g */
  NULL,                                           /* h */
  NULL,                                           /* i */
  NULL,                                           /* j */
  NULL,                                           /* k */
  NULL,                                           /* l */
  NULL,                                           /* m */
  NULL,                                           /* n */
  NULL,                                           /* o */
  NULL,                                           /* p */
  NULL,                                           /* q */
  NULL,                                           /* r */
  NULL,                                           /* s */
  NULL,                                           /* t */
  NULL,                                           /* u */
  NULL,                                           /* v */
  NULL,                                           /* w */
  NULL,                                           /* x */
  NULL,                                           /* y */
  NULL,                                           /* z */
};


guint
gtk_shentry_get_type ()
{
  static guint shentry_type = 0;

  if (!shentry_type)
    {
      GtkTypeInfo shentry_info =
      {
	"GtkSHEntry",
	sizeof (GtkSHEntry),
	sizeof (GtkSHEntryClass),
	(GtkClassInitFunc) gtk_shentry_class_init,
	(GtkObjectInitFunc) gtk_shentry_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      shentry_type = gtk_type_unique (gtk_widget_get_type (), &shentry_info);
    }

  return shentry_type;
}

static void
gtk_shentry_class_init (GtkSHEntryClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;

  parent_class = gtk_type_class (gtk_widget_get_type ());

  shentry_signals[INSERT_TEXT] =
    gtk_signal_new ("insert_text",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSHEntryClass, insert_text),
		    gtk_shentry_marshal_signal_1,
		    GTK_TYPE_NONE, 3,
		    GTK_TYPE_STRING, GTK_TYPE_INT,
		    GTK_TYPE_POINTER);
  shentry_signals[DELETE_TEXT] =
    gtk_signal_new ("delete_text",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSHEntryClass, delete_text),
		    gtk_shentry_marshal_signal_2,
		    GTK_TYPE_NONE, 2,
		    GTK_TYPE_INT, GTK_TYPE_INT);
  shentry_signals[CHANGED] =
    gtk_signal_new ("changed",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSHEntryClass, changed),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  shentry_signals[SET_TEXT] =
    gtk_signal_new ("set_text",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSHEntryClass, set_text),
		    gtk_signal_default_marshaller,
		    GTK_TYPE_NONE, 0);
  shentry_signals[ACTIVATE] =
    gtk_signal_new ("activate",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GtkSHEntryClass, activate),
		    gtk_shentry_marshal_signal_3,
		    GTK_TYPE_NONE, 1, GTK_TYPE_POINTER);

  gtk_object_class_add_signals (object_class, shentry_signals, LAST_SIGNAL);

  object_class->destroy = gtk_shentry_destroy;

  widget_class->realize = gtk_shentry_realize;
  widget_class->unrealize = gtk_shentry_unrealize;
  widget_class->draw_focus = gtk_shentry_draw_focus;
  widget_class->size_request = gtk_shentry_size_request;
  widget_class->size_allocate = gtk_shentry_size_allocate;
  widget_class->draw = gtk_shentry_draw;
  widget_class->expose_event = gtk_shentry_expose;
  widget_class->button_press_event = gtk_shentry_button_press;
  widget_class->button_release_event = gtk_shentry_button_release;
  widget_class->motion_notify_event = gtk_shentry_motion_notify;
  widget_class->key_press_event = gtk_shentry_key_press;
  widget_class->focus_in_event = gtk_shentry_focus_in;
  widget_class->focus_out_event = gtk_shentry_focus_out;
  widget_class->selection_clear_event = gtk_shentry_selection_clear;
  widget_class->selection_received = gtk_shentry_selection_received;

  class->insert_text = gtk_real_shentry_insert_text;
  class->delete_text = gtk_real_shentry_delete_text;
  class->changed = gtk_shentry_adjust_scroll;
  class->set_text = NULL; 	  /* user defined handling */
  class->activate = NULL; 	  /* user defined handling */
}

static void
gtk_shentry_init (GtkSHEntry *shentry)
{
  GTK_WIDGET_SET_FLAGS (shentry, GTK_CAN_FOCUS);

  shentry->text_area = NULL;
  shentry->backing_pixmap = NULL;
  shentry->text = NULL;
  shentry->justification=GTK_JUSTIFY_LEFT;
  shentry->text_size = 0;
  shentry->text_length = 0;
  shentry->text_max_length = 0;
  shentry->text_max_size = 0;
  shentry->current_pos = 0;
  shentry->selection_start_pos = 0;
  shentry->selection_end_pos = 0;
  shentry->scroll_offset = 0;
  shentry->have_selection = FALSE;
  shentry->timer = 0;
  shentry->visible = 1;
  shentry->editable = 1;
  shentry->clipboard_text = NULL;

#ifdef USE_XIM
  shentry->ic = NULL;
#endif

  if (!clipboard_atom)
    clipboard_atom = gdk_atom_intern ("CLIPBOARD", FALSE);

  gtk_selection_add_handler (GTK_WIDGET(shentry), GDK_SELECTION_PRIMARY,
			     GDK_TARGET_STRING, gtk_shentry_selection_handler,
			     NULL);
  gtk_selection_add_handler (GTK_WIDGET(shentry), clipboard_atom,
			     GDK_TARGET_STRING, gtk_shentry_selection_handler,
			     NULL);

  if (!text_atom)
    text_atom = gdk_atom_intern ("TEXT", FALSE);

  gtk_selection_add_handler (GTK_WIDGET(shentry), GDK_SELECTION_PRIMARY,
			     text_atom,
			     gtk_shentry_selection_handler,
			     NULL);
  gtk_selection_add_handler (GTK_WIDGET(shentry), clipboard_atom,
			     text_atom,
			     gtk_shentry_selection_handler,
			     NULL);

  if (!ctext_atom)
    ctext_atom = gdk_atom_intern ("COMPOUND_TEXT", FALSE);

  gtk_selection_add_handler (GTK_WIDGET(shentry), GDK_SELECTION_PRIMARY,
			     ctext_atom,
			     gtk_shentry_selection_handler,
			     NULL);
  gtk_selection_add_handler (GTK_WIDGET(shentry), clipboard_atom,
			     ctext_atom,
			     gtk_shentry_selection_handler,
			     NULL);
}

GtkWidget*
gtk_shentry_new ()
{
  return GTK_WIDGET (gtk_type_new (gtk_shentry_get_type ()));
}

GtkWidget*
gtk_shentry_new_with_max_length (guint16 max)
{
  GtkSHEntry *shentry;
  shentry = gtk_type_new (gtk_shentry_get_type ());
  shentry->text_max_length = max;
  return GTK_WIDGET (shentry);
}

GtkWidget*
gtk_shentry_new_with_max_size (guint16 max)
{
  GtkSHEntry *shentry;
  shentry = gtk_type_new (gtk_shentry_get_type ());
  shentry->text_max_size = max;
  return GTK_WIDGET (shentry);
}

void
gtk_shentry_set_justification (GtkSHEntry *shentry,
		             guint justification)
{
  shentry->justification=justification;
}

void
gtk_shentry_set_text (GtkSHEntry *shentry,
		    const gchar *text,
                    guint justification)
{
  gint tmp_pos;

  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  gtk_real_shentry_delete_text (shentry, 0, shentry->text_length);

  tmp_pos = 0;
  gtk_shentry_insert_text (shentry, text, strlen (text), &tmp_pos);

  shentry->selection_start_pos = 0;
  shentry->selection_end_pos = 0;
  shentry->justification = justification;

/*  shentry->current_pos=justification==GTK_JUSTIFY_RIGHT ? 0 : strlen(text); 
*/
  shentry->current_pos=strlen(text); 


  if (GTK_WIDGET_DRAWABLE (shentry)){
    gtk_shentry_adjust_scroll (shentry);
    gtk_shentry_draw_text (shentry);
  }

  gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[SET_TEXT]);
}

void
gtk_shentry_append_text (GtkSHEntry *shentry,
		       const gchar *text)
{
  gint tmp_pos;

  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  tmp_pos = shentry->text_length;
  gtk_shentry_insert_text (shentry, text, strlen (text), &tmp_pos);
  shentry->current_pos = tmp_pos;

  gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[SET_TEXT]);
}

void
gtk_shentry_prepend_text (GtkSHEntry *shentry,
			const gchar *text)
{
  gint tmp_pos;

  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  tmp_pos = 0;
  gtk_shentry_insert_text (shentry, text, strlen (text), &tmp_pos);
  shentry->current_pos = tmp_pos;

  gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[SET_TEXT]);
}

void
gtk_shentry_set_position (GtkSHEntry *shentry,
			gint      position)
{
  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  if ((position == -1) || (position > shentry->text_length))
    shentry->current_pos = shentry->text_length;
  else
    shentry->current_pos = position;
}

void
gtk_shentry_set_visibility (GtkSHEntry *shentry,
			  gboolean visible)
{
  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  shentry->visible = visible;
}

void
gtk_shentry_set_editable(GtkSHEntry *shentry,
		       gboolean editable)
{
  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));
  shentry->editable = editable;
  gtk_shentry_queue_draw(shentry);
}

gchar*
gtk_shentry_get_text (GtkSHEntry *shentry)
{
  static char empty_str[2] = "";

  g_return_val_if_fail (shentry != NULL, NULL);
  g_return_val_if_fail (GTK_IS_SHENTRY (shentry), NULL);

  if (!shentry->text)
    return empty_str;
  return shentry->text;
}


static void
gtk_shentry_marshal_signal_1 (GtkObject      *object,
			    GtkSignalFunc   func,
			    gpointer        func_data,
			    GtkArg         *args)
{
  GtkSHEntrySignal1 rfunc;

  rfunc = (GtkSHEntrySignal1) func;

  (* rfunc) (object, GTK_VALUE_STRING (args[0]), GTK_VALUE_INT (args[1]),
	     GTK_VALUE_POINTER (args[2]), func_data);
}

static void
gtk_shentry_marshal_signal_2 (GtkObject      *object,
			    GtkSignalFunc   func,
			    gpointer        func_data,
			    GtkArg         *args)
{
  GtkSHEntrySignal2 rfunc;

  rfunc = (GtkSHEntrySignal2) func;

  (* rfunc) (object, GTK_VALUE_INT (args[0]), GTK_VALUE_INT (args[1]),
	     func_data);
}

static void
gtk_shentry_marshal_signal_3 (GtkObject      *object,
			    GtkSignalFunc   func,
			    gpointer        func_data,
			    GtkArg         *args)
{
  GtkSHEntrySignal3 rfunc;

  rfunc = (GtkSHEntrySignal3) func;

  (* rfunc) (object, GTK_VALUE_POINTER (args[0]), func_data);
}

static void
gtk_shentry_destroy (GtkObject *object)
{
  GtkSHEntry *shentry;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (object));

  shentry = GTK_SHENTRY (object);

#ifdef USE_XIM
  if (shentry->ic)
    {
      gdk_ic_destroy (shentry->ic);
      shentry->ic = NULL;
    }
#endif

  if (shentry->timer)
    gtk_timeout_remove (shentry->timer);

  if (shentry->text)
    g_free (shentry->text);
  shentry->text = NULL;

  if (shentry->backing_pixmap)
    gdk_pixmap_unref (shentry->backing_pixmap);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gtk_shentry_realize (GtkWidget *widget)
{
  GtkSHEntry *shentry;
  GdkWindowAttr attributes;
  gint attributes_mask;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (widget));

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  shentry = GTK_SHENTRY (widget);

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);
  attributes.event_mask = gtk_widget_get_events (widget);
  attributes.event_mask |= (GDK_EXPOSURE_MASK |
			    GDK_BUTTON_PRESS_MASK |
			    GDK_BUTTON_RELEASE_MASK |
			    GDK_BUTTON1_MOTION_MASK |
			    GDK_BUTTON3_MOTION_MASK |
			    GDK_POINTER_MOTION_HINT_MASK |
			    GDK_ENTER_NOTIFY_MASK |
			    GDK_LEAVE_NOTIFY_MASK |
			    GDK_KEY_PRESS_MASK);
  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

  widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);
  gdk_window_set_user_data (widget->window, shentry);

/*  attributes.x = widget->style->klass->xthickness + INNER_BORDER;
  attributes.y = widget->style->klass->ythickness + INNER_BORDER;
  attributes.width = widget->allocation.width - attributes.x * 2;
  attributes.height = widget->allocation.height - attributes.y * 2;
  attributes.cursor = shentry->cursor = gdk_cursor_new (GDK_XTERM);
  attributes_mask |= GDK_WA_CURSOR;
*/

  attributes.x = INNER_BORDER;
  attributes.y = INNER_BORDER;
  attributes.width = widget->allocation.width - 2 * INNER_BORDER;
  attributes.height = widget->allocation.height - 2 * INNER_BORDER ;
  attributes.cursor = shentry->cursor = gdk_cursor_new (GDK_XTERM);
  attributes_mask |= GDK_WA_CURSOR;


  shentry->text_area = gdk_window_new (widget->window, &attributes, attributes_mask);
  gdk_window_set_user_data (shentry->text_area, shentry);

  widget->style = gtk_style_attach (widget->style, widget->window);

  gdk_window_set_background (widget->window, &widget->style->white);
  gdk_window_set_background (shentry->text_area, &widget->style->white);

  gdk_gc_new (widget->window);  

#ifdef USE_XIM
  if (gdk_im_ready ())
    {
      GdkPoint spot;
      GdkRectangle rect;
      gint width, height;
      GdkEventMask mask;
      GdkIMStyle style;
      GdkIMStyle supported_style = GdkIMPreeditNone | GdkIMPreeditNothing |
			GdkIMPreeditPosition |
			GdkIMStatusNone | GdkIMStatusNothing;

      if (widget->style && widget->style->font->type != GDK_FONT_FONTSET)
	supported_style &= ~GdkIMPreeditPosition;

      style = gdk_im_decide_style (supported_style);
      switch (style & GdkIMPreeditMask)
	{
	case GdkIMPreeditPosition:
	  if (widget->style && widget->style->font->type != GDK_FONT_FONTSET)
	    {
	      g_warning ("over-the-spot style requires fontset");
	      break;
	    }
	  gdk_window_get_size (shentry->text_area, &width, &height);
	  rect.x = 0;
	  rect.y = 0;
	  rect.width = width;
	  rect.height = height;
	  spot.x = 0;
	  spot.y = height;
	  shentry->ic = gdk_ic_new (shentry->text_area, shentry->text_area,
  			       style,
			       "spotLocation", &spot,
			       "area", &rect,
			       "fontSet", GDK_FONT_XFONT (widget->style->font),
			       NULL);
	  break;
	default:
	  shentry->ic = gdk_ic_new (shentry->text_area, shentry->text_area,
				  style, NULL);
	}
     
      if (shentry->ic == NULL)
	g_warning ("Can't create input context.");
      else
	{
	  GdkColormap *colormap;

	  mask = gdk_window_get_events (shentry->text_area);
	  mask |= gdk_ic_get_events (shentry->ic);
	  gdk_window_set_events (shentry->text_area, mask);

	  if ((colormap = gtk_widget_get_colormap (widget)) !=
	    	gtk_widget_get_default_colormap ())
	    {
	      gdk_ic_set_attr (shentry->ic, "preeditAttributes",
	      		       "colorMap", GDK_COLORMAP_XCOLORMAP (colormap),
			       NULL);
	    }
	  gdk_ic_set_attr (shentry->ic,"preeditAttributes",
		     "foreground", widget->style->fg[GTK_STATE_NORMAL].pixel,
		     "background", widget->style->white.pixel,
		     NULL);
	}
    }
#endif

  shentry->fg_gc = gdk_gc_new (widget->window);  
  shentry->bg_gc = gdk_gc_new (widget->window);  

  gdk_gc_set_foreground(shentry->fg_gc, &widget->style->white);
  gdk_gc_set_foreground(shentry->bg_gc, &widget->style->black);

  gdk_window_show (shentry->text_area);
}

static void
gtk_shentry_unrealize (GtkWidget *widget)
{
  GtkSHEntry *shentry;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (widget));

  GTK_WIDGET_UNSET_FLAGS (widget, GTK_REALIZED);
  shentry = GTK_SHENTRY (widget);

  gtk_style_detach (widget->style);

  if (shentry->text_area)
    {
      gdk_window_set_user_data (shentry->text_area, NULL);
      gdk_window_destroy (shentry->text_area);
      gdk_cursor_destroy (shentry->cursor);
    }
  if (widget->window)
    {
      gdk_window_set_user_data (widget->window, NULL);
      gdk_window_destroy (widget->window);
    }

  shentry->text_area = NULL;
  widget->window = NULL;
}

static void
gtk_shentry_draw_focus (GtkWidget *widget)
{
  gint width, height;
  gint x, y;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (widget));

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      gtk_shentry_draw_cursor (GTK_SHENTRY (widget));
    }
}

static void
gtk_shentry_size_request (GtkWidget      *widget,
			GtkRequisition *requisition)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (widget));
  g_return_if_fail (requisition != NULL);

  requisition->width = MIN_SHENTRY_WIDTH + (widget->style->klass->xthickness + INNER_BORDER) * 2;
  requisition->height = (widget->style->font->ascent +
			 widget->style->font->descent +
			 (widget->style->klass->ythickness + INNER_BORDER) * 2);
}

static void
gtk_shentry_size_allocate (GtkWidget     *widget,
			 GtkAllocation *allocation)
{
  GtkSHEntry *shentry;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (widget));
  g_return_if_fail (allocation != NULL);

  widget->allocation = *allocation;
  shentry = GTK_SHENTRY (widget);

  if (GTK_WIDGET_REALIZED (widget))
    {
      gdk_window_move_resize (widget->window,
			      allocation->x,
			      allocation->y,
			      allocation->width, allocation->height);
      gdk_window_move_resize (shentry->text_area,
			      INNER_BORDER,
			      INNER_BORDER,
			      allocation->width - 2 * INNER_BORDER ,
			      allocation->height - 2 * INNER_BORDER );


      shentry->scroll_offset = 0;
      gtk_shentry_adjust_scroll (shentry);

#ifdef USE_XIM
      if (shentry->ic && (gdk_ic_get_style (shentry->ic) & GdkIMPreeditPosition))
	{
	  gint width, height;
	  GdkRectangle rect;

	  gdk_window_get_size (shentry->text_area, &width, &height);
	  rect.x = 0;
	  rect.y = 0;
	  rect.width = width;
	  rect.height = height;
	  gdk_ic_set_attr (shentry->ic, "preeditAttributes", "area", &rect, NULL);
	}
#endif

    }
}

static void
gtk_shentry_draw (GtkWidget    *widget,
		GdkRectangle *area)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (widget));
  g_return_if_fail (area != NULL);

  if (GTK_WIDGET_DRAWABLE (widget))
    {
      gtk_widget_draw_focus (widget);
      gtk_shentry_draw_text (GTK_SHENTRY (widget));
    }
}

static gint
gtk_shentry_expose (GtkWidget      *widget,
		  GdkEventExpose *event)
{
  GtkSHEntry *shentry;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHENTRY (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  shentry = GTK_SHENTRY (widget);

  if (widget->window == event->window)
    gtk_widget_draw_focus (widget);
  else if (shentry->text_area == event->window)
    gtk_shentry_draw_text (GTK_SHENTRY (widget));

  return FALSE;
}

static gint
gtk_shentry_button_press (GtkWidget      *widget,
			GdkEventButton *event)
{
  GtkSHEntry *shentry;
  gint tmp_pos;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHENTRY (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (ctext_atom == GDK_NONE)
    ctext_atom = gdk_atom_intern ("COMPOUND_TEXT", FALSE);

  shentry = GTK_SHENTRY (widget);
  if (!GTK_WIDGET_HAS_FOCUS (widget))
    gtk_widget_grab_focus (widget);

  if (event->button == 1)
    {
      switch (event->type)
	{
	case GDK_BUTTON_PRESS:
	  gtk_grab_add (widget);

	  tmp_pos = gtk_shentry_position (shentry, event->x + shentry->scroll_offset);
	  gtk_shentry_select_region (shentry, tmp_pos, tmp_pos);
	  shentry->current_pos = shentry->selection_start_pos;
	  break;

	case GDK_2BUTTON_PRESS:
	  gtk_select_word (shentry);
	  break;

	case GDK_3BUTTON_PRESS:
	  gtk_select_line (shentry);
	  break;

	default:
	  break;
	}
    }
  else if (event->type == GDK_BUTTON_PRESS)
    {
      if (event->button == 2)
	{
	  if (shentry->selection_start_pos == shentry->selection_end_pos ||
	      shentry->have_selection)
	    shentry->current_pos = gtk_shentry_position (shentry, event->x + shentry->scroll_offset);
	  gtk_selection_convert (widget, GDK_SELECTION_PRIMARY,
				 ctext_atom, event->time);
	}
      else
	{
	  gtk_grab_add (widget);

	  tmp_pos = gtk_shentry_position (shentry, event->x + shentry->scroll_offset);
	  gtk_shentry_select_region (shentry, tmp_pos, tmp_pos);
	  shentry->have_selection = FALSE;
	  shentry->current_pos = shentry->selection_start_pos;
	}
    }

  return FALSE;
}

static gint
gtk_shentry_button_release (GtkWidget      *widget,
			  GdkEventButton *event)
{
  GtkSHEntry *shentry;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHENTRY (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  if (event->button == 1)
    {
      shentry = GTK_SHENTRY (widget);
      gtk_grab_remove (widget);

      shentry->have_selection = FALSE;
      if (shentry->selection_start_pos != shentry->selection_end_pos)
	{
	  if (gtk_selection_owner_set (widget,
				       GDK_SELECTION_PRIMARY,
				       event->time))
	    {
	      shentry->have_selection = TRUE;
	      gtk_shentry_queue_draw (shentry);
	    }
	}
      else
	{
	  if (gdk_selection_owner_get (GDK_SELECTION_PRIMARY) == widget->window)
	    gtk_selection_owner_set (NULL, GDK_SELECTION_PRIMARY, event->time);
	}
    }
  else if (event->button == 3)
    {
      gtk_grab_remove (widget);
      if (gdk_selection_owner_get (GDK_SELECTION_PRIMARY) == widget->window)
	gtk_selection_owner_set (NULL, GDK_SELECTION_PRIMARY, event->time);
    }

  return FALSE;
}

static gint
gtk_shentry_motion_notify (GtkWidget      *widget,
			 GdkEventMotion *event)
{
  GtkSHEntry *shentry;
  gint x;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHENTRY (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  shentry = GTK_SHENTRY (widget);

  x = event->x;
  if (event->is_hint || (shentry->text_area != event->window))
    gdk_window_get_pointer (shentry->text_area, &x, NULL, NULL);

  shentry->selection_end_pos = gtk_shentry_position (shentry, event->x + shentry->scroll_offset);
  shentry->current_pos = shentry->selection_end_pos;
  gtk_shentry_adjust_scroll (shentry);
  gtk_shentry_queue_draw (shentry);

  return FALSE;
}

static gint
gtk_shentry_key_press (GtkWidget   *widget,
		     GdkEventKey *event)
{
  GtkSHEntry *shentry;
  gint return_val;
  gint key;
  gint tmp_pos;
  gint extend_selection;
  gint extend_start;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHENTRY (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  shentry = GTK_SHENTRY (widget);
  return_val = FALSE;

  if(!GTK_WIDGET_HAS_FOCUS(widget)) return FALSE;

  if(shentry->editable == FALSE){
    switch(event->keyval){
	case GDK_Left: case GDK_Right: case GDK_Up: case GDK_Down:
        case GDK_Tab: case GDK_Return:
         break;
        default:
         return TRUE;     
    }
  } 

  extend_selection = event->state & GDK_SHIFT_MASK;
  extend_start = FALSE;

  if (extend_selection)
    {
      if (shentry->selection_start_pos == shentry->selection_end_pos)
	{
	  shentry->selection_start_pos = shentry->current_pos;
	  shentry->selection_end_pos = shentry->current_pos;
	}
      
      extend_start = (shentry->current_pos == shentry->selection_start_pos);
    }

  switch (event->keyval)
    {
    case GDK_BackSpace:
      return_val = TRUE;
      if(!shentry->text){
          gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[ACTIVATE],
event);
	  break;
      }
      if(strlen(shentry->text)==0)
      {
          gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[ACTIVATE],
event);
	  break;
      }
      if (shentry->selection_start_pos != shentry->selection_end_pos)
	gtk_delete_selection (shentry);
      else if (event->state & GDK_CONTROL_MASK)
	gtk_delete_backward_word (shentry);
      else
	gtk_delete_backward_character (shentry);
      break;
    case GDK_Clear:
      return_val = TRUE;
      gtk_delete_line (shentry);
      break;
     case GDK_Insert:
       return_val = TRUE;
       if (event->state & GDK_SHIFT_MASK)
	 {
	   extend_selection = FALSE;
	   gtk_shentry_paste_clipboard (shentry, event);
	 }
       else if (event->state & GDK_CONTROL_MASK)
	 {
	   gtk_shentry_copy_clipboard (shentry, event);
	 }
       else
	 {
	   /* gtk_toggle_insert(shentry) -- IMPLEMENT */
	 }
       break;
    case GDK_Delete:
      return_val = TRUE;
      if (shentry->selection_start_pos != shentry->selection_end_pos)
	gtk_delete_selection (shentry);
      else
	{
	  if (event->state & GDK_CONTROL_MASK)
	    gtk_delete_line (shentry);
	  else if (event->state & GDK_SHIFT_MASK)
	    gtk_shentry_cut_clipboard (shentry, event);
	  else
	    gtk_delete_forward_character (shentry);
	}
      break;
    case GDK_Home:
      return_val = TRUE;
      gtk_move_beginning_of_line (shentry);
      break;
    case GDK_End:
      return_val = TRUE;
      gtk_move_end_of_line (shentry);
      break;
    case GDK_Left:
      return_val = TRUE;
      if(!shentry->text){
          gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[ACTIVATE],
event);
	  break;
      }
      if(strlen(shentry->text)==0)
      {
          gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[ACTIVATE],
event);
	  break;
      }
      gtk_move_backward_character (shentry);
      break;
    case GDK_Right:
      return_val = TRUE;
      if(!shentry->text){
          gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[ACTIVATE],
event);
	  break;
      }
      if(strlen(shentry->text)==0)
      {
          gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[ACTIVATE],
event);
	  break;
      }
      gtk_move_forward_character (shentry);
      break;
    case GDK_Shift_L:
    case GDK_Shift_R:
    case GDK_Return:
    case GDK_Tab:
    case GDK_Escape:
    case GDK_Up:
    case GDK_Down:
      return_val = TRUE;
      gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[ACTIVATE],
event);
      break;
    default:
      if ((event->keyval >= 0x20) && (event->keyval <= 0xFF))
	{
	  key = event->keyval;

	  if (event->state & GDK_CONTROL_MASK)
	    {
	      if ((key >= 'A') && (key <= 'Z'))
		key -= 'A' - 'a';

	      if ((key >= 'a') && (key <= 'z') && control_keys[key - 'a'])
		{
		  (* control_keys[key - 'a']) (shentry, event);
		  return_val = TRUE;
		}
	      break;
	    }
	  else if (event->state & GDK_MOD1_MASK)
	    {
	      if ((key >= 'A') && (key <= 'Z'))
		key -= 'A' - 'a';

	      if ((key >= 'a') && (key <= 'z') && alt_keys[key - 'a'])
		{
		  (* alt_keys[key - 'a']) (shentry, event);
		  return_val = TRUE;
		}
	      break;
	    }
	}
      if (event->length > 0)
	{
	  extend_selection = FALSE;
	  gtk_delete_selection (shentry);

	  tmp_pos = shentry->current_pos;
	  gtk_shentry_insert_text (shentry, event->string, event->length, &tmp_pos);
	  shentry->current_pos = tmp_pos;

	  return_val = TRUE;
	}
      break;
    }

  if (return_val)
    {
      if (extend_selection)
	{
	  if (shentry->current_pos < shentry->selection_start_pos)
	    shentry->selection_start_pos = shentry->current_pos;
	  else if (shentry->current_pos > shentry->selection_end_pos)
	    shentry->selection_end_pos = shentry->current_pos;
	  else
	    {
	      if (extend_start)
		shentry->selection_start_pos = shentry->current_pos;
	      else
		shentry->selection_end_pos = shentry->current_pos;
	    }
	}
      else
	{
	  shentry->selection_start_pos = 0;
	  shentry->selection_end_pos = 0;
	}

      /* alex stuff */
      if (shentry->selection_start_pos != shentry->selection_end_pos)
	{
	  if (gtk_selection_owner_set (widget, GDK_SELECTION_PRIMARY, event->time))
	    {
	      shentry->have_selection = TRUE;
	      gtk_shentry_queue_draw (shentry);
	    }
	}
      else
	{
	  if (gdk_selection_owner_get (GDK_SELECTION_PRIMARY) == widget->window)
	    gtk_selection_owner_set (NULL, GDK_SELECTION_PRIMARY, event->time);
	}
      /* end of alex stuff */
      
      gtk_shentry_adjust_scroll (shentry);
      gtk_shentry_queue_draw (shentry);
    }

  return return_val;
}

static gint
gtk_shentry_focus_in (GtkWidget     *widget,
		    GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHENTRY (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  GTK_WIDGET_SET_FLAGS (widget, GTK_HAS_FOCUS);
  gtk_widget_draw_focus (widget);

#ifdef USE_XIM
  if (GTK_SHENTRY(widget)->ic)
    gdk_im_begin (GTK_SHENTRY(widget)->ic, GTK_SHENTRY(widget)->text_area);
#endif

  return FALSE;
}

static gint
gtk_shentry_focus_out (GtkWidget     *widget,
		     GdkEventFocus *event)
{
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHENTRY (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  GTK_WIDGET_UNSET_FLAGS (widget, GTK_HAS_FOCUS);
  gtk_widget_draw_focus (widget);

#ifdef USE_XIM
  gdk_im_end ();
#endif

  return FALSE;
}

static gint
gtk_shentry_selection_clear (GtkWidget         *widget,
			   GdkEventSelection *event)
{
  GtkSHEntry *shentry;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_SHENTRY (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  /* Let the selection handling code know that the selection
   * has been changed, since we've overriden the default handler */
  gtk_selection_clear (widget, event);

  shentry = GTK_SHENTRY (widget);

  if (event->selection == GDK_SELECTION_PRIMARY)
    {
      if (shentry->have_selection)
	{
	  shentry->have_selection = FALSE;
	  gtk_shentry_queue_draw (shentry);
	}
    }
  else if (event->selection == clipboard_atom)
    {
      g_free (shentry->clipboard_text);
      shentry->clipboard_text = NULL;
    }

  return FALSE;
}

static void
gtk_shentry_selection_handler (GtkWidget        *widget,
			     GtkSelectionData *selection_data,
			     gpointer          data)
{
  GtkSHEntry *shentry;
  gint selection_start_pos;
  gint selection_end_pos;

  guchar *str;
  gint length;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (widget));

  shentry = GTK_SHENTRY (widget);

  if (selection_data->selection == GDK_SELECTION_PRIMARY)
    {
      selection_start_pos = MIN (shentry->selection_start_pos, shentry->selection_end_pos);
      selection_end_pos = MAX (shentry->selection_start_pos, shentry->selection_end_pos);
      str = &shentry->text[selection_start_pos];
      length = selection_end_pos - selection_start_pos;
    }
  else				/* CLIPBOARD */
    {
      if (!shentry->clipboard_text)
	return;			/* Refuse */

      str = shentry->clipboard_text;
      length = strlen (shentry->clipboard_text);
    }
  
  if (selection_data->target == GDK_SELECTION_TYPE_STRING)
    {
      gtk_selection_data_set (selection_data,
                              GDK_SELECTION_TYPE_STRING,
                              8*sizeof(gchar), str, length);
    }
  else if (selection_data->target == text_atom ||
           selection_data->target == ctext_atom)
    {
      guchar *text;
      gchar c;
      GdkAtom encoding;
      gint format;
      gint new_length;

      c = str[length];
      str[length] = '\0';
      gdk_string_to_compound_text (str, &encoding, &format, &text, &new_length);
      gtk_selection_data_set (selection_data, encoding, format, text, new_length);
      gdk_free_compound_text (text);
      str[length] = c;
    }
}

static void
gtk_shentry_selection_received  (GtkWidget         *widget,
			       GtkSelectionData  *selection_data)
{
  GtkSHEntry *shentry;
  gint reselect;
  gint old_pos;
  gint tmp_pos;
  enum {INVALID, STRING, CTEXT} type;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (widget));

  shentry = GTK_SHENTRY (widget);

  if (selection_data->type == GDK_TARGET_STRING)
    type = STRING;
  else if (selection_data->type == ctext_atom)
    type = CTEXT;
  else
    type = INVALID;

  if (type == INVALID || selection_data->length < 0)
    {
    /* avoid infinite loop */
    if (selection_data->target != GDK_TARGET_STRING)
      gtk_selection_convert (widget, selection_data->selection,
			     GDK_TARGET_STRING, GDK_CURRENT_TIME);
    return;
  }

  reselect = FALSE;

  if ((shentry->selection_start_pos != shentry->selection_end_pos) && 
      (!shentry->have_selection || 
       (selection_data->selection == clipboard_atom)))
    {
      reselect = TRUE;

      /* Don't want to call gtk_delete_selection here if we are going
       * to reclaim the selection to avoid extra server traffic */
      if (shentry->have_selection)
	{
	  gtk_shentry_delete_text (shentry,
				 MIN (shentry->selection_start_pos, shentry->selection_end_pos),
				 MAX (shentry->selection_start_pos, shentry->selection_end_pos));
	}
      else
	gtk_delete_selection (shentry);
    }

  tmp_pos = old_pos = shentry->current_pos;

  switch (type)
    {
    case STRING:
      selection_data->data[selection_data->length] = 0;
      gtk_shentry_insert_text (shentry, selection_data->data,
			     strlen (selection_data->data), &tmp_pos);
      shentry->current_pos = tmp_pos;
      break;
    case CTEXT:
      {
	gchar **list;
	gint count;
	gint i;

	count = gdk_text_property_to_text_list (selection_data->type,
						selection_data->format, 
	      					selection_data->data,
						selection_data->length,
						&list);
	for (i=0; i<count; i++) 
	  {
	    gtk_shentry_insert_text (shentry, list[i], strlen (list[i]), &tmp_pos);
	    shentry->current_pos = tmp_pos;
	  }
	if (count > 0)
	  gdk_free_text_list (list);
      }
      break;
    case INVALID:		/* quiet compiler */
      break;
    }

  if (reselect)
    gtk_shentry_select_region (shentry, old_pos, shentry->current_pos);

  gtk_shentry_queue_draw (shentry);
}

static void
gtk_shentry_make_backing_pixmap (GtkSHEntry *shentry, gint width, gint height)
{
  gint pixmap_width, pixmap_height;

  if (!shentry->backing_pixmap)
    {
      /* allocate */
      shentry->backing_pixmap = gdk_pixmap_new (shentry->text_area,
					      width, height,
					      -1);
    }
  else
    {
      /* reallocate if sizes don't match */
      gdk_window_get_size (shentry->backing_pixmap,
			   &pixmap_width, &pixmap_height);
      if ((pixmap_width != width) || (pixmap_height != height))
	{
	  gdk_pixmap_unref (shentry->backing_pixmap);
	  shentry->backing_pixmap = gdk_pixmap_new (shentry->text_area,
						  width, height,
						  -1);
	}
    }
}

static void
gtk_shentry_draw_text (GtkSHEntry *shentry)
{
  GtkWidget *widget;
  GtkStateType selected_state;
  gint selection_start_pos;
  gint selection_end_pos;
  gint selection_start_xoffset;
  gint selection_end_xoffset;
  gint width, height;
  gint y;
  GdkDrawable *drawable;
  gint use_backing_pixmap;

  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  if (shentry->timer)
    {
      gtk_timeout_remove (shentry->timer);
      shentry->timer = 0;
    }

  if (!shentry->visible)
    {
      gtk_shentry_draw_cursor (shentry);
      return;
    }

  if (GTK_WIDGET_DRAWABLE (shentry))
    {
      widget = GTK_WIDGET (shentry);
      gdk_window_get_size (shentry->text_area, &width, &height);

      if (!shentry->text)
	{	  
	  gdk_draw_rectangle (shentry->text_area,
			      shentry->bg_gc,
			      TRUE,
			      0, 0,
			      width,
			      height);

	  if (shentry->editable)
	    gtk_shentry_draw_cursor (shentry);
	  return;
	}

      /*
	If the widget has focus, draw on a backing pixmap to avoid flickering
	and copy it to the text_area.
	Otherwise draw to text_area directly for better speed.
      */
      use_backing_pixmap = GTK_WIDGET_HAS_FOCUS (widget) && (shentry->text != NULL);
      if (use_backing_pixmap)
	{
	  gtk_shentry_make_backing_pixmap (shentry, width, height);
	  drawable = shentry->backing_pixmap;
	  gdk_draw_rectangle (drawable,
			      shentry->bg_gc,
			      TRUE,
			      0, 0,
			      width,
			      height);
	}
      else
	{
	  drawable = shentry->text_area;
	  gdk_draw_rectangle (shentry->text_area,
			      shentry->bg_gc,
			      TRUE,
			      0, 0,
			      width,
			      height);

	}

/* CENTER */
/*      y = height; 
      y = (height - (widget->style->font->ascent + widget->style->font->descent)) / 2;
      y += widget->style->font->ascent;
*/
/* BOTTOM */
      y = height;
      y = (y - widget->style->font->ascent - widget->style->font->descent);
      y += widget->style->font->ascent;

      if (shentry->selection_start_pos != shentry->selection_end_pos)
	{
	  selected_state = GTK_STATE_SELECTED;
	  if (!shentry->have_selection)
	    selected_state = GTK_STATE_ACTIVE;

	  selection_start_pos = MIN (shentry->selection_start_pos, shentry->selection_end_pos);
	  selection_end_pos = MAX (shentry->selection_start_pos, shentry->selection_end_pos);

	  selection_start_xoffset = gdk_text_width (widget->style->font,
						    shentry->text,
						    selection_start_pos);
	  selection_end_xoffset = gdk_text_width (widget->style->font,
						  shentry->text,
						  selection_end_pos);

	  if (selection_start_pos > 0)
	    gdk_draw_text (drawable, widget->style->font,
			   shentry->fg_gc,
			   -shentry->scroll_offset, y,
			   shentry->text, selection_start_pos);

	  gdk_draw_rectangle (drawable,
			      widget->style->bg_gc[selected_state],
			      TRUE,
			      -shentry->scroll_offset + selection_start_xoffset,
			      0,
			      selection_end_xoffset - selection_start_xoffset,
			      -1);

	  gdk_draw_text (drawable, widget->style->font,
			 widget->style->fg_gc[selected_state],
			 -shentry->scroll_offset + selection_start_xoffset, y,
			 shentry->text + selection_start_pos,
			 selection_end_pos - selection_start_pos);

	  if (selection_end_pos < shentry->text_length)
	    gdk_draw_string (drawable, widget->style->font,
	                     shentry->fg_gc,
			     -shentry->scroll_offset + selection_end_xoffset, y,
			     shentry->text + selection_end_pos);
	}
      else
	{
	  GdkGCValues values;
	  
	  gdk_gc_get_values (widget->style->fg_gc[GTK_STATE_NORMAL], &values);
	  gdk_draw_string (drawable, widget->style->font,
			   shentry->fg_gc,
			   -shentry->scroll_offset, y,
			   shentry->text);
	}

      if (shentry->editable)
	gtk_shentry_draw_cursor_on_drawable (shentry, drawable);

      if (use_backing_pixmap)
	gdk_draw_pixmap(shentry->text_area,
			widget->style->fg_gc[GTK_STATE_NORMAL],
			shentry->backing_pixmap,
			0, 0, 0, 0, width, height);	  

    }
}

static void
gtk_shentry_draw_cursor (GtkSHEntry *shentry)
{
  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  gtk_shentry_draw_cursor_on_drawable (shentry, shentry->text_area);
}

static void
gtk_shentry_draw_cursor_on_drawable (GtkSHEntry *shentry, GdkDrawable *drawable)
{
  GtkWidget *widget;
  GdkGC *gc;
  gint xoffset, yoffset;
  gint text_area_height;
  gint height;

  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  if (GTK_WIDGET_DRAWABLE (shentry))
    {
      widget = GTK_WIDGET (shentry);

      if (shentry->current_pos > 0 && shentry->visible)
	xoffset = gdk_text_width (widget->style->font, shentry->text, shentry->current_pos);
      else
	xoffset = 0;
      xoffset -= shentry->scroll_offset;

      if (GTK_WIDGET_HAS_FOCUS (widget) &&
	  (shentry->selection_start_pos == shentry->selection_end_pos))
	gc = shentry->fg_gc;
      else
	gc = shentry->bg_gc;

      gdk_window_get_size (shentry->text_area, NULL, &text_area_height);

      height = widget->style->font->ascent+widget->style->font->descent;
      yoffset= text_area_height - height ;

      gdk_draw_line (drawable, gc, xoffset, yoffset, xoffset, yoffset+height);
#ifdef USE_XIM
      if (gdk_im_ready() && shentry->ic && 
	  gdk_ic_get_style (shentry->ic) & GdkIMPreeditPosition)
	{
	  GdkPoint spot;

	  spot.x = xoffset;
	  spot.y = (text_area_height + (widget->style->font->ascent - widget->style->font->descent) + 1) / 2;
	  gdk_ic_set_attr (shentry->ic, "preeditAttributes", "spotLocation", &spot, NULL);
	}
#endif 
    }
}

static void
gtk_shentry_queue_draw (GtkSHEntry *shentry)
{
  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  if (!shentry->timer)
    shentry->timer = gtk_timeout_add (DRAW_TIMEOUT, gtk_shentry_timer, shentry);

}

static gint
gtk_shentry_timer (gpointer data)
{
  GtkSHEntry *shentry;

  g_return_val_if_fail (data != NULL, FALSE);

  shentry = GTK_SHENTRY (data);
  shentry->timer = 0;
  gtk_shentry_draw_text (shentry);

  return FALSE;
}

static gint
gtk_shentry_position (GtkSHEntry *shentry,
		    gint      x)
{
  gint return_val;
  gint char_width;
  gint sum;
  gint i;
  gint len;

  g_return_val_if_fail (shentry != NULL, 0);
  g_return_val_if_fail (GTK_IS_SHENTRY (shentry), 0);

  i = 0;
  sum = 0;

  if (x > sum)
    {
      for (; i < shentry->text_length; i+=len)
	{
	  len = mblen (shentry->text+i, MB_CUR_MAX);
	  /* character not supported in current locale is included */
	  if (len < 1) 
	    len = 1;
	  char_width = gdk_text_width (GTK_WIDGET (shentry)->style->font, 
	  			       shentry->text + i, len);

	  if (x < (sum + char_width / 2))
	    break;
	  sum += char_width;
	}
    }

  return_val = i;

  return return_val;
}

void
gtk_shentry_adjust_scroll (GtkSHEntry *shentry)
{
  gint xoffset;
  gint text_area_width;
  gint text_area_height;
  gint text_width;
  gint char_width;

  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  if (!shentry->text_area)
    return;

  gdk_window_get_size (shentry->text_area, &text_area_width, &text_area_height);
  char_width = gdk_char_width (GTK_WIDGET(shentry)->style->font,(gchar)'X');
  switch(shentry->justification){

   case GTK_JUSTIFY_LEFT:

/* LEFT JUSTIFICATION */

    if (shentry->current_pos > 0)
      xoffset = gdk_text_width (GTK_WIDGET (shentry)->style->font, shentry->text, shentry->current_pos);
    else
      xoffset = 0;

    xoffset -= shentry->scroll_offset;
    if (xoffset < 0)
      shentry->scroll_offset += xoffset;
    else if (xoffset > text_area_width){
      if(shentry->text_max_size != 0 &&
                     text_area_width+char_width<=shentry->text_max_size){
       GTK_WIDGET(shentry)->allocation.width=text_area_width+char_width;
       gtk_shentry_size_allocate(GTK_WIDGET(shentry),
                          (GtkAllocation *)(&GTK_WIDGET(shentry)->allocation));
       gtk_widget_queue_draw(GTK_WIDGET(shentry));
      }else{
       shentry->scroll_offset += (xoffset - text_area_width) + 1;
      }
    }

    break;

   case GTK_JUSTIFY_RIGHT:

/* RIGHT JUSTIFICATION FOR NUMBERS */

    if(shentry->text){

     text_width = gdk_text_width (GTK_WIDGET (shentry)->style->font, shentry->text, strlen(shentry->text));
 
     shentry->scroll_offset=  -(text_area_width - text_width);
     if(shentry->scroll_offset > 0){
      if(shentry->text_max_size != 0 &&
                      text_area_width+char_width<=shentry->text_max_size){
       GTK_WIDGET(shentry)->allocation.x=GTK_WIDGET(shentry)->allocation.x+
                                         GTK_WIDGET(shentry)->allocation.width-
                                         (text_area_width+char_width);
       GTK_WIDGET(shentry)->allocation.width=text_area_width+char_width;
       gtk_shentry_size_allocate(GTK_WIDGET(shentry),
                          (GtkAllocation *)(&GTK_WIDGET(shentry)->allocation));
       gtk_widget_queue_draw(GTK_WIDGET(shentry));
      }
      else
      {
       shentry->scroll_offset= -(text_area_width  
 - gdk_text_width (GTK_WIDGET (shentry)->style->font, shentry->text, shentry->current_pos)) + 1;
       if(shentry->scroll_offset < 0) shentry->scroll_offset = 0;
      }
     }
    }
    else
     shentry->scroll_offset=0;
  
    break;

   case GTK_JUSTIFY_CENTER:

    if(shentry->text){

     text_width = gdk_text_width (GTK_WIDGET (shentry)->style->font, shentry->text, strlen(shentry->text));
 
     shentry->scroll_offset=  -(text_area_width - text_width)/2;
     if(shentry->scroll_offset > 0){
      if(shentry->text_max_size != 0 &&
                      text_area_width+char_width<=shentry->text_max_size){
       GTK_WIDGET(shentry)->allocation.x=GTK_WIDGET(shentry)->allocation.x+
                                       GTK_WIDGET(shentry)->allocation.width/2-
                                       (text_area_width+char_width)/2;
       GTK_WIDGET(shentry)->allocation.width=text_area_width+char_width;
       gtk_shentry_size_allocate(GTK_WIDGET(shentry),
                          (GtkAllocation *)(&GTK_WIDGET(shentry)->allocation));
       gtk_widget_queue_draw(GTK_WIDGET(shentry));
      }
      else
      {
       shentry->scroll_offset= -(text_area_width  
 - gdk_text_width (GTK_WIDGET (shentry)->style->font, shentry->text, shentry->current_pos)) + 1;
       if(shentry->scroll_offset < 0) shentry->scroll_offset = 0;
      }
     }
    }
    else
     shentry->scroll_offset=0;
  
    break;

   }
}

static void
gtk_shentry_grow_text (GtkSHEntry *shentry)
{
  gint previous_size;
  gint i;

  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  previous_size = shentry->text_size;
  if (!shentry->text_size)
    shentry->text_size = 128;
  else
    shentry->text_size *= 2;
  shentry->text = g_realloc (shentry->text, shentry->text_size);

  for (i = previous_size; i < shentry->text_size; i++)
    shentry->text[i] = '\0';
}

static void
gtk_shentry_insert_text (GtkSHEntry *shentry,
		       const gchar *new_text,
		       gint      new_text_length,
		       gint     *position)
{
  gchar buf[64];
  gchar *text;

  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  if (new_text_length <= 64)
    text = buf;
  else
    text = g_new (gchar, new_text_length);

  strncpy (text, new_text, new_text_length);

  gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[INSERT_TEXT],
		   text, new_text_length, position);
  gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[CHANGED]);

  if (new_text_length > 64)
    g_free (text);
}

static void
gtk_shentry_delete_text (GtkSHEntry *shentry,
		       gint      start_pos,
		       gint      end_pos)
{
  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[DELETE_TEXT],
		   start_pos, end_pos);
  gtk_signal_emit (GTK_OBJECT (shentry), shentry_signals[CHANGED]);
}

static void
gtk_real_shentry_insert_text (GtkSHEntry *shentry,
			    const gchar *new_text,
			    gint      new_text_length,
			    gint     *position)
{
  gchar *text;
  gint start_pos;
  gint end_pos;
  gint last_pos;
  gint i;

  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  /* Make sure we do not exceed the maximum size of the shentry. */
  if (shentry->text_max_length != 0 &&
      new_text_length + shentry->text_length > shentry->text_max_length)
    new_text_length = shentry->text_max_length - shentry->text_length;

  /* Don't insert anything, if there was nothing to insert. */
  if (new_text_length <= 0)
    return;

  start_pos = *position;
  end_pos = start_pos + new_text_length;
  last_pos = new_text_length + shentry->text_length;

  if (shentry->selection_start_pos >= *position)
    shentry->selection_start_pos += new_text_length;
  if (shentry->selection_end_pos >= *position)
    shentry->selection_end_pos += new_text_length;

  while (last_pos >= shentry->text_size)
    gtk_shentry_grow_text (shentry);

  text = shentry->text;
  for (i = last_pos - 1; i >= end_pos; i--)
    text[i] = text[i- (end_pos - start_pos)];
  for (i = start_pos; i < end_pos; i++)
    text[i] = new_text[i - start_pos];

  shentry->text_length += new_text_length;
  *position = end_pos;
}

static void
gtk_real_shentry_delete_text (GtkSHEntry *shentry,
			    gint      start_pos,
			    gint      end_pos)
{
  gchar *text;
  gint deletion_length;
  gint i;

  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  if (shentry->selection_start_pos > start_pos)
    shentry->selection_start_pos -= MIN(end_pos, shentry->selection_start_pos) - start_pos;
  if (shentry->selection_end_pos > start_pos)
    shentry->selection_end_pos -= MIN(end_pos, shentry->selection_end_pos) - start_pos;

  if ((start_pos < end_pos) &&
      (start_pos >= 0) &&
      (end_pos <= shentry->text_length))
    {
      text = shentry->text;
      deletion_length = end_pos - start_pos;

      for (i = end_pos; i < shentry->text_length; i++)
        text[i - deletion_length] = text[i];

      for (i = shentry->text_length - deletion_length; i < shentry->text_length; i++)
        text[i] = '\0';

      shentry->text_length -= deletion_length;
      shentry->current_pos = start_pos;
    }
}


static void
gtk_move_forward_character (GtkSHEntry *shentry)
{
  gint len;

  if (shentry->current_pos < shentry->text_length)
    {
      len = mblen (shentry->text+shentry->current_pos, MB_CUR_MAX);
      shentry->current_pos += (len>0)? len:1;
    }
  if (shentry->current_pos > shentry->text_length)
    shentry->current_pos = shentry->text_length;
}

static gint
move_backward_character (gchar *str, gint index)
{
  gint i;
  gint len;

  if (index <= 0)
    return -1;
  for (i=0,len=0; i<index; i+=len)
    {
      len = mblen (str+i, MB_CUR_MAX);
      if (len<1)
	return i;
    }
  return i-len;
}

static void
gtk_move_backward_character (GtkSHEntry *shentry)
{
  /* this routine is correct only if string is state-independent-encoded */

  if (0 < shentry->current_pos)
    {
      shentry->current_pos = move_backward_character (shentry->text,
      						    shentry->current_pos);
      if (shentry->current_pos < 0)
	shentry->current_pos = 0;
    }
}

static void
gtk_move_forward_word (GtkSHEntry *shentry)
{
  gchar *text;
  gint i;
  wchar_t c;
  gint len;

  if (shentry->text)
    {
      text = shentry->text;
      i = shentry->current_pos;

      len = mbtowc (&c, text+i, MB_CUR_MAX);
      if (!iswalnum(c))
	for (; i < shentry->text_length; i+=len)
	  {
	    len = mbtowc (&c, text+i, MB_CUR_MAX);
	    if (len < 1 || iswalnum(c))
	      break;
	  }
  
      for (; i < shentry->text_length; i+=len)
	{
	  len = mbtowc (&c, text+i, MB_CUR_MAX);
	  if (len < 1 || !iswalnum(c))
	    break;
	}

      shentry->current_pos = i;
      if (shentry->current_pos > shentry->text_length)
	shentry->current_pos = shentry->text_length;
    }
}

static void
gtk_move_backward_word (GtkSHEntry *shentry)
{
  gchar *text;
  gint i;
  wchar_t c;
  gint len;

  if (shentry->text)
    {
      text = shentry->text;
      i=move_backward_character(text, shentry->current_pos);
      if (i < 0) /* Per */
	{
	  shentry->selection_start_pos = 0;
	  shentry->selection_end_pos = 0;
	  return;
	}

      len = mbtowc (&c, text+i, MB_CUR_MAX);
      if (!iswalnum(c))
        for (; i >= 0; i=move_backward_character(text, i))
	  {
	    len = mbtowc (&c, text+i, MB_CUR_MAX);
	    if (iswalnum(c))
	      break;
	  }

      for (; i >= 0; i=move_backward_character(text, i))
	{
	  len = mbtowc (&c, text+i, MB_CUR_MAX);
	  if (!iswalnum(c))
	    {
	      i += len;
	      break;
	    }
	}

      if (i < 0)
        i = 0;

      shentry->current_pos = i;
    }
}

static void
gtk_move_beginning_of_line (GtkSHEntry *shentry)
{
  shentry->current_pos = 0;
}

static void
gtk_move_end_of_line (GtkSHEntry *shentry)
{
  shentry->current_pos = shentry->text_length;
}

static void
gtk_delete_forward_character (GtkSHEntry *shentry)
{
  gint old_pos;

  old_pos = shentry->current_pos;
  gtk_move_forward_character (shentry);
  gtk_shentry_delete_text (shentry, old_pos, shentry->current_pos);
}

static void
gtk_delete_backward_character (GtkSHEntry *shentry)
{
  gint old_pos;

  old_pos = shentry->current_pos;
  gtk_move_backward_character (shentry);
  gtk_shentry_delete_text (shentry, shentry->current_pos, old_pos);
}

static void
gtk_delete_forward_word (GtkSHEntry *shentry)
{
  gint old_pos;

  old_pos = shentry->current_pos;
  gtk_move_forward_word (shentry);
  gtk_shentry_delete_text (shentry, old_pos, shentry->current_pos);
}

static void
gtk_delete_backward_word (GtkSHEntry *shentry)
{
  gint old_pos;

  old_pos = shentry->current_pos;
  gtk_move_backward_word (shentry);
  gtk_shentry_delete_text (shentry, shentry->current_pos, old_pos);
}

static void
gtk_delete_line (GtkSHEntry *shentry)
{
  gtk_shentry_delete_text (shentry, 0, shentry->text_length);
}

static void
gtk_delete_to_line_end (GtkSHEntry *shentry)
{
  gtk_shentry_delete_text (shentry, shentry->current_pos, shentry->text_length);
}

static void
gtk_delete_selection (GtkSHEntry *shentry)
{
  if (shentry->selection_start_pos != shentry->selection_end_pos)
    gtk_shentry_delete_text (shentry,
			   MIN (shentry->selection_start_pos, shentry->selection_end_pos),
			   MAX (shentry->selection_start_pos, shentry->selection_end_pos));

  shentry->selection_start_pos = 0;
  shentry->selection_end_pos = 0;

  if (shentry->have_selection)
    {
      shentry->have_selection = FALSE;
      if (gdk_selection_owner_get (GDK_SELECTION_PRIMARY) == GTK_WIDGET (shentry)->window)
	gtk_selection_owner_set (NULL, GDK_SELECTION_PRIMARY, GDK_CURRENT_TIME);
    }
}

static void
gtk_select_word (GtkSHEntry *shentry)
{
  gint start_pos;
  gint end_pos;

  gtk_move_backward_word (shentry);
  start_pos = shentry->current_pos;
  end_pos = shentry->current_pos;

  gtk_move_forward_word (shentry);
  end_pos = shentry->current_pos;

  gtk_shentry_select_region (shentry, start_pos, end_pos);
}

static void
gtk_select_line (GtkSHEntry *shentry)
{
  gtk_shentry_select_region (shentry, 0, shentry->text_length);
  shentry->current_pos = shentry->selection_end_pos;
}

void
gtk_shentry_select_region (GtkSHEntry *shentry,
		   gint      start,
		   gint      end)
{
  shentry->have_selection = TRUE;

  shentry->selection_start_pos = start;
  shentry->selection_end_pos = end;

  gtk_widget_queue_draw (GTK_WIDGET (shentry));
}

static void
gtk_shentry_cut_clipboard (GtkSHEntry *shentry, GdkEventKey *event)
{
  gtk_shentry_copy_clipboard (shentry, event);
  gtk_delete_selection (shentry);

  gtk_shentry_queue_draw (shentry);
}

static void
gtk_shentry_copy_clipboard (GtkSHEntry *shentry, GdkEventKey *event)
{
  gint selection_start_pos; 
  gint selection_end_pos;

  selection_start_pos = MIN (shentry->selection_start_pos, shentry->selection_end_pos);
  selection_end_pos = MAX (shentry->selection_start_pos, shentry->selection_end_pos);
 
  if (selection_start_pos != selection_end_pos)
    {
      if (gtk_selection_owner_set (GTK_WIDGET (shentry),
				   clipboard_atom,
				   event->time))
	{
	  char c;

	  c = shentry->text[selection_end_pos];
	  shentry->text[selection_end_pos] = 0;
	  shentry->clipboard_text = g_strdup (shentry->text + selection_start_pos);
	  shentry->text[selection_end_pos] = c;
	}
    }
}

static void
gtk_shentry_paste_clipboard (GtkSHEntry *shentry, GdkEventKey *event)
{
  gtk_selection_convert (GTK_WIDGET(shentry), 
			 clipboard_atom, ctext_atom, event->time);
}

void
gtk_shentry_set_max_length (GtkSHEntry *shentry, guint16 max)
{
  g_return_if_fail (shentry != NULL);
  g_return_if_fail (GTK_IS_SHENTRY (shentry));

  shentry->text_max_length = max;
}
