/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 * Copyright (C) 1998-1999 Adrian E. Feiguin <adrian@ifir.ifir.edu.ar>
 *
 * GtkSheet widget for Gtk+, by Adrian E. Feiguin, July 1998.
 * Based on GtkClist widget by Jay Painter, but major changes.
 * Memory allocation routines inspired on SC (Spreadsheet Calculator)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#ifndef __GTK_SHEET_H__
#define __GTK_SHEET_H__


#include <gdk/gdk.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkalignment.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkbutton.h>
#include <gtk/gtkhscrollbar.h>
#include <gtk/gtkvscrollbar.h>
#include <gtk/gtkmain.h>
#include <gdk/gdkkeysyms.h>
#include "gtksheetentry.h"


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


typedef enum{GTK_SHEET_FOREGROUND,
	     GTK_SHEET_BACKGROUND,
             GTK_SHEET_FONT,
	     GTK_SHEET_JUSTIFICATION,
	     GTK_SHEET_BORDER,
             GTK_SHEET_BORDER_COLOR,
             GTK_SHEET_IS_EDITABLE,
             GTK_SHEET_IS_VISIBLE} GtkSheetAttrType;

/* sheet->state */

enum {GTK_SHEET_NORMAL,
      GTK_SHEET_ROW_SELECTED,
      GTK_SHEET_COLUMN_SELECTED,
      GTK_SHEET_RANGE_SELECTED};
     
enum
{
  CELL_LEFT_BORDER    = 1 << 0, 
  CELL_RIGHT_BORDER    = 1 << 1, 
  CELL_TOP_BORDER    = 1 << 2, 
  CELL_BOTTOM_BORDER    = 1 << 3, 
}; 

/* sheet flags */
enum                    
{
  SHEET_IS_LOCKED       = 1 << 0,
  SHEET_IS_FROZEN       = 1 << 1,                                     
  SHEET_IN_XDRAG        = 1 << 2,                                        
  SHEET_IN_YDRAG        = 1 << 3,                                        
  SHEET_IN_DRAG         = 1 << 4,                                        
  SHEET_IN_SELECTION    = 1 << 5,
  SHEET_IN_RESIZE       = 1 << 6,
  SHEET_IN_CLIP         = 1 << 7,
  SHEET_ROW_FROZEN      = 1 << 8,  /* set rows to be resizeable */
  SHEET_COLUMN_FROZEN   = 1 << 9,  /* set cols to be resizeable */
  SHEET_AUTORESIZE      = 1 << 10,  /* resize column if text width is great than column width */
  SHEET_ADJUST_TEXT     = 1 << 11, /* fix text to column width */
  SHEET_ROW_TITLES_VISIBLE = 1 << 12,
  SHEET_COL_TITLES_VISIBLE = 1 << 13,
}; 

#define GTK_SHEET(obj)          GTK_CHECK_CAST (obj, gtk_sheet_get_type (), GtkSheet)
#define GTK_SHEET_CLASS(klass)  GTK_CHECK_CLASS_CAST (klass, gtk_sheet_get_type (), GtkSheetClass)
#define GTK_IS_SHEET(obj)       GTK_CHECK_TYPE (obj, gtk_sheet_get_type ())

#define GTK_SHEET_FLAGS(sheet)             (GTK_SHEET (sheet)->flags)
#define GTK_SHEET_SET_FLAGS(sheet,flag)    (GTK_SHEET_FLAGS (sheet) |= (flag))
#define GTK_SHEET_UNSET_FLAGS(sheet,flag)  (GTK_SHEET_FLAGS (sheet) &= ~(flag))

#define GTK_SHEET_IS_LOCKED(sheet)   (GTK_SHEET_FLAGS (sheet) & SHEET_IS_LOCKED)
#define GTK_SHEET_IS_FROZEN(sheet)   (GTK_SHEET_FLAGS (sheet) & SHEET_IS_FROZEN)
#define GTK_SHEET_IN_XDRAG(sheet)     (GTK_SHEET_FLAGS (sheet) & SHEET_IN_XDRAG)
#define GTK_SHEET_IN_YDRAG(sheet)     (GTK_SHEET_FLAGS (sheet) & SHEET_IN_YDRAG)
#define GTK_SHEET_IN_DRAG(sheet)     (GTK_SHEET_FLAGS (sheet) & SHEET_IN_DRAG)
#define GTK_SHEET_IN_SELECTION(sheet) (GTK_SHEET_FLAGS (sheet) & SHEET_IN_SELECTION)
#define GTK_SHEET_IN_RESIZE(sheet) (GTK_SHEET_FLAGS (sheet) & SHEET_IN_RESIZE)
#define GTK_SHEET_IN_CLIP(sheet) (GTK_SHEET_FLAGS (sheet) & SHEET_IN_CLIP)
#define GTK_SHEET_ROW_FROZEN(sheet)   (GTK_SHEET_FLAGS (sheet) & SHEET_ROW_FROZEN)
#define GTK_SHEET_COLUMN_FROZEN(sheet)   (GTK_SHEET_FLAGS (sheet) & SHEET_COLUMN_FROZEN)
#define GTK_SHEET_AUTORESIZE(sheet)   (GTK_SHEET_FLAGS (sheet) & SHEET_AUTORESIZE)
#define GTK_SHEET_ADJUST_TEXT(sheet)   (GTK_SHEET_FLAGS (sheet) & SHEET_ADJUST_TEXT)
#define GTK_SHEET_ROW_TITLES_VISIBLE(sheet)   (GTK_SHEET_FLAGS (sheet) & SHEET_ROW_TITLES_VISIBLE)
#define GTK_SHEET_COL_TITLES_VISIBLE(sheet)   (GTK_SHEET_FLAGS (sheet) & SHEET_COL_TITLES_VISIBLE)

typedef struct _GtkSheet GtkSheet;
typedef struct _GtkSheetClass GtkSheetClass;
typedef struct _GtkSheetRow GtkSheetRow;
typedef struct _GtkSheetColumn GtkSheetColumn;
typedef struct _GtkSheetCell GtkSheetCell;
typedef struct _GtkSheetRange GtkSheetRange;
typedef struct _GtkSheetButton       GtkSheetButton;
typedef struct _GtkSheetCellAttr     GtkSheetCellAttr;
typedef struct _GtkSheetAttr     GtkSheetAttr;
typedef struct _GtkSheetCellBorder     GtkSheetCellBorder;

struct _GtkSheetButton
{
  gint state;
  gchar *label;
};

struct _GtkSheetCellBorder
{
  gint8 mask;
  gint width;
  GdkLineStyle line_style;
  GdkCapStyle cap_style;
  GdkJoinStyle join_style;
  GdkColor color;
};

struct _GtkSheetCellAttr
{
  gint justification;
  GdkFont *font;
  GdkColor foreground;
  GdkColor background;
  GtkSheetCellBorder border;
  gint is_editable:1;
  gint is_visible:1;
};

struct _GtkSheetCell
{
  GdkRectangle area;
  gint row;
  gint col;

  GtkSheetCellAttr attributes;

  gchar *text;

};

struct _GtkSheetRange
{
  gint row0,col0;
  gint rowi,coli;
};

struct _GtkSheetAttr
{
  GtkSheetRange range;
  GtkSheetAttrType type;

  union {
          gint justification;
          GdkFont *font;
          GdkColor foreground;
          GdkColor background;
          GtkSheetCellBorder border;
          gint is_editable:1;
          gint is_visible:1;
  } value;

};


struct _GtkSheetRow
{
 gchar *name;
 gint height;
 gint top_ypixel;

 GtkSheetButton button;
 gint is_sensitive:1;
 gint is_visible:1;
};

struct _GtkSheetColumn
{
 gchar *name;
 gint width;
 gint left_xpixel;

 GtkSheetButton button;

 gint left_text_column; /* min left column displaying text on this column */
 gint right_text_column; /* max right column displaying text on this column */

 gint is_sensitive:1;
 gint is_visible:1;
};


struct _GtkSheet{
  GtkContainer container;

  guint16 flags;

  /* allocation rectangle after the container_border_width
     and the width of the shadow border */
  GdkRectangle internal_allocation;

  gchar *name;

  GtkSheetRow *row;
  GtkSheetColumn *column;

  /* max number of diplayed cells */
  gint maxrow;
  gint maxcol;

  /* Displayed range */

  GtkSheetRange view; 

  /* sheet data: dynamically allocated array of cell pointers */
  GtkSheetCell ***data;

  /* max number of allocated cells */
  gint maxallocrow;
  gint maxalloccol;

  /* active cell */
  GtkSheetCell active_cell;
  GtkWidget *sheet_entry;

  /* expanding selection */
  GtkSheetCell selection_cell;

  /* timer for automatic scroll during selection */  
  gint32 timer;
  /* timer for flashing clipped range */
  gint32 clip_timer;
  gint interval;

  /* global selection button */
  GtkWidget *button;

  /* sheet state */
  gint state;

  /* selected range */
  GtkSheetRange range;

  /* sheet attributes */
  GtkSheetAttr *attributes;
  gint maxrange;

  /*the scrolling window and it's height and width to
   * make things a little speedier */
  GdkWindow *sheet_window;
  gint sheet_window_width;
  gint sheet_window_height;

  /* sheet backing pixmap */  
  GdkWindow *pixmap;    

  /* offsets for scrolling */
  gint hoffset;
  gint voffset;
  
  /* border shadow style */
  GtkShadowType shadow_type;
  
  /* Column Titles */
  GdkRectangle column_title_area;
  GdkWindow *column_title_window;
  GtkWidget **column_button;

  /* Row Titles */
  GdkRectangle row_title_area;
  GdkWindow *row_title_window;
  GtkWidget **row_button;

  /*scrollbars*/
  GtkWidget *vscrollbar;
  GtkWidget *hscrollbar;
  guint8 vscrollbar_policy;
  guint8 hscrollbar_policy;

  /* xor GC for the verticle drag line */
  GdkGC *xor_gc;

  /* gc for drawing unselected cells */
  GdkGC *fg_gc;
  GdkGC *bg_gc;

  /* cursor used to indicate dragging */
  GdkCursor *cursor_drag;

  /* the current x-pixel location of the xor-drag vline */
  gint x_drag;

  /* the current y-pixel location of the xor-drag hline */
  gint y_drag;

  /* current cell being dragged */
  GtkSheetCell drag_cell;
  /* current range being dragged */
  GtkSheetRange drag_range;

  /* clipped range */
  GtkSheetRange clip_range;
};

struct _GtkSheetClass
{
 GtkContainerClass parent_class;

 void (*select_row) (GtkSheet *sheet,
	             gint row, gint column);

 void (*select_column) (GtkSheet *sheet,
	                gint row, gint column);

 void (*select_range) (GtkSheet *sheet,
	               GtkSheetRange *range);

 void (*clip_range) (GtkSheet *sheet,
	               GtkSheetRange *clip_range);

 void (*resize_range)  (GtkSheet *sheet,
	                 GtkSheetRange *old_range,
                         GtkSheetRange *new_range);

 void (*move_range)    (GtkSheet *sheet,
	                 GtkSheetRange *old_range,
                         GtkSheetRange *new_range);

 void (*activate_cell) (GtkSheet *sheet,
	               gint row, gint column);

 void (*set_cell) (GtkSheet *sheet,
	               gint row, gint column);

 void (*changed) (GtkSheet *sheet,
	               gint row, gint column);


 gint scrollbar_spacing;
};
  
guint gtk_sheet_get_type (void);

/* create en new sheet */
GtkWidget *
gtk_sheet_new (int rows, int columns, gchar *title);

/* create a new sheet with autoformat for numbers */
GtkWidget *
gtk_sheet_new_with_autoformat (int rows, int columns, gchar *title);

/* create a new browser sheet. It cells can not be edited */
GtkWidget *
gtk_sheet_new_browser (int rows, int columns, gchar *title);

/* set sheet title */
void
gtk_sheet_set_title (GtkSheet *sheet, gchar *title);

/* set border style */
void
gtk_sheet_set_border (GtkSheet * sheet,
		      GtkShadowType border);

/* freeze all visual updates of the sheet.
 * Then thaw the sheet after you have made a number of changes.
 * The updates will occure in a more efficent way than if
 * you made them on a unfrozen sheet */
void
gtk_sheet_freeze(GtkSheet *sheet);
void
gtk_sheet_thaw(GtkSheet *sheet);

/* set column title */ 
void
gtk_sheet_set_column_title (GtkSheet * sheet,
			    gint column,
			    gchar * title);

/* set row title */
void
gtk_sheet_set_row_title (GtkSheet * sheet,
			    gint row,
			    gchar * title);

/* set button label */
void
gtk_sheet_row_button_add_label(GtkSheet *sheet, gint row, char *label);
void
gtk_sheet_column_button_add_label(GtkSheet *sheet, gint column, char *label);

/* scroll the viewing area of the sheet to the given column
 * and row; row_align and col_align are between 0-1 representing the
 * location the row should appear on the screnn, 0.0 being top or left,
 * 1.0 being bottom or right; if row or column is negative then then there
 * is no change */
void
gtk_sheet_moveto (GtkSheet * sheet,
		  gint row,
		  gint column,
	          gfloat row_align,
                  gfloat col_align);

/* show/hide column/row titles window */
void
gtk_sheet_show_column_titles(GtkSheet *sheet);
void
gtk_sheet_show_row_titles(GtkSheet *sheet);
void
gtk_sheet_hide_column_titles(GtkSheet *sheet);
void
gtk_sheet_hide_row_titles(GtkSheet *sheet);

/* set column button sensitivity. If sensitivity is TRUE it responds selecting  
 * the corresponding column, otherwise it acts as a title */
 void 
gtk_sheet_column_set_sensitivity(GtkSheet *sheet, gint column, gint sensitive);

/* set sensitivity for all column buttons */
void
gtk_sheet_columns_set_sensitivity(GtkSheet *sheet, gint sensitive);


/* set row button sensitivity. If sensitivity is TRUE it responds selecting  
 * the corresponding row, otherwise it acts as a title */
void 
gtk_sheet_row_set_sensitivity(GtkSheet *sheet, gint row,  gint sensitive);

/* set sensitivity for all row buttons */
void
gtk_sheet_rows_set_sensitivity(GtkSheet *sheet, gint sensitive);

/* set column visibility. The default value is TRUE. If FALSE, the 
 * column is hidden */
void
gtk_sheet_column_set_visibility(GtkSheet *sheet, gint column, gint visible);

/* set row visibility. The default value is TRUE. If FALSE, the 
 * row is hidden */
void
gtk_sheet_row_set_visibility(GtkSheet *sheet, gint row, gint visible);

/* select the row. The range is then highlighted, and the bounds are stored
 * in sheet->range  */
void
gtk_sheet_select_row (GtkSheet * sheet,
		      gint row);

/* select the column. The range is then highlighted, and the bounds are stored
 * in sheet->range  */
void
gtk_sheet_select_column (GtkSheet * sheet,
		         gint column);

/* save selected range to "clipboard" */
void
gtk_sheet_clip_range (GtkSheet *sheet, GtkSheetRange range);
/* free clipboard */
void
gtk_sheet_unclip_range(GtkSheet *sheet);

/* get scrollbars adjustment */
GtkAdjustment *
gtk_sheet_get_vadjustment (GtkSheet * sheet);
GtkAdjustment *
gtk_sheet_get_hadjustment (GtkSheet * sheet);

/* set scrollbars policy */
void
gtk_sheet_set_policy (GtkSheet * sheet,
		      GtkPolicyType vscrollbar_policy,
		      GtkPolicyType hscrollbar_policy);
void
gtk_sheet_set_update_policy (GtkSheet * sheet,
		      GtkUpdateType vscrollbar_policy,
		      GtkUpdateType hscrollbar_policy);

/* highlight the selected range and store bounds in sheet->range */
void gtk_sheet_select_range(GtkSheet *sheet, GtkSheetRange *range); 

/* obvious */
void gtk_sheet_unselect_range(GtkSheet *sheet, GtkSheetRange *range); 

/* draw cell background and frame */
void
gtk_sheet_cell_draw_default (GtkSheet *sheet, gint row, gint column);

/* draw cell border */
void
gtk_sheet_cell_draw_border (GtkSheet *sheet, gint row, gint column, gint mask);

/* draw cell contents */
void
gtk_sheet_cell_draw_label (GtkSheet *sheet, gint row, gint column);

/* draw cell */
void
gtk_sheet_cell_draw (GtkSheet *sheet, gint row, gint column);

/* draw visible part of range. If range==NULL then draw the whole screen */
void
gtk_sheet_range_draw(GtkSheet *sheet, GtkSheetRange *range);

/* highlight the visible part of the selected range */
void
gtk_sheet_range_draw_selection(GtkSheet *sheet, GtkSheetRange range);

/* set active cell where the entry will be displayed */
void
gtk_sheet_set_active_cell (GtkSheet *sheet, gint row, gint column);

/* set cell contents and allocate memory if needed */
void 
gtk_sheet_set_cell(GtkSheet *sheet, int row, int col, int justification,
                   gchar *text);

/* get cell contents */
gchar *     
gtk_sheet_cell_get_text (GtkSheet *sheet, gint row, gint col);


/* clear cell contents */
void gtk_sheet_cell_clear(GtkSheet *sheet, gint row, gint col);

/* clear range contents. If range==NULL the whole sheet will be cleared */
void gtk_sheet_range_clear(GtkSheet *sheet, GtkSheetRange *range);

/* get cell state: GTK_STATE_NORMAL, GTK_STATE_SELECTED */
int
gtk_sheet_cell_get_state (GtkSheet *sheet, int row, int col);


/* get row and column correspondig to the given position in the screen */
gint
gtk_sheet_get_pixel_info (GtkSheet * sheet,
			  gint x,
			  gint y,
			  gint * row,
			  gint * column);

/* get area of a given cell */
gint
gtk_sheet_get_cell_info (GtkSheet *sheet,
                         gint row,
                         gint column,
                         GdkRectangle *area);

/* set column width */
void
gtk_sheet_set_column_width (GtkSheet * sheet,
			    gint column,
			    gint width);

/* set row height */
void
gtk_sheet_set_row_height (GtkSheet * sheet,
			    gint row,
			    gint height);

/* append ncols columns to the end of the sheet */
void
gtk_sheet_add_column(GtkSheet *sheet, gint ncols);

/* append nrows row to the end of the sheet */
void
gtk_sheet_add_row(GtkSheet *sheet, gint nrows);

/* insert nrows rows before the given row and pull right */
void
gtk_sheet_insert_rows(GtkSheet *sheet, gint row, gint nrows);

/* insert ncols columns before the given col and pull down */
 void
gtk_sheet_insert_columns(GtkSheet *sheet, gint col, gint ncols);

/* delete nrows rows starting in row */
void
gtk_sheet_delete_rows(GtkSheet *sheet, gint row, gint nrows);

/* delete ncols columns starting in col */
void
gtk_sheet_delete_columns(GtkSheet *sheet, gint col, gint ncols);

/* set abckground color of the given range */
void
gtk_sheet_range_set_background(GtkSheet *sheet, GtkSheetRange range, GdkColor *color);

/* set foreground color (text color) of the given range */
void
gtk_sheet_range_set_foreground(GtkSheet *sheet, GtkSheetRange range, GdkColor *color);

/* set text justification (GTK_JUSTIFY_LEFT, RIGHT, CENTER) of the given range.
 * The default value is GTK_JUSTIFY_LEFT. If autoformat is on, the
 * default justification for numbers is GTK_JUSTIFY_RIGHT */
void
gtk_sheet_range_set_justification(GtkSheet *sheet, GtkSheetRange range, gint just);

/* set if cell contents can be edited or not in the given range:
 * accepted values are TRUE or FALSE. */
void
gtk_sheet_range_set_editable(GtkSheet *sheet, GtkSheetRange range, gint editable);

/* set if cell contents are visible or not in the given range:
 * accepted values are TRUE or FALSE.*/
void
gtk_sheet_range_set_visible(GtkSheet *sheet, GtkSheetRange range, gint visible);

/* set cell border style in the given range.
 * mask values are CELL_LEFT_BORDER, CELL_RIGHT_BORDER, CELL_TOP_BORDER,
 * CELL_BOTTOM_BORDER
 * width is the width of the border line in pixels 
 * line_style is the line_style for the border line */
void
gtk_sheet_range_set_border(GtkSheet *sheet, GtkSheetRange range, gint mask, 
gint width, gint line_style);

/* set border color for the given range */
void
gtk_sheet_range_set_border_color(GtkSheet *sheet, GtkSheetRange range, GdkColor *color);

/* set font for the given range */
/* NOT IMPLEMENTED. I leave this for someone with experience on X fonts */
void
gtk_sheet_range_set_font(GtkSheet *sheet, ...);


/* get cell attributes of the given cell */
void
gtk_sheet_get_attributes(GtkSheet *sheet, gint row, gint col, GtkSheetCellAttr *attributes);



#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* __GTK_SHEET_H__ */


