#include "pkg.h"

#include "popt.h"
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "config.h"

int 
main (int argc, char **argv)
{
  int want_my_version = 0;
  int want_version = 0;
  int want_libs = 0;
  int want_cflags = 0;
  int want_l_libs = 0;
  int want_L_libs = 0;
  int want_I_cflags = 0;
  int want_list = 0;
  int result;
  GSList *packages = NULL;
  char *search_path;
  gboolean need_newline;
  char *variable_name = NULL;
  int want_exists = 0;
  char *required_atleast_version = NULL;
  char *required_exact_version = NULL;
  char *required_max_version = NULL;
  char *required_pkgconfig_version = NULL;
  
  const char *pkgname;
  Package *pkg;

  poptContext opt_context;
  
  struct poptOption options_table[] = {
    { "version", 0, POPT_ARG_NONE, &want_my_version, 0,
      "output version of pkg-config" },
    { "modversion", 0, POPT_ARG_NONE, &want_version, 0,
      "output version for package" },
    { "atleast-pkgconfig-version", 0, POPT_ARG_STRING, &required_pkgconfig_version, 0,
      "require given version of pkg-config", "VERSION" },
    { "libs", 0, POPT_ARG_NONE, &want_libs, 0,
      "output all linker flags" },
    { "libs-only-l", 0, POPT_ARG_NONE, &want_l_libs, 0,
      "output -l flags" },
    { "libs-only-L", 0, POPT_ARG_NONE, &want_L_libs, 0,
      "output -L flags" },
    { "cflags", 0, POPT_ARG_NONE, &want_cflags, 0,
      "output all pre-processor and compiler flags" },
    { "cflags-only-I", 0, POPT_ARG_NONE, &want_I_cflags, 0,
      "output -I flags" },
    { "variable", 0, POPT_ARG_STRING, &variable_name, 0,
      "get the value of a variable", "VARIABLENAME" },
    { "exists", 0, POPT_ARG_NONE, &want_exists, 0,
      "return 0 if the module(s) exist" },
    { "atleast-version", 0, POPT_ARG_STRING, &required_atleast_version, 0,
      "return 0 if the module is at least version VERSION", "VERSION" },
    { "exact-version", 0, POPT_ARG_STRING, &required_exact_version, 0,
      "return 0 if the module is at exactly version VERSION", "VERSION" },
    { "max-version", 0, POPT_ARG_STRING, &required_max_version, 0,
      "return 0 if the module is at no newer than version VERSION", "VERSION" },
    { "list-all", 0, POPT_ARG_NONE, &want_list, 0,
      "list all known packages" },
    POPT_AUTOHELP
    { NULL, 0, 0, NULL, 0 }
  };

  search_path = getenv ("PKG_CONFIG_PATH");
  if (search_path)
    {
      char **search_dirs;
      char **iter;

      search_dirs = g_strsplit (search_path, ":", -1);

      iter = search_dirs;
      while (*iter)
        {
          add_search_dir (*iter);

          ++iter;
        }

      g_strfreev (search_dirs);
    }

  
  opt_context = poptGetContext (NULL, argc, argv,
                                options_table, 0);

  result = poptGetNextOpt (opt_context);
  if (result != -1)
    {
      fprintf(stderr, "%s: %s\n",
	      poptBadOption(opt_context, POPT_BADOPTION_NOALIAS),
	      poptStrerror(result));
      return 0;
    }

  if (want_my_version)
    {
      printf ("%s\n", VERSION);
      return 0;
    }

  if (required_pkgconfig_version)
    {
      if (compare_versions (VERSION, required_pkgconfig_version) >= 0)
        return 0;
      else
        return 1;
    }
  
  package_init ();

  if (want_list)
    {
      print_package_list ();
      return 0;
    }

  while (1)
    {
      pkgname = poptGetArg (opt_context);
      if (pkgname == NULL)
	break;
      
      pkg = get_package (pkgname);

      if (pkg == NULL)
        {
          fprintf (stderr, "Unknown package '%s'\n", pkgname);
          exit (1);
        }

      packages = g_slist_prepend (packages, pkg);
    }
  
  packages = g_slist_reverse (packages);

  if (packages == NULL)
    {
      fprintf (stderr, "Must specify package names on the command line\n");
      
      exit (1);
    }
  
  if (want_exists)
    return 0; /* if we got here, all the packages existed. */
  
  if (want_version)
    {
      GSList *tmp;
      tmp = packages;
      while (tmp != NULL)
        {
          Package *pkg = tmp->data;

          printf ("%s\n", pkg->version);

          tmp = g_slist_next (tmp);
        }
    }

  if (required_exact_version)
    {
      Package *pkg = packages->data;
      
      if (compare_versions (pkg->version, required_exact_version) == 0)
        return 0;
      else
        return 1;
    }
  else if (required_atleast_version)
    {
      Package *pkg = packages->data;
      
      if (compare_versions (pkg->version, required_atleast_version) >= 0)
        return 0;
      else
        return 1;
    }
  else if (required_max_version)
    {
      Package *pkg = packages->data;
      
      if (compare_versions (pkg->version, required_max_version) <= 0)
        return 0;
      else
        return 1;
    }
  
  /* Print all flags; then print a newline at the end. */
  need_newline = FALSE;

  if (variable_name)
    {
      char *str = packages_get_var (packages, variable_name);
      printf ("%s", str);
      g_free (str);
      need_newline = TRUE;
    }
  
  if (want_I_cflags)
    {
      char *str = packages_get_I_cflags (packages);
      printf ("%s ", str);
      g_free (str);
      need_newline = TRUE;
    }
  else if (want_cflags)
    {
      char *str = packages_get_all_cflags (packages);
      printf ("%s ", str);
      g_free (str);
      need_newline = TRUE;
    }
  
  if (want_l_libs)
    {
      char *str = packages_get_l_libs (packages);
      printf ("%s ", str);
      g_free (str);
      need_newline = TRUE;
    }
  else if (want_L_libs)
    {
      char *str = packages_get_L_libs (packages);
      printf ("%s ", str);
      g_free (str);
      need_newline = TRUE;
    }
  else if (want_libs)
    {
      char *str = packages_get_all_libs (packages);
      printf ("%s ", str);
      g_free (str);
      need_newline = TRUE;
    }

  if (need_newline)
    printf ("\n");
  
  return 0;
}
