/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * GtkAspectFrame: Ensure that the child window has a specified aspect ratio
 *    or, if obey_child, has the same aspect ratio as its requested size
 *
 *     Copyright Owen Taylor                          4/9/97
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 * Modified by the GTK+ Team and others 1997-2000.  See the AUTHORS
 * file for a list of people on the GTK+ Team.  See the ChangeLog
 * files for a list of changes.  These files are distributed with
 * GTK+ at ftp://ftp.gtk.org/pub/gtk/. 
 */

#include "gtkaspectframe.h"

enum {
  ARG_0,
  ARG_XALIGN,
  ARG_YALIGN,
  ARG_RATIO,
  ARG_OBEY_CHILD
};

static void gtk_aspect_frame_class_init               (GtkAspectFrameClass *klass);
static void gtk_aspect_frame_init                     (GtkAspectFrame      *aspect_frame);
static void gtk_aspect_frame_set_arg                  (GtkObject           *object,
						       GtkArg              *arg,
						       guint                arg_id);
static void gtk_aspect_frame_get_arg                  (GtkObject           *object,
						       GtkArg              *arg,
						       guint                arg_id);
static void gtk_aspect_frame_compute_child_allocation (GtkFrame            *frame,
						       GtkAllocation       *child_allocation);

#define MAX_RATIO 10000.0
#define MIN_RATIO 0.0001

static GtkFrameClass *parent_class = NULL;

GtkType
gtk_aspect_frame_get_type (void)
{
  static GtkType aspect_frame_type = 0;
  
  if (!aspect_frame_type)
    {
      static const GtkTypeInfo aspect_frame_info =
      {
	"GtkAspectFrame",
	sizeof (GtkAspectFrame),
	sizeof (GtkAspectFrameClass),
	(GtkClassInitFunc) gtk_aspect_frame_class_init,
	(GtkObjectInitFunc) gtk_aspect_frame_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };
      
      aspect_frame_type = gtk_type_unique (GTK_TYPE_FRAME, &aspect_frame_info);
    }
  
  return aspect_frame_type;
}

static void
gtk_aspect_frame_class_init (GtkAspectFrameClass *class)
{
  GtkObjectClass *object_class;
  GtkFrameClass *frame_class;
  
  parent_class = gtk_type_class (GTK_TYPE_FRAME);

  object_class = GTK_OBJECT_CLASS (class);
  frame_class = GTK_FRAME_CLASS (class);
  
  object_class->set_arg = gtk_aspect_frame_set_arg;
  object_class->get_arg = gtk_aspect_frame_get_arg;

  frame_class->compute_child_allocation = gtk_aspect_frame_compute_child_allocation;

  gtk_object_add_arg_type ("GtkAspectFrame::xalign", GTK_TYPE_FLOAT,
			   GTK_ARG_READWRITE, ARG_XALIGN);
  gtk_object_add_arg_type ("GtkAspectFrame::yalign", GTK_TYPE_FLOAT,
			   GTK_ARG_READWRITE, ARG_YALIGN);
  gtk_object_add_arg_type ("GtkAspectFrame::ratio", GTK_TYPE_FLOAT,
			   GTK_ARG_READWRITE, ARG_RATIO);
  gtk_object_add_arg_type ("GtkAspectFrame::obey_child", GTK_TYPE_BOOL,
			   GTK_ARG_READWRITE, ARG_OBEY_CHILD);  
}

static void
gtk_aspect_frame_init (GtkAspectFrame *aspect_frame)
{
  aspect_frame->xalign = 0.5;
  aspect_frame->yalign = 0.5;
  aspect_frame->ratio = 1.0;
  aspect_frame->obey_child = TRUE;
}

static void
gtk_aspect_frame_set_arg (GtkObject *object,
			  GtkArg    *arg,
			  guint      arg_id)
{
  GtkAspectFrame *aspect_frame = GTK_ASPECT_FRAME (object);
  
  switch (arg_id)
    {
    case ARG_XALIGN:
      gtk_aspect_frame_set (aspect_frame,
			    GTK_VALUE_FLOAT (*arg),
			    aspect_frame->yalign,
			    aspect_frame->ratio,
			    aspect_frame->obey_child);
      break;
    case ARG_YALIGN:
      gtk_aspect_frame_set (aspect_frame,
			    aspect_frame->xalign,
			    GTK_VALUE_FLOAT (*arg),
			    aspect_frame->ratio,
			    aspect_frame->obey_child);
      break;
    case ARG_RATIO:
      gtk_aspect_frame_set (aspect_frame,
			    aspect_frame->xalign,
			    aspect_frame->yalign,
			    GTK_VALUE_FLOAT (*arg),
			    aspect_frame->obey_child);
      break;
    case ARG_OBEY_CHILD:
      gtk_aspect_frame_set (aspect_frame,
			    aspect_frame->xalign,
			    aspect_frame->yalign,
			    aspect_frame->ratio,
			    GTK_VALUE_BOOL (*arg));
      break;
    }
}

static void
gtk_aspect_frame_get_arg (GtkObject *object,
			  GtkArg    *arg,
			  guint      arg_id)
{
  GtkAspectFrame *aspect_frame = GTK_ASPECT_FRAME (object);
  
  switch (arg_id)
    {
    case ARG_XALIGN:
      GTK_VALUE_FLOAT (*arg) = aspect_frame->xalign;
      break;
    case ARG_YALIGN:
      GTK_VALUE_FLOAT (*arg) = aspect_frame->yalign;
      break;
    case ARG_RATIO:
      GTK_VALUE_FLOAT (*arg) = aspect_frame->ratio;
      break;
    case ARG_OBEY_CHILD:
      GTK_VALUE_BOOL (*arg) = aspect_frame->obey_child;
      break;
    default:
      arg->type = GTK_TYPE_INVALID;
      break;
    }
}

GtkWidget*
gtk_aspect_frame_new (const gchar *label,
		      gfloat       xalign,
		      gfloat       yalign,
		      gfloat       ratio,
		      gboolean     obey_child)
{
  GtkAspectFrame *aspect_frame;

  aspect_frame = gtk_type_new (gtk_aspect_frame_get_type ());

  aspect_frame->xalign = CLAMP (xalign, 0.0, 1.0);
  aspect_frame->yalign = CLAMP (yalign, 0.0, 1.0);
  aspect_frame->ratio = CLAMP (ratio, MIN_RATIO, MAX_RATIO);
  aspect_frame->obey_child = obey_child != FALSE;

  gtk_frame_set_label (GTK_FRAME(aspect_frame), label);

  return GTK_WIDGET (aspect_frame);
}

void
gtk_aspect_frame_set (GtkAspectFrame *aspect_frame,
		      gfloat          xalign,
		      gfloat          yalign,
		      gfloat          ratio,
		      gboolean        obey_child)
{
  g_return_if_fail (aspect_frame != NULL);
  g_return_if_fail (GTK_IS_ASPECT_FRAME (aspect_frame));
  
  xalign = CLAMP (xalign, 0.0, 1.0);
  yalign = CLAMP (yalign, 0.0, 1.0);
  ratio = CLAMP (ratio, MIN_RATIO, MAX_RATIO);
  obey_child = obey_child != FALSE;
  
  if ((aspect_frame->xalign != xalign) ||
      (aspect_frame->yalign != yalign) ||
      (aspect_frame->ratio != ratio) ||
      (aspect_frame->obey_child != obey_child))
    {
      GtkWidget *widget = GTK_WIDGET(aspect_frame);
      
      aspect_frame->xalign = xalign;
      aspect_frame->yalign = yalign;
      aspect_frame->ratio = ratio;
      aspect_frame->obey_child = obey_child;
      
      gtk_widget_queue_resize (widget);
    }
}

static void
gtk_aspect_frame_compute_child_allocation (GtkFrame      *frame,
					   GtkAllocation *child_allocation)
{
  GtkAspectFrame *aspect_frame = GTK_ASPECT_FRAME (frame);
  GtkBin *bin = GTK_BIN (frame);
  gdouble ratio;

  if (bin->child && GTK_WIDGET_VISIBLE (bin->child))
    {
      GtkAllocation full_allocation;
      
      if (aspect_frame->obey_child)
	{
	  GtkRequisition child_requisition;

	  gtk_widget_get_child_requisition (bin->child, &child_requisition);
	  if (child_requisition.height != 0)
	    {
	      ratio = ((gdouble) child_requisition.width /
		       child_requisition.height);
	      if (ratio < MIN_RATIO)
		ratio = MIN_RATIO;
	    }
	  else if (child_requisition.width != 0)
	    ratio = MAX_RATIO;
	  else
	    ratio = 1.0;
	}
      else
	ratio = aspect_frame->ratio;

      parent_class->compute_child_allocation (frame, &full_allocation);
      
      if (ratio * full_allocation.height > full_allocation.width)
	{
	  child_allocation->width = full_allocation.width;
	  child_allocation->height = full_allocation.width / ratio + 0.5;
	}
      else
	{
	  child_allocation->width = ratio * full_allocation.height + 0.5;
	  child_allocation->height = full_allocation.height;
	}
      
      child_allocation->x = full_allocation.x + aspect_frame->xalign * (full_allocation.width - child_allocation->width);
      child_allocation->y = full_allocation.y + aspect_frame->yalign * (full_allocation.height - child_allocation->height);
    }
  else
    parent_class->compute_child_allocation (frame, child_allocation);
}
