/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998, 1999 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	<gle/gleconfig.h>

#include	"glegarg.h"
#include	<string.h>
#include	<stdio.h>


/* --- prototypes --- */
static void		gle_garg_callback	(GleGArg      *garg,
						 GleGArgCbType cb_type);


/* --- variables --- */
static GHashTable	*gle_arg_info_ht = NULL;
static GHashTable	*gle_child_arg_info_ht = NULL;
static GMemChunk	*gle_garg_mem_chunk = NULL;
static GMemChunk	*gle_arg_info_mem_chunk = NULL;
static GQuark		 quark_garg_cb = 0;


/* --- functions --- */
GleGArg*
gle_garg_new (const GleArgInfo *info)
{
  GtkType	object_type;
  GleGArg	*garg;

  g_return_val_if_fail (info != NULL, NULL);
  g_return_val_if_fail (info->arg != NULL, NULL);
  g_return_val_if_fail (info->arg->type != GTK_TYPE_INVALID, NULL);
  g_return_val_if_fail (info->arg->type != GTK_TYPE_NONE, NULL);

  object_type = gtk_type_from_arg_name (info->arg->name, NULL);
  g_return_val_if_fail (gtk_type_is_a (object_type, GTK_TYPE_OBJECT), NULL);

  if (!gle_garg_mem_chunk)
    gle_garg_mem_chunk = g_mem_chunk_create (GleGArg, GLE_GARGS_PREALLOC, G_ALLOC_AND_FREE);
  garg = g_chunk_new0 (GleGArg, gle_garg_mem_chunk);
  garg->arg_name = info->arg->name;
  garg->info = info;
  garg->widget = NULL;
  garg->setting_failed = FALSE;
  garg->needs_set = FALSE;
  garg->saved_arg.type = GTK_TYPE_INVALID;
  garg->saved_arg.name = info->arg->name;
  garg->object_arg.type = GTK_TYPE_INVALID;
  garg->object_arg.name = info->arg->name;
  
  gle_garg_reset (garg);
  
  garg->setting_failed = FALSE;
  garg->needs_set = FALSE;
  
  return garg;
}

void
gle_garg_reset (GleGArg	*garg)
{
  gboolean changed;

  g_return_if_fail (garg != NULL);

  changed = gle_arg_copy_value (garg->info->arg, &garg->object_arg);
  garg->needs_set |= changed;
  garg->setting_failed = FALSE;

  if (changed)
    gle_garg_callback (garg, GLE_GARG_CB_ARG_RESET);
}

gboolean
gle_garg_defaults (GleGArg *garg)
{
  g_return_val_if_fail (garg != NULL, FALSE);

  return gle_arg_values_equal (garg->info->arg, &garg->object_arg);
}

void
gle_garg_save (GleGArg *garg)
{
  g_return_if_fail (garg != NULL);

  garg->setting_failed = FALSE;
  if (garg->object_arg.type > GTK_TYPE_NONE)
    {
      if (gle_arg_copy_value (&garg->object_arg, &garg->saved_arg))
	gle_garg_callback (garg, GLE_GARG_CB_ARG_SAVE);
    }
  else
    {
      gle_arg_reset (&garg->object_arg);
      gle_arg_reset (&garg->saved_arg);
      garg->needs_set = FALSE;
      gle_garg_callback (garg, GLE_GARG_CB_ARG_SAVE);
    }
}

void
gle_garg_restore (GleGArg *garg)
{
  g_return_if_fail (garg != NULL);

  garg->setting_failed = FALSE;
  if (garg->saved_arg.type > GTK_TYPE_NONE)
    {
      gboolean changed;

      changed = gle_arg_copy_value (&garg->saved_arg, &garg->object_arg);
      garg->needs_set |= changed;
      if (changed)
	gle_garg_callback (garg, GLE_GARG_CB_ARG_RESTORE);
    }
  else
    {
      gle_arg_reset (&garg->saved_arg);
      gle_arg_reset (&garg->object_arg);
      garg->needs_set = FALSE;
      gle_garg_callback (garg, GLE_GARG_CB_ARG_RESTORE);
    }
}

void
gle_garg_free (GleGArg*        garg)
{
  g_return_if_fail (garg != NULL);
  g_return_if_fail (garg->widget == NULL);

  gle_arg_reset (&garg->saved_arg);
  gle_arg_reset (&garg->object_arg);
  
  g_chunk_free (garg, gle_garg_mem_chunk);
}

void
gle_garg_get (GleGArg        *garg,
	      GtkObject      *object,
	      GtkWidget	     *child)
{
  g_return_if_fail (garg != NULL);
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_OBJECT (object));
  g_return_if_fail (GLE_GARG_IS_READABLE (garg));
  g_return_if_fail (gtk_type_is_a (GTK_OBJECT_TYPE (object), GLE_GARG_CLASS_TYPE (garg)));

  if (GLE_GARG_IS_CHILD_ARG (garg))
    {
      g_return_if_fail (GTK_IS_CONTAINER (object));
      g_return_if_fail (child != NULL);
      g_return_if_fail (GTK_IS_WIDGET (child));
    }
  
  gle_arg_reset (&garg->object_arg);
  garg->object_arg.type = garg->info->arg->type;

  if (GLE_GARG_IS_CHILD_ARG (garg))
    gtk_container_child_getv (GTK_CONTAINER (object), child, 1, &garg->object_arg);
  else
    gtk_object_getv (object, 1, &garg->object_arg);
  garg->needs_set = FALSE;
  garg->setting_failed = FALSE;
  gle_garg_callback (garg, GLE_GARG_CB_ARG_GET);
}

void
gle_garg_set (GleGArg        *garg,
	      GtkObject      *object,
	      GtkWidget	     *child)
{
  g_return_if_fail (garg != NULL);
  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_OBJECT (object));
  g_return_if_fail (GLE_GARG_IS_WRITABLE (garg));
  g_return_if_fail (gtk_type_is_a (GTK_OBJECT_TYPE (object), GLE_GARG_CLASS_TYPE (garg)));


  if (GLE_GARG_IS_CHILD_ARG (garg))
    {
      g_return_if_fail (GTK_IS_CONTAINER (object));
      g_return_if_fail (child != NULL);
      g_return_if_fail (GTK_IS_WIDGET (child));
    }

  if (garg->object_arg.type > GTK_TYPE_NONE)
    {
      GtkArg arg;

      g_memmove (&arg, &garg->object_arg, sizeof (arg));

      if (GLE_GARG_IS_CHILD_ARG (garg))
	gtk_container_child_setv (GTK_CONTAINER (object), child, 1, &garg->object_arg);
      else
	gtk_object_setv (object, 1, &arg);
      garg->setting_failed = !(arg.type > GTK_TYPE_NONE);
    }
  else
    garg->setting_failed = TRUE;
  garg->needs_set = FALSE;
  gle_garg_callback (garg, GLE_GARG_CB_ARG_SET);
}

void
gle_garg_widget_set_cb (GtkWidget              *widget,
			GleGArgCbFunc           cb_func)
{
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_WIDGET (widget));

  if (!quark_garg_cb)
    quark_garg_cb = g_quark_from_static_string (GLE_PRIVATE_KEY (garg-cb));

  gtk_object_set_data_by_id (GTK_OBJECT (widget), quark_garg_cb, cb_func);
}

static void
gle_garg_callback (GleGArg      *garg,
		   GleGArgCbType cb_type)
{
  if (garg->widget)
    {
      GleGArgCbFunc cb;

      cb = gtk_object_get_data_by_id (GTK_OBJECT (garg->widget), quark_garg_cb);
      if (cb)
	cb (garg, cb_type);
    }
}

gint
gle_arg_copy_value (const GtkArg   *src_arg,
		    GtkArg         *dest_arg)
{
  gboolean changed;

  g_return_val_if_fail (src_arg != NULL, FALSE);
  g_return_val_if_fail (dest_arg != NULL, FALSE);
  g_return_val_if_fail (src_arg->type > GTK_TYPE_NONE, FALSE);

  /* coying a GtkArg means to copy the data portion and setting the
   * type, the name is left untouched
   */

  if (dest_arg->type == src_arg->type)
    changed = !gle_arg_values_equal (src_arg, dest_arg);
  else
    changed = TRUE;

  if (changed)
    {
      gle_arg_reset (dest_arg);
      g_memmove (&dest_arg->d, &src_arg->d, sizeof (dest_arg->d));
      if (GTK_FUNDAMENTAL_TYPE (src_arg->type) == GTK_TYPE_STRING)
	GTK_VALUE_STRING (*dest_arg) = g_strdup (GTK_VALUE_STRING (*src_arg));
      dest_arg->type = src_arg->type;
    }
  
  return changed;
}

void
gle_arg_reset (GtkArg *arg)
{
  g_return_if_fail (arg != NULL);

  if (arg->type != GTK_TYPE_INVALID)
    {
      if (GTK_FUNDAMENTAL_TYPE (arg->type) == GTK_TYPE_STRING)
	g_free (GTK_VALUE_STRING (*arg));
      arg->type = GTK_TYPE_INVALID;
    }
}

const GleArgInfo*
gle_arg_get_info (const gchar *arg_name,
		  gboolean     is_child_arg)
{
  GleArgInfo *info;
  
  g_return_val_if_fail (arg_name != NULL, NULL);
  
  if (!gle_arg_info_ht)
    {
      gle_arg_info_ht = g_hash_table_new (g_str_hash,
					  g_str_equal);
      gle_child_arg_info_ht = g_hash_table_new (g_str_hash,
						g_str_equal);
      info = NULL;
    }
  else if (is_child_arg)
    info = g_hash_table_lookup (gle_child_arg_info_ht, arg_name);
  else
    info = g_hash_table_lookup (gle_arg_info_ht, arg_name);
  
  if (!info)
    {
      GtkType		object_type;
      GtkObject		*object;
      GtkWidget		*child;
      GtkArg		*args;
      guint		n_args;
      guint32		*flags;

      object_type = gtk_type_from_arg_name (arg_name, NULL);
      if (is_child_arg)
	g_return_val_if_fail (gtk_type_is_a (object_type, GTK_TYPE_CONTAINER), NULL);
      else
	g_return_val_if_fail (gtk_type_is_a (object_type, GTK_TYPE_OBJECT), NULL);

      object = gtk_type_new (object_type);
      gtk_object_ref (object);
      gtk_object_sink (object);
      n_args = 0;
      flags = NULL;
      child = NULL;

      if (is_child_arg)
	{
	  GtkType child_type;

	  gtk_object_default_construct (object);
	  args = gtk_container_query_child_args (GTK_OBJECT_TYPE (object), &flags, &n_args);
	  child_type = gtk_container_child_type (GTK_CONTAINER (object));
	  if (child_type)
	    {
	      child = gtk_widget_new (child_type, NULL);
	      gtk_container_add (GTK_CONTAINER (object), child);
	    }
	}
      else
	args = gtk_object_query_args (GTK_OBJECT_TYPE (object), &flags, &n_args);

      for (; n_args > 0; n_args--)
	if (args[n_args - 1].type)
	  {
	    memset (&args[n_args - 1].d, 0, sizeof (args[n_args - 1].d));

	    if (is_child_arg)
	      g_assert ((flags[n_args - 1] & GTK_ARG_CHILD_ARG) != 0);
	    else
	      g_assert ((flags[n_args - 1] & GTK_ARG_CHILD_ARG) == 0);
	    
	    if (flags[n_args - 1] & GTK_ARG_READABLE)
	      {
		if (!is_child_arg)
		  {
		    GtkType ftype;

		    ftype = GTK_FUNDAMENTAL_TYPE (args[n_args - 1].type);
		    if (ftype == GTK_TYPE_BOXED ||
			ftype == GTK_TYPE_POINTER ||
			ftype == GTK_TYPE_OBJECT)
		      GTK_VALUE_POINTER (args[n_args - 1]) = NULL;
		    else
		      gtk_object_getv (object, 1, &args[n_args - 1]);
		  }
		else if (child)
		  {
		    if (GTK_FUNDAMENTAL_TYPE (args[n_args - 1].type) == GTK_TYPE_OBJECT)
		      GTK_VALUE_OBJECT (args[n_args - 1]) = NULL;
		    else
		      gtk_container_child_getv (GTK_CONTAINER (object),
						child,
						1, &args[n_args - 1]);
		  }
		else
		  g_assert_not_reached ();
	      }

	    if (!gle_arg_info_mem_chunk)
	      gle_arg_info_mem_chunk = g_mem_chunk_create (GleArgInfo, 256, G_ALLOC_ONLY);
	    info = g_chunk_new0 (GleArgInfo, gle_arg_info_mem_chunk);
	    info->class_type = object_type;
	    info->arg_flags = flags[n_args - 1];
	    info->seq_num = n_args - 1;
	    info->arg = &args[n_args - 1];

	    if (is_child_arg)
	      g_hash_table_insert (gle_child_arg_info_ht, info->arg->name, info);
	    else
	      g_hash_table_insert (gle_arg_info_ht, info->arg->name, info);
	  }

      g_free (flags);
      gtk_object_default_construct (object);
      gtk_object_destroy (object);
      gtk_object_unref (object);

      if (is_child_arg)
	info = g_hash_table_lookup (gle_child_arg_info_ht, arg_name);
      else
	info = g_hash_table_lookup (gle_arg_info_ht, arg_name);
    }

  return info;
}

void
gle_arg_set_from_string (GtkArg         *arg,
			 const gchar    *value_string)
{
  gchar *string;

  g_return_if_fail (arg != NULL);
  g_return_if_fail (value_string != NULL);

  string = g_strdup (value_string);

  switch (GTK_FUNDAMENTAL_TYPE (arg->type))
    {
    case  GTK_TYPE_INVALID:
      g_warning ("gle_arg_set_from_string() used with invalid type");
      break;
    case  GTK_TYPE_CHAR:
      if (strlen (string) > 1)
	{
	  GTK_VALUE_INT (*arg) = 0;
	  sscanf (string, "%d", &GTK_VALUE_INT (*arg));
	  GTK_VALUE_CHAR (*arg) = GTK_VALUE_INT (*arg);
	}
      else
	GTK_VALUE_CHAR (*arg) = string[0];
      break;
    case  GTK_TYPE_BOOL:
      g_strdown (string);
      if ((string[0] > '0' && string[0] <= '9') ||
	  g_str_equal (string, "true") ||
	  g_str_equal (string, "yes"))
	GTK_VALUE_BOOL (*arg) = TRUE;
      else
	GTK_VALUE_BOOL (*arg) = FALSE;
      break;
    case  GTK_TYPE_INT:
      GTK_VALUE_INT (*arg) = 0;
      sscanf (string, "%d", &GTK_VALUE_INT (*arg));
      break;
    case  GTK_TYPE_UINT:
      GTK_VALUE_UINT (*arg) = 0;
      sscanf (string, "%u", &GTK_VALUE_UINT (*arg));
      break;
    case  GTK_TYPE_LONG:
      GTK_VALUE_LONG (*arg) = 0;
      sscanf (string, "%ld", &GTK_VALUE_LONG (*arg));
      break;
    case  GTK_TYPE_ULONG:
      GTK_VALUE_ULONG (*arg) = 0;
      sscanf (string, "%lu", &GTK_VALUE_ULONG (*arg));
      break;
    case  GTK_TYPE_FLOAT:
      GTK_VALUE_FLOAT (*arg) = 0;
      sscanf (string, "%f", &GTK_VALUE_FLOAT (*arg));
      break;
    case  GTK_TYPE_DOUBLE:
      GTK_VALUE_DOUBLE (*arg) = 0;
      sscanf (string, "%lf", &GTK_VALUE_DOUBLE (*arg));
      break;
    case  GTK_TYPE_STRING:
      GTK_VALUE_STRING (*arg) = g_strdup (value_string);
      break;
    default:
      g_warning ("gle_arg_set_from_string() used with type <%s>", gtk_type_name (arg->type));
      break;
    }
  g_free (string);
}

gint
gle_arg_values_equal (const GtkArg	*arg1,
		      const GtkArg	*arg2)
{
  gboolean equal;
  
  g_return_val_if_fail (arg1 != NULL, FALSE);
  g_return_val_if_fail (arg2 != NULL, FALSE);
  g_return_val_if_fail (GTK_FUNDAMENTAL_TYPE (arg1->type) ==
			GTK_FUNDAMENTAL_TYPE (arg2->type), FALSE);
  
  switch (GTK_FUNDAMENTAL_TYPE (arg1->type))
    {
    case  GTK_TYPE_INVALID:
      equal = TRUE;
      break;
    case  GTK_TYPE_CHAR:
      equal = GTK_VALUE_CHAR (*arg1) == GTK_VALUE_CHAR (*arg2);
      break;
    case  GTK_TYPE_BOOL:
      equal = (GTK_VALUE_BOOL (*arg1) != FALSE) == (GTK_VALUE_BOOL (*arg2) != FALSE);
      break;
    case  GTK_TYPE_INT:
      equal = GTK_VALUE_INT (*arg1) == GTK_VALUE_INT (*arg2);
      break;
    case  GTK_TYPE_UINT:
      equal = GTK_VALUE_UINT (*arg1) == GTK_VALUE_UINT (*arg2);
      break;
    case  GTK_TYPE_LONG:
      equal = GTK_VALUE_LONG (*arg1) == GTK_VALUE_LONG (*arg2);
      break;
    case  GTK_TYPE_ULONG:
      equal = GTK_VALUE_ULONG (*arg1) == GTK_VALUE_ULONG (*arg2);
      break;
    case  GTK_TYPE_FLOAT:
      equal = GTK_VALUE_FLOAT (*arg1) == GTK_VALUE_FLOAT (*arg2);
      break;
    case  GTK_TYPE_DOUBLE:
      equal = GTK_VALUE_DOUBLE (*arg1) == GTK_VALUE_DOUBLE (*arg2);
      break;
    case  GTK_TYPE_STRING:
      if (!GTK_VALUE_STRING (*arg1) ||
	  !GTK_VALUE_STRING (*arg2))
	equal = GTK_VALUE_STRING (*arg1) == GTK_VALUE_STRING (*arg2);
      else
	equal = g_str_equal (GTK_VALUE_STRING (*arg1), GTK_VALUE_STRING (*arg2));
      break;
    case  GTK_TYPE_ENUM:
      equal = GTK_VALUE_ENUM (*arg1) == GTK_VALUE_ENUM (*arg2);
      break;
    case  GTK_TYPE_FLAGS:
      equal = GTK_VALUE_FLAGS (*arg1) == GTK_VALUE_FLAGS (*arg2);
      break;
    case  GTK_TYPE_BOXED:
      equal = GTK_VALUE_BOXED (*arg1) == GTK_VALUE_BOXED (*arg2);
      break;
    case  GTK_TYPE_FOREIGN:
      equal = (GTK_VALUE_FOREIGN (*arg1).data == GTK_VALUE_FOREIGN (*arg2).data &&
	       GTK_VALUE_FOREIGN (*arg1).notify == GTK_VALUE_FOREIGN (*arg2).notify);
      break;
    case  GTK_TYPE_CALLBACK:
      equal = (GTK_VALUE_CALLBACK (*arg1).marshal == GTK_VALUE_CALLBACK (*arg2).marshal &&
	       GTK_VALUE_CALLBACK (*arg1).data == GTK_VALUE_CALLBACK (*arg2).data &&
	       GTK_VALUE_CALLBACK (*arg1).notify == GTK_VALUE_CALLBACK (*arg2).notify);
      break;
    case  GTK_TYPE_ARGS:
      equal = (GTK_VALUE_ARGS (*arg1).n_args == GTK_VALUE_ARGS (*arg2).n_args &&
	       GTK_VALUE_ARGS (*arg1).args == GTK_VALUE_ARGS (*arg2).args);
      break;
    case  GTK_TYPE_OBJECT:
      equal = GTK_VALUE_OBJECT (*arg1) == GTK_VALUE_OBJECT (*arg2);
      break;
    case  GTK_TYPE_POINTER:
      equal = GTK_VALUE_POINTER (*arg1) == GTK_VALUE_POINTER (*arg2);
      break;
    case  GTK_TYPE_SIGNAL:
      equal = (GTK_VALUE_SIGNAL (*arg1).f == GTK_VALUE_SIGNAL (*arg2).f &&
	       GTK_VALUE_SIGNAL (*arg1).d == GTK_VALUE_SIGNAL (*arg2).d);
      break;
    case  GTK_TYPE_C_CALLBACK:
      equal = (GTK_VALUE_C_CALLBACK (*arg1).func == GTK_VALUE_C_CALLBACK (*arg2).func &&
	       GTK_VALUE_C_CALLBACK (*arg1).func_data == GTK_VALUE_C_CALLBACK (*arg2).func_data);
      break;
    default:
      g_warning ("gle_arg_values_equal() used with type `%s'", gtk_type_name (arg1->type));
      equal = FALSE;
      break;
    }
  
  return equal;
}
