/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998, 1999 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	<gle/gleconfig.h>

#include	"gleplist.h"
#include	"glegobject.h"
#include	"glestock.h"
#include	"glercargs.h"
#include	<gdk/gdk.h>
#include	<string.h>
#include	<stdio.h>


/* --- signals --- */
enum {
  SIGNAL_BUTTON_CLICK,
  SIGNAL_CHECK_TAG,
  SIGNAL_LAST
};
typedef	gboolean	(*SignalButtonClick)	(GtkObject  *object,
						 GleGObject *gobject,
						 guint	     root_x,
						 guint	     root_y,
						 guint	     button,
						 guint	     time,
						 gpointer    func_data);
typedef	GQuark		(*SignalCheckTag)	(GtkObject  *object,
						 GleGObject *gobject,
						 gpointer    func_data);


/* --- prototypes --- */
static void		gle_plist_class_init		(GlePListClass	 *class);
static void		gle_plist_init			(GlePList	 *plist);
static void		gle_plist_destroy		(GtkObject	 *object);
static GlePListPix*	gle_plist_get_pix		(GlePList	 *plist,
							 GQuark		  pix_id);
static void		gle_plist_realize		(GtkWidget	 *widget);
static void		gle_plist_unrealize		(GtkWidget	 *widget);
static gint		gle_plist_button_press_event	(GtkWidget	 *widget,
							 GdkEventButton	 *event);
static void		gle_plist_row_data_remove	(GlePListRowData *row_data);
static void		gle_plist_row_data_new		(GlePList	 *plist,
							 GleGObject	 *gobject,
							 gint		  row);
static void		gle_plist_row_data_update	(GlePListRowData *row_data);
static void		gle_plist_row_data_destroyed	(GlePListRowData *row_data);


/* --- variables --- */
static guint	plist_init_col_width[GLE_PLIST_N_COLS] =
{
  30, 20, 120, 90, 200,
};
static GtkCListClass	*parent_class = NULL;
static GlePListClass	*gle_plist_class = NULL;
static guint		 plist_signals[SIGNAL_LAST] = { 0 };


/* --- functions --- */
GtkType
gle_plist_get_type (void)
{
  static GtkType plist_type = 0;
  
  if (!plist_type)
    {
      GtkTypeInfo plist_info =
      {
	"GlePList",
	sizeof (GlePList),
	sizeof (GlePListClass),
	(GtkClassInitFunc) gle_plist_class_init,
	(GtkObjectInitFunc) gle_plist_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };
      
      plist_type = gtk_type_unique (GTK_TYPE_CLIST, &plist_info);
    }
  
  return plist_type;
}

static void
gle_plist_marshal_button_click (GtkObject      *object,
				GtkSignalFunc  func,
				gpointer       func_data,
				GtkArg	       *args)
{
  gboolean *handled;
  SignalButtonClick sfunc = (SignalButtonClick) func;
  
  handled = GTK_RETLOC_BOOL (args[5]);
  
  *handled = sfunc (object,
		    GTK_VALUE_POINTER (args[0]),
		    GTK_VALUE_UINT (args[1]),
		    GTK_VALUE_UINT (args[2]),
		    GTK_VALUE_UINT (args[3]),
		    GTK_VALUE_UINT (args[4]),
		    func_data);
}

static void
gle_plist_marshal_check_tag (GtkObject	    *object,
			     GtkSignalFunc   func,
			     gpointer	     func_data,
			     GtkArg	    *args)
{
  GQuark *handled;
  SignalCheckTag sfunc = (SignalCheckTag) func;
  
  handled = GTK_RETLOC_UINT (args[1]);
  
  *handled = sfunc (object,
		    GTK_VALUE_POINTER (args[0]),
		    func_data);
}

static void
gle_plist_class_init (GlePListClass  *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  
  gle_plist_class = class;
  parent_class = gtk_type_class (GTK_TYPE_CLIST);
  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  
  plist_signals[SIGNAL_BUTTON_CLICK] =
    gtk_signal_new ("button-click",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GlePListClass, button_click),
		    gle_plist_marshal_button_click,
		    GTK_TYPE_BOOL, 5,
		    GLE_TYPE_GOBJECT,
		    GTK_TYPE_UINT,
		    GTK_TYPE_UINT,
		    GTK_TYPE_UINT,
		    GTK_TYPE_UINT);
  plist_signals[SIGNAL_CHECK_TAG] =
    gtk_signal_new ("check-tag",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GlePListClass, check_tag),
		    gle_plist_marshal_check_tag,
		    GTK_TYPE_UINT, 1,
		    GLE_TYPE_GOBJECT);
  gtk_object_class_add_signals (object_class, plist_signals, SIGNAL_LAST);
  
  object_class->destroy = gle_plist_destroy;
  
  widget_class->realize = gle_plist_realize;
  widget_class->unrealize = gle_plist_unrealize;
  widget_class->button_press_event = gle_plist_button_press_event;
  
  class->row_data_mem_chunk_ref = 0;
  class->row_data_mem_chunk = NULL;
  class->button_click = NULL;
  class->check_tag = NULL;
}

static void
gle_plist_init (GlePList *plist)
{
  GtkCList *clist;
  guint i;
  gchar *titles[GLE_PLIST_N_COLS] = { NULL };
  
  titles[GLE_PLIST_COL_ICON] = "Type";
  titles[GLE_PLIST_COL_TAG] = "?";
  titles[GLE_PLIST_COL_GNAME] = "GName";
  titles[GLE_PLIST_COL_OTYPE] = "Type Name";
  titles[GLE_PLIST_COL_PROXY] = "Proxy";
  
  clist = GTK_CLIST (plist);
  
  plist->shown_row_data = NULL;
  plist->row_datas = NULL;
  plist->pix_cache = NULL;
  plist->unknown_xpm = gle_stock_get_xpm_unknown ();
  
  /* setup clist
   */
  gtk_clist_construct (clist, GLE_PLIST_N_COLS, titles);
  gtk_clist_set_selection_mode (clist, GTK_SELECTION_EXTENDED);
  gtk_clist_column_titles_passive (clist);
  for (i = 0; i < GLE_PLIST_N_COLS; i++)
    gtk_clist_set_column_width (clist, i, plist_init_col_width[i]);
  gtk_clist_set_column_justification (clist, GLE_PLIST_COL_GNAME, GTK_JUSTIFY_RIGHT);
  gtk_clist_set_column_justification (clist, GLE_PLIST_COL_TAG, GTK_JUSTIFY_CENTER);
  
  /* setup widget
   */
  gtk_widget_set_usize (GTK_WIDGET (plist),
			GLE_RCVAL_PLIST_WIDTH > 0 ? GLE_RCVAL_PLIST_WIDTH : -1,
			GLE_RCVAL_PLIST_HEIGHT > 0 ? GLE_RCVAL_PLIST_HEIGHT : -1);
  
  /* row data memchunks
   */
  if (!gle_plist_class->row_data_mem_chunk)
    gle_plist_class->row_data_mem_chunk =
      g_mem_chunk_new ("GlePListRowData mem chunks",
		       sizeof (GlePListRowData),
		       sizeof (GlePListRowData) * 16,
		       G_ALLOC_AND_FREE);
  gle_plist_class->row_data_mem_chunk_ref += 1;
}

static void
gle_plist_realize (GtkWidget *widget)
{
  GlePList *plist;
  
  plist = GLE_PLIST (widget);
  
  GTK_WIDGET_CLASS (parent_class)->realize (widget);
  
  gle_plist_update_all (plist);
}

static void
gle_plist_unrealize (GtkWidget *widget)
{
  GlePList *plist;
  GlePListPix *pix;
  
  plist = GLE_PLIST (widget);
  
  GTK_WIDGET_CLASS (parent_class)->unrealize (widget);
  
  pix = plist->pix_cache;
  plist->pix_cache = NULL;
  while (pix)
    {
      GlePListPix *prev;
      
      gdk_pixmap_unref (pix->pixmap);
      gdk_bitmap_unref (pix->pixmap);
      
      prev = pix;
      pix = prev->next;
      g_free (prev);
    }
}

static GlePListPix*
gle_plist_get_pix (GlePList    *plist,
		   GQuark	quark)
{
  gchar **xpm = NULL;
  GlePListPix *pix;
  GtkWidget *widget;
  
  if (!GTK_WIDGET_REALIZED (plist) || quark == 0)
    return NULL;
  
  for (pix = plist->pix_cache; pix; pix = pix->next)
    if (pix->id == quark)
      return pix;
  
  xpm = gle_stock_id_get_xpm (quark);
  
  if (!xpm)
    xpm = plist->unknown_xpm;
  
  widget = GTK_WIDGET (plist);
  
  pix = g_new0 (GlePListPix, 1);
  pix->next = plist->pix_cache;
  plist->pix_cache = pix;
  pix->id = quark;
  pix->pixmap = gdk_pixmap_create_from_xpm_d (widget->window,
					      &pix->bitmap,
					      &widget->style->bg[GTK_STATE_PRELIGHT],
					      xpm);
  return pix;
}

GtkWidget*
gle_plist_new (void)
{
  GlePList *plist;
  
  plist = gtk_type_new (GLE_TYPE_PLIST);
  
  return GTK_WIDGET (plist);
}

static void
gle_plist_destroy (GtkObject *object)
{
  GlePList *plist;
  
  plist = GLE_PLIST (object);
  
  GTK_OBJECT_CLASS (parent_class)->destroy (object);
  
  gle_plist_class->row_data_mem_chunk_ref -= 1;
  if (gle_plist_class->row_data_mem_chunk_ref == 0)
    {
      g_mem_chunk_destroy (gle_plist_class->row_data_mem_chunk);
      gle_plist_class->row_data_mem_chunk = NULL;
    }
}

void
gle_plist_clear (GlePList	*plist)
{
  g_return_if_fail (plist != NULL);
  g_return_if_fail (GLE_IS_PLIST (plist));
  
  gtk_clist_clear (GTK_CLIST (plist));
}

void
gle_plist_freeze (GlePList	 *plist)
{
  g_return_if_fail (plist != NULL);
  g_return_if_fail (GLE_IS_PLIST (plist));
  
  gtk_clist_freeze (GTK_CLIST (plist));
}

void
gle_plist_thaw (GlePList       *plist)
{
  g_return_if_fail (plist != NULL);
  g_return_if_fail (GLE_IS_PLIST (plist));
  
  gtk_clist_thaw (GTK_CLIST (plist));
}

GleGObject*
gle_plist_get_gobject (GlePList	      *plist,
		       gint	       row)
{
  GlePListRowData *row_data;
  
  g_return_val_if_fail (plist != NULL, NULL);
  g_return_val_if_fail (GLE_IS_PLIST (plist), NULL);
  
  row_data = gtk_clist_get_row_data (GTK_CLIST (plist), row);
  
  return row_data ? row_data->gobject : NULL;
}

GList*
gle_plist_list_selected (GlePList *plist)
{
  GtkCList *clist;
  GList *list;
  GList *selection = NULL;
  
  g_return_val_if_fail (plist != NULL, NULL);
  g_return_val_if_fail (GLE_IS_PLIST (plist), NULL);
  
  clist = GTK_CLIST (plist);
  for (list = g_list_last (clist->selection); list; list = list->prev)
    selection = g_list_prepend (selection,
				gle_plist_get_gobject (plist,
						       GPOINTER_TO_INT (list->data)));
  
  return selection;
}

GList*
gle_plist_list_all (GlePList *plist)
{
  GtkCList *clist;
  guint i;
  GList *selection = NULL;
  
  g_return_val_if_fail (plist != NULL, NULL);
  g_return_val_if_fail (GLE_IS_PLIST (plist), NULL);
  
  clist = GTK_CLIST (plist);
  for (i = 0; i < clist->rows; i++)
    selection = g_list_prepend (selection, gle_plist_get_gobject (plist, i));
  
  return selection;
}

gboolean
gle_plist_mark (GlePList       *plist,
		GleGObject     *proxy)
{
  GtkCList *clist;
  GlePListRowData *row_data;
  
  g_return_val_if_fail (plist != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_PLIST (plist), FALSE);
  
  if ((plist->shown_row_data == NULL && proxy == NULL) ||
      (plist->shown_row_data && proxy == plist->shown_row_data->gobject))
    return proxy != NULL;
  
  clist = GTK_CLIST (plist);
  
  gtk_clist_freeze (clist);
  
  if (plist->shown_row_data)
    {
      row_data = plist->shown_row_data;
      plist->shown_row_data = NULL;
      gle_plist_row_data_update (row_data);
    }
  
  if (proxy)
    {
      GSList *slist;
      
      row_data = NULL;
      for (slist = plist->row_datas; slist; slist = slist->next)
	{
	  row_data = slist->data;
	  
	  if (row_data->gobject == proxy)
	    break;
	}
      if (row_data && row_data->gobject != proxy)
	row_data = NULL;
      
      if (row_data)
	{
	  gint row;
	  
	  plist->shown_row_data = row_data;
	  gle_plist_row_data_update (row_data);
	  
	  row = gtk_clist_find_row_from_data (clist, row_data);
	  if (gtk_clist_row_is_visible (GTK_CLIST (plist), row) != GTK_VISIBILITY_FULL)
	    gtk_clist_moveto (GTK_CLIST (plist),
			      row, -1,
			      0.5, 0);
	}
      else
	proxy = NULL;
    }
  
  gtk_clist_thaw (clist);
  
  return proxy != NULL;
}

static void
gle_plist_row_data_remove (GlePListRowData *row_data)
{
  GtkCList *clist;
  gint row;
  
  clist = GTK_CLIST (row_data->plist);
  row = gtk_clist_find_row_from_data (clist, row_data);
  gtk_clist_remove (clist, row);
}

static void
gle_plist_row_data_new (GlePList   *plist,
			GleGObject *gobject,
			gint	    row)
{
  gchar *text[GLE_PLIST_N_COLS];
  GlePListRowData *row_data;
  guint i;
  
  for (i = 0; i < GLE_PLIST_N_COLS; i++)
    text[i] = NULL;
  
  text[GLE_PLIST_COL_ICON] = NULL;
  text[GLE_PLIST_COL_TAG] = NULL;
  text[GLE_PLIST_COL_OTYPE] = gtk_type_name (GLE_GOBJECT_OTYPE (gobject));
  text[GLE_PLIST_COL_PROXY] = gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gobject));
  text[GLE_PLIST_COL_GNAME] = "";
  
  row_data = g_mem_chunk_alloc (gle_plist_class->row_data_mem_chunk);
  row_data->plist = plist;
  plist->row_datas = g_slist_prepend (plist->row_datas, row_data);
  row_data->gobject = gobject;
  row = gtk_clist_insert (GTK_CLIST (plist), row, text);
  gtk_clist_set_row_data_full (GTK_CLIST (plist),
			       row,
			       row_data,
			       (GtkDestroyNotify) gle_plist_row_data_destroyed);
  
  GLE_NOTIFIER_INSTALL_DATA (gobject, "set_gname", gle_plist_row_data_update, row_data);
  GLE_NOTIFIER_INSTALL_DATA (gobject, "associate", gle_plist_row_data_update, row_data);
  GLE_NOTIFIER_INSTALL_DATA (gobject, "disassociate", gle_plist_row_data_update, row_data);
  GLE_NOTIFIER_INSTALL_DATA (gobject, "pre_destroy", gle_plist_row_data_remove, row_data);
  
  gle_plist_row_data_update (row_data);
}

static void
gle_plist_insert (GlePList   *plist,
		  GleGObject *proxy,
		  gint	      row)
{
  GlePListRowData *row_data = NULL;
  GSList *slist;
  
  for (slist = plist->row_datas; slist; slist = slist->next)
    {
      row_data = slist->data;
      
      if (row_data->gobject == proxy)
	break;
    }
  if (row_data && row_data->gobject != proxy)
    row_data = NULL;
  
  if (row_data)
    {
      g_warning ("GlePList: `%s' proxy %s \"%s\" already contained",
		 gtk_type_name (GLE_GOBJECT_OTYPE (proxy)),
		 gtk_type_name (GLE_GOBJECT_PROXY_TYPE (proxy)),
		 proxy->gname);
      return;
    }
  
  gle_plist_row_data_new (plist, proxy, row);
}

void
gle_plist_append (GlePList	 *plist,
		  GleGObject	 *proxy)
{
  g_return_if_fail (plist != NULL);
  g_return_if_fail (GLE_IS_PLIST (plist));
  g_return_if_fail (proxy != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (proxy));
  
  gle_plist_insert (plist, proxy, -1);
}

void
gle_plist_prepend (GlePList	  *plist,
		   GleGObject	  *proxy)
{
  g_return_if_fail (plist != NULL);
  g_return_if_fail (GLE_IS_PLIST (plist));
  g_return_if_fail (proxy != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (proxy));
  
  gle_plist_insert (plist, proxy, 0);
}

static void
gle_plist_row_data_update (GlePListRowData *row_data)
{
  if (row_data)
    {
      GlePList *plist;
      GtkCList *clist;
      GleGObject *gobject;
      GlePListPix* pix;
      gint row;
      gint row_height;
      GQuark tag_id;
      
      plist = row_data->plist;
      clist = GTK_CLIST (plist);
      gobject = row_data->gobject;
      row = gtk_clist_find_row_from_data (clist, row_data);
      row_height = clist->row_height;
      
      if (plist->shown_row_data == row_data)
	gtk_clist_set_foreground (clist, row, &GTK_WIDGET (clist)->style->base[GTK_STATE_PRELIGHT]);
      else
	gtk_clist_set_foreground (clist, row, NULL);
      
      gtk_clist_set_text (clist, row, GLE_PLIST_COL_GNAME, gobject->gname);
      
      if (GLE_GOBJECT_IS_INSTANTIATED (gobject))
	pix = gle_plist_get_pix (plist, g_quark_try_string (gtk_type_name (GLE_GOBJECT_OTYPE (gobject))));
      else
	pix = gle_plist_get_pix (plist, g_quark_try_string (GLE_XPM_PROXY));
      if (!pix)
	gtk_clist_set_text (clist, row, GLE_PLIST_COL_ICON, NULL);
      else
	{
	  gint h;
	  
	  gtk_clist_set_pixmap (clist, row, GLE_PLIST_COL_ICON, pix->pixmap, pix->bitmap);
	  gdk_window_get_size (pix->pixmap, NULL, &h);
	  row_height = MAX (row_height, h);
	}
      
      tag_id = 0;
      gtk_signal_emit (GTK_OBJECT (plist),
		       plist_signals[SIGNAL_CHECK_TAG],
		       gobject,
		       &tag_id);
      pix = gle_plist_get_pix (plist, tag_id);
      if (!pix)
	gtk_clist_set_text (clist, row, GLE_PLIST_COL_TAG, NULL);
      else
	{
	  gint h;
	  
	  gtk_clist_set_pixmap (clist, row, GLE_PLIST_COL_TAG, pix->pixmap, pix->bitmap);
	  gdk_window_get_size (pix->pixmap, NULL, &h);
	  row_height = MAX (row_height, h);
	}
      
      if (row_height > clist->row_height)
	gtk_clist_set_row_height (clist, row_height);
    }
}

void
gle_plist_update (GlePList   *plist,
		  GleGObject *proxy)
{
  GlePListRowData *row_data = NULL;
  GSList *slist;
  
  g_return_if_fail (plist != NULL);
  g_return_if_fail (GLE_IS_PLIST (plist));
  g_return_if_fail (proxy != NULL);
  g_return_if_fail (GLE_IS_GOBJECT (proxy));
  
  for (slist = plist->row_datas; slist; slist = slist->next)
    {
      row_data = slist->data;
      
      if (row_data->gobject == proxy)
	break;
    }
  if (row_data && row_data->gobject != proxy)
    row_data = NULL;
  
  if (!row_data)
    {
      g_warning ("GlePList: no such `%s' proxy %s \"%s\"",
		 gtk_type_name (GLE_GOBJECT_OTYPE (proxy)),
		 gtk_type_name (GLE_GOBJECT_PROXY_TYPE (proxy)),
		 proxy->gname);
      return;
    }
  
  gle_plist_row_data_update (row_data);
  
  /*
    gdk_window_get_size (pixmap, NULL, &height);
    row_height = MAX (row_height, height);
    gtk_clist_set_row_height (shell->clist, row_height);
  */
  
}

static void
gle_plist_row_data_destroyed (GlePListRowData *row_data)
{
  GlePList *plist;
  GtkCList *clist;
  GleGObject *gobject;
  gint row;
  
  plist = row_data->plist;
  clist = GTK_CLIST (plist);
  gobject = row_data->gobject;
  
  if (plist->shown_row_data == row_data)
    plist->shown_row_data = NULL;
  
  gtk_clist_freeze (clist);
  
  row = gtk_clist_find_row_from_data (clist, row_data);
  gtk_clist_remove (clist, row);
  
  GLE_NOTIFIER_REMOVE_FD (gobject, gle_plist_row_data_update, row_data);
  GLE_NOTIFIER_REMOVE_FD (gobject, gle_plist_row_data_remove, row_data);
  
  gtk_clist_thaw (clist);
  
  row_data->plist = NULL;
  plist->row_datas = g_slist_remove (plist->row_datas, row_data);
  g_mem_chunk_free (gle_plist_class->row_data_mem_chunk, row_data);
}

void
gle_plist_remove (GlePList	 *plist,
		  GleGObject	 *proxy)
{
  GlePListRowData *row_data = NULL;
  GtkCList *clist;
  GSList *slist;
  gint row;
  
  g_return_if_fail (plist != NULL);
  g_return_if_fail (GLE_IS_PLIST (plist));
  
  clist = GTK_CLIST (plist);
  
  for (slist = plist->row_datas; slist; slist = slist->next)
    {
      row_data = slist->data;
      
      if (row_data->gobject == proxy)
	break;
    }
  if (row_data && row_data->gobject != proxy)
    row_data = NULL;
  
  if (!row_data)
    {
      g_warning ("GlePList: no such `%s' proxy %s \"%s\"",
		 gtk_type_name (GLE_GOBJECT_OTYPE (proxy)),
		 gtk_type_name (GLE_GOBJECT_PROXY_TYPE (proxy)),
		 proxy->gname);
      return;
    }
  
  row = gtk_clist_find_row_from_data (clist, row_data);
  gtk_clist_remove (clist, row);
}

void
gle_plist_update_all (GlePList	     *plist)
{
  GtkCList *clist;
  GSList *slist;
  
  g_return_if_fail (plist != NULL);
  g_return_if_fail (GLE_IS_PLIST (plist));
  
  clist = GTK_CLIST (plist);
  
  gtk_clist_freeze (clist);
  
  for (slist = plist->row_datas; slist; slist = slist->next)
    gle_plist_row_data_update (slist->data);
  
  gtk_clist_thaw (clist);
}

static gint
gle_plist_button_press_event (GtkWidget		*widget,
			      GdkEventButton	*event)
{
  GlePList *plist;
  gboolean handled;
  
  plist = GLE_PLIST (widget);
  
  handled = FALSE;
  if (event->type == GDK_BUTTON_PRESS && event->button)
    {
      gint row;
      
      if (gle_clist_selection_info (GTK_CLIST (plist),
				    event->window,
				    event->x, event->y,
				    &row, NULL))
	{
	  GlePListRowData *row_data;
	  
	  row_data = gtk_clist_get_row_data (GTK_CLIST (plist), row);
	  if (row_data)
	    {
	      gint return_val = FALSE;
	      
	      gtk_signal_emit (GTK_OBJECT (plist),
			       plist_signals[SIGNAL_BUTTON_CLICK],
			       row_data->gobject,
			       (guint) event->x_root,
			       (guint) event->y_root,
			       (guint) event->button,
			       (guint) event->time,
			       &return_val);
	      handled |= return_val != FALSE;
	    }
	}
    }
  if (!handled && GTK_WIDGET_CLASS (parent_class)->button_press_event)
    handled = GTK_WIDGET_CLASS (parent_class)->button_press_event (widget, event);
  
  return handled;
}
