// This may look like C code, but it is really -*- C++ -*-
// 
// <copyright> 
// 
// Copyright (c) 1993 
// Institute for Information Processing and Computer Supported New Media (IICM), 
// Graz University of Technology, Austria. 
// 
// </copyright> 
//
//
// <file> 
// 
// Name:        vwstuff.h
// 
// Purpose:     definition of the base class for all Hyper-G viewers
//              definition of the base class for everyone who tells 
//              a Hyper-G viewer what to do 
//              
// 
// Created:     26 Jul 93   Joerg Faschingbauer
// 
// Modified:    
// 
// </file> 
#ifndef hg_viewer_vwstuff_h
#define hg_viewer_vwstuff_h


#include <hyperg/viewer/lib.h>

#include <hyperg/utils/types.h>
#include <hyperg/hyperg/object.h>
#include <hyperg/hyperg/language.h>




class HgViewerManager ;


// <class> 
//  
// Name:             HgViewer
// 
// Purpose:          base class for any kind of Hyper-G viewer
// 
// Public Interface: HgViewer (HgViewerManager*)
//                      constructor; a pointer to his boss is passed, upon 
//                      whom the viewer may invoke some operations (e.g. if 
//                      he requests following a link)
// 
//                   ~HgViewer()
//                      destructor
// 
//                   virtual void load (const char* doc, const char* anchors)
//                      load a document an intermix the anchors
// 
//                   virtual void browse (const char* dest)
//                      browse the previously loaded document with some destination anchors
// 
//                   virtual void terminate()
//                      blah
// 
//                   virtual void holdDocument (const RString&)
// 
//                   virtual void iconify()
//                   virtual void deiconify()
//                   virtual void moveto (float, float) 
//                   virtual void resize (float, float) 
//                   virtual void map()
//                   virtual void unmap()
//                   virtual void raise()
//                   virtual void lower()
//                      these correspond to the X functions
// 
//                   virtual void setLanguage (int)
//                      blah
// 
//                   virtual void saveResources() ;
//                      the viewer saves its specific X attributes
// 
//                   virtual void doEdit (const RString& object, const RString& anchors)
//                   virtual void newDoc()
//                   virtual void getPos (RString&)
// 
//                   HgViewer::VwError error()
//                      returns the error state of the HgViewer
// 
// </class> 
class HgViewer {
public:
   HgViewer (HgViewerManager*) ;
   virtual ~HgViewer() {}

   void load (ObjectID id, const char* doc, const char* anchors, char* info=nil) ;

   virtual void load (const char* doc, const char* anchors, char* info=nil) = 0 ;
   virtual void browse (const char* dest) = 0 ;
   virtual void terminate() = 0 ;
   virtual void cancelRequest (const char*) {}

   virtual void holdDocument (const char*) {}
   virtual void iconify() {} 
   virtual void deiconify() {}
   virtual void moveto (float, float) {}
   virtual void resize (float, float) {}
   virtual void map() {}
   virtual void unmap() {}
   virtual void raise() {}
   virtual void lower() {}

   virtual void setLanguage (HgLanguage::Language) {}

   virtual void saveResources() {}
   
   virtual void doEdit (const RString& /*object*/, const RString& /*anchors*/) {}
   virtual void newDoc (int /*refno*/, const RString& /*info*/) {}
   virtual void getPos (RString&) {}

   const RString& options() const { return options_; }
   virtual int port() const = 0 ;

   void setCurrentObjectID (ObjectID oid) { current_id_ = oid; }
   ObjectID currentObjectID() const { return current_id_; }

   void setRefNo (int refno) { refno_ = refno; }
   int refNo() const { return refno_; }

   enum VwError {
      OK,
      NOCONNECTION,
      TIMEOUT,
      EXECFAIL,
      UNDEF,
      CONNCLOSED,

      INCOMPLETELOAD,

      NERRORS
      } ;
   static boolean isFatal (VwError) ;

   static const char* errorDescription (VwError, 
                                        HgLanguage::Language language=HgLanguage::Default) ;

   VwError error()  { return error_; }

protected:
   HgViewerManager* manager_ ;
   VwError error_ ;
   static const char* err_descr_[] ;

   RString options_ ;

private:
   ObjectID current_id_ ;
   int refno_ ;
} ;

inline HgViewer :: HgViewer (HgViewerManager* manager)
: manager_ (manager) ,
  error_ (OK),
  refno_(-1) {}

inline void HgViewer :: load (ObjectID id, const char* doc, const char* anchors, char* info) {
   current_id_ = id ;
   load (doc, anchors, info) ;
}



// <class> 
//  
// Name:             HgViewerManager
// 
// Purpose:          base class for everyone who tells a HgViewer what to do next 
// 
// Public Interface: 
// 
// - ~HgViewerManager()
//   destructor
// 
// - virtual void followLink (const RString& the_link, HgViewer*)
//   is called by a HgViewer (most probably) who wants to tell
//   his boss he should follow a link and invoke the appropriate 
//   action on himself (HgViewer*)
// 
// - virtual void viewerError (HgViewer*)
//   the boss (this) is notified about an error of a viewer (i.e. HgViewer*)
// 
// - virtual void viewerTerminated (const RString&, HgViewer*)
//   the boss (this) is notified that a viewer (HgViewer*) has terminated.
//   Additionally the Viewer tells him what object was viewed lastly.
// 
// </class> 
class HgViewerManager {
public:
   virtual ~HgViewerManager() {}

   virtual void followLink (const RString&, HgViewer*) {}
   virtual void viewerError (HgViewer*) {}
   virtual void viewerTerminated (const RString&, HgViewer*) {}

   virtual void defineSourceAnchor (const RString&, HgViewer*) {}
   virtual void defineDestAnchor (const RString&, HgViewer*) {}

   virtual void deleteLink (const RString&, HgViewer*) {}

   virtual void showReferences (const RString& /*object*/, HgViewer*) {}
   virtual void showAnnotations (const RString& /*object*/, HgViewer*) {}
   virtual void showParents (const RString& /*object*/, HgViewer*) {}
   virtual void showAttributes (const RString& /*object*/, HgViewer*) {}
   virtual void back (HgViewer*) {}
   virtual void forward (HgViewer*) {}
   virtual void history (HgViewer*) {}
   virtual void hold (const RString& /*object*/, HgViewer*) {}

   virtual void edit (const RString& /*object*/, HgViewer*) {}
   virtual void upload (const RString& /*object*/, const RString& /*anchors*/,
                        const RString& /*host*/, int /*port*/,
                        HgViewer*, int /*refno*/=-1) {}
   virtual void cancelEdit (const RString& /*object*/, HgViewer*) {}
   virtual void modifyAttribute (const RString&) {}
   virtual void reloadDocument (const RString&) {}
} ;





#endif


