#include <stdio.h>

/* a list of the offsets of the start of the stroke data for each	*
*  character.  Numbers are offset + 1 because this used to be FORTRAN.	*
*  0 means a control character						*/
static short ugs_offsets[256] = {
	/*data          ctbx1*/
	3452,3452,3452,3452,3452,3452,3452,3452,3452,3452,3452,3452,
	3452,3452,3452,3452,3452,3452,3452,3452,3452,3452,3452,3452,
	3452,3452,3452,3452,3452,3452,3452,3452,3448,2598,2646,3307,
	3268,3138,3168,2641,2672,2691,3439,2790,2566,2787,2560,2669,
	2205,2243,2252,2294,2338,2348,2383,2429,2456,2514,2574,2585,
	2832,2820,2836,2611,3215,   1,  14,  54,  85, 112, 129, 144,
	180, 199, 208, 226, 245, 256, 277, 292, 334, 359, 419, 458,
	491, 504, 524, 535, 552, 567, 582,2710,3818,2719,3564,3597,
	/*data          ctbx2*/
	2953, 595, 631, 660, 687, 718, 747, 766, 821, 843, 857, 879,
	898, 907, 942, 964, 998,1029,1058,1077,1108,1122,1144,1155,
	1172,1187,1205,2728,2784,2752,3593,3452,2856,3033,2802,2807,
	3600,2980,2916,3615,2776,2780,3384,2795,2840,3316,3759,3783,
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,
	0,   0,   0,   0,2890,2655,3022,2971,2962,2944,3651,3064,
	2929,3666,3805,3812,3815,3430,2848,3683,3463,3468,3473,3479,
	3485,3498,3511,3524,3537,3554,2877,3715,3737,3359,2825,2903,
	/*data          ctbx3*/
	3689,   1,  14,1218,1229, 112, 582, 180,1240, 199, 226,1290,
	256, 277,1301, 292,1326, 334,1341, 491,1357,1386, 552,1428,
	1463,3702,3449,3455,3450,3454,3451,3453,3456,1502,1539,1591,
	1616,1658,1681,1711,1740,1780,1794,1819,1839,1863,1883,1924,
	1954,1971,2000,2032,2045,2074,2112,2133,2164,3457,3458,3459,
	3460,3461,3462,3112
};

/* list of the number of strokes in each character.			*
*  for control characters, the entry is the switch index.		*/
static short ugs_nstrokes[256] = {
	/*data          ctby1*/
	1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
	1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,   1,
	1,   1,   1,   1,   1,   1,   1,   1,   1,  13,   9,   9,
	39,  30,  47,   5,  19,  19,   9,   5,   8,   3,   6,   3,
	38,   9,  42,  44,  10,  35,  46,  27,  58,  46,  11,  13,
	4,   5,   4,  30,  53,  13,  40,  31,  27,  17,  15,  36,
	19,   9,  18,  19,  11,  21,  15,  42,  25,  60,  39,  33,
	13,  20,  11,  17,  15,  15,  13,   9,   3,   9,  29,   3,
	/*data          ctby2*/
	9,  36,  29,  27,  31,  29,  19,  55,  22,  14,  22,  19,
	9,  35,  22,  34,  31,  29,  19,  31,  14,  22,  11,  17,
	15,  18,  13,  24,   3,  24,   4,   1,  21,  31,   5,  13,
	15,  42,  13,  36,   4,   4,  46,   7,   8,  43,  24,  22,
	1,   2,   3,   4,   5,   6,   7,   8,   9,  10,  11,  12,
	13,  14,  15,  16,  13,  14,  11,   9,   9,   9,  15,  48,
	15,  17,   7,   3,   3,   9,   8,   6,   5,   5,   6,   6,
	13,  13,  13,  13,  17,  10,  13,  22,  22,  25,   7,  13,
	/*data          ctby3*/
	13,  13,  40,  11,  11,  17,  13,  19,  50,   9,  19,  11,
	21,  15,  25,  42,  15,  25,  16,  13,  29,  42,  15,  35,
	39,  13,   1,   1,   1,   1,   1,   1,   1,  37,  52,  25,
	42,  23,  30,  29,  40,  14,  25,  20,  24,  20,  41,  30,
	17,  29,  32,  13,  29,  38,  21,  31,  41,   1,   1,   1,
	1,   1,   1,  26
};

/*  A LIST OF THE STROKES, THAT IS, THE BLANKING BIT (BBT),	*
*  DELTA X (DLX), AND DELTA Y (DLY).  THE DATA IS ENCODED AS	*
*  16384*BBT+128*(DLX+64)+(DLY+64).				*/
static short ugs_strokes[3820] = {
	/*data          strx01*/
	8011,23723, 9173,25515, 7378,25390, 6854,25792, 6586,25408,
	9024,25408, 6858, 7371,24619, 8405,24619, 7765,26176,25023,
	24767,24766,24638,24510,24511,24255, 8266,24895,24767,24766,
	24638,24510,24511,24383, 7232,25664,25023,24767,24766,24637,
	24510,24511,24255,23104, 9803,24895,24767,24766,24637,24510,
	24511,24383, 7754, 8904,24765,24646,24509,24386,24257,24384,
	24255,24382,24510,24509,24635,24765,24766,24894,25023,24896,
	25025,24898,24770, 7248,24383,24382,24510,24509,24635,24765,
	/*data          strx02*/
	24766,24894,24895, 8138, 7371,24619, 8405,24619, 7765,25920,
	25023,24894,24766,24765,24635,24509,24510,24382,24255,23360,
	9557,24895,24894,24766,24765,24635,24509,24510,24382,24383,
	8010, 7371,24619, 8405,24619, 9039,24632, 6990,26688,24634,
	24518, 6838,25408, 6965,26688,24646,24506, 7242, 7371,24619,
	8405,24619, 9039,24632, 6990,26688,24634,24518, 6838,25408,
	6965,25536, 8138, 8904,24765,24646,24509,24386,24257,24384,
	24255,24382,24510,24509,24635,24765,24766,24894,25023,24896,
	/*data          strx03*/
	25025,24898, 7378,24383,24382,24510,24509,24635,24765,24766,
	24894,24895, 9160,24632, 8392,24632, 7752,25536, 6978, 7371,
	24619, 8405,24619, 9813,24619, 8405,24619, 6101,25536, 9024,
	25536, 6198,26176, 6197,25536, 9024,25536, 6986, 7371,24619,
	8405,24619, 7765,25536, 7339,25536, 6986, 8011,24623,24509,
	24383,24384,24385,24514,24642,24769,24767,24511, 9041,24623,
	24509,24511, 8149,25536, 6965, 7371,24619, 8405,24619, 9941,
	22963, 8900,25652, 7116,25652, 6229,25536, 9024,25408, 5803,
	/*data          strx04*/
	25536, 9024,25408, 6858, 7371,24619, 8405,24619, 7765,25536,
	7339,26560,24646,24506, 6986, 7371,24619, 8405,25390, 7378,
	25515, 9173,23723, 9173,24619, 8405,24619, 5973,25152, 9920,
	25152, 5547,25408, 9280,25536, 6986, 7371,24619, 8405,26157,
	6737,26157, 8277,24619, 6229,25152, 9408,25408, 5803,25408,
	8650, 8011,24255,24382,24510,24508,24637,24764,24766,24894,
	25023,24896,25025,24898,24770,24772,24643,24516,24514,24386,
	24257,24384, 8256,24383,24382,24510,24508,24637,24764,24766,
	/*data          strx05*/
	24894,24895, 8512,24897,24898,24770,24772,24643,24516,24514,
	24386,24385, 7989, 7371,24619, 8405,24619, 7765,26176,25023,
	24767,24766,24637,24510,24511,24255,23616, 9291,24895,24767,
	24766,24637,24510,24511,24383, 6710,25536, 8394, 8011,24255,
	24382,24510,24508,24637,24764,24766,24894,25023,24896,25025,
	24898,24770,24772,24643,24516,24514,24386,24257,24384, 8256,
	24383,24382,24510,24508,24637,24764,24766,24894,24895, 8512,
	24897,24898,24770,24772,24643,24516,24514,24386,24385, 7597,
	/*data          strx06*/
	24641,24770,24897,24768,24895,24766,24761,24767,24896,24770,
	24641, 7621,24764,24766,24767,24768,24769, 7117, 7371,24619,
	8405,24619, 7765,26176,25023,24767,24766,24638,24510,24511,
	24255,23616, 9290,24895,24767,24766,24638,24510,24511,24383,
	6709,25536, 8523,24895,24767,25017,24767,24768,24769, 7368,
	24766,24889,24767,24896,24770,24641, 6983, 8776,24771,24634,
	24515,24386,24257,24256,24255,24382,24638,24766,24767,24895,
	25406,24895,24894, 6473,24894,24895,25406,24895,24767,24766,
	/*data          strx07*/
	24636,24382,24255,24256,24257,24386,24515,24634,24771, 8775,
	7883,24619, 8405,24619, 7381,24506,24646,26560,24634,24518,
	6955,25536, 7498, 7371,24625,24765,24894,25023,24896,25025,
	24898,24771,24655, 6592,24625,24765,24894,24895, 7125,25536,
	9152,25408, 6965, 7115,25515, 7509,25390, 9170,23723, 7125,
	25408, 9024,25408, 6837, 7243,25131, 7893,25008, 8784,24107,
	8789,25131, 7893,25008, 8784,24107, 6357,25536, 9408,25408,
	6837, 7115,26283, 6741,26283, 8277,22827, 8021,25408, 9024,
	/*data          strx08*/
	25408, 5931,25408, 9024,25408, 6858, 7115,25525,24630, 7509,
	25525,24630, 9173,23733, 6987,25408, 9152,25408, 6571,25536,
	7626, 8779,22955,10069,22955, 8277,24506,24646,26432, 6443,
	26432,24646,24506, 7242, 7362,24639,24512,24641,24769,24897,
	25152,24895,24767,24766,24633,24766,24767, 7884,24631,24766,
	24895,24768, 7754,24511,23871,24255,24510,24638,24766,25023,
	25024,24897,24898, 7365,24383,24510,24638,24766,24895, 8394,
	7371,24619, 8405,24619, 8267,24898,24897,24896,25023,24894,
	/*data          strx09*/
	24765,24638,24509,24382,24255,24384,24385,24386, 9035,24895,
	24894,24765,24638,24509,24382,24383, 6997,25152, 8629, 8641,
	24511,24767,24769,24641,24386,24385,24256,24255,24382,24509,
	24638,24765,24894,25023,24896,25025,24898, 7371,24383,24382,
	24509,24638,24765,24894,24895, 8010, 8651,24619, 8405,24619,
	8139,24386,24385,24384,24255,24382,24509,24638,24765,24894,
	25023,24896,24897,24898, 7499,24383,24382,24509,24638,24765,
	24894,24895, 8661,25152, 8107,25152, 6986, 7230,26176,24642,
	/*data          strx10*/
	24514,24513,24385,24256,24255,24382,24509,24638,24765,24894,
	25023,24896,25025,24898, 8133,24643,24514, 7617,24383,24382,
	24509,24638,24765,24894,24895, 8010, 8010,24511,24767,24769,
	24641,24513,24384,24383,24510,24622, 8661,24511,24510,24622,
	7758,25664, 7218,25536, 7242, 7748,24383,24511,24510,24638,
	24766,24767,24895,24896,24897,24769,24770,24642,24514,24513,
	24385,24384, 7999,24510,24636,24766, 9024,24770,24644,24514,
	8383,24769,24897,24639,24384, 7097,24511,24510,24639,24766,
	/*data          strx11*/
	25023,25280,25023,24767, 6597,24767,25023,25280,25023,24766,
	24639,24510,24255,23872,24257,24514,24641,24770,25025, 8394,
	7371,24619, 8405,24619, 8267,24898,25025,24896,25023,24766,
	24629, 7758,24895,24766,24629, 6485,25152, 7723,25536, 8768,
	25536, 6986, 7371,24511,24767,24769,24513, 8249,24626, 8398,
	24626, 7758,25152, 7730,25536, 6986, 7499,24511,24767,24769,
	24513, 8377,24622,24510,24383,24384,24513,24641,24769,24767,
	24511, 8788,24622,24510,24511, 8149,25152, 7228, 7371,24619,
	/*data          strx12*/
	8405,24619, 9550,23350, 8900,25400, 7368,25400, 6485,25152,
	9145,25408, 6066,25536, 8768,25408, 6986, 7371,24619, 8405,
	24619, 7765,25152, 7723,25536, 6986, 7364,24626, 8398,24626,
	8267,24898,25025,24896,25023,24766,24629, 7758,24895,24766,
	24629, 8395,24898,25025,24896,25023,24766,24629, 7758,24895,
	24766,24629, 5070,25152, 7730,25536, 8768,25536, 8768,25536,
	6986, 7364,24626, 8398,24626, 8267,24898,25025,24896,25023,
	24766,24629, 7758,24895,24766,24629, 6478,25152, 7730,25536,
	/*data          strx13*/
	8768,25536, 6986, 7876,24255,24382,24509,24638,24765,24894,
	25023,24896,25025,24898,24771,24642,24515,24386,24257,24384,
	8256,24383,24382,24509,24638,24765,24894,24895, 8512,24897,
	24898,24771,24642,24515,24386,24385, 7868, 7364,24619, 8405,
	24619, 8274,24898,24897,24896,25023,24894,24765,24638,24509,
	24382,24255,24384,24385,24386, 9035,24895,24894,24765,24638,
	24509,24382,24383, 6990,25152, 7723,25536, 8273, 8644,24619,
	8405,24619, 8146,24386,24385,24384,24255,24382,24509,24638,
	/*data          strx14*/
	24765,24894,25023,24896,24897,24898, 7499,24383,24382,24509,
	24638,24765,24894,24895, 8633,25536, 6865, 7364,24626, 8398,
	24626, 8264,24771,24898,24897,25024,24767,24639,24511,24513,
	24769, 6721,25152, 7730,25536, 7754, 8386,24770,24636,24514,
	24513,24385,24128,24383,24511,24638,24767,24895,25278,24895,
	24767, 6855,24767,24895,25278,24895,24767,24637,24511,24383,
	24128,24385,24513,24514,24636,24770, 8392, 7371,24623,24765,
	24895,24896,24897,24770, 7378,24623,24765,24767, 7502,25664,
	/*data          strx15*/
	7356, 7364,24629,24766,25023,24896,25025,24898, 6987,24629,
	24766,24895, 9166,24626, 8398,24626, 6350,25152, 9152,25152,
	8114,25152, 6986, 7108,25394, 7630,25268, 9036,23858, 7246,
	25408, 8768,25408, 6844, 7236,25138, 7886,25013, 8779,24114,
	8782,25138, 7886,25013, 8779,24114, 6350,25536, 9408,25408,
	6844, 7236,26034, 6990,26034, 8270,23090, 8014,25408, 8768,
	25408, 6194,25408, 8768,25408, 6986, 7236,25394, 7630,25268,
	9036,23858,24380,24382,24383,24512,24513,24769,24767, 8020,
	/*data          strx16*/
	25408, 8768,25408, 6844, 8516,23218, 9806,23218, 8270,24508,
	24644,26176, 6706,26176,24644,24508, 7242, 7371,24619, 8405,
	24619, 7765,26560,24634,24518, 6443,25536, 7882, 8011,23595,
	9301,25643, 7250,25518, 6465,26432, 6335,26688, 6986, 8011,
	24255,24382,24510,24508,24637,24764,24766,24894,25023,24896,
	25025,24898,24770,24772,24643,24516,24514,24386,24257,24384,
	8256,24383,24382,24510,24508,24637,24764,24766,24894,24895,
	8512,24897,24898,24770,24772,24643,24516,24514,24386,24385,
	/*data          strx17*/
	7737,24633, 9031,24633, 7492,25408, 7487,25408, 7744, 8011,
	23723, 9173,25515, 7378,25390, 6336,25408, 9024,25408, 6858,
	7244,24507,10309,24507, 6972,24507, 9285,24507, 6972,24507,
	10309,24507, 6485,26432, 6463,26432, 6968,25408, 7487,25408,
	6968,26432, 6463,26432, 7241, 7371,24619, 8405,24619, 9813,
	24619, 8405,24619, 6101,27200, 5675,25536, 9024,25536, 6986,
	7115,25526,23605, 8277,25526, 7370,26560,24762,24390, 6572,
	26304, 6463,26560,24774,24378, 7370, 6982,24642,24770,24769,
	/*data          strx18*/
	24896,24767,24766,24764,24626, 7378,24898,24896,24894, 9406,
	24642,24514,24513,24384,24511,24510,24508,24626, 9170,24386,
	24384,24382, 7598,25536, 7498, 8011,24619, 8405,24619, 7888,
	24255,24511,24510,24637,24766,24767,25023,25280,25025,24769,
	24770,24643,24514,24513,24257,24000, 8256,24383,24511,24510,
	24637,24766,24767,24895, 8896,24897,24769,24770,24643,24514,
	24513,24385, 7493,25536, 7339,25536, 7626, 8139,24619, 8405,
	24619, 6990,24769,24895,24764,24766,24767,24895, 7369,24767,
	/*data          strx19*/
	24764,24766,24767,25023,25024,25025,24769,24770,24772,24769,
	7351,24897,24769,24770,24772,24897,24767, 6599,25536, 7339,
	25536, 7754, 7097,24765,25152,24388,24388,24515,24644,24771,
	24898,25025,25152,25023,24894,24765,24636,24509,24380,24380,
	25152,24771, 6593,24515,24516,24644,24771,24898,24897, 8768,
	24895,24894,24765,24636,24508,24509, 6717,25024, 9280,25024,
	7241, 8004,24255,24382,24510,24509,24637,24766,25023,24896,
	24897,25027,24899,24900,24771, 7104,24383,24382,24510,24509,
	/*data          strx20*/
	24637,24766,24895, 8654,24896,24895,24766,24888,24766,24767,
	7374,24767,24766,24888,24766,24895,24768, 7114, 8395,24255,
	24382,24380,24509,24508,24506,24504, 9692,24383,24382,24380,
	24509,24508,24506,24504, 9564,24896,24895,24767,24637,24510,
	24511,24255,24128, 9033,24894,24637,24510,24511,24383, 7744,
	25151,24894,24766,24637,24510,24511,24255,24384,24385,24513,
	24515, 8775,25023,24894,24766,24637,24510,24511,24383, 8010,
	6849,24898,24897,24896,24895,24767,24765,24636,24508,24248,
	/*data          strx21*/
	7635,24897,25152,24895, 9282,24509,24510,23993,24251,24380,
	9685,24509,24510,24121, 7880, 8387,24385,24384,24255,24381,
	24509,24637,24766,24767,24895,24896,25025,24899,24771,24643,
	24514,24133,24514,24642,24769,24896,24895,24894, 7355,24383,
	24381,24509,24636,24766, 8767,24897,24899,24771,24644,24514,
	24387,24514,24642,24769,24896,25022, 7095, 8515,24385,24256,
	24255,24382,24509,24637,24765,24767,25023,25024,24897, 7757,
	24383,24382,24509,24637,24765,24767,24895, 7751,25664, 7363,
	/*data          strx22*/
	8139,24383,24511,24639,24767,25023,25280,24641,24255,24126,
	24254,24253,24509,24638,24766,25022,25022,24766,24638,24511,
	24384,24513, 8917,24125,24253,24509,24638,24766,24894, 8138,
	6848,24770,24898,25024,24767,24638,24508,24377, 8398,24767,
	24638,24508,24377, 8647,24900,24898,24897,24896,24895,24767,
	24637,24507,24245, 8405,24894,24637,24507,24245, 7889, 8139,
	24255,24381,24510,24509,24507,24636,24766,24895,24896,25025,
	24899,24770,24771,24773,24644,24514,24385,24384, 8256,24383,
	/*data          strx23*/
	24381,24510,24509,24507,24636,24766,24767, 8512,24897,24899,
	24770,24771,24773,24644,24514,24513, 7222,25792, 7359, 7492,
	24377,24508,24638,24767,25024,24898,24770, 7882,24377,24508,
	24638,24767, 7626, 7492,24114, 8910,24114, 9934,24767,24768,
	24513,24384,24383,24124,24383,24384, 8512,24895,24890,24767,
	7624,24767,24890,24767,24896,24897,24899, 7110, 7115,24896,
	24895,24767,24766,25394,24766,24767, 6741,24894,24766,25394,
	24766,24895,24768, 7246,23602, 9294,23730, 8906, 7620,23851,
	/*data          strx24*/
	9173,23851, 8914,24506,24637,24894,24896,24897,24898,24899,
	8520,24245,24638,24767,25024,24898,24770, 7882,24245,24638,
	24767, 7498, 7492,24370, 8654,24506,24507,24509, 9934,24508,
	24380, 8776,24509,24510,24381,24382,24254,24383,24255, 8142,
	25152, 8380, 8139,24383,24511,24639,24767,25023,25024, 7872,
	24127,24383,24510,24638,24894,25023,25024, 8137,24255,24383,
	24510,24638,24894,24895, 8256,24127,24383,24510,24638,24894,
	25278,24767,24638,24383,24384, 8654,24255,24383,24510,24638,
	/*data          strx25*/
	24894,25150, 7629, 7876,24255,24381,24509,24637,24766,24767,
	24895,24896,25025,24899,24771,24643,24514,24513,24385,24384,
	8256,24383,24381,24509,24636,24766, 8767,24897,24899,24771,
	24644,24514, 7357, 7875,24115, 8781,24243, 9421,24627, 8269,
	24755, 6475,24898,25025,26304, 5949,24897,25025,26304, 6973,
	7227,24765,24767,24895,24896,25025,24899,24771,24643,24514,
	24513,24385,24384,24255,24381,24509,24114, 9543,24897,24899,
	24771,24644,24514, 7745,24383,24381,24509,24114, 9041, 9156,
	/*data          strx26*/
	23360,24255,24381,24509,24637,24766,24767,24895,24896,25025,
	24899,24771,24643,24514,24513,24385, 8000,24383,24381,24509,
	24636,24766, 8767,24897,24899,24771,24644,24514, 8256,25408,
	6973, 8131,24243, 8653,24371, 7371,24898,25025,26048, 6205,
	24897,25025,26048, 6973, 6848,24770,24898,25024,24767,24638,
	24378,24637,24894, 8014,24767,24638,24378,24637,24767,24895,
	24768,25025,24898,24899,24771,24643,24514,24511,24767,24765,
	8125,24774, 7102, 8523,23844, 9180,23588, 8661,24127,24382,
	/*data          strx27*/
	24509,24637,24766,24894,25023,25024,25153,24898,24771,24643,
	24514,24386,24257,24256, 8256,24255,24382,24509,24637,24766,
	24894,24895, 8640,25025,24898,24771,24643,24514,24386,24385,
	7868, 6980,24896,24895,24766,25265,24766,24767, 6997,24767,
	24766,25265,24766,24895,24896, 8405,24510,24381,23349,24381,
	24510, 8913, 8651,23844, 9180,23588, 7377,24770,24898,25024,
	24767,24638,24507,24637,24894,25024,24897,25027,24899, 6598,
	24767,24638,24507,24637,24766,24895,25024,24897,24898,24899,
	/*data          strx28*/
	24770,24902, 6972, 7232,24898,25025,24513,24383,24381,24509,
	24637,24765,24767,24896,24897,24899,24771, 7101,24766,24767,
	24896,24897,24898, 8259,24637,24765,24767,24896,24897,24899,
	24771,24643,24515,24513,24511,24895,24766, 7098,24766,24767,
	24896,24897,24898, 7238, 7883,24255,24381,24507,24637,24763,
	24893,25023,24896,25025,24899,24773,24643,24517,24387,24257,
	24384, 8256,24383,24511,24510,24507,24637,24763,24766,24767,
	24895, 8512,24897,24769,24770,24773,24643,24517,24514,24513,
	/*data          strx29*/
	24385, 7861, 7495,24897,25027,24619, 8148,24620, 7744,25792,
	7370, 7239,24767,24511,24513,24641,24770,24769,25025,25152,
	25023,24767,24766,24638,24510,24254,23998,24383,24382,24509,
	24637, 9429,24895,24767,24766,24638,24510,24254,24126, 7609,
	24769,24896,25278,25024,24897,24769, 6848,25277,25152,24769,
	24770,24642, 7109, 7239,24767,24511,24513,24641,24770,24769,
	25025,25152,25023,24766,24637,24510,24255,24256, 8649,24895,
	24766,24637,24510,24383, 8256,24895,24894,24766,24637,24510,
	/*data          strx30*/
	24511,24255,24128,24257,24513,24514,24641,24769,24767,24511,
	9670,24765,24637,24510,24511,24383, 7754, 8265,24621, 8405,
	24619, 8277,23217,26688, 7098,25536, 7242, 7371,24374, 8256,
	24898,25025,25024,25023,24894,24765,24638,24509,24382,24255,
	24256,24257,24513,24514,24641,24769,24767,24511, 9162,24895,
	24894,24765,24638,24509,24382,24383, 7509,25920, 6975,25280,
	25281, 7349, 8648,24511,24767,24769,24641,24514,24385,24256,
	24255,24382,24510,24508,24634,24765,24894,25023,24896,25025,
	/*data          strx31*/
	24898,24771,24641,24515,24386,24257,24512,24255,24382,24509,
	9038,24383,24382,24510,24508,24634,24765,24894,24895, 8512,
	24897,24898,24771,24641,24515,24386,24385, 7869, 7115,24634,
	8258,24770,24898,24896,25277,24896,24769,24770, 6590,24897,
	24896,25278, 8771,24637,24509,24123,24510,24509,24635, 9039,
	23995,24510,24509,24635, 8138, 7755,24255,24510,24637,24766,
	25023,25152,25025,24770,24643,24514,24257,24128, 8256,24383,
	24510,24637,24766,24895, 8768,24897,24770,24643,24514,24385,
	/*data          strx32*/
	7735,24255,24511,24510,24636,24766,24767,25023,25152,25025,
	24769,24770,24644,24514,24513,24257, 7744,24383,24511,24510,
	24636,24766,24767,24895, 8768,24897,24769,24770,24644,24514,
	24513,24385, 7742, 8772,24509,24382,24255,24512,24257,24386,
	24515,24641,24771,24898,25025,24896,25023,24894,24765,24634,
	24508,24510,24382,24255,24256,24385,24514,24641,24769,24767,
	24511, 8773,24385,24386,24515,24641,24771,24898,24897, 8512,
	24895,24894,24765,24634,24508,24510,24382,24383, 8010, 7352,
	/*data          strx33*/
	24511,24767,24769,24513, 7368, 7350,24513,24769,24767,24638,
	24510,24511, 7502, 7364,24511,24767,24769,24513, 8244,24511,
	24767,24769,24513, 7368, 7364,24511,24767,24769,24513, 8242,
	24513,24769,24767,24638,24510,24511, 7502, 7371,24510,24756,
	24780,24514, 8254,24634, 8245,24511,24767,24769,24513, 7368,
	7239,24767,24511,24513,24641,24770,24769,24897,25024,25023,
	24767,24766,24638,24510,24511,24126,24637, 8398,24895,24767,
	24766,24638,24510,24382, 7991,24511,24767,24769,24513, 7880,
	/*data          strx34*/
	7243,24505, 8519,24377, 7356, 7243,24505, 8519,24377, 9415,
	24505, 8519,24377, 7356, 7499,24383,24510,24638,24766,24895,
	24896,24897,24770,24642,24514,24385,24384, 7733, 9295,22304,
	9297, 8143,24382,24381,24380,24507,24636,24763,24892,24893,
	24894, 8030,24380,24509,24507,24636,24763,24765,24892, 7375,
	7119,24894,24893,24892,24763,24636,24507,24380,24381,24382,
	8542,24892,24765,24763,24636,24507,24509,24380, 7887, 7247,
	24608, 8416,24608, 8160,25536, 7328,25536, 7121, 7887,24608,
	/*data          strx35*/
	8416,24608, 7392,25536, 7328,25536, 7249, 7887,24253,24509,
	24638,24765,25021, 8012,24509,24636,24765, 8510,24254,25022,
	8256,24253,24509,24638,24765,25021, 8012,24509,24636,24765,
	7631, 7375,25021,24765,24638,24509,24253, 8524,24765,24636,
	24509, 7998,25022,24254, 8256,25021,24765,24638,24509,24253,
	8524,24765,24636,24509, 7631, 8015,23728,25520, 7249, 7247,
	25520,23728, 8017, 7247,24608, 7249, 7231,26944, 7233, 8392,
	24622, 7113,26944, 7233, 8263,24623, 7241,26688, 6199,26688,
	/*data          strx36*/
	7242, 7238,26418, 8270,22834, 9032, 8392,24511,24767,24769,
	24513, 7095,26944, 7097,24511,24767,24769,24513, 8392, 7234,
	26944, 5946,26944, 7236, 9288,22830, 8012,26944, 5946,26944,
	7236, 9288,22583,26679, 7242, 7240,26679,22583, 9290, 9291,
	22585,26681, 6206,26688, 6203,26688, 7242, 7243,26681,22585,
	8254,26688, 6203,26688, 7242, 9402,24384,24385,24386,24260,
	24513,24385,24384,24383,24510,24638,24766,24895,24896,24897,
	24769,25028,24898,24897,24896, 7228, 9287,23744,24127,24383,
	/*data          strx37*/
	24382,24509,24638,24765,24894,24895,25151,25536, 7241, 7239,
	24633,24764,24766,24894,25023,24896,25025,24898,24770,24772,
	24647, 7225, 7239,25536,25151,24895,24894,24765,24638,24509,
	24382,24383,24127,23744, 9289, 7223,24647,24772,24770,24898,
	25025,24896,25023,24894,24766,24764,24633, 7241, 9287,23744,
	24127,24383,24382,24509,24638,24765,24894,24895,25151,25536,
	6216,26176, 7745, 9153,25022,24254, 7879,25275,23995, 6725,
	26816, 7361, 7493,24899,24893, 7357,25285,25275, 7621,24623,
	/*data          strx38*/
	7754, 7617,24254,25022, 8647,23995,25275, 7621,26816, 7233,
	7481,24893,24899, 7363,25275,25285, 7628,24623, 7753, 8639,
	24515,24513,24385,24384,24255,24381,24509,24637,24766,24767,
	24895,24896,25025,24898,24771,24773,24645,24515,24513,24385,
	24256,24383,24511,24639,24768,24641, 8507,24383,24381,24509,
	24636,24766, 8767,24897,24898,24771,24773,24645,24515,24386,
	7605, 6987,25643, 7381,25517, 9299,23595, 7253,26688, 6335,
	26432, 7094, 9422,24511,24767,24769,24641,24513,24384,24383,
	/*data          strx39*/
	24382,24510,24509,24508,24372,24508,24510, 9308,24510,24508,
	24372,24508,24509,24510,24382,24383,24384,24513,24641,24769,
	24767,24511, 9424, 9422,24511,24767,24769,24641,24513,24384,
	24383,24382,24510,24509,24508,24372,24508,24510, 9308,24510,
	24508,24372,24508,24509,24510,24382,24383,24384,24513,24641,
	24769,24767,24511, 9302,24255,24382,24509,24638,24765,24894,
	25023,24896,25025,24898,24771,24642,24515,24386,24257,24384,
	8378, 9534,24510,24383,24384,24385,24513,24260,24513,24385,
	/*data          strx40*/
	24384,24383,24510,24638,24766,24895,24896,24897,24769,25028,
	24769,24897,24896,24895,24766,24638, 7106, 9419,22315, 8917,
	24894,24638,24510,24383,24384,24386,24642,24770,24897,24896,
	24895,25023,25024,25025,24897, 7730,24383,24510,24638,24894,
	24896,24897,24770,24642,24386,24384, 7619, 9411,24511,24767,
	24769,24641,24513,24512,24511,24510,24379,24381,24382,24383,
	24256,24257,24514,24643,24770,25412,24898,24770,24642,24514,
	24385,24383,24510,24638,24765,24893,25273,24894,25023,24768,
	/*data          strx41*/
	24769,24641, 6334,24385,24514,24643,24770,24898, 8262,24766,
	25653,24894,24895, 7370, 9027,24514,24385,24256,24383,24511,
	24509,24637,24766,24895,25024,24897,24770, 7624,24382,24509,
	24637,24766,24767, 9163,24504,24638,24895,24896,24898,24771,
	24642,24515,24514,24386,24385,24257,24256,24255,24383,24382,
	24510,24509,24637,24765,24766,24894,24895,25023,25024,25025,
	24897,24769, 8013,24504,24638,24767, 7749, 7759,24611, 8797,
	24611, 8790,24511,24767,24769,24641,24386,24257,24128,24255,
	/*data          strx42*/
	24382,24638,24766,24767,24895,25406,24895,24894, 6473,24894,
	24895,25406,24895,24767,24766,24636,24382,24255,24128,24257,
	24386,24641,24769,24767,24511, 8775, 8139,23716, 9948,23716,
	7505,26432, 6330,26432, 7238, 8136,24511,24767,24769,24641,
	24514,24385,24384,24383,24510,24638,24766,24894,25277, 7365,
	25277,24894,24766,24638,24510,24382, 7756,24382,24510,24638,
	24766,24894,25277, 7365,25277,24894,24766,24638,24510,24383,
	24384,24385,24514,24641,24769,24767,24511, 8142, 7755,24510,
	/*data          strx43*/
	24766,24770,24514, 8256,24612, 8273,24509,24754,24782,24515,
	7492,24895,24897,24385,24383, 8256,26176, 7744,24895,24897,
	24385,24383, 7484, 7755,24510,24766,24770,24514, 8256,24626,
	8260,24510,24892,24510,24514,24900,24514, 8252,24626, 8260,
	24510,24766,24770,24514, 7505,24895,24897,24385,24383, 8256,
	26176, 7744,24895,24897,24385,24383, 7218,24895,24897,24385,
	24383, 8256,26176, 7744,24895,24897,24385,24383, 7498, 8651,
	24619, 6613,26304, 7222,25664, 6581,26304, 7242, 7622,25394,
	/*data          strx44*/
	8270,23858, 7754,26426, 8262,22842, 8900, 7232, 6720, 6208,
	5696, 5184, 4672, 4160, 3648, 3136, 5196, 5172, 5192, 5176,
	5188, 5180, 7232,26688, 7240,24624, 8264, 7224,26704, 6208,
	26672, 7240, 8248,25672,23624,23608,25656, 8264, 7224,26688,
	24656,22592,24624, 9288, 7232,25152,25156, 8260,24636,25148,
	8768,24128,24124, 8252,24644,24132, 8768, 7224,25156,24648,
	7748,25148,25664, 8772,24124,24632, 8764,24132,23616, 8772,
	7232,26688, 7240,24624, 8768,23616, 7748,24648, 8772,25664,
	/*data          strx45*/
	8764,24632, 7236, 7224,26704, 6208,26672, 8260,24124, 7232,
	24132, 8264,25156, 9280,25148, 7228, 7744,24128, 8264,25148,
	8768,24644, 9280,24124, 8252,25152, 8248,24132, 7744,24636,
	7232,25156, 8772, 7228,24648,25156,25664,25148,24632,24124,
	23616,24132, 9284, 8644,24511,24767,24769,24641,24386,24385,
	24256,24255,24382,24509,24638,24765,24894,25023,24896,25025,
	24898, 7371,24383,24382,24509,24638,24765,24894,24895, 8401,
	24620, 7882, 7234,26944,24636, 7234, 6964,26944, 6988, 7235,
	/*data          strx46*/
	24770,24897,24896,25023,24895,25023,24896,24897,24770, 5945,
	26944, 5948,26944, 7237, 7237,24767,24511,24513,24641,24770,
	24769,24897,25024,25023,24767,24766,24638,24510,24511,24126,
	8395,24895,24767,24766,24638,24510,24382, 8013,24510,24753,
	24783,24514, 8254,24632, 8246,24511,24767,24769,24513, 7880,
	7745,24254,25022, 8647,23995,25275, 7621,26944, 7621,25275,
	23995, 8647,25022,24254, 7747, 9287,23744,24127,24383,24382,
	24509,24638,24765,24894,24895,25151,25536, 6216,26176, 8138,
	/*data          strx47*/
	23980, 8523, 6987,25515,25557, 6714,25920, 7227, 9287,23232,
	24383,24382,24510,24638,24766,24894,24895,26048, 6204,26688,
	7241, 7239,26048,24895,24894,24766,24638,24510,24382,24383,
	23232, 8252,26688, 7241, 7879,24255,24382,24509,24638,24765,
	24894,25023,24896,25025,24898,24771,24642,24515,24386,24257,
	24384, 8382,24630, 7621,25920, 7360, 7879,24255,24382,24509,
	24638,24765,24894,25023,24896,25025,24898,24771,24642,24515,
	24386,24257,24384, 8893,23608, 8264,25656, 7492, 7499,24619,
	/*data          strx48*/
	8405,24619, 8267,24898,25025,24896,25023,24766,24629, 7758,
	24895,24766,24629, 6485,25152, 7723,25536, 8768,25536, 7636,
	22840, 9406, 7115,24896,24895,24767,24766,25394,24766,24767,
	6741,24894,24766,25394,24766,24895,24768, 7246,23602, 9294,
	23730, 9556,23351, 8895, 7108,25152,25396, 7372,25522,25819,
	6703, 6704,27456, 6736, 6737,27456, 6703, 6991,26912, 6993
	};


/* convert a ugs character string into a list of strokes.  Return the	*
*  number of strokes found						*/
strokes(text,x,y,bbits)
    char *text;			/* input string */
    float *x,*y;		/* arrays of x and y values */
    int *bbits;			/* array of blanking bits */
{
float scale;			/* current scale factor */
float xcur,ycur;		/* current x and y values */
float xold,yold,scaleold;	/* saved state */
register int npts,		/* no. of strokes	*/
	tmp,			/* temporary */
	off,			/* offset of char in stroke table */
	ns,			/* no. of strokes in this character */
	k;			/* stroke counter */
int c;				/* one character from the string */

    scale = 3.0;	/* Kludge.  This depends on the hard 4096
				in the rest of the code.  */
    npts = 0;
    xcur = ycur = 0.0;

    while((c=(int)(*text)) != '\0'){
	if(c < 0)c += 256;	/* make sure it is positive */
	off = ugs_offsets[c] - 1;	/* -1 because Fortran used [1:...] */
	ns  = ugs_nstrokes[c];

	if(off < 0)switch(ns){	/* control character */
	    case 1:	/* start superscript */
		xcur -= (8.0*scale)/3.0;
		ycur += 12.0*scale;
		scale = (2.0*scale)/3.0;
		break;
	    case 2:	/* end superscript */
		scale = (3.0*scale)/2.0;
		xcur += 4.0*scale;	/* black magic */
		ycur -= 12.0*scale;
		break;
	    case 3:	/* start subscript */
		xcur -= (8.0*scale)/3.0;
		ycur -= 12.0*scale;
		scale = (2.0*scale)/3.0;
		break;
	    case 4:	/* end subscript */
		scale = (3.0*scale)/2.0;
		xcur += 4.0*scale;
		ycur += 12.0*scale;
		break;
	    case 5:	/* save current position */
		xold = xcur;
		yold = ycur;
		scaleold = scale;
		break;
	    case 6:	/* return to saved position */
		xcur = xold;
		ycur = yold;
		scale = scaleold;
		break;
	    /* there are more control characters available! */
	    default:	break;
	}

	else for(k=0;k<ns;k++){	/* loop over strokes */
	    tmp = (int)ugs_strokes[off++];
	    bbits[npts] = tmp >> 14;
	    xcur += scale*(float)(((tmp & 0x3F80) >>7)-64);
	    ycur += scale*(float)(((tmp & 0x007F)    )-64);
	    if(k == ns-1)xcur += scale*24.0;
	    x[npts] = xcur;
	    y[npts] = ycur;
	    npts++;

	}
	text++;

    }
    return(--npts);
}
