/*
 * A simplified bc for doubles. Semicolons `;' used as expression separators.
 * ex. "sin(3.14159); 7*4 + 3; pow(sin(1), 2) + pow(cos(1), 2);" etc.
 *
 * 2.8.1991 Kenneth Oksanen <cessu@niksula.hut.fi>
 *   Cleaned up and reorganizing.
 *
 * 9.6.1991 Kenneth Oksanen <cessu@niksula.hut.fi>
 *   Cleaned up the code a bit, added more comments.
 *
 * 25.5.1991 Kenneth Oksanen <cessu@niksula.hut.fi>
 *   First Version.
 */

#include <stdio.h>
#include <strings.h>
#include <ctype.h>
#include <math.h>


/*
 * These should be in some utility library and merely #included
 */

typedef enum { FALSE = 0, TRUE = 1 } boolean_t;

#define EXIT_NO_ERROR 0
#define EXIT_USER_ERROR 1
#define EXIT_PROGRAM_ERROR 2


/*
 * Info about predefined functions stored in a NULL-ending vector.
 */

struct {
  char *name;
  int arity;
  double (*function)();
} function_vector[] = {
  { "sin",  1, sin },
  { "cos",  1, cos },
  { "atan", 1, atan },
  { "sqrt", 1, sqrt },
  { "exp",  1, exp },
  { "ln",   1, log }, 
  { "pow",  2, pow }, 
  { NULL,   0, NULL }
};

/*
 * Not elegant, but abbreviates things. Should be more or equal to
 * the amount of chars in longest name in function_vector[] .
 */
#define MAX_FUNCTION_NAME_LENGTH 10


typedef enum { 
  END_OF_FILE,     OPEN_PAREN,     CLOSE_PAREN,     SEMICOLON,
  COMMA,           PLUS_OP,        MINUS_OP,        TIMES_OP,
  DIVIDES_OP,      FUNCTION_NAME,  CONSTANT,        END = 0
} token_type_t;


typedef struct {
  token_type_t token_type;
  union {
    double numeric;
    int function_index;
  } value;
} token_t;


struct {
  int ch;
  token_type_t token_type;
} one_char_tokens[] = {
  { '(', OPEN_PAREN },  { ')', CLOSE_PAREN },   { ';', SEMICOLON },
  { ',', COMMA      },  { '+', PLUS_OP     },   { '-', MINUS_OP  },
  { '*', TIMES_OP   },  { '/', DIVIDES_OP  },   { '\0', END      }
};


/* The lexer. Gets the next token, analyzes it and stores info about it
 * in the given address.
 */

void get_next_token(token)
token_t *token;
{
  int ch, i;

#define VALUED_RETURN(type,field,analyzed_value,pushback,ch) \
  do { \
    if (pushback) \
      ungetc(ch, stdin); \
    token->token_type = type; \
    token->value.field = analyzed_value; \
    return; \
  } while (FALSE)

#define RETURN(type,pushback) VALUED_RETURN(type, numeric, 0, pushback, EOF)

  /*
   * Read away all whitespaces. If isspace weren't a macro, this would be
   *
   *   while (isspace(ch = getchar()))
   *     ;
   *
   * Notice that isspace() works correctly also on EOF.
   */
  do {
    ch = getchar();
  } while (isspace(ch));

  if (ch == EOF)
    RETURN(END_OF_FILE, FALSE);

  /*
   * ch was not EOF, now test if it is one of the one character long tokens
   * defined in one_char_tokens[].
   */
  for (i = 0; one_char_tokens[i].ch; i++)
    if (one_char_tokens[i].ch == ch)
      RETURN(one_char_tokens[i].token_type, FALSE);

  /*
   * If ch is a digit, the token to be read it a number.
   */
  if (isdigit(ch)) {
    double value = 0, unit_decimal_value;

    do { /* Get the integer part ... */
      value = 10 * value + ch - '0';
      ch = getchar();
    } while (isdigit(ch));
    if (ch != '.') 
      VALUED_RETURN(CONSTANT, numeric, value, TRUE, ch);
    unit_decimal_value = 1; /* ... and the possible decimal part. */
    ch = getchar();
    while (isdigit(ch)) {
      unit_decimal_value /= 10;
      value = value + unit_decimal_value * (ch - '0');
      ch = getchar();
    }
    VALUED_RETURN(CONSTANT, numeric, value, TRUE, ch);
  }

  /*
   * ch starts a function name, read it ...
   */
  if (isalpha(ch)) {
    char s[MAX_FUNCTION_NAME_LENGTH+1];

    /*
     * A function name is [a-zA-Z][a-zA-Z0-9]* as a regexp. Read such
     * a regexp as long as it is possible.
     */
    for (i = 0; i < MAX_FUNCTION_NAME_LENGTH && isalnum(ch); i++) {
      s[i] = ch;
      ch = getchar();
    }
    s[i] = '\0';

    /*
     * Search for the name in function_vector[] and return the index.
     */
    for (i = 0; function_vector[i].name != NULL; i++)
      if (!strcmp(function_vector[i].name, s))
	VALUED_RETURN(FUNCTION_NAME, function_index, i, TRUE, ch);

    /*
     * If name didn't match any of the predefined functions, report an error.
     */
    fprintf(stderr, "unrecognized function name '%s'\n", s);
    exit(EXIT_USER_ERROR);
  }
  
  /* The input could not be recognized.
   */
  fprintf(stderr, "garbled input\n");
  exit(EXIT_USER_ERROR);

#undef RETURN
#undef VALUED_RETURN
  /*NOTREACHED*/
}


/*
 * The lexer ends and parser starts here.
 */

static token_t lookahead;                  /* Initialized in main() */

#define SHIFT() get_next_token(&lookahead) /* Read next token */

#define MATCH(t,message) \
  do { \
    if (lookahead.token_type != t) { \
      fprintf(stderr, "expected %s\n", message); \
      exit(EXIT_USER_ERROR); \
    } \
    SHIFT(); \
  } while (FALSE)

static double parse_and_eval_expr();


/*
 * The highest precedence level corresponds to this function
 */

double parse_and_eval_factor()
{
  double tmp;
  int i;

  switch (lookahead.token_type) {
  case CONSTANT:
    tmp = lookahead.value.numeric;
    SHIFT();
    return tmp;
  case OPEN_PAREN:
    SHIFT();
    tmp = parse_and_eval_expr();
    MATCH(CLOSE_PAREN, "')'");
    return tmp;
  case MINUS_OP:
    SHIFT();
    return -parse_and_eval_factor();
  case FUNCTION_NAME:
    i = lookahead.value.function_index;
    SHIFT();
    MATCH(OPEN_PAREN, "'('");
    tmp = parse_and_eval_expr();
    switch (function_vector[i].arity) {
    case 1: /* Unary function */
      MATCH(CLOSE_PAREN, ")");
      return function_vector[i].function(tmp);
    case 2: /* binary function */
      MATCH(COMMA, "','");
      tmp = function_vector[i].function(tmp, parse_and_eval_expr());
      MATCH(CLOSE_PAREN, "')'");
      return tmp;
    default:
      fprintf(stderr, "Function arities above 2 not implemented.\n");
      exit(EXIT_PROGRAM_ERROR);
    }
    /*NOTREACHED*/
    break;
  default:
    fprintf(stderr, "syntax error\n");
    exit(EXIT_USER_ERROR);
  }
  /*NOTREACHED*/
}


double parse_and_eval_term()
{
  double value;
  token_type_t operator;

  value = parse_and_eval_factor();
  while ((operator = lookahead.token_type) == TIMES_OP || 
	 operator == DIVIDES_OP) {
    SHIFT();
    if (operator == TIMES_OP)
      value *= parse_and_eval_factor();
    else
      value /= parse_and_eval_factor();
  }
  return value;
}


double parse_and_eval_expr()
{
  double value;
  token_type_t operator;

  value = parse_and_eval_term();
  while ((operator = lookahead.token_type) == PLUS_OP ||
	 operator == MINUS_OP) {
    SHIFT();
    if (operator == PLUS_OP)
      value += parse_and_eval_term();
    else
      value -= parse_and_eval_term();
  }
  return value;
}


int main()
{
  SHIFT();

  while (lookahead.token_type != END_OF_FILE) {
    printf("%lf\n", parse_and_eval_expr());
    if (lookahead.token_type == SEMICOLON)
      SHIFT();
    else if (lookahead.token_type != END_OF_FILE) {
      fprintf(stderr, "expected ';' or '^D' (end of file)");
      exit(EXIT_USER_ERROR);
    }
  }
  exit(EXIT_NO_ERROR);
  /*NOTREACHED*/
}
