(*
   RPCSun.m3
   Sun RPC
   David Nichols, Xerox PARC
   July, 1991

   $Id: RPCSun.m3,v 1.6 1992/05/20 22:05:38 nichols Exp $
*)

(* Copyright (c) 1991, 1992 Xerox Corporation.  All rights reserved.

   Use and copying of this software and preparation of derivative works
   based upon this software are permitted.  Any distribution of this
   software or derivative works must comply with all applicable United
   States export control laws.  This software is made available AS IS, and
   Xerox Corporation makes no warranty about the software, its performance
   or its conformity to any specification. *)

UNSAFE MODULE RPCSun EXPORTS RPCSun, RPCSunPriv;

IMPORT M3toC, PortMapper, Random, RPC, Thread, Uin, Unetdb, Usocket, XDR;


(*
 * Binding info
 *)

(* Encapsulate some binding information.  Only TCP is supported right now.
   If the port is 0, the runtime either discovers it (by calling the remote
   port mapper) or allocates one (for servers). *)
PROCEDURE CreateBindingInfo (hostAddr   : INTEGER;
                             progNum    : INTEGER;
                             progVersion: INTEGER;
                             port                   := 0;
                             protocol               := Protocol.TCP):
  BindingInfo =
  VAR bi: BindingInfo;
  BEGIN
    bi := NEW(BindingInfo);
    bi.hostAddr := hostAddr;
    bi.port := port;
    bi.progNum := progNum;
    bi.progVersion := progVersion;
    bi.proto := protocol;
    RETURN bi;
  END CreateBindingInfo;

PROCEDURE DecodeBindingInfo (    b          : BindingInfo;
                             VAR hostAddr   : INTEGER;
                             VAR progNum    : INTEGER;
                             VAR progVersion: INTEGER;
                             VAR port       : INTEGER;
                             VAR protocol   : Protocol     ) =
  BEGIN
    hostAddr := b.hostAddr;
    progNum := b.progNum;
    progVersion := b.progVersion;
    port := b.port;
    protocol := b.proto;
  END DecodeBindingInfo;

(*
 * Client services
 *)

(* Create a client from an address. *)
PROCEDURE ImportService (bi: BindingInfo): Client
  RAISES {RPC.Failed, Thread.Alerted} =
  BEGIN
    IF bi.port = 0 THEN
      bi.port := LookupPort(bi);
      (* If still zero, then the program isn't registered. *)
      IF bi.port = 0 THEN
        RAISE
          RPC.Failed(NEW(RPC.Failure, info := "program not registered"));
      END;
    END;
    CASE bi.proto OF
      Protocol.UDP => RETURN UDPImportService(bi);
    | Protocol.TCP => RETURN TCPImportService(bi);
    END;
  END ImportService;

(*
 * Server services
 *)

(* Standard export (from RPCSun.i3).  Caller provides program number,
   version number, protocol and optional port number, and runtime chooses
   port number and allocates a socket. *)
PROCEDURE Export (sp        : ServerProc;
                  prog, vers: INTEGER;
                  protocol  : Protocol;
                  port                     := 0): BindingInfo
  RAISES {Erred, RPC.Failed, Thread.Alerted} =
  VAR s: INTEGER;
  BEGIN
    CASE protocol OF
      Protocol.UDP =>
        s := GetUDPSocket(port);
        RETURN ExportUDP(sp, prog, vers, s);
    | Protocol.TCP =>
        s := GetListeningSocket(port);
        RETURN ExportTCPListener(sp, prog, vers, s);
    END;
  END Export;


(*
 * Misc. marshalling routines.
 *)

(* Encode a call header. *)
PROCEDURE PutCallHeader (s                    : XDR.Sink;
                         xid, prog, vers, proc: INTEGER;
                         cred, verf           : Credentials)
  RAISES {XDR.Failed, Thread.Alerted} =
  BEGIN
    XDR.PutInteger(s, xid);
    XDR.PutInteger(s, CALLMSG);
    XDR.PutInteger(s, RPCVERS);
    XDR.PutInteger(s, prog);
    XDR.PutInteger(s, vers);
    XDR.PutInteger(s, proc);
    PutAuth(s, cred);
    PutAuth(s, verf);
  END PutCallHeader;

PROCEDURE GetCallHeader (    s                    : XDR.Source;
                         VAR xid, prog, vers, proc: INTEGER;
                         VAR cred, verf           : Credentials )
  RAISES {RPC.Failed, XDR.Failed, Thread.Alerted, HeaderError} =
  BEGIN
    xid := XDR.GetInteger(s);
    IF XDR.GetInteger(s) # CALLMSG THEN
      RAISE RPC.Failed(NEW(RPC.Failure, info := "not a call msg"));
    END;
    IF XDR.GetInteger(s) # RPCVERS THEN RAISE HeaderError; END;
    prog := XDR.GetInteger(s);
    vers := XDR.GetInteger(s);
    proc := XDR.GetInteger(s);
    cred := GetAuth(s);
    verf := GetAuth(s);
  END GetCallHeader;

PROCEDURE PutReplyHeader (s                 : XDR.Sink;
                          xid, accept, code : INTEGER;
                          authWhy, low, high              := 0;
                          verf              : Credentials := NIL)
  RAISES {XDR.Failed, Thread.Alerted} =
  BEGIN
    XDR.PutInteger(s, xid);
    XDR.PutInteger(s, REPLYMSG);
    XDR.PutInteger(s, accept);
    CASE accept OF              <* NOWARN *>
      MSG_ACCEPTED =>
        PutAuth(s, verf);
        XDR.PutInteger(s, code);
        CASE code OF            <* NOWARN *>
          ACCEPT_SUCCESS, ACCEPT_PROG_UNAVAIL, ACCEPT_PROC_UNAVAIL,
            ACCEPT_GARBAGE_ARGS => (* no data *)
        | ACCEPT_PROG_MISMATCH =>
            XDR.PutInteger(s, low);
            XDR.PutInteger(s, high);
        END;
    | MSG_DENIED =>
        XDR.PutInteger(s, code);
        CASE code OF            <* NOWARN *>
          REJECT_RPC_MISMATCH =>
            XDR.PutInteger(s, low);
            XDR.PutInteger(s, high);
        | REJECT_AUTH_ERROR => XDR.PutInteger(s, authWhy);
        END;
    END;
  END PutReplyHeader;

PROCEDURE GetReplyHeader (    s                    : XDR.Source;
                          VAR xid                  : INTEGER;
                          VAR accept, code, authWhy: INTEGER;
                          VAR low, high            : INTEGER;
                          VAR verf                 : Credentials )
  RAISES {RPC.Failed, XDR.Failed, Thread.Alerted} =
  BEGIN
    xid := XDR.GetInteger(s);
    IF XDR.GetInteger(s) # REPLYMSG THEN
      RAISE
        RPC.Failed(NEW(RPC.Failure, info := "protocol error: msg type"));
    END;
    accept := XDR.GetInteger(s);
    CASE accept OF
      MSG_ACCEPTED =>
        verf := GetAuth(s);
        code := XDR.GetInteger(s);
        CASE code OF
          ACCEPT_SUCCESS, ACCEPT_PROG_UNAVAIL, ACCEPT_PROC_UNAVAIL,
            ACCEPT_GARBAGE_ARGS => (* no data *)
        | ACCEPT_PROG_MISMATCH =>
            low := XDR.GetInteger(s);
            high := XDR.GetInteger(s);
        ELSE
          RAISE RPC.Failed(NEW(RPC.Failure,
                               info := "protocol error: bad acceptance"));
        END;
    | MSG_DENIED =>
        code := XDR.GetInteger(s);
        CASE code OF
          REJECT_RPC_MISMATCH =>
            low := XDR.GetInteger(s);
            high := XDR.GetInteger(s);
        | REJECT_AUTH_ERROR => authWhy := XDR.GetInteger(s);
        ELSE
          RAISE RPC.Failed(
                  NEW(RPC.Failure, info := "protocol error: bad denial"));
        END;
    ELSE
      RAISE RPC.Failed(
              NEW(RPC.Failure, info := "protocol error: bad reply type"));
    END;
  END GetReplyHeader;

(* Encode credentials.  For now, we just send null ones. *)
PROCEDURE PutAuth (s: XDR.Sink; <*UNUSED*> cred: Credentials)
  RAISES {XDR.Failed, Thread.Alerted} =
  BEGIN
    XDR.PutInteger(s, AUTH_NULL);
    XDR.PutInteger(s, 0);
    (* No bytes of data. *)
  END PutAuth;

(* Decode credentials.  For now, we only understand null ones. *)
PROCEDURE GetAuth (s: XDR.Source): Credentials
  RAISES {XDR.Failed, Thread.Alerted} =
  VAR
    kind, length, n: INTEGER;
    a              : ARRAY [0 .. 127] OF CHAR;
  BEGIN
    kind := XDR.GetInteger(s);
    length := XDR.GetInteger(s);
    (* Gobble up the opaque stuff and ignore it. *)
    WHILE length > 0 DO
      n := MIN(length, 128);
      XDR.GetBytes(s, a);
      length := length - n;
    END;
    RETURN NIL;
  END GetAuth;

(*
 * Port mapper.
 *)

PROCEDURE LookupPort (b: BindingInfo): INTEGER
  RAISES {RPC.Failed, Thread.Alerted} =
  VAR
    proto: INTEGER;
    port : INTEGER;
    pbi  : BindingInfo;         (* port mapper binding info *)
    pm: PortMapper.PMAP_VERSClient; (* port mapper client *)
  BEGIN
    pbi := CreateBindingInfo(b.hostAddr, PMProg, PMVers,
                             PortMapper.PMAP_PORT, Protocol.UDP);
    pm := PortMapper.ImportPMAP_VERS(pbi);
    CASE b.proto OF
      Protocol.UDP => proto := PortMapper.IPPROTO_UDP;
    | Protocol.TCP => proto := PortMapper.IPPROTO_TCP;
    END;
    TRY
      port := pm.GetPort(b.progNum, b.progVersion, proto, b.port);
    EXCEPT
      RPC.Failed (v) =>
        RAISE
          RPC.Failed(NEW(RPC.Failure, info := "can't contact portmapper",
                         subArg := v))
    END;
    pm.GetClient().Destroy();
    RETURN port;
  END LookupPort;

CONST
  TransientProgNumBase = 16_40000000; (* Start of SunRPC transient program
                                         numbers. *)
  TransientProgNumMax = 16_5FFFFFFF; (* First number past legal SunRPC
                                        transient program number. *)
  TransientStartMax = 16_48000000; (* max value we'll pick randomly *)
TYPE
  Registration = REF RECORD
                       next      : Registration;
                       prog, vers: INTEGER;
                     END;
VAR registrations: Registration := NIL;

PROCEDURE PortMapperRegister (VAR (*in/out*) prog      : INTEGER;
                                             vers, port: INTEGER;
                                             protocol  : Protocol )
  RAISES {Erred, RPC.Failed, Thread.Alerted} =
  VAR
    proto: INTEGER;
    pbi  : BindingInfo;         (* port mapper binding info *)
    pm: PortMapper.PMAP_VERSClient; (* port mapper client *)
    r : Registration;
  BEGIN
    pbi := CreateBindingInfo(
             0, PMProg, PMVers, PortMapper.PMAP_PORT, Protocol.UDP);
    pm := PortMapper.ImportPMAP_VERS(pbi);
    CASE protocol OF
      Protocol.UDP => proto := PortMapper.IPPROTO_UDP;
    | Protocol.TCP => proto := PortMapper.IPPROTO_TCP;
    END;
    IF prog = TransientProgram THEN
      (* Find a transient program number to use and register it.  We do
         this by successively picking transient program numbers and trying
         to register them until one succeeds.  NOTE: Unlike the
         non-transient case, we should NOT try to unregister anything. *)
      prog := PickTransientProgramNumber();
      WHILE NOT pm.Set(prog, vers, proto, port) DO
        INC(prog);              (* Try the next one up. *)
        IF prog >= TransientProgNumMax THEN
          RAISE
            RPC.Failed(
              NEW(RPC.Failure,
                  info := "can't find a free transient program number"));
        END;
      END;
    ELSE
      (* If we've never registered this prog/vers for any protocol before,
         then we need to unregister it at the portmapper.  We're assuming
         that we'll never have one process export for UDP and another for
         TCP for the same prog/vers pair, but the semantics of
         pmapproc_unset sort of guarantee that, anyway. *)
      r := registrations;
      WHILE r # NIL AND (r.prog # prog OR r.vers # vers) DO
        r := r.next;
      END;
      IF r = NIL THEN
        r := NEW(Registration, next := registrations, prog := prog,
                 vers := vers);
        registrations := r;
        EVAL pm.Unset(prog, vers, proto, 0);
      END;
      (* Now register ourselves. *)
      IF NOT pm.Set(prog, vers, proto, port) THEN
        RAISE RPC.Failed(NEW(RPC.Failure,
                             info := "can't register with port mapper"));
      END;
    END;
    pm.GetClient().Destroy();
  END PortMapperRegister;

VAR rand: Random.T := NIL;
PROCEDURE PickTransientProgramNumber (): INTEGER =
  (* Return a "randomly" picked transient program number. *)
  BEGIN
    IF rand = NIL THEN rand := Random.New(-1); END;
    RETURN Random.Subrange(rand, TransientProgNumBase, TransientStartMax);
  END PickTransientProgramNumber;

(*
 * Host name lookup.
 *)

(* Convenience routine for host name translation.  Returns the first *host
   order* internet address of of the name provided that it finds with
   gethostbyname call. *)
PROCEDURE LookupHost (hostName: TEXT): INTEGER RAISES {RPC.Failed} =
  VAR
    pi: UNTRACED REF INTEGER;
    he: Unetdb.struct_hostent_star;
  BEGIN
    he := Unetdb.gethostbyname(M3toC.TtoS(hostName));
    IF he = NIL OR he.h_addr_list = NIL THEN
      RAISE RPC.Failed(NEW(RPC.Failure, info := "gethostbyname failed"));
    END;
    pi := LOOPHOLE(he.h_addr_list^, UNTRACED REF INTEGER);
    RETURN Uin.ntohl(pi^);
  END LookupHost;


(*
 * Socket manipulation.
 *)

VAR myHostAddr: INTEGER;

PROCEDURE SetAddr (VAR addr: Uin.struct_sockaddr_in; host, port: INTEGER) =
  BEGIN
    addr.sin_family := Usocket.AF_INET;
    addr.sin_addr := Uin.union_in_addr{Uin.htonl(host)};
    addr.sin_port := Uin.htons(port);
    FOR i := FIRST(addr.sin_zero) TO LAST(addr.sin_zero) DO
      addr.sin_zero[i] := VAL(0, CHAR);
    END;
  END SetAddr;

(* Deduce port number from a socket. *)
PROCEDURE GetHostPortFromSocket (s: INTEGER; VAR host, port: INTEGER)
  RAISES {Erred, RPC.Failed} =
  VAR
    addr: Uin.struct_sockaddr_in;
    len : INTEGER                := BYTESIZE(addr);
  BEGIN
    IF Usocket.getsockname(s, ADR(addr), ADR(len)) = -1 THEN
      RAISE RPC.Failed(NEW(RPC.Failure, info := "getsockname failed"));
    END;
    IF addr.sin_family # Usocket.AF_INET THEN
      RAISE Erred(NEW(Error, info := "not an Internet socket"));
    END;
    host := myHostAddr;
    port := Uin.ntohs(addr.sin_port);
  END GetHostPortFromSocket;

BEGIN
  myHostAddr := GetLocalInetAddr();
END RPCSun.
