/* Copyright (c) Stichting Mathematisch Centrum, Amsterdam, 1986. */

/* B error message handling */

/* There are two kinds of errors:
	1) parsing, when the line in error is in a buffer
	2) execution, when the line in error is a parse-tree, and must
	   therefore be reconstructed.
*/

#include "b.h"
#include "bmem.h"
#include "bint.h"
#include "feat.h"
#include "bobj.h"
#include "i0err.h"
#include "i2par.h"
#include "i3env.h"
#include "i3scr.h"
#include "i3sou.h"

#ifdef GFX
#include "bgfx.h"
#endif

Visible bool still_ok= Yes;
Visible bool mess_ok= Yes;	/* if Yes print error message */
Visible bool interrupted= No;
Visible bool can_interrupt= Yes;

Visible parsetree curline= Vnil;
Visible value curlino;

Visible FILE *errfile= stderr;	/* may be changed in initerr() */

/*********************************************************************/

Hidden Procedure nline() {
	fflush(stdout); /* should be i3scr.c's ofile, but doesnot matter */
	if (cntxt == In_read && rd_interactive)
		at_nwl= Yes;
	if (!at_nwl)
		putnewline(errfile);
	at_nwl= Yes;
}

Visible intlet errlino= 0;

Hidden intlet pr_line(at) bool at; {
	/*prints the line that tx is in, with an arrow pointing to the column
	  that tx is at.
	*/
	txptr lx= fcol(); intlet ap= -1, p= 0; char c;
	txptr ax= tx;
	
	if (!at) do ax--; while (Space(Char(ax)));
	while (!Eol(lx) && Char(lx) != Eotc) {
		if (lx == ax) ap= p;
		c= *lx++;
		if (c == '\t') {
			do { putchr(errfile, ' '); } while (((++p)%4)!=0);
		} else { putchr(errfile, c); p++; }
	}
	putnewline(errfile);
	if (ap < 0) ap= p;
	for (p= 0; p < ap+4; p++) putchr(errfile, ' ');
	putstr(errfile, "^\n");
}

#define IN_COMMAND	MESS(3100, " in your command\n")
#define IN_READ		MESS(3101, " in your expression to be read\n")
#define IN_EDVAL	MESS(3102, " in your edited value\n")
#define IN_TARVAL	MESS(3103, " in your location %s\n")
#define IN_PRMNV	MESS(3104, " in your permanent environment\n")
#define IN_WSGROUP	MESS(3105, " in your workspace index\n")
#define IN_UNIT		MESS(3106, " in your how-to %s\n")
#define IN_UNIT_LINE	MESS(3107, " in line %d of your how-to %s\n")
#define IN_INPUT	MESS(3108, "*** (detected after reading 1 line of your input file standard input)\n")
#define IN_INPUT_LINE	MESS(3109, "*** (detected after reading %d lines of your input file standard input)\n")
#define IN_FILE		MESS(3110, "*** (detected after reading 1 line of your input file %s)\n")
#define IN_FILE_LINE	MESS(3111, "*** (detected after reading %d lines of your input file %s)\n")

Hidden Procedure show_where(in_node, at, node)
	bool in_node, at; parsetree node; {

	int line_no= in_node ? intval(curlino) : lino;
	show_line(in_node, at, node, line_no);
	if (!interactive && ifile == sv_ifile && !unit_file())
		show_f_line();
}

Hidden Procedure show_line(in_node, at, node, line_no)
	bool in_node, at; parsetree node; int line_no; {
	
	switch (cntxt) {
		case In_command: putmess(errfile, IN_COMMAND); break;
		case In_read: putmess(errfile, IN_READ); break;
		case In_edval: putmess(errfile, IN_EDVAL); break;
		case In_tarval:
			putSmess(errfile, IN_TARVAL, strval(errtname));
			break;
		case In_prmnv: putmess(errfile, IN_PRMNV); break;
		case In_wsgroup: putmess(errfile, IN_WSGROUP); break;
		case In_unit: show_howto(line_no); break;
		default:
			putstr(errfile, "???\n");
			return;
	}
	if (!in_node || Valid(node)) putstr(errfile, "    ");
	if (in_node) display(errfile, node, Yes);
	else pr_line(at);
}

Hidden value unitname(line_no) int line_no; {
	if (Valid(uname) && Is_text(uname)) {
		def_perm(last_unit, uname);
		errlino= line_no;
		return Permname(uname);
	}
	else free_perm(last_unit);
	return mk_text("");
}

Hidden Procedure show_howto(line_no) int line_no; {
	value name= unitname(line_no);
	if (line_no == 1)
		putSmess(errfile, IN_UNIT, strval(name));
	else
		putDSmess(errfile, IN_UNIT_LINE, line_no, strval(name));
	release(name);
}

Hidden bool unit_file() {
	value *aa;
	return cntxt == In_unit &&
		Valid(uname) && Is_text(uname) && p_exists(uname, &aa);
}

Hidden Procedure show_f_line() {
	if (f_lino == 1 && iname == Vnil) 
		putmess(errfile, IN_INPUT);
	else if (f_lino == 1)
		putSmess(errfile, IN_FILE, strval(iname));
	else if (iname == Vnil)
		putDSmess(errfile, IN_INPUT_LINE, f_lino, "");
	else
		putDSmess(errfile, IN_FILE_LINE, f_lino, strval(iname));
	if (iname != Vnil && i_lino > 0) {
		if (i_lino == 1)
			putmess(errfile, IN_INPUT);
		else
			putDSmess(errfile, IN_INPUT_LINE, i_lino, "");
	}
}

#define PROBLEM		MESS(3112, "*** The problem is:")

Visible Procedure syserr(m) int m; {
	static bool beenhere= No;
	if (beenhere) immexit(-1);
	beenhere= Yes;
	nline();
#ifdef DEBUG
#ifdef macintosh
	Debugger();
#endif
#endif
	putmess(errfile, MESS(3113, "*** Sorry, ABC system malfunction\n"));
	putmess(errfile, PROBLEM);
	putstr(errfile, " ");
	putmess(errfile, m); 
	putnewline(errfile);
	bye(-1);
}

#ifndef macintosh
	/* MacABC uses an alert to make sure the user gets the message */

Visible Procedure memexh() {
	static bool beenhere= No;
	if (beenhere) immexit(-1);
	beenhere= Yes;
	nline();
	putmess(errfile, MESS(3114, "*** Sorry, memory exhausted"));
/* show_where(Yes, Yes); don't know if in node or not; to fix */
	putnewline(errfile);
	bye(-1);
}

#endif /*macintosh*/

Hidden Procedure message(m1, m2, in_node, at, arg)
	int m1, m2;
	bool in_node, at; 
	value arg;
{
	still_ok= No;
	if (!mess_ok)
		return;
	nline();
	putmess(errfile, m1);
	show_where(in_node, at, curline);
	putmess(errfile, PROBLEM);
	putstr(errfile, " ");
	putSmess(errfile, m2, Valid(arg) ? strval(arg) : "");
	putnewline(errfile);
	fflush(errfile);
	at_nwl=Yes;
}

#define UNDERSTAND	MESS(3115, "*** There's something I don't understand")

#define RESOLVE		MESS(3116, "*** There's something I can't resolve")

#define COPE		MESS(3117, "*** Can't cope with problem")

#define RECONCILE	MESS(3118, "*** Cannot reconcile the types")

Visible Procedure pprerrV(m, v) int m; value v; {
	if (still_ok)
		message(UNDERSTAND, m, No, No, v);
}

Visible Procedure pprerr(m) int m; {
	if (still_ok)
		message(UNDERSTAND, m, No, No, Vnil);
}

Visible Procedure parerrV(m, v) int m; value v; {
	if (still_ok)
		message(UNDERSTAND, m, No, Yes, v);
}

Visible Procedure parerr(m) int m; {
	if (still_ok)
		message(UNDERSTAND, m, No, Yes, Vnil);
}

Visible Procedure fixerrV(m, v) int m; value v; {
	if (still_ok)
		message(RESOLVE, m, Yes, Yes, v);
}

Visible Procedure fixerr(m) int m; {
	if (still_ok)
		message(RESOLVE, m, Yes, Yes, Vnil);
}

Visible Procedure typerrV(m, v) int m; value v; {
	if (still_ok)
		message(RECONCILE, m, Yes, Yes, v);
}

Visible Procedure interrV(m, v) int m; value v; {
	if (still_ok)
		message(COPE, m, Yes, No, v);
}

Visible Procedure interr(m) int m; {
	if (still_ok)
		message(COPE, m, Yes, No, Vnil);
}

Visible Procedure checkerr() {
	still_ok= No;
	nline();
	putmess(errfile, MESS(3119, "*** Your check failed"));
	show_where(Yes, No, curline);
	fflush(errfile);
	at_nwl= Yes;
}

Visible Procedure int_signal() {
	if (can_interrupt) {
		interrupted= Yes; still_ok= No;
		if (cntxt == In_wsgroup || cntxt == In_prmnv)
			immexit(-1);
	}
	if (!interactive) {
		if (ifile != stdin) fclose(ifile);
		bye(1);
	}
	nline();
	putmess(errfile, MESS(3120, "*** interrupted\n"));
	fflush(errfile);
	if (can_interrupt) {
		if (cntxt == In_read) {
			set_context(&read_context);
			copy(uname);
		}
	}
	at_nwl= Yes;
}

Visible bool testing= No;

Visible Procedure bye(ex) int ex; {
#ifdef GFX
	if (gfx_mode != TEXT_MODE)
		exit_gfx();
#endif
	at_nwl= Yes;
/*	putperm(); */ /* shall be called via endall() */
	endall();
	immexit(ex);
}

extern bool in_vtrm;

Visible Procedure immexit(status) int status; {
	if (in_vtrm)
		endterm();
	exit(status);
}

Visible Procedure initerr() {
	still_ok= Yes; interrupted= No; curline= Vnil; curlino= zero;
#ifdef TTY_ERRFILE
	/* The idea of the following is, that we cannot use stderr
	 * for "abc cmd.file >out 2>err", since errors for READ
	 * commands must be visible for the user (who is entering
	 * them interactively, as reported in rd_interactive).
	 * The current solution is unix dependent; but stderr redirection
	 * seems impossible on non-unix anyway.
	 * When the first such system shows up it might be necessary
	 * to change all fprintf(errfile,...)'s to prerr's that print
	 * to the proper device (console or stderr file).
	 */
	if (rd_interactive && (errfile= fopen("/dev/tty", "w")) == NULL)
		errfile= stderr;
#endif
}

