/* Copyright (c) Stichting Mathematisch Centrum, Amsterdam, 1986. */

/* Sources: maintaining units and values on external files */

#include "b.h"
#include "bint.h"
#include "feat.h"
#include "bmem.h"
#include "bobj.h"
#include "bfil.h"
#include "i2par.h"
#include "i2nod.h"
#include "i3env.h"
#include "i3scr.h"
#include "i3in2.h"
#include "i3sou.h"

#ifdef TYPE_CHECK
value stc_code();
#endif
#ifdef unix
#define CK_WS_WRITABLE
#endif

Visible value b_perm= Vnil;
	/* The table that maps tags to their file names */
Visible value b_units= Vnil;
	/* The table that maps tags to their internal repr. */

#define Is_filed(v) (Is_indirect(v))

#define t_exists(name, aa)	(in_env(prmnv->tab, name, aa))

Visible Procedure def_target(name, t) value name, t; {
	e_replace(t, &prmnv->tab, name);
}

#define free_target(name)	(e_delete(&prmnv->tab, name))

/************************** UNITS ************************************/

#define Is_funprd(u)		(Is_function(u) || Is_predicate(u))
#define Is_predefined(u)	(Is_funprd(u) && Funprd(u)->pre != Use)

#define USR_ALL		'1'
#define USR_PARSED	'2'

Hidden Procedure freeunits(which) literal which; {
	intlet k, len;
	value vkey, vassoc;
	
	len= length(b_units);
	for (k= len-1; k >= 0; --k) {
		/* Reverse loop so deletions don't affect the numbering! */
		vkey= *key(b_units, k);
		vassoc= *assoc(b_units, k);
		switch (which) {
		case USR_ALL:
			if (!Is_predefined(vassoc)) free_unit(vkey);
			break;
		case USR_PARSED:
			if (!Is_predefined(vassoc) &&
					!How_to(vassoc)->unparsed)
				free_unit(vkey);
			break;
		}
	}
}

Visible Procedure rem_unit(u) parsetree u; {
	value pname= get_pname(u);
	free_unit(pname);
	release(pname);
}

/********************************************************************** */

Visible value permkey(name, type) value name; literal type; {
	char t[2];
	value v, w;
	
	if (!Valid(name))
		return Vnil;
	t[0]= type; t[1]= '\0';
	w= mk_text(t);
	v= concat(w, name); release(w);
	return v;
}

Visible string lastunitname() {
	value *aa;
	
	if (p_exists(last_unit, &aa))
		return sstrval(Permname(*aa));
	return NULL;
}

#define CANTGETFNAME	MESS(4000, "cannot create file name for %s")

Hidden value get_ufname(pname, silently) value pname; bool silently; {
	value fname;
	value *aa;
	
	if (p_exists(pname, &aa))
		fname= copy(*aa);
	else {
		value name= Permname(pname);
		literal type= Permtype(pname);
		
		fname= new_fname(name, type);
		if (Valid(fname))
			def_perm(pname, fname);
		else if (!silently)
			interrV(CANTGETFNAME, name);
		release(name);
	}
	return fname;
}

Hidden bool p_version(name, type, pname) value name, *pname; literal type; {
	value *aa;
	*pname= permkey(name, type);
	if (p_exists(*pname, &aa)) return Yes;
	release(*pname); *pname= Vnil;
	return No;
}

Hidden bool u_version(name, type, pname) value name, *pname; literal type; {
	value *aa;
	*pname= permkey(name, type);
	if (u_exists(*pname, &aa)) return Yes;
	release(*pname); *pname= Vnil;
	return No;
}

Hidden bool tar_version(name, pname) value name, *pname; {
	value *aa;
	if (p_version(name, Tar, pname))
		return Yes;
	else if (t_exists(name, &aa)) {
		*pname= permkey(name, Tar);
		return Yes;
	}
	else return No;
}

Hidden Procedure del_perm(pname) value pname; {
	value *aa;
	if (p_exists(pname, &aa)) {
		f_delete(*aa);
		idelpos(*aa);	/* delete file from positions file */
		free_perm(pname);
	}
}

/***********************************************************************/

Hidden bool is_loaded(pname, aa) value pname, **aa; {
	value u= Vnil, npname= Vnil, *a, get_unit();
	if (u_exists(pname, &a)) {
		if (Is_predefined(*a) && p_exists(pname, aa)) {
			/* loading userdefined over predefined */;
		}
		else {
			*aa= a; 
			return Yes; /* already loaded */
		}
	}
	else if (!p_exists(pname, aa)) {
		return No;
	}
	ifile= fopen(strval(**aa), "r");
	if (ifile == NULL) {
		vs_ifile();
		return No;
	}
	Eof= No;
	first_ilev();
	u= get_unit(&npname, Yes, No);
	if (still_ok) def_unit(npname, u);
	fclose(ifile);
	vs_ifile();
	Eof= No;
	if (still_ok && !u_exists(pname, aa)) {
		value name= Permname(pname);; 
		release(uname); uname= copy(pname);
		curline= How_to(u)->unit; curlino= one;
		interrV(MESS(4001, "filename and how-to name incompatible for %s"), name);
		release(name);
	}
	release(u); release(npname);
	return still_ok;
}

/* Does the unit exist without faults? */

Visible bool is_unit(name, type, aa) value name, **aa; literal type; {
	value pname;
	context c; bool is;
	sv_context(&c);
	cntxt= In_unit;
	pname= permkey(name, type);
	is= is_loaded(pname, aa);
	release(pname);
	set_context(&c);
	return is;
}

/***********************************************************************/

#define CANT_WRITE	MESS(4002, "cannot create file %s; need write permission in directory")

#define CANT_READ	MESS(4003, "unable to find file")

Hidden Procedure u_name_type(v, name, type) parsetree v; value *name;
		literal *type; {
	intlet adic;
	switch (Nodetype(v)) {
		case HOW_TO:	*type= Cmd; break;
		case YIELD:	adic= intval(*Branch(v, FPR_ADICITY));
				*type= adic==0 ? Zfd : adic==1 ? Mfd : Dfd;
				break;
		case TEST:	adic= intval(*Branch(v, FPR_ADICITY));
				*type= adic==0 ? Zpd : adic==1 ? Mpd : Dpd;
				break;
		default:	syserr(MESS(4004, "wrong nodetype of how-to"));
	}
	*name= copy(*Branch(v, UNIT_NAME));
}

Hidden value get_unit(pname, filed, editing) value *pname; bool filed, editing;
{
	value name; literal type;
	parsetree u= unit(No, editing);
	if (u == NilTree)
		return Vnil;
	u_name_type(u, &name, &type);
	*pname= permkey(name, type);
	release(name);
	switch (Nodetype(u)) {
		case HOW_TO:	return mk_how(u, filed);
		case YIELD:	return mk_fun(type, Use, u, filed);
		case TEST:	return mk_prd(type, Use, u, filed);
		default:	return Vnil; /* Keep lint happy */
	}
}

Visible value get_pname(v) parsetree v; {
	value pname, name; literal type;
	u_name_type(v, &name, &type);
	pname= permkey(name, type);
	release(name);
	return pname;
}

Hidden Procedure get_heading(h, pname) parsetree *h; value *pname; {
	*h= unit(Yes, No);
	*pname= still_ok ? get_pname(*h) : Vnil;
}

/********************************************************************** */

/* Check for certain types of name conflicts.
   The checks made are:
   - unit with the same name
   - function and predicate with the same name (and different or same
     adicity)
   - function or predicate with the same name as a target
   - zeroadic and monadic unit with the same name
   - zeroadic and dyadic unit with the same name.
*/

#define CR_EXIST	MESS(4005, "there is already a how-to with this name")

#define CR_TAR		MESS(4006, "there is already a permanent location with this name")

#define ED_EXIST	MESS(4007, "*** the how-to name is already in use;\n*** should the old how-to be discarded?\n*** (if not you have to change the how-to name)\n")

#define ED_TAR		MESS(4008, "*** the how-to name is already in use for a permanent location;\n*** should that location be deleted?\n*** (if not you have to change the how-to name)\n")

/* name_conflict() is called if a unit is created (HOW TO ? : command) */

Hidden bool name_conflict(pname) value pname; {
	value npname;
	if (smash(pname, &npname)) {
		interr(Permtype(npname) == Tar ? CR_TAR : CR_EXIST);
		if (Permtype(pname) != Tar)
			def_perm(last_unit, npname);
		release(npname);
		return Yes;
	}
	return No;
}

/* name_clash() is called if a unit is edited through the ':' command */

Hidden bool name_clash(pname) value pname; {
	value npname;
	
	if (!Valid(pname))
		return No;
	while (smash(pname, &npname)) {
		if (!do_discard(npname)) {
			release(npname);
			return Yes;
		}
		/* continue: there can be both a monadic and a	*/
		/* 	     dyadic version 			*/
		release(npname); npname= Vnil;
	}
	return No;
}

Hidden bool do_discard(pname) value pname; {
	bool istarg= Permtype(pname) == Tar;
	
	if (is_intended(istarg ? ED_TAR : ED_EXIST)) {
		if (istarg) {
			value name= Permname(pname);
			del_target(name);
			release(name);
		}
		else {
			free_unit(pname);
			del_perm(pname);
		}
		return Yes;
	}
	return No;
}

Hidden bool smash(pname, npname) value pname, *npname; {
	value name, *aa;
	literal u_type, v_type;
	bool sm;

	if (p_exists(pname, &aa)) {
		*npname= copy(pname);
		return Yes;
	}
	u_type= Permtype(pname);
	if (u_type == Cmd) {
		*npname= Vnil;
		return No;
	}
	name= Permname(pname);
	sm= p_version(name, Zfd, npname) ||
		p_version(name, Mfd, npname) ||
		p_version(name, Dfd, npname) ||
		p_version(name, Zpd, npname) ||
		p_version(name, Mpd, npname) ||
		p_version(name, Dpd, npname) ||
		tar_version(name, npname);
	release(name);
	if (!sm) {
		release(*npname); *npname= Vnil;
		return No;
	}
	v_type= Permtype(*npname);
	switch (u_type) {
		case Mfd: sm= v_type != Dfd; break;
		case Dfd: sm= v_type != Mfd; break;
		case Mpd: sm= v_type != Dpd; break;
		case Dpd: sm= v_type != Mpd; break;
		default: sm= Yes; break;
	}
	if (!sm) {
		release(*npname); *npname= Vnil;
		return No;
	}
	return Yes;
}

/***********************************************************************/

/* Create a unit via the editor or from the input stream. */

Visible Procedure create_unit() {
	value pname= Vnil; parsetree heading= NilTree;
	if (!interactive) {
		value v= get_unit(&pname, No, No);
		if (still_ok) def_unit(pname, v);
		release(v); release(pname);
		return;
	}
	get_heading(&heading, &pname);
	curline= heading; curlino= one; /* For all error messages */
	if (still_ok && !name_conflict(pname)) {
		value fname= get_ufname(pname, No);

		if (Valid(fname)) {
			FILE *fp= fopen(strval(fname), "w");
			if (fp == NULL)
				interrV(CANT_WRITE, fname);
			else {
				txptr tp= fcol();
				do { fputc(Char(tp), fp); }
				while (Char(tp++) != '\n');
				fputc('\n', fp);
				f_close(fp);
				ed_unit(&pname, &fname, Yes);
			}
		}
		release(fname);
	}
	release(pname); release(heading);
}


/***********************************************************************/

/* Edit a unit. The name of the unit is either given, or is defaulted
   to the last unit edited or the last unit that gave an error, whichever
   was most recent.
   It is possible for the user to mess things up with the w command, for
   instance, but this is not checked. It is allowed to rename the unit though,
   or delete it completely. If the file is empty, the unit is disposed of.
   Otherwise, the name and adicity are determined and if these have changed,
   the new unit is written out to a new file, and the original deleted.
   Thus the original is not saved.

   The function edit_unit parses the command line and does some
   high-level bookkeeping; ed_unit does the lower-level bookkeeping;
   f_edit is called to pass control to the editor and wait till it
   finishes its job.  Note that the editor reads the unit from the file
   and writes it back (if changed); there is no sharing of data
   structures such as parse trees in this version of the system.

   Renaming, deleting, or changing the adicity of a test or yield
   unfortunately requires all other units to be thrown away internally
   (by freeunits), since the unit parse trees may be wrong. For instance,
   consider the effect on the following of making a formerly monadic
   function f, into a zeroadic function:
	WRITE f root 2
*/

#define CANT_EDIT	MESS(4009, "I find nothing editible here")

Visible value last_unit= Vnil;

Visible Procedure edit_unit() {
	value name= Vnil, pname= Vnil; 
	value fname, *aa;
	value which_funprd();
	char *kw;

	if (Ceol(tx)) {
		if (!p_exists(last_unit, &aa))
			parerr(MESS(4010, "no current how-to"));
		else pname= copy(*aa);
	}
	else if (is_cmdname(ceol, &kw)) {
		name= mk_text(kw);
		pname= permkey(name, Cmd);
	}
	else if (is_tag(&name))
		pname= which_funprd(name);
	else
		parerr(CANT_EDIT);

	if (still_ok && ens_filed(pname, &fname)) {
		ed_unit(&pname, &fname, No);
		release(fname);
	}
	release(name); release(pname);
}

#define ED_MONDYA	MESS(4011, "*** do you want to visit the version with %c or %c operands?\n")
#define ONE_PAR '1'
#define TWO_PAR '2'

Hidden value which_funprd(name) value name; {
	/* There may be two units with the same name (functions
	   or predicates of different adicity).  Check if this
	   is the case, and if so, ask which one is meant.
	*/
	value pname, v= Vnil;
	char qans;
	
	if (p_version(name, Zfd, &pname) || p_version(name, Zpd, &pname))
		return pname;
	if (p_version(name, Mfd, &pname) || p_version(name, Mpd, &pname)) {
		if (p_version(name, Dfd, &v) || p_version(name, Dpd, &v)) {
			qans= q_answer(ED_MONDYA, ONE_PAR, TWO_PAR);
			if (qans == ONE_PAR) {
				release(v);
				return pname;
			}
			else if (qans == TWO_PAR) {
				release(pname);
				return copy(v);
			}
			else {
				/* interrupted */
				still_ok = No;
				return pname;
			}
		}
		else {
			release(v);
			return pname;
		}
	}
	if (p_version(name, Dfd, &pname))
		return pname;
	if (p_version(name, Dpd, &pname))
		return pname;

	/* be prepared to find at least one not-filed how-to;
	 * this does not find all of them;
	 * and it doesn't allow any conflicting with already existing ones.
	 */
	
	if (u_version(name, Zfd, &pname) ||
	    u_version(name, Mfd, &pname) ||
	    u_version(name, Dfd, &pname) ||
	    u_version(name, Zpd, &pname) ||
	    u_version(name, Mpd, &pname) ||
	    u_version(name, Dpd, &pname)
	)
		return pname;

	return permkey(name, Dpd);
	/* If it doesn't exist, ens_filed will complain. */
}
	
#define NO_U_WRITE	MESS(4012, "*** you have no write permission in this workspace:\n*** you may not change the how-to\n*** do you still want to display the how-to?\n")

/* Edit a unit.  Parameters are the prmnv key and the file name.
   This is called in response to the ':' command and when a new unit is
   created (the header of the new unit must already be written to the
   file).
   Side effects are many, e.g. on prmnv: the unit may be deleted or
   renamed.  When renamed, the original unit is lost.
   The unit is reparsed after editing.  A check is made for illegal
   name conflicts (e.g., a zeroadic and a monadic unit of the same
   name), and this is resolved by forcing the user to edit the unit
   again. In that case the edit is done on a temporary file.
   The new unit name is kept as the current unit name; when the unit is
   deleted the current unit name is set to Vnil. */

Hidden bool clash;

#define First_edit (!clash)

#ifdef TYPE_CHECK
Hidden value old_typecode= Vnil;
#define Sametypes(old, new) ((!Valid(old) && !Valid(new)) || \
		(Valid(old) && Valid(new) && compare(old, new) == 0))
#endif

Hidden Procedure ed_unit(pname, fname, creating) value *pname, *fname;
		bool creating;
{
#ifdef CK_WS_WRITABLE
	if (!wsp_writable() && !is_intended(NO_U_WRITE)) return;
#endif
#ifdef CLEAR_MEM
	clear_perm();
		/* To give the editor as much space as possible, remove
		   all parse trees and target values from memory.
		   (targets that have been modified are first written
		   out, of course).
		*/
#endif
	clash= No;
#ifdef TYPE_CHECK
	old_typecode= stc_code(*pname);
	if (!creating) del_types();
#endif
	do edunit(pname, fname, creating); while (clash);
#ifdef SAVE_PERM
	put_perm(b_perm);
#endif
#ifdef TYPE_CHECK
	release(old_typecode);
#endif
}

Hidden Procedure edunit(p_pname, p_fname, creating) value *p_pname, *p_fname;
		bool creating; {
	value pname= *p_pname, fname= *p_fname;
	value npname= Vnil, u;
	bool new_def, changed, samehead;
#ifdef TYPE_CHECK
	value new_typecode;
#endif

	release(uname); uname= copy(pname);
	changed= f_edit(fname, err_line(pname), ':', creating && First_edit)
		 || creating;
	errlino= 0;
	if (First_edit && !changed) {
		/* Remember it as current unit: */
		def_perm(last_unit, pname);
#ifdef TYPE_CHECK
		if (!creating) adjust_types(Yes);
#endif
		return;
	}
	if (!still_there(fname)) {
		free_original(pname);
#ifdef TYPE_CHECK
		if (!creating) adjust_types(No);
#endif
		idelpos(fname);	/* delete file from positions file */
		free_perm(last_unit);
		clash= No;
		return;
	}
	first_ilev();
	u= get_unit(&npname, Yes, Yes);
		/* the second Yes means the user may edit the heading;
		 * therefore no type check now in unit() */
	fclose(ifile); vs_ifile(); Eof= No;
	
	if (First_edit && same_heading(pname, npname, u)) {
		new_def= Yes;
		samehead= Yes;
	}
	else {
		samehead= No;
		free_original(pname);
		if (!name_clash(npname) && rnm_file(fname, npname))
			clash= No;
		else {
			/* edit again with npname and temp fname */
			release(*p_pname);
			*p_pname= copy(npname);
			if (First_edit) {
				value tfile= mk_text(temp1file);
				f_rename(fname, tfile);
				imovpos(fname, tfile);
				/* move position in positions file */
				release(*p_fname);
				*p_fname= tfile;
			}
			clash= Yes;
		}
		new_def= !clash;
	}
	if (new_def) {
		/* changed heading now def_perm()'ed, so now typecheck */
#ifdef TYPE_CHECK
		type_check((Is_funprd(u) ? Funprd(u)->unit : How_to(u)->unit));
		new_typecode= stc_code(npname);
		if (!creating)
			adjust_types(samehead &&
				     Sametypes(old_typecode, new_typecode));
		release(new_typecode);
#endif
		if (still_ok) def_unit(npname, u);
		else free_unit(npname);
		def_perm(last_unit, npname);
	}
	release(npname); release(u);
}

Hidden Procedure free_original(pname) value pname; {
	if (First_edit) {
		free_unit(pname); 
		free_perm(pname);
		freeunits(USR_PARSED);
	}
}

#define cmd_unit(pname)	(Permtype(pname) == Cmd)

Hidden bool same_heading(pname, npname, u_new) value pname, npname, u_new; {
	value *aa;
	
	if (!Valid(u_new) || !Valid(npname))
		return No;
	else if (compare(pname, npname) != 0)
		return No;
	else if (!cmd_unit(pname))
		return Yes;
	else if (!u_exists(pname, &aa))
		return Yes;
	else {
		parsetree old= How_to(*aa)->unit;
		parsetree new= How_to(u_new)->unit;
		parsetree old_kw, old_fml, old_next;
		parsetree new_kw, new_fml, new_next;
		
		old= *Branch(old, HOW_FORMALS);
		new= *Branch(new, HOW_FORMALS);
		do {
			old_kw= *Branch(old, FML_KEYW);
			old_fml= *Branch(old, FML_TAG);
			old_next= *Branch(old, FML_NEXT);
			new_kw= *Branch(new, FML_KEYW);
			new_fml= *Branch(new, FML_TAG);
			new_next= *Branch(new, FML_NEXT);
			
			if (compare(old_kw, new_kw) != 0)
				return No;
			else if (old_fml == NilTree && new_fml != NilTree)
				return No;
			else if (old_fml != NilTree && new_fml == NilTree)
				return No;
			else if (old_next == NilTree && new_next != NilTree)
				return No;
			else if (old_next != NilTree && new_next == NilTree)
				return No;
			old= old_next;
			new= new_next;
		}
		while (old != NilTree);
		return Yes;
	}
}

#define CANT_GET_FNAME	MESS(4013, "*** cannot create file name;\n*** you have to change the how-to name\n")

Hidden bool rnm_file(fname, pname) value fname, pname; {
	value nfname;
	
	nfname= (Valid(pname) ? get_ufname(pname, Yes) : Vnil);
	
	if (Valid(nfname)) {
		f_rename(fname, nfname);
		imovpos(fname, nfname); /* move position in positions file */
		release(nfname);
		return Yes;
	}
	else {
		putmess(errfile, CANT_GET_FNAME);
		return No;
	}
}

/* Find out if the file exists, and is not empty. Some editors don't
   allow a file to be edited to empty, but insist it should be at least
   one empty line.  Therefore, a file with one, empty, line is also
   considered empty.
   As a side effect, if the file is 'still there', ifile is set to it
   and it remains open, positioned at the beginning.
   (A previous version of this function would leave it positioned after
   an initial \n, if there was one; this version just rewinds the file.)
   */

Hidden bool still_there(fname) value fname; {
	int k;

	ifile= fopen(strval(fname), "r");
	if (ifile == NULL) {
		vs_ifile();
		return No;
	} else {
		if ((k= getc(ifile)) == EOF ||
				(k == '\n' && (k= getc(ifile)) == EOF)) {
			fclose(ifile);
			f_delete(fname);
			vs_ifile();
			return No;
		}
		rewind(ifile);
		return Yes;
	}
}

/* Ensure the unit is filed. If the unit was read non-interactively (eg passed
   as a parameter to abc), it is only held in store.
   Editing it puts it into a file. This is the safest way to copy a unit from
   one workspace to another.
*/

#define NO_HOWTO MESS(4014, "%s isn't a how-to in this workspace")

Hidden bool ens_filed(pname, fname) value pname, *fname; {
	value *aa;
	if (p_exists(pname, &aa)) {
		*fname= copy(*aa);
		return Yes;
	} else if (!u_exists(pname, &aa) || How_to(*aa)->unit == NilTree) {
		value name= Permname(pname);
		pprerrV(NO_HOWTO, name);
		release(name);
		return No;
	} else {
		how *du= How_to(*aa); FILE *fp;
		if (du->filed == Yes) {
			syserr(MESS(4015, "ens_filed()"));
			return No;
		}
		*fname= get_ufname(pname, No);
		if (!Valid(*fname))
			return No;
		fp= fopen(strval(*fname), "w");
		if (!fp) {
			interrV(CANT_WRITE, *fname);
			release(*fname);
			return No;
		} else {
			display(fp, du->unit, No);
			f_close(fp);
			du->filed= Yes;
			return Yes;
		}
	}
}

Hidden int err_line(pname) value pname; {
	value *aa;
	if (!p_exists(last_unit, &aa) || compare(*aa, pname) != 0)
		return 0;
	else
		return errlino;
}

/************************** VALUES ***************************************/
/* The permanent environment in the old format was kept as a single file */
/* but this caused slow start ups if the file was big.			 */
/* Thus the new version stores each permanent target on a separate file, */
/* that furthermore is only loaded on demand.				 */
/* To achieve this, a directory is kept of the permanent tags and their  */
/* file names. Care has to be taken that disaster occurring in		 */
/* the middle of an update of this directory does the least harm.	 */
/* Having the directory refer to a non-existent file is considered less  */
/* harmful than leaving a file around that can never be accessed, for	 */
/* instance, so a file is deleted before its directory entry,		 */
/* and so forth.							 */
/*************************************************************************/

Visible value errtname= Vnil;

Hidden Procedure tarfiled(name, v) value name, v; {
	value p= mk_indirect(v);
	def_target(name, p);
	release(p);
}

Visible value last_target= Vnil; /* last edited target */

Visible Procedure del_target(name) value name; {
	value pname= permkey(name, Tar);
	value *aa;
	free_target(name);
	del_perm(pname);
	if (p_exists(last_target, &aa) && (compare(name, *aa) == 0))
		free_perm(last_target);
	release(pname);
}

Hidden value get_tfname(name) value name; {
	value fname;
	value pname= permkey(name, Tar);
	value *aa;
	
	if (p_exists(pname, &aa))
		fname= copy(*aa);
	else {
		fname= new_fname(name, Tar);
		if (Valid(fname))
			def_perm(pname, fname);
		else
			interrV(CANTGETFNAME, name);
	}
	release(pname);
	return fname;
}

Visible Procedure edit_target() {
	value name= Vnil;
	value fname, *aa;
	if (Ceol(tx)) {
		if (!p_exists(last_target, &aa))
			parerr(MESS(4016, "no current location"));
		else
			name= copy(*aa);
	} else if (!is_tag(&name))
		parerr(CANT_EDIT);
	if (still_ok && ens_tfiled(name, &fname)) {
		ed_target(name, fname);
		release(fname);
	}
	release(name);
}

#define NO_T_WRITE	MESS(4017, "*** you have no write permission in this workspace:\n*** you may not change the location\n*** do you still want to display the location?\n")

/* Edit a target. The value in the target is written to the file,
   and then removed from the internal permanent environment so that
   if a syntax error occurs when reading the value back, the value is
   absent from the internal permanent environment.
   Thus when editing the file to correct the syntax error, the
   file doesn't get overwritten.
   The contents may be completely deleted in which case the target is
   deleted. */

Hidden Procedure ed_target(name, fname) value name, fname; {
	value v;

#ifdef CK_WS_WRITABLE
	if (!wsp_writable() && !is_intended(NO_T_WRITE)) return;
#endif
#ifdef CLEAR_MEM
	clear_perm(); /* To give the editor as much space as possible */
#endif
	def_perm(last_target, name);
	if (!f_edit(fname, 0, '=', No))
		/* File is unchanged */
		return;
	if (!still_there(fname)) {
		del_target(name);
#ifdef SAVE_PERM
		put_perm(b_perm);
#endif
		return;
	}
	fclose(ifile); /* Since still_there leaves it open */
	/* vs_ifile(); ? */
	v= getval(fname, In_edval);
	if (still_ok) def_target(name, v);
	release(v);
}

#define NO_TARGET MESS(4018, "%s isn't a location in this workspace")

Visible bool ens_tfiled(name, fname) value name, *fname; {
	value *aa;
	if (!t_exists(name, &aa)) {
		pprerrV(NO_TARGET, name);
		return No;
	} else {
		*fname= get_tfname(name);
		if (!Valid(*fname))
			return No;
		if (!Is_filed(*aa)) {
			release(errtname); errtname= copy(name);
			putval(*fname, *aa, No, In_tarval);
			tarfiled(name, *aa);
		}
		return Yes;
	}
}

/***************************** Values on files ****************************/

Visible value getval(fname, ct) value fname; literal ct; {
	char *buf; int k; parsetree w, code= NilTree; value v= Vnil;
	ifile= fopen(strval(fname), "r");
	if (ifile) {
		txptr fcol_save= first_col, tx_save= tx; context c;
		sv_context(&c);
		cntxt= ct;
		buf= (char *) getmem((unsigned)(f_size(ifile)+2)*sizeof(char));
		first_col= tx= ceol= buf;
		while ((k= getc(ifile)) != EOF)
			if (k != '\n') *ceol++= k;
		*ceol= '\n';
		fclose(ifile); vs_ifile();
		w= expr(ceol);
		if (still_ok) fix_nodes(&w, &code);
		curline= w; curlino= one;
		v= evalthread(code); 
		if (!env_ok(v)) {
			release(v);
			v= Vnil;
		}
		curline= Vnil;
		release(w);
		freemem((ptr) buf);
		set_context(&c);
		first_col= fcol_save; tx= tx_save;
	} else {
		interr(CANT_READ);
		vs_ifile();
	}
	return v;
}

Hidden bool env_ok(v) value v; {
	if (cntxt == In_prmnv || cntxt == In_wsgroup) {
		if (!Is_table(v)) {
			interr(MESS(4019, "value is not a table"));
			return No;
		}
		else if (!Is_ELT(v) && !Is_text(*key(v, 0))) {
			interr(MESS(4020, "in t[k], k is not a text"));
			return No;
		}
	}
	return Yes;
}

Visible bool permchanges;

Visible Procedure initperm() {
	if (F_exists(permfile)) {
		value fn, name;
		intlet k, len;
		value v, pname;
		
		fn= mk_text(permfile);
		v= getval(fn, In_prmnv);
		release(fn);
		if (Valid(v)) {
			release(b_perm);
			b_perm= v;
		}
		len= length(b_perm);
		for (k= 0; k < len; k++) {
			pname= *key(b_perm, k);
			if (Permtype(pname) == Tar) {
				name= Permname(pname);
				tarfiled(name, Vnil);
				release(name);
			}
		}
	}
	permchanges= No;
}

Visible Procedure putval(fname, v, silently, ct) value fname, v;
		bool silently; literal ct; {
	value fn= copy(fname);
	FILE *fp;
	bool was_ok= still_ok;
	context c;

	sv_context(&c);
	cntxt= ct;
	curline= Vnil;
	curlino= one;
#ifdef unix
	release(fn); fn= mk_text(tempfile);
#endif
	fp= fopen(strval(fn), "w");
	if (fp != NULL) {
		redirect(fp);
		still_ok= Yes;
		wri(v, No, No, Yes); newline();
		f_close(fp);
		redirect(stdout);
#ifdef unix
		if (still_ok) f_rename(fn, fname);
#endif
	}
	else if (!silently) interrV(CANT_WRITE, fn);
	still_ok= was_ok;
	release(fn);
	set_context(&c);
}

Visible Procedure endperm() {
	static bool active;
	bool was_ok= still_ok;
	
	if (active)
		return;
	active= Yes;
	still_ok= Yes;
	put_targs();
	put_perm(b_perm);
	still_ok= was_ok;
	active= No;
}

Hidden Procedure put_targs() {
	int k, len;
	value v, name;
	
	len= Valid(prmnv->tab) ? length(prmnv->tab) : 0;
	for (k= 0; k < len; k++) {
		v= copy(*assoc(prmnv->tab, k));
		name= copy(*key(prmnv->tab, k));
		if (!Is_filed(v)) {
			value fname= get_tfname(name);
			if (Valid(fname)) {
				release(errtname); errtname= copy(name);
				putval(fname, v, Yes, In_tarval);
			}
			release(fname);
		}
		tarfiled(name, Vnil);
		release(v); release(name);
	}
}

Visible Procedure put_perm(v) value v; {
	value fn;
	intlet len;
	
	if (!permchanges || !Valid(v))
		return;
	fn= mk_text(permfile);
	/* Remove the file if the permanent environment is empty */
	len= length(v);
	if (len == 0)
		f_delete(fn);
	else
		putval(fn, v, Yes, In_prmnv);
	release(fn);
	permchanges= No;
}

Visible Procedure clear_perm() {
	freeunits(USR_ALL);
	endperm();
}

Visible Procedure initsou() {
	release(b_units); b_units= mk_elt();
	release(last_unit); last_unit= mk_text(":");
	release(last_target); last_target= mk_text("=");
	release(b_perm); b_perm= mk_elt();
}

Visible Procedure endsou() {
	if (terminated)
		return;	/* hack; to prevent seemingly endless QUIT */
	release(b_units); b_units= Vnil;
	release(b_perm); b_perm= Vnil;
	release(last_unit); last_unit= Vnil;
	release(last_target); last_target= Vnil;
}

/*
 * lst_uhds() displays the first line of the unit without a possible
 * present simple command
 */
 
#define MORE MESS(4021, "Press [SPACE] for more, [RETURN] to exit list")
extern int winheight;
bool ask_for();

Visible Procedure lst_uhds() {
	intlet k, len;
	value pname, *aa;
	how *u;
	int nprinted= 0;
	bool more= Yes;
	
	len= length(b_perm);
	for (k= 0; k<len && still_ok && more; ++k) {
		pname= *key(b_perm, k);
		if (!Is_text(pname) || Permtype(pname) == Tar) 
			continue;
		/* reduce disk access: */
		if (u_exists(pname, &aa) && !Is_predefined(*aa))
			display(stdout, How_to(*aa)->unit, Yes);
		else
			lst_fileheading(*assoc(b_perm, k));
		fflush(stdout);
		if (++nprinted >= winheight) {
			more= ask_for(MORE);
			nprinted= 0;
		}
	}
	/* not interactive units */
	len= length(b_units);
	for (k= 0; k<len && still_ok && more; ++k) {
		u= How_to(*assoc(b_units, k));
		if (u -> filed == No && !p_exists(*key(b_units, k), &aa)) {
			display(stdout, u -> unit, Yes);
			fflush(stdout);
			if (++nprinted >= winheight) {
				more= ask_for(MORE);
				nprinted= 0;
			}
		}

	}
}

Hidden Procedure lst_fileheading(v) value v; {
	FILE *fn;
	char *line;
	char *pcolon, *pc;

	if (!Is_text(v))
		return;
	fn= fopen(strval(v), "r");
	if (!fn)
		return;
	if ((line= f_getline(fn)) != NULL) {
		pcolon= strchr(line, C_COLON);
		if (pcolon != NULL) {
			pc= ++pcolon;
			while (Space(*pc)) ++pc;
			if (*pc != C_COMMENT && *pc != '\n') {
				/* single command after colon;
				 * don't show it.
				 */
				*(pcolon+1)= '\n';
				*(pcolon+2)= '\0';
			}
		}
		putstr(stdout, line);
		freestr(line);
	}
	fclose(fn);
}
