//
// Copyright (C) 1991 Texas Instruments Incorporated.
//
// Permission is granted to any individual or institution to use, copy, modify,
// and distribute this software, provided that this complete copyright and
// permission notice is maintained, intact, in all copies and supporting
// documentation.
//
// Texas Instruments Incorporated provides this software "as is" without
// express or implied warranty.
//
// Created: MBN 07/11/89 -- Initial design
// Updated: MBN 08/10/89 -- Initial implementation
// Updated: MBN 08/11/89 -- Inherit from Generic
// Updated: MBN 09/19/89 -- Added conditional exception handling
// Updated: MBN 12/21/89 -- Added optional argument to set_compare method
// Updated: MBN 02/27/90 -- Added operator= for pointer argument
// Updated: MJF 03/12/90 -- Added group names to RAISE
//
// The D_Node<Type,nchild> class implements parameterized  nodes of  a  dynamic
// size for N-ary  trees.   This node class  is parameterized for  the type and
// some initial  "N",  the   number  of  subtrees  each  node may   have.   The
// D_Node<Type,nchild>  class  is dynamic   in  the  sense that the   number of
// subtrees allowed for each  node is not  fixed.  D_Node<Type,nchild> utilizes
// the Vector<Type>  class,  supporting runtime growth charactersitics.    As a
// result, the D_Node<Type,nchild> class  should be  used as the node  type for
// the N_Tree<Node,Type,nchild> class when  the number of subtrees is variable,
// unknown at  compile time, or   needs to  increase on   a per-node  basis  at
// runtime.  This capability is suited for heirarchical trees such as  might be
// used  in an   organization  chart.   In  addition,   specialization  of  the
// N_Tree<Node,Type,nchild>  class  would  allow   for  the   relatively   easy
// implementation of a Diagram class.
//
// There are three public constructors for the  D_Node class to allow  the user
// to  create nodes and  control the building and   structure of an  N-ary tree
// where the  ordering   can  have a  specific  meaning.   The  first takes  no
// arguments and initializes the pointer  and data  slots to NULL.   The second
// takes an argument of type Type and initializes  the data slot to that value.
// The third takes a reference to another D_Node object and duplicates the data
// value, but not its children.
//
// The private  data section  contains just two  slots.  The  first is a Vector
// object of some initial size "N"  that  contains pointers to  D_Node objects,
// one for each  child.  The second  is a data slot  of type Type  to hold  the
// value of the data item.
//
// Methods are provided to set and get the node data value, determine if a node
// is a leaf or the root of some subtree,  and implement member-wise assignment
// from one  D_Node to another via  the overloaded operator=. In  addition, the
// brackets operator is overloaded to  provided  a mechanism to efficiently get
// and set individual  subtree pointers in  a specific node,  thus allowing the
// user to   control  the  orgranization  and  structure   of a tree.  Finally,
// insertion, and removal  methods to allow the  user to control  placement and
// ordering of sub-trees of a node are available.
//

#ifndef D_NODEH					// If no definition for class
#define D_NODEH

#ifndef VECTORH					// If no Vector class defined
#include <cool/Vector.h>
#endif

template <class Type, int nchild> D_Node {
  class D_Node<Type,nchild>;			// Forward reference class
  class N_Tree<D_Node,Type,nchild>;		// Forward reference class
  typedef Boolean (*Type ## _D_Node_Compare)(const Type&, const Type&);
  typedef D_Node<Type,nchild>* D_Node_##Type##_##nchild##_p; // Pointer to class
  DECLARE Vector<D_Node<Type,nchild>*>;		// Vector of pointers to D_Node
}
 
template <class Type, int nchild>
class D_Node<Type,nchild> : public Generic {
private:
  Vector<D_Node<Type,nchild>*> sub_trees;	// Vector of subtree pointers
  Type data;					// Slot to hold data value
  static Type ## _D_Node_Compare compare_s;	// Compare function for class

  void index_error (int);			// Raise exception
  void before_error (int);			// Raise exception
  void after_error (int);			// Raise exception

  friend class N_Tree<D_Node,Type,nchild>;	// Friend class to access data
  friend int Type ## _default_D_Node_compare (const Type&, const Type&);

public:
  D_Node<Type,nchild> ();			// Simple constructor
  D_Node<Type,nchild> (const Type& value);	// Constructor with data value
  D_Node<Type,nchild> (D_Node<Type,nchild>&);	// Constructor with reference

  ~D_Node<Type,nchild> ();			// Destructor
  inline void set (const Type&);		// Set node data to value
  inline Type& get () CONST;			// Get node data value
  Boolean is_leaf () CONST;			// TRUE if node has no children
  inline D_Node_##Type##_##nchild##_p& operator[] (int); // Set/Get pointers
  inline int num_subtrees() CONST;		 // number of subtree slots in node
  
  inline void set_compare (Type ## _D_Node_Compare = NULL); // Set compare
  inline Boolean operator== (const Type&) CONST;     // Overload operator==
  inline Boolean operator!= (const Type&) CONST;     // Overload operator!=
  inline Boolean operator< (const Type&) CONST;	     // Overload operator<
  inline Boolean operator> (const Type&) CONST;	     // Overload operator>
  inline Boolean operator<= (const Type&) CONST;     // Overload operator<=
  inline Boolean operator>= (const Type&) CONST;     // Overload operator>=

  D_Node<Type,nchild>& operator= (const D_Node<Type,nchild>&); // Assignment
  inline D_Node<Type,nchild>& operator= (const D_Node<Type,nchild>*);
  Boolean insert_before (const D_Node<Type,nchild>&, int);     // Insert before 
  Boolean insert_after (const D_Node<Type,nchild>&, int);      // Insert after
};

template <class Type, int nchild> D_Node {
  IMPLEMENT Vector<D_Node<Type,nchild>*>;	// Vector of pointers to D_Node
}


// default_D_Node_compare -- Default node comparison function utilizing builtin
//                           less than, equal, and greater than operators
// Input:                    Reference to two Type data values
// Output:                   -1, 0, or 1 if less than, equal to, or greater than

template <class Type, int nchild> D_Node {
  int Type##_default_D_Node_compare (const Type& v1, const Type& v2) {
    if (v1 == v2)				// If data items equal
      return 0;					// Return zero
    if (v1 < v2)				// If this less than data
      return -1;				// Return negative one
    return 1;					// Else return positive one
  }
}


// D_Node -- Simple constructor that allocates enough storage for a vector of
//           pointers to D_Node objects
// Input:    None
// Output:   None

template <class Type, int nchild> 
D_Node<Type,nchild>::D_Node<Type,nchild> () {
  // fix growth ratio instead
  sub_trees.set_alloc_size(nchild);
  for (int i = 0; i < nchild; i++)		// For each pointer in vector
    this->sub_trees.push(NULL);			// Insure NULL pointer value
  if (this->compare_s == NULL)			// If no compare function
    this->compare_s = &Type##_default_D_Node_compare; // Default
}


// D_Node -- Simple constructor that allocates enough storage for a vector of
//           pointers to D_Node objects and assigns an initial data value
// Input:    Data slot value
// Output:   None

template <class Type, int nchild> 
D_Node<Type,nchild>::D_Node<Type,nchild> (const Type& value) {
  this->data = value;				// Copy initial data value
  for (int i = 0; i < nchild; i++)		// For each pointer in vector
    this->sub_trees.push(NULL);			// Insure NULL pointer value
  if (this->compare_s == NULL)			// If no compare function
    this->compare_s = &Type##_default_D_Node_compare; // Default
}


// D_Node -- Constructor that duplicates the values of another D_Node object
// Input:    Reference to D_Node
// Output:   None

template <class Type, int nchild> 
D_Node<Type,nchild>::D_Node<Type,nchild> (D_Node<Type,nchild>& n) {
  this->data = n.data;				// Copy initial data value
  for (int i = 0; i < nchild; i++)		// For each pointer in vector
    this->sub_trees.push(n.sub_trees[i]);	// Copy value
  this->compare_s = n.compare_s;		// Set compare method
}


// ~D_Node -- Destructor for the D_Node<Type,nchild> class
// Input:     None
// Output:    None

template <class Type, int nchild> 
D_Node<Type,nchild>::~D_Node<Type,nchild> () {
  for (int i = 0; i < nchild; i++) {		// For each pointer in vector
    // delete (this->sub_trees[i]);     C++2.0 couldn't handle this!!!!!
    D_Node<Type,nchild>* tmp = (this->sub_trees[i]);
    delete tmp; 		// Invoke destructor
  }
}

// num_subtrees -- Returns number of slots available for subtrees in node
// Input:          None
// Output:         int length of the Vector allocated  for Subtrees.

template <class Type, int nchild>
inline int D_Node<Type,nchild>::num_subtrees () CONST {
  return sub_trees.length();
}


// set -- Set value of data slot in node
// Input: Reference to data slot value
// Output: None

template <class Type, int nchild> 
inline void D_Node<Type,nchild>::set (const Type& value) {
  this->data = value;				// Set data slot value
}


// get -- Get value of data slot in node
// Input: None
// Output: Reference to data slot value

template <class Type, int nchild> 
inline Type& D_Node<Type,nchild>::get () CONST {
  return (this->data);				// Return data slot value
}


// set_compare -- Specify the comparison function to be used in logical tests
//                of node data values
// Input:         Pointer to a compare function
// Output:        None

template <class Type, int nchild> 
inline void D_Node<Type,nchild>::set_compare (Type ## _D_Node_Compare c) {
  if (c == NULL)
    this->compare_s = &Type##_default_D_Node_compare; // Default equality
  else
    this->compare_s = c;			// Else use one provided
}


// operator== -- Overload the equality operator to use the compare function
// Input:        Constant reference to Type value
// Output:       TRUE/FALSE

template <class Type, int nchild> 
inline Boolean D_Node<Type,nchild>::operator== (const Type& value) CONST {
  return ((((*this->compare_s)(this->data,value)) == 0) ? TRUE : FALSE);
}


// operator!= -- Overload the inequality operator to use the compare function
// Input:        Constant reference to Type value
// Output:       TRUE/FALSE

template <class Type, int nchild> 
inline Boolean D_Node<Type,nchild>::operator!= (const Type& value) CONST {
  return ((((*this->compare_s)(this->data,value)) == 0) ? FALSE : TRUE);
}


// operator< -- Overload the less than operator to use the compare function
// Input:       Constant reference to Type value
// Output:      TRUE/FALSE

template <class Type, int nchild> 
inline Boolean D_Node<Type,nchild>::operator< (const Type& value) CONST {
  return ((((*this->compare_s)(this->data,value)) < 0) ? TRUE : FALSE);
}


// operator> -- Overload the greater than operator to use the compare function
// Input:       Constant reference to Type value
// Output:      TRUE/FALSE

template <class Type, int nchild> 
inline Boolean D_Node<Type,nchild>::operator> (const Type& value) CONST {
  return ((((*this->compare_s)(this->data,value)) > 0) ? TRUE : FALSE);
}


// operator<= -- Overload the less than or equal operator to use the compare
//               function
// Input:        Constant reference to Type value
// Output:       TRUE/FALSE

template <class Type, int nchild> 
inline Boolean D_Node<Type,nchild>::operator<= (const Type& value) CONST {
  return ((((*this->compare_s)(this->data,value)) > 0) ? FALSE : TRUE);
}


// operator>= -- Overload the greater than or equal operator to use the compare
//               function
// Input:        Constant reference to Type value
// Output:       TRUE/FALSE

template <class Type, int nchild> 
inline Boolean D_Node<Type,nchild>::operator>= (const Type& value) CONST {
  return ((((*this->compare_s)(this->data,value)) < 0) ? FALSE : TRUE);
}


// is_leaf -- Determine if node has any children
// Input:     None
// Output:    TRUE if no children, else FALSE

template <class Type, int nchild> 
Boolean D_Node<Type,nchild>::is_leaf () CONST {
  for (int i = 0; i < nchild; i++)
    if (this->sub_trees[i])
      return (FALSE);
  return TRUE;
}


// operator[] -- Overload the brackets operator to provide a mechanism to set
//               and/or get a sub-tree pointer of a node whose zero-relative
//               index is specified from left to right
// Input:        Zero-relative index into vector of sub-tree pointers
// Output:       Reference to a pointer value

template <class Type, int nchild> 
inline D_Node_##Type##_##nchild##_p& D_Node<Type,nchild>::operator[](int index) {
#if ERROR_CHECKING
  if (index >= nchild)				// If index out of range
    this->index_error (index);			// Raise exception
#endif
  return (this->sub_trees[index]);
}


// operator= -- Overload the assignment operator to copy all values from one
//              node object to another. This routine could potentially result
//              in a complete deep copy, since for each valid sub_tree pointer,
//              a new node is allocated and its sub_tree pointers copied.
// Input:       Reference to D_Node
// Output:      Rererence to updated D_Node

template <class Type, int nchild> 
D_Node<Type,nchild>& D_Node<Type,nchild>::operator= (const D_Node<Type,nchild>& n) {
  for (int i = 0; i < nchild; i++) {		// For each pointer in vector
    D_Node<Type,nchild>* temp;			// Temporary pointer
    if (n.sub_trees[i]) {			// If valid sub_tree pointer
      temp = new D_Node<Type,nchild> (*(n.sub_trees[i])); // Duplicate node
      this->sub_trees[i] = temp;		// Point to subtree
    }
    else
      this->sub_trees[i] = NULL;		// Else NULL pointer
  }
  this->set(n.get());				// Copy data value
  return *this;					// Return reference
}


// operator= -- Overload the assignment operator to copy all values from one
//              node object to another. This routine could potentially result
//              in a complete deep copy, since for each valid sub_tree pointer,
//              a new node is allocated and its sub_tree pointers copied.
// Input:       Pointer to  D_Node
// Output:      Rererence to updated D_Node

template <class Type, int nchild> 
inline D_Node<Type,nchild>& D_Node<Type,nchild>::operator= (const D_Node<Type,nchild>* n) {
  return this->operator=(*n);
}


// insert_before -- Insert sub-tree pointer to child before the specified
//                  zero-relative sub-tree index (numbered from left to right)
// Input:           Pointer to child node, zero-relative index
// Output:          TRUE/FALSE

template <class Type, int nchild> 
Boolean D_Node<Type,nchild>::insert_before (const D_Node<Type,nchild>& n, int index) {
#if ERROR_CHECKING
  if (index < 0) {				// If index out of range
    this->before_error (index);			// Raise exception
    return FALSE;				// Return failure status
  }
#endif
  this->sub_trees.insert_before(&n, index);	// Pointer to new sub-tree
  return TRUE;					// Return success status
}


// insert_after -- Insert sub-tree pointer to child after the specified
//                 zero-relative sub-tree index (numbered from left to right)
// Input:          Pointer to child node, zero-relative index
// Output:         TRUE/FALSE

template <class Type, int nchild> 
Boolean D_Node<Type,nchild>::insert_after (const D_Node<Type,nchild>& n, int index) {
#if ERROR_CHECKING
  if (index < 0) {				// If index out of range
    this->after_error (index);			// Raise exception
    return FALSE;				// Return failure status
  }
#endif
  this->sub_trees.insert_after(&n, index);	// Pointer to new sub-tree
  return TRUE;					// Return success status
}


// index_error -- Raise exception for D_Node::operator[]
// Input:         Invalid index
// Output:        None

template <class Type, int nchild> 
void D_Node<Type,nchild>::index_error (int n) {
  RAISE (Error, SYM(D_Node), SYM(Out_Of_Range),
	 "D_Node<%s,%d>::operator[]: Index %d out of range", #Type,
	 nchild, n);
}


// before_error -- Raise exception for D_Node::insert_before()
// Input:          Invalid index
// Output:         None

template <class Type, int nchild> 
void D_Node<Type,nchild>::before_error (int n) {
  RAISE (Error, SYM(D_Node), SYM(Out_Of_Range),
	 "D_Node<%s,%d>::insert_before(): Index %d out of range", #Type,
	 nchild, n);
}


// after_error -- Raise exception for D_Node::insert_after()
// Input:         Invalid index
// Output:        None

template <class Type, int nchild> 
void D_Node<Type,nchild>::after_error (int n) {
  RAISE (Error, SYM(D_Node), SYM(Out_Of_Range),
	 "D_Node<%s,%d>::insert_after: Invalid index %d", #Type,
	 nchild, n);
}

#endif						// End D_NODEH #if
