// This may look like C code, but it is really -*- C++ -*-
// 
// Copyright (C) 1988 University of Illinois, Urbana, Illinois
// Copyright (C) 1989 University of Colorado, Boulder, Colorado
// Copyright (C) 1990 University of Colorado, Boulder, Colorado
//
// written by Dirk Grunwald (grunwald@foobar.colorado.edu)
//
#include "HardwareContext.h"
#include "HardwareContextP.h"
#include "CpuMultiplexor.h"
#include "Thread.h"
#include "assert.h"
#include <stream.h>

#if	defined(sparc)
extern "C" {
#include <sun4/asm_linkage.h>
#include <sun4/trap.h>
}
#endif

//
//	Many machines have the same (or very similar) stack format.
//	The 68K & 32K are examples of such machines.
//
//	The const *registers* defines the number of additional longwords
//	needed on the stack past the last frame pointer. This value needs
//	to jibe with the one in HardwareContext-<arch>.s.
//

const long MagicStackMarker = 0x464f4f20;	// this says 'FOO '

static int expectedBytesSavedInContext = 112;
static int numberOfBytesSavedInContext = 0;
static int numberOfQuadsSavedInContext = 0;

HardwareContext::HardwareContext (int check, unsigned size)
{
    checkStackLimits = check;

    stackBase = 0;
    stackEnd = 0;
    stackMax = 0;
    stackSize = 0;
    stackCheck = 0;
    stackMallocAt = 0;

    if (size > 0) {
	stackSize = size;
	stackMallocAt = new void *[ stackSize ];
	//
	// stackBase should point to the first writeable cell of the
	// new stack.
	//
	stackEnd = stackMallocAt;
	stackBase = &stackMallocAt[stackSize-1];
    }
    else {
	//
	// Have the main process figure out how many registers
	// (actually, Quads) are pushed when we save an entire context.
	// Do this by calling magicSwitchTo with itself. This has
	// the side-effect of storing fp & sp in the context.
	//
    }
}

HardwareContext::reclaimStack()
{
    if ( stackMallocAt ) {
	if ( check ) {
	    mprotect(stackMallocAt, pageSizeInBytes, PROT_READ | PROT_WRITE);
	}
	free(stackMallocAt);
	stackMallocAt = 0;
    }
}

void
HardwareContext::magicSwitchTo(HardwareContext *to)
{
    cerr << "switch from " << hex(long(this));
    cerr << " to " << hex(long(to)) << "\n";
    if ( setjmp(context) == 0 ) {
	longjmp(to -> context,1);
    }
}

void
HardwareContext::stackOverflow()
{
    register unsigned depth = stackBase - getSp();
    if (stackMax < depth) {
	stackMax = depth;
    }
    if ( stackMax >= stackSize ) {
	cerr << "\nStack overflow\n";
	cerr << " getSp() = " << hex(long(getSp()));
	cerr << " and stackBase = " << hex(long(stackBase)) << "\n";
	cerr << *this << "\n";
	cerr << "Current task is \n";
	cerr << *(CurrentThread());
	cerr << "\n";
	cerr.flush();
    }
    assert( stackMax < stackSize );
    assert( *stackCheck == MagicStackMarker );
}

void
HardwareContext::buildHiddenReturnFrame(void *fp, void *sp)
{
    if (setjmp(context) == 0) {
	cerr << "buildHiddenReturnFrame returns to temp for HC ";
	cerr << hex(long(this)) << "\n";
	cerr << " ..and should eventually jump to thread ...";
	cerr << hex(long(contextOf)) << "\n";
	Thread *who = (Thread *) contextOf;
	cerr << "Thread name is " << who -> name();

#if	defined(sparc)
	asm("ta	%0" : : "i" ta	ST_FLUSH_WINDOWS		! Flush all other active windows
	asm("ld	%0,%%sp" : : "r" (sp));
	asm("ld	%0,%%fp" : : "r" (fp));
#endif

    }
    else {
	cerr << "buildHiddenReturnFrame calls startoff HC ";
	cerr << hex(long(this)) << "\n";
	cerr << " ..and jumps to thread ...";
	cerr << hex(long(contextOf)) << "\n";

	Thread *who = (Thread *) contextOf;
	cerr << "Thread name is " << who -> name();

	contextOf -> startOff();
	exit(1);
    }
}

void
HardwareContext::buildReturnFrame(void *returnThis, voidFuncP returnAddress)
{
    void *sp;
    void *fp;

    contextOf = (Thread *) returnThis;

    cerr << "buildReturnFrame calls setjmp for HC " << hex(long(this));
    cerr << " w/thread " << hex(long(returnThis)) << "\n";

    Thread *who = (Thread *) returnThis;
    cerr << "Thread name is " << who -> name();

    if ( setjmp(temporary) == 0) {
#if	defined(sparc)
	asm("ta	%0" : : "I" (ST_FLUSH_WINDOWS));
	asm("st	%%sp,%0" : "=g" (sp));
	asm("st	%%fp,%0" : "=g" (fp));
	asm("ld	%0,%%fp" : : "g" (stackBase));
	void *foo = stackBase - MINFRAME;
	asm("ld	%0,%%sp" : : "g" (foo));
#endif
	buildHiddenReturnFrame(fp,sp);
	contextOf = (Thread *) returnThis;
#if 0
#if defined(sparc)
	context[2] = stackBase;
#else
	fix this
#endif
#endif
    }
    cerr << "buildReturnFrame exits\n";
}

void
HardwareContext::classPrintOn(ostream& s)
{
    s << "[HardwareContext] Stack spans " << hex(long(stackEnd));
    s << " to " << hex(long(stackBase));
    s << " used is  " << (stackMax) << " of " << stackSize << "\n";
    s << "[HardwareContext] fp = " << hex(long(fp));
    s << " sp = " << hex(long(sp));
    long p = *( (long *) fp );
    s << " @fp = " << hex(p) << "\n";
}

static
void PrintSingleStackFrame()
{
}
